#!/p/perl/bin/perl -w

# snake.pl was written by John H. Aughey (jha@aughey.com)
# Distribute freely

use strict;
use Math::Spline;
no strict 'refs'; # to allow us to pull variables out using symbolic references
use OpenGL;
use Tk;
use Tk::NoteBook;
use Tk::VStack;

my $dirty=1;
my $waittime=100;
my $looping=0;
my $init=0;

my $PI=3.14159265358979323846;

my $curframe=0;
my @framelist;
my $lastframe=0;

# Set the coloring for each side
# The sides which aren't seen color black
my @side1=(0,0,0);
my @side2=(0,0,0);
# Color the two triangle sides;
my @side3=(0,0,1);
my @side4=(0,0,1);
# And the square top
my @side5=(1,0,0);

my($cameraxrot,$camerayrot,$camerazrot)=(20,270,0);
my($cameraxlookat,$cameraylookat,$camerazlookat)=(0,0,0);
my $viewangle=20;
my ($nearclip,$farclip)=(.1,100);
my($snakexrot,$snakeyrot,$snakezrot)=(0,0,0);
my($snakexpos,$snakeypos,$snakezpos)=(0,.4,0);
my($distance)=(40);
my $linkgap=0.00;
#my $incrstep=5;
my $incrstep=30;
my @linkrotation;
# initialize the link rotation
my $i;
for($i=0;$i<23;$i++) {
    $linkrotation[$i]=0;
}

parsefile($ARGV[0]) if(defined $ARGV[0]);
rewind();

my $main=new MainWindow();
$main->title("Snake Animator");
my $notebook=$main->NoteBook->pack(-side => 'right');
my $camerapage=$notebook->add("camera",-label => "Camera");
my $lenspage=$notebook->add("lens",-label => "Lens");
my $snakepage=$notebook->add("snake",-label => "Snake");
my $povpage=$notebook->add("pov",-label => "Povray Controls");

my $resolution=0.01;

# Camera Page
my $rightframe=$camerapage->Frame()->pack();
my $frame;
$frame=makeslider($rightframe,0,360,$resolution,"Camera X Rotation",\$cameraxrot,\&dirty);
$frame->pack(-side => 'top');
$frame=makeslider($rightframe,0,360,$resolution,"Camera Y Rotation",\$camerayrot,\&dirty);
$frame->pack(-side => 'top');
$frame=makeslider($rightframe,0,360,$resolution,"Camera Z Rotation",\$camerazrot,\&dirty);
$frame->pack(-side => 'top');
$frame=makeslider($rightframe,-10,10,$resolution,"Camera X Lookat",\$cameraxlookat,\&dirty);
$frame->pack(-side => 'top');
$frame=makeslider($rightframe,-10,10,$resolution,"Camera Y Lookat",\$cameraylookat,\&dirty);
$frame->pack(-side => 'top');
$frame=makeslider($rightframe,-10,10,$resolution,"Camera Z Lookat",\$camerazlookat,\&dirty);
$frame->pack(-side => 'top');
$frame=makeslider($rightframe,0,50,$resolution,"Distance",\$distance,\&dirty);
$frame->pack(-side => 'top');
my $continous=0;

# Pov Page
$rightframe=$povpage->Frame()->pack();
$rightframe->Button(-text => 'Write Scene',
		    -command => \&writescene)->pack(-side => 'top');
my $rendersize=1;
my @rendersizex=(100,320,500,800);
my @rendersizey=(100,240,500,600);
my $newframenum=0;
$frame=makeslider($rightframe,0,$lastframe,1,"Current Frame",\$newframenum,\&newframe);
$frame->pack();
$rightframe->Checkbutton(-text => 'Continuous',
			 -variable => \$continous,
			 -command => \&docontinous)->pack(-side => 'top');
$frame->pack();
$rightframe->Radiobutton(-text => 'Thumbnail',
			 -value => 0,
			 -variable => \$rendersize,
			 -command => \&setviewport)->pack(-side => 'top');
$rightframe->Radiobutton(-text => '320x240',
			 -value => 1,
			 -variable => \$rendersize,
			 -command => \&setviewport)->pack(-side => 'top');
$rightframe->Radiobutton(-text => '500x500',
			 -value => 2,
			 -variable => \$rendersize,
			 -command => \&setviewport)->pack(-side => 'top');
$rightframe->Radiobutton(-text => '800x600',
			 -value => 3,
			 -variable => \$rendersize,
			 -command => \&setviewport)->pack(-side => 'top');
my $hiquality=0;
$rightframe->Radiobutton(-text => 'Low-Quality',
			 -value => 0,
			 -variable => \$hiquality)->pack(-side => 'top');
$rightframe->Radiobutton(-text => 'Hi-Quality',
			 -value => 1,
			 -variable => \$hiquality)->pack(-side => 'top');
my $rendersequence=0;
my $sequencenum=0;
$rightframe->Checkbutton(-text => "Render sequence",
			 -variable => \$rendersequence)->pack(-side => 'top');
my $viewrendering=1;
$rightframe->Checkbutton(-text => "View rendering",
			 -variable => \$viewrendering)->pack(-side => 'top');
my $filename="scene";
$rightframe->Entry(-textvariable => \$filename)->pack(-side => 'top');
$rightframe->Button(-text => 'Povray Render',
		    -command => \&povrayrender)->pack(-side => 'top');
my $timetocomplete=0;
my $totaltime=0;
$rightframe->Label(-text => "Total time so far")->pack(-side => 'top');
$rightframe->Label(-textvariable => \$totaltime)->pack(-side => 'top');
$rightframe->Label(-text => "Estimated time to complete (hours)")->pack(-side => 'top');
$rightframe->Label(-textvariable => \$timetocomplete)->pack(-side => 'top');

# Lens page
$rightframe=$lenspage->Frame()->pack();
$frame=makeslider($rightframe,0,90,$resolution,"Camera View Angle",\$viewangle,\&setviewport);
$frame->pack(-side => 'top');
$frame=makeslider($rightframe,0,90,$resolution,"Near Clipping Plane",\$nearclip,\&setviewport);
$frame->pack(-side => 'top');
$frame=makeslider($rightframe,0,90,$resolution,"Far Clipping Plane",\$farclip,\&setviewport);
$frame->pack(-side => 'top');

# Snake page
$rightframe=$snakepage->Frame()->pack();
$frame=makeslider($rightframe,-10,10,$resolution,"Snake X Position",\$snakexpos,\&dirty);
$frame->pack(-side => 'top');
$frame=makeslider($rightframe,-10,10,$resolution,"Snake Y Position",\$snakeypos,\&dirty);
$frame->pack(-side => 'top');
$frame=makeslider($rightframe,-10,10,$resolution,"Snake Z Position",\$snakezpos,\&dirty);
$frame->pack(-side => 'top');
$frame=makeslider($rightframe,0,360,$resolution,"Snake X Rotation",\$snakexrot,\&dirty);
$frame->pack(-side => 'top');
$frame=makeslider($rightframe,0,360,$resolution,"Snake Y Rotation",\$snakeyrot,\&dirty);
$frame->pack(-side => 'top');
$frame=makeslider($rightframe,0,360,$resolution,"Snake Z Rotation",\$snakezrot,\&dirty);
$frame->pack(-side => 'top');
$frame=makeslider($rightframe,0,1,$resolution,"Gap",\$linkgap);
$frame->pack(-side => 'top');

my $snakenotebook=$main->NoteBook->pack(-side => 'right');
my $snakenum;
for($snakenum=0;$snakenum<2;$snakenum++) {
    my $snakepage=$snakenotebook->add("snake$snakenum",-label => "Snake $snakenum");

    my $leftframe=$snakepage->Frame()->pack(-side => 'left');
    for($i=0;$i<12;$i++) {
	$frame=makenarrowslider($leftframe,0,360,"${i}th Rotation",\$linkrotation[$i]);
	$frame->pack(-side => 'top');
    }
    $leftframe=$snakepage->Frame()->pack(-side => 'left');
    for($i=12;$i<23;$i++) {
	$frame=makenarrowslider($leftframe,0,360,"${i}th Rotation",\$linkrotation[$i]);
	$frame->pack(-side => 'top');
    }
}

$main->update();
glpOpenWindow(width => 500, height => 500,
	      attributes=>[GLX_RGBA,GLX_DOUBLEBUFFER,GLX_DEPTH_SIZE,1]);
glEnable(GL_DEPTH_TEST);
glEnable(GL_LIGHTING);
glEnable(GL_COLOR_MATERIAL);
glLightfv(GL_LIGHT0,GL_POSITION,pack("f4",10,10,10));
glEnable(GL_LIGHT0);

glMatrixMode(GL_PROJECTION);
glLoadIdentity();
glFrustum(-.50,.50,-.50,.50,2.0,500.0);
glMatrixMode(GL_MODELVIEW);

my @clearcolor=(.8,.8,.8,1);
glClearColor(@clearcolor);
glPolygonMode(GL_BACK,GL_LINE);

my($link,$axis,$grid)=(1,2,3);

# Build a link of the snake
glNewList($link,GL_COMPILE);
glBegin(GL_QUADS);
# Back side
glColor3f(@side1);
glNormal3f(0,0,-1);
glVertex3f(-.5,.5,-.5);
glVertex3f(.5,.5,-.5);
glVertex3f(.5,-.5,-.5);
glVertex3f(-.5,-.5,-.5);
glEnd();

glBegin(GL_QUADS);
# Top
glColor3f(@side2);
glNormal3f(0,1,0);
glVertex3f(.5,.5,-.5);
glVertex3f(-.5,.5,-.5);
glVertex3f(-.5,.5,.5);
glVertex3f(.5,.5,.5);
glEnd();

glBegin(GL_QUADS);
# angled side
glColor3f(@side5);
glNormal3f(0,-sqrt(2),sqrt(2));
glVertex3f(-.5,-.5,-.5);
glVertex3f(.5,-.5,-.5);
glVertex3f(.5,.5,.5);
glVertex3f(-.5,.5,.5);
glEnd();

glBegin(GL_TRIANGLES);
# positive x triangle
glColor3f(@side3);
glNormal3f(1,0,0);
glVertex3f(.5,.5,-.5);
glVertex3f(.5,.5,.5);
glVertex3f(.5,-.5,-.5);
glEnd();

glBegin(GL_TRIANGLES);
# Negative x triangle
glColor3f(@side4);
glNormal3f(-1,0,0);
glVertex3f(-.5,.5,-.5);
glVertex3f(-.5,-.5,-.5);
glVertex3f(-.5,.5,.5);
glEnd();

glEndList();

# Build the axis object
glNewList($axis,GL_COMPILE);
glBegin(GL_LINES);
glColor3f(1,0,0);
glVertex3f(0,0,0);
glVertex3f(10,0,0);
glEnd();
glBegin(GL_LINES);
glColor3f(0,1,0);
glVertex3f(0,0,0);
glVertex3f(0,10,0);
glEnd();
glBegin(GL_LINES);
glColor3f(0,0,1);
glVertex3f(0,0,0);
glVertex3f(0,0,-10);
glEnd();
glEndList();

# build the grid object
glNewList($grid,GL_COMPILE);
glBegin(GL_LINES);
for($i=-10;$i <= 10;$i+=1) {
    glVertex3f($i,0,-10);
    glVertex3f($i,0,10);
    
    glVertex3f(-10,0,$i);
    glVertex3f(10,0,$i);
}
glEnd();
glEndList();

$init=1;
setviewport();
render();
MainLoop();

sub setviewport {
    return if(!$init);
    my $ratio=$rendersizey[$rendersize]/$rendersizex[$rendersize];

    glViewport(0,500-500*$ratio,500,500*$ratio);
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    my $horz=(sin(degtorad($viewangle/2))/cos(degtorad($viewangle/2)))*$nearclip;
    glFrustum(-$horz,$horz,-$horz*$ratio,$horz*$ratio,$nearclip,$farclip);
    glMatrixMode(GL_MODELVIEW);
    dirty(0);
}

sub docontinous {
    my $starttime=time();
    while($continous && $curframe <= $lastframe) {
	setpositions();
	$main->update();
	render(1);
	if($rendersequence) {
	    povrayrender();
	    # calculate how much longer
	    my $nowtime=time();
	    $totaltime=($nowtime-$starttime)/(60*60);
	    my $timeperframe=($nowtime-$starttime)/($curframe+1);
	    $timetocomplete=$timeperframe*($lastframe-$curframe)/(60*60)
	}
	$curframe++;
	$newframenum=$curframe;
    }
    $sequencenum=0;
}

sub render {
    my ($force)=@_;
    return if(!$init);
    if(!$force) {
	if(!$dirty) {
	    $looping=0;
	    return;
	}
    }
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
    glLoadIdentity();
    glTranslatef(0,0,-$distance);
    glRotatef($cameraxrot,1.0,0.0,0.0);
    glRotatef($camerayrot,0.0,1.0,0.0);
    glRotatef($camerazrot,0.0,0.0,1.0);
    glTranslatef(-$cameraxlookat,-$cameraylookat,$camerazlookat);
    glLineWidth(3);
    glCallList($axis);

    glLineWidth(1);
    glColor3f(0,0,0);
    glCallList($grid);

    glRotatef(180,0,1,0);
    glTranslatef(-$snakexpos,$snakeypos,$snakezpos);
    glRotatef($snakezrot,0,0,1);
    glRotatef($snakeyrot,0,1,0);
    glRotatef($snakexrot,1,0,0);
    drawsnake();
    glFlush();
    glXSwapBuffers();

    if(!$force) {
	$dirty=0;
	after($waittime,[\&render,0]);
    }
}

sub povrayrender {
    writescene();
    my $options;
    my $background="&";
    if($hiquality) {
	$options="hiquality.ini";
    } else {
	$options="";
    }
    if($viewrendering) {
	$options="$options +D1";
    }
    if($rendersequence) {
	$options="$options +O$filename$curframe.tga -P -X";
	$background="";
	$sequencenum++;
    }
    system("povray +W$rendersizex[$rendersize] +H$rendersizey[$rendersize] $options $options +i scene.pov $background");
}

sub writescene {
    open(FILE,">snake.pivits") || die "open: snake.pivits: $!\n";
    for($i=0;$i<23;$i++) {
	print FILE "#declare Pivit$i = -$linkrotation[$i]\n";
    }
    print FILE "#declare Gap = $linkgap\n";
    close(FILE);
    open(FILE,">snake.transform") || die "open: snake.rotate: $!\n";
    print FILE "rotate <$snakexrot,0,0>\n";
    print FILE "rotate <0,-$snakeyrot,0>\n";
    print FILE "rotate <0,0,-$snakezrot>\n";
    print FILE "translate <$snakexpos,$snakeypos,$snakezpos>\n";
    close FILE;
    # Call rotatetoabs to get the camera in the right place
    my($cameraxpos,$cameraypos,$camerazpos)=rotatetoabs();
    open(FILE,">snake.camera") || die "open: snake.camera: $!\n";
    print FILE "perspective\n";
    print FILE "location <$cameraxpos,$cameraypos,$camerazpos>\n";
    print FILE "up <0,1,0>\n";
    print FILE "right <",$rendersizex[$rendersize]/$rendersizey[$rendersize],",0,0>\n";
    print FILE "angle ",$viewangle,"\n";
    print FILE "look_at <$cameraxlookat,$cameraylookat,$camerazlookat>\n";
    close(FILE);
}

sub writesnakepov {
    open(FILE,">snake.pov") || die "open: snake.pov: $!\n";
    my $i;
    print FILE "#declare Snake = object {\n";
    print FILE "union {\n";
    for($i=0;$i<11;$i++) {
	print FILE "union {\n";
    }
    print FILE "object {\n";
    print FILE "Link // pivit 22\nrotate <0,180,0>\nrotate <90,0,0>\nrotate <0,Pivit22,0>\ntranslate <0,1+Gap,0>\n}\n";
    for($i=22;$i>=12;$i--) {
	print FILE "Link // pivit ",$i-1,"\nrotate <0,180,0>\nrotate <90,0,0>\nrotate <0,Pivit",$i-1,",0>\ntranslate <0,1+Gap,0>\n}\n";
    }
LEFT:
    for($i=0;$i<10;$i++) {
	print FILE "union {\n";
    }
    print FILE "object {\n";
    print FILE "Link // pivit 0\nrotate <0,180,0>\nrotate <90,0,0>\nrotate <0,0,Pivit0>\ntranslate <0,0,-(1+Gap)>\n}\n";
    for($i=1;$i<11;$i++) {
	print FILE "Link // pivit ",$i,"\nrotate <0,180,0>\nrotate <90,0,0>\nrotate <0,0,Pivit$i>\ntranslate <0,0,-(1+Gap)>\n}\n";
    }
    print FILE "object{Link\n}}\nrotate <45,0,0>\n}\n";
    close(FILE);
}

sub rotatetoabs {
    my($x,$y,$z)=(0,0,$distance);
    ($x,$y,$z)=rotatex($x,$y,$z,degtorad($cameraxrot));
    ($x,$y,$z)=rotatey($x,$y,$z,degtorad($camerayrot));
    ($x,$y,$z)=rotatez($x,$y,$z,degtorad($camerazrot));
    return($x+$cameraxlookat,$y+$cameraylookat,-($z-$camerazlookat));
}

sub drawsnake {
    my $i;
    glPushMatrix();

    glPushMatrix();
    glRotatef(45,1,0,0);
    for($i=12;$i<24;$i++) {
	glTranslatef(0,1+$linkgap,0);
	glRotatef(90,1,0,0);
	glRotatef(180,0,1,0);
	glRotatef($linkrotation[$i-1],0,0,1);
	glCallList($link);
    }
    glPopMatrix();

    glPushMatrix();
    glRotatef(45,1,0,0);
    for($i=10;$i>=0;$i--) {
	glTranslatef(0,0,-(1+$linkgap));
	glRotatef(90,1,0,0);
	glRotatef(180,0,1,0);
	glRotatef($linkrotation[$i],0,1,0);
	glCallList($link);
    }
    glPopMatrix();

    glRotatef(45,1,0,0);
    glCallList($link);

    glPopMatrix();
}

# This function makes a slider bar with a title in a frame which controls
# a certain variable.
sub makeslider {
    my($window,$min,$max,$resolution,$text,$variable,$sub)=@_;
    
    my $frame = $window->Frame(-relief => 'raised', -border => 1);
    $frame->Label(-text => $text)->pack(-side => 'top');
    
    my $interval=($max-$min)/5;
    my $slider=$frame->Scale(-orient => 'horizontal',
                             '-length' => 320,
                             -from => $min,
                             -to => $max,
                             -tickinterval => $interval,
                             -variable => $variable,
                             -resolution => $resolution,
                             -command => $sub
                             );
    $slider->pack(-side => 'top');
    return $frame;
}

# This function makes a slider bar with a title in a frame which controls
# a certain variable.
sub makenarrowslider {
    my($window,$min,$max,$text,$variable)=@_;
    
    my $frame = $window->Frame(-relief => 'raised', -border => 1);
    $frame->Label(-text => $text)->pack(-side => 'left');
    
    my $interval=($max-$min)/5;
    my $slider=$frame->Scale(-orient => 'horizontal',
                             '-length' => 320,
                             -from => $min,
                             -to => $max,
                             -tickinterval => $interval,
                             -variable => $variable,
                             -resolution => .01,
                             -command => \&dirty
                             );
    $slider->pack(-side => 'left');
    return $frame;
}

sub dirty {
    $dirty=1;
    if(! $looping) {
	$looping=1;
	after($waittime,[\&render,0]);
    }
}

sub rotatex {
    my($x,$y,$z,$angle)=@_;
    my $newy=$y*cos($angle)+$z*sin($angle);
    my $newz=$y*-sin($angle)+$z*cos($angle);
    return ($x,$newy,$newz);
}

sub rotatey {
    my($x,$y,$z,$angle)=@_;
    my $newx=$x*cos($angle)+$z*-sin($angle);
    my $newz=$x*sin($angle)+$z*cos($angle);
    return ($newx,$y,$newz);
}

sub rotatez {
    my($x,$y,$z,$angle)=@_;
    my $newx=$x*cos($angle)+$y*sin($angle);
    my $newy=$x*-sin($angle)+$y*cos($angle);
    return($newx,$newy,$z);
}

sub degtorad {
    return $_[0]*((2.0*$PI)/360);
}

sub radtodec {
    return $_[0]*(360/(2*$PI));
}

sub parsefile {
    open(FILE,"<$_[0]") || die "open: $_[0]: $!\n";
    my($variable0,$variable1,$variable2,$variable3,$variable4,$variable5,
       $variable6,$variable7,$variable8,$variable9);
    my($mark0,$mark1,$mark2,$mark3,$mark4,$mark5,
       $mark6,$mark7,$mark8,$mark9);
    my($mark10,$mark11,$mark12,$mark13,$mark14,$mark15,
       $mark16,$mark17,$mark18,$mark19);
    my @tweakablevars=(\$cameraxrot,\$camerayrot,\$camerazrot,\$cameraxlookat,\$cameraylookat,\$camerazlookat,\$viewangle,\$nearclip,\$farclip,\$snakexrot,\$snakeyrot,\$snakezrot,\$snakexpos,\$snakeypos,\$snakezpos,\$distance);
    my @tweakablenames=qw(cameraxrot camerayrot camerazrot cameraxlookat cameraylookat camerazlookat viewangle nearclip farclip snakexrot snakeyrot snakezrot snakexpos snakeypos snakezpos distance);

    my(@camerax,@cameray,@cameraz);
    while(<FILE>) {
	last if(/^__END__/);
	s/\#.*//g;
	my @arguments=split(' ');
	next if(!defined $arguments[0]);
	if($arguments[0] eq "eval") {
	    shift(@arguments);
	    my $str=join(' ',@arguments);
	    eval $str;
	    next;
	}
	if($#arguments != 3) {
	    print STDERR "Error line $.:  Argument count should be 3\n";
	    next;
	}
	# Run eval on the frame counts and the rotation angle in case
	# they're expressions
	$arguments[0]=eval $arguments[0];
	$arguments[1]=eval $arguments[1];
	$arguments[3]=eval $arguments[3];
#	print "arguments are ",join(' ',@arguments),"\n";
	my $variable;
	my $clip=1;
	if($arguments[2]=~/^joint(\d+)/) {
	    $variable=\$linkrotation[$1];
	} elsif($arguments[2] eq 'cameraxrot') {
	    push(@camerax,$arguments[0]);
	    push(@camerax,$arguments[1]);
	    push(@camerax,$arguments[3]);
	    next;
	} elsif($arguments[2] eq 'camerayrot') {
	    push(@cameray,$arguments[0]);
	    push(@cameray,$arguments[1]);
	    push(@cameray,$arguments[3]);
	    next;
	} elsif($arguments[2] eq 'camerazrot') {
	    push(@cameraz,$arguments[0]);
	    push(@cameraz,$arguments[1]);
	    push(@cameraz,$arguments[3]);
	    next;
	} else {
	    my $i;
	    my $match=0;
	    for($i=0;$i<=$#tweakablevars;$i++) {
		if($arguments[2] eq $tweakablenames[$i]) {
		    $clip=0 if($arguments[2]=~/pos/ || $arguments[2]=~/lookat/);
		    $variable=$tweakablevars[$i];
		    $match=1;
		    last;
		}
	    }
	    if(! $match) {
		print "Error line $.: $arguments[2] an unknown variable\n";
		next;
	    }
	}
	my $i;
	my $step=incrstep($$variable,$arguments[3],
			  $arguments[1]-$arguments[0]+1,$clip);
	for($i=$arguments[0];$i<=$arguments[1];$i++) {
	    $lastframe=$i if($i > $lastframe);
	    my $array=$framelist[$i];
	    if(!defined $array) {
		my @newarray=();
		$framelist[$i]=\@newarray;
		$array=\@newarray;
	    }
	    my $prevarray=$framelist[$i-1];
	    if(!defined $prevarray) {
		my @newarray=();
		$framelist[$i-1]=\@newarray;
		$prevarray=\@newarray;
	    }
	    push(@{$prevarray},[$variable,$$variable]);
	    $$variable=$$variable + $step;
	    if($clip) {
		if($$variable < 0) {
		    $$variable=360 + $$variable;
		} elsif($$variable > 360) {
		    $$variable=$$variable - 360;
		}
	    }
	    push(@{$array},[$variable,$$variable]);
	}
	$curframe=$arguments[1]+1;
    }
    rewind();
    
    dospline(\$cameraxrot,@camerax);
    dospline(\$camerayrot,@cameray);
}

sub dospline {
    my($variable,@array)=@_;
    my $i;
    my(@x,@y);

    my $curposition=$$variable;
    push(@x,0);
    push(@y,$curposition);
    for($i=0;$i<=$#array;$i+=3) {
	my($begframe,$frame,$position)=($array[$i],$array[$i+1],$array[$i+2]);
	my $step=incrstep($curposition,$position,1,1);
#	print "begframe=$begframe, frame=$frame, position=$position, step=$step\n";
#	push(@x,$begframe);
#	push(@y,$curposition);
	push(@x,$frame);
	$curposition+=$step;
	push(@y,$curposition);
    }
    my $spline=new Math::Spline(\@x,\@y);
    for($i=0;$i<=$lastframe;$i++) {
	my $array=$framelist[$i];
	if(!defined $array) {
	    my @newarray=();
	    $framelist[$i]=\@newarray;
	    $array=\@newarray;
	}
	my $yval=$spline->evaluate($i);
	while($yval > 360) {
	    $yval=$yval-360;
	}
	while($yval < 0) {
	    $yval=360+$yval;
	}
	# This is a hack
	$yval=0 if($variable == \$cameraxrot && $yval > 180);
	push(@{$array},[$variable,$yval]);
    }
}

sub incrstep {
    my($curangle,$wantangle,$frames,$clip)=@_;
    my $dir;
    my $diff=abs($curangle-$wantangle);
    if($wantangle > $curangle) {
	$dir=1;
    } else {
	$dir=-1;
    }
    if($clip) {
	if($diff > 180) {
	    $diff=360-$diff;
	    $dir=-$dir;
	}
    }
    return ($dir*($diff/$frames));
}

sub rewind {
    # This is a nasty trick to fix an out of phase frame count.  We're
    # forcing it to the last frame and working our way backwards.  We'll
    # Hit every frame which has changed this way.
    $curframe=$lastframe;
    $newframenum=0;
    newframe();
}

sub setpositions {
    my $ref=$framelist[$curframe];
    my $newref;
    foreach $newref (@$ref) {
	my($variable,$value)=@$newref;
	$$variable=$value;
    }
}

# Called when we move the frame slider bar
sub newframe {
    my $i;
    my $dir;
    if($newframenum > $curframe) {
	$dir=1;
    } else {
	$dir=-1;
    }
    for($i=$curframe;$i!=$newframenum;$i+=$dir) {
	$curframe=$i;
	setpositions(-1);
	setpositions(1);
    }
    $curframe=$newframenum;
    setpositions();
    dirty();
}
