// Persistence of Vision Ray Tracer Scene Description File
// File: CatThrow.inc
// Vers: 3.1
// Desc: A scene file showing two hands throwing a paperwad back and forth
// Date: 10/13/98
// Auth: Steve Sloan II

#include "catobjs.inc"

/*******************/
/* Hand parameters */
/*******************/

/**************/
/* Hand poses */
/**************/

/* Closed Hand */
#declare pinky_angle      = 60;
#declare ring_angle       = 60;
#declare middle_angle     = 60;
#declare index_angle      = 60;
#declare thumb_angle      = 15;
#declare thumb_hor_angle  = 40;

/* Flexing Hand */
#declare pinky_angle      = 60*SineWaveClock(clock);
#declare ring_angle       = 60*SineWaveClock(clock);
#declare middle_angle     = 60*SineWaveClock(clock);
#declare index_angle      = 60*SineWaveClock(clock);
#declare thumb_angle      = 15*SineWaveClock(clock);
#declare thumb_hor_angle  = 40*SineWaveClock(clock);

/* Throwing Hand */
#declare pinky_angle      = 10;
#declare ring_angle       =  5;
#declare middle_angle     =  0;
#declare index_angle      =  0;
#declare thumb_angle      = 25;
#declare thumb_hor_angle  = 30;

/* Create the hand */
#include "cathand.inc"

/* Elbow locations of the two */
/* arms throwing the paperwad */
#declare armpos1 = <-1*in, 16*in, -3.25*ft>;
#declare armpos2 = < 1*in, 16*in,  3.25*ft>;

/* Length of the forearm */
#declare forearm_radius    =        12.3*in;

/* Angle limits of motion, for the hands as */
/* they throw the paperwad back and forth   */
#declare hand_start_angle1 =             50;
#declare hand_end_angle1   =             35;
#declare hand_start_angle2 =            -50;
#declare hand_end_angle2   =            -35;

/* A constant fudge-factor used to make the paperwad's throwing */
/* speed get the "wad" from one hand to the other               */
#declare throw_speed       =          23.45;

#declare ThrowingHands = 
union {
    #switch (throw_clock)
        /* Throwing the paperwad */
        #range (0, throw_time)
            union {
                object { Arm rotate 90*x }
                object { Paperwad translate <1, 1, 0.75>*in translate forearm_radius*z }
                #declare current_angle = interp_values(hand_start_angle1, hand_end_angle1,
                                             throw_clock/throw_time);
                rotate current_angle*x
                translate armpos1
            }
            object {
                Arm rotate 90*x rotate 180*y
                rotate hand_start_angle2*x
                translate armpos2
            }
        #break
        
        /* After the paperwad leaves the (first) hand */
        #range (throw_time, catch_time)
            #declare startvel = get_startvelvector (hand_start_angle1 - hand_end_angle1,
                                    hand_end_angle1, forearm_radius, throw_time);
    
            // The arm moves back down slightly, after release
            object {
                Arm
                rotate 90*x
                #declare current_clock = (throw_clock - throw_time) /
                                         (catch_time - throw_time);
                #declare current_angle = interp_values(hand_end_angle1, hand_start_angle1,
                                                       current_clock);
                rotate current_angle*x
                translate armpos1
            }
            
            // The other arm moves upward to catch the paperwad
            object {
                Arm rotate 90*x rotate 180*y
                #declare current_clock = (throw_clock - throw_time) /
                                         (catch_time - throw_time);
                #declare current_angle = interp_values(hand_start_angle2, hand_end_angle2,
                                                       current_clock);
                rotate current_angle*x
                translate armpos2
            }
            
            // The paperwad flies up with projectile motion
            object {
                Paperwad translate <1, 1, 0.75>*in
                translate forearm_radius*z
                rotate hand_end_angle1*x
                translate armpos1
                translate projectile(throw_speed*startvel, throw_clock - throw_time)
            }
        #break
        
        #range (catch_time, catch_end_time)
            #declare startvel = get_startvelvector (hand_start_angle1 - hand_end_angle1,
                                    hand_end_angle1, forearm_radius, throw_time);
            object {
                Arm
                rotate 90*x
                rotate hand_start_angle1*x
                translate armpos1
            }
            
            union {
                object { Arm rotate 90*x }
                object { Paperwad translate <1, 1, 0.75>*in translate forearm_radius*z }
                rotate 180*y
                #declare current_clock = (throw_clock - catch_time) /
                                         (catch_end_time - catch_time);
                #declare current_angle = interp_values(hand_end_angle2, hand_start_angle2,
                                                       current_clock);
                rotate current_angle*x
                translate armpos2
            }
        #break
    
        /* Throwing the paperwad (other arm) */
        #range (catch_end_time, throw2_time)
            object {
                Arm rotate 90*x
                rotate hand_start_angle1*x
                translate armpos1
            }
            union {
                object { Arm rotate 90*x }
                object { Paperwad translate <1, 1, 0.75>*in translate forearm_radius*z }
                rotate 180*y
                #declare current_clock = (throw_clock - catch_time) /
                                         (throw2_time - catch_time);
                #declare current_angle = interp_values(hand_start_angle2, hand_end_angle2,
                                                       current_clock);
                rotate current_angle*x
                translate armpos2
            }
        #break
        
        /* After the paperwad leaves the (second) hand */
        #range (throw2_time, catch2_time)
            #declare startvel = get_startvelvector (hand_start_angle2 - hand_end_angle2,
                                    hand_end_angle2, forearm_radius,
                                    throw2_time - catch_end_time);
    
            // The other arm moves upward to catch the paperwad
            object {
                Arm rotate 90*x
                #declare current_clock = (throw_clock - throw2_time) /
                                         (catch2_time - throw2_time);
                #declare current_angle = interp_values(hand_start_angle1, hand_end_angle1,
                                                       current_clock);
                rotate current_angle*x
                translate armpos1
            }
            
            // The arm moves back down slightly, after release
            object {
                Arm
                rotate 90*x rotate 180*y
                #declare current_clock = (throw_clock - throw2_time) /
                                         (catch2_time - throw2_time);
                #declare current_angle = interp_values(hand_end_angle2, hand_start_angle2,
                                                       current_clock);
                rotate current_angle*x
                translate armpos2
            }
            
            #if (cat_action = CatWatch | throw_time <= (throw2_time + catch2_time)/2)
            
            // The paperwad flies up with projectile motion
            object {
                Paperwad translate <1, 1, 0.75>*in
                translate forearm_radius*z
                rotate 180*y
                rotate hand_end_angle2*x
                translate armpos2
                #declare wad_pos = projectile(throw_speed*startvel,
                                              throw_clock - throw2_time);
                translate wad_pos
            }
            
            #end
        #break
        
        #else // (catch2_time, scene_runtime)
            #declare startvel = get_startvelvector (hand_start_angle1 - hand_end_angle1,
                                    hand_end_angle1, forearm_radius, throw_time);
            union {
                object { Arm rotate 90*x }
                #if (cat_action = CatWatch)
                    object { Paperwad translate <1, 1, 0.75>*in translate forearm_radius*z }
                #end
                #declare current_clock = (throw_clock - catch2_time) /
                                         (scene_runtime - catch2_time);
                #declare current_angle = interp_values(hand_end_angle1, hand_start_angle1,
                                                       current_clock);
                rotate current_angle*x
                translate armpos1
            }
    
            object {
                Arm
                rotate 90*x
                rotate 180*y
                rotate hand_start_angle2*x
                translate armpos2
            }
//        #break
    #end
}
