// Cosms.pov  
// Microcosms  1998 by Neil Alexander
// This is the main control file for the Microcosms animation.
// This file defines camera placements and motions, lighting
// effects, and motion control for animated objects.
// As well as the include files below, this file uses a tga
// image map created from Fabric.pov
// House.inc uses:
// Trim.inc, Chair.inc, Bookcase.inc
// and image maps Rug2.gif and Gable.gif

#include "house.inc"
#include "cassidy.inc"
#include "flea.inc"
#include "birds.inc"
#include "mite.inc"
#include "cell.inc"

#declare Scene = 1;  // (1 to 8)

#switch (Scene)

  #case (1) //***************** Scene 1 ***********************
  // 3 seconds, frames 1 to 72
  // 1/2 sec open from black, 1/2 sec still
  // birds fly by, 1.5 sec
    
    global_settings { max_trace_level 7 }
        
    #declare K = clock*6;   // fade-in from black, first 0.5 sec
    #if (K>1)
      #declare K = 1;
      #end
      
    #declare B = clock*3/2 - 0.5; // birds, 2nd & 3rd sec
          
    #declare F1 = B*7;  // wing flap rate
        
    camera { 
      angle 50 
      location <-280, -30, 96> 
      look_at <10, 140, 0>  
      }
      
    sky_sphere { Sky }

    object { Cloudsky }

    object { Sun }
    
    light_source {  // fill-in light
      <-50000, 100000, 200000>
      color rgb 0.2
      shadowless
      } 
      
    light_source {  // fill-in light
      <-100000, 10000, 10000>
      color rgb 0.2
      shadowless
      }
    
    object { House(0) }

    object { Livingroom }

    object { House(1)
      translate <0,24,-256>
      }
      
    object { Cassidy (<0,0,0>)
      rotate <0,10,0>
      translate <10,0,56>
      } 
      
   object { Bird(F1)
     scale 2
     rotate y*90
     translate -z*280*B  // straight line travel along axis
     rotate <10,10,0>       // direction
     translate <-36,50,145>    // start point
     }
     
   object { Bird(F1-0.375)
     scale 2
     rotate y*90
     translate -z*280*B  // straight line travel along axis
     rotate <10,25,0>       // direction
     translate <-36,80,165>    // start point
     }  
      
    #if (K<1)
    intersection {
      plane {z,0  // fade in from black
        texture {
          pigment { wood
            color_map {
              [(K*0.5) color rgbt <0,0,0,1>]
              [(K*0.55) color rgbt 0]
              }
            scale 18.5
            }
          }
        }
      box {<-11,-11,-1>,<11,11,1>
        texture { pigment { color rgbt 1}}
        }
      translate z*8
      rotate <-29.1,108.3,0>
      translate <-280,-30,96>
      no_shadow
      hollow
      }
    #end // end fade-in  

    
    #break  // end Scene 1
    
  #case (2) //***************** Scene 2 ***********************
  // 2 seconds, frames 73 to 120
  // Cut to closer shot, reflection of more birds,
  // cat watching 
    
    #declare F1 = clock*7;  // flap rate
    
    camera { 
      angle 50 
      location <-130, 20, 100> // <-280, -30, 96> 
      look_at <10, 72, 42>     // <10, 140, 0> 
      }
      
    sky_sphere { Sky }

    object { Cloudsky }

    object { Sun }
    
    light_source {  // fill-in light
      <-50000, 100000, 200000>
      color rgb 0.2
      shadowless
      } 
      
    light_source {  // fill-in light
      <-100000, 10000, 10000>
      color rgb 0.2
      shadowless
      }
    
    object { House(0) }

    object { Livingroom }

    object { House(1)
      translate <0,24,-256>
      }
      
    #declare Bx = 160 - cos(135*clock*pi/180)*135; // Bird's x-pos relative to cat
    #declare Bz = -76 + sin(135*clock*pi/180)*135; // Bird's y-pos relative to cat
    #declare Cx = atan2(45,(sqrt((Bx*Bx)+(Bz*Bz))))*180/pi; // Cat's head tilt
    #declare Cy = atan2(Bz,Bx)*180/pi;                      // Cat's head rot
    
    #if (Cx>50)
      #declare Cx = 50;
      #end 
      
    #if (Cx<-50)
      #declare Cx = -50;
      #end
      
    #if (Cy>50)
      #declare Cy = 50; 
      #end
    
    #if (Cy<-50)
      #declare Cy = -50;
      #end
                        
    #if (clock<0.25)   // Cassidy looks up as bird enters frame, then follows it
      #declare K = clock*4;
      #declare Cx = K * Cx;
      #declare Cy = K * Cy;
      #end
    
    object { Cassidy (<-Cx,Cy,0>)
      rotate <0,10,0>
      translate <10,0,56>
      } 
        
   object { Bird(F1)
     scale 2
     rotate <30,-90,0>
     translate x*135    // radius of curve
     rotate y*-135*clock       // distance along curve
     translate <-150,70,-20>    // start point
     } 
       
    #break  // end Scene 2 

  #case (3) //***************** Scene 3 ***********************
  // 2 seconds, frames 121 to 168 
  // Inside livingroom, cat looking out, moving head (1 sec) 
  // zoom in (0.5 sec), flea jumps from back of cat to chair (0.5 sec)
  
    #declare F = clock*4-3;  // 0 to 1 over final 0.5 sec
    #if (F<0)
      #declare F = 0;
      #end 
    #declare A = clock*4-2;  // 0 to 1 over first half of final 1 sec
    #if (A<0)
      #declare A = 0;
      #end 
    #if (A>1)
      #declare A = 1;
      #end
    #declare Fx = F*14*cos(radians(60));
    #declare Fy = (F-F*F)*8;
    #declare Fz = F*14*sin(radians(60));
    #declare Fpos = <13.25+Fx,20+Fy,56-Fz>;
    
    #declare C1 = clock*12;    // Cat head movement
    #if (C1>1)
      #declare C1 = 1;
      #end
    #declare C1 = sin(pi/2*C1); // fast look up, pause
    #declare C2 = clock*6-1;
    #if (C2<0)
      #declare C2 = 0;
      #end
    #if (C2>1)
      #declare C2 = 1;
      #end
    #declare C2 = (cos(pi*(C2+1))+1)/2; // slow-fast-slow
    #declare C3 = clock*12-4;
    #if (C3<0)
      #declare C3 = 0;
      #end
    #if (C3>1)
      #declare C3 = 1;
      #end
    #declare C3 = sin(pi/2*C3); 
    
    
        
    camera {  // inside
      angle 60/pow(30,A)
      location <6, 36, 100> // focal blur with zoom
      look_at (<15.5,21,52> + 2*Fpos)/3
      aperture A
      focal_point Fpos
      blur_samples 10
      }
    
    object { Sun }
    
    light_source {  // fill-in light
      <3, 60, 30>
      color rgb 0.2
      shadowless
      }
      
    light_source {  // fill-in light
      <40, 110, 155>
      color rgb 0.3
      }
      
    object { House(0) }

    object { Livingroom }

    object { Cassidy (<-C1*30+C2*10+C3*30,-C1*35+C2*45+C3*28,0>)
      translate <10.5,0,56>
      }
      
    #declare Fp = 1-(sin(F*pi));
    object { Flea (Fp)
      rotate y*60
      translate Fpos
      }   
      
      
    #break  // end Scene 3
    
  #case (4) //***************** Scene 4 ***********************
  // 6 seconds, frames 169 to 312 
  // scale 1 unit = 0.01"
  // Close-up of chair fabric - flea lands, absorbs shock, bounces (0.5s)
  // flea pauses (1s), crouches, and jumps away(0.5s). Mite is crawling along (0.25s).
  // Zoom in fast on mite(0.75s), slow pan to front(2.5s), fast into mouth (0.5s)
    
    #declare Miteloc = <0.0625+2.8125*clock,0,-6.25>;
    
    // Camera motions
    
    #declare M = clock*2.6667;   // pan around right (0 to 2.25 )
    #if (M>1)
      #declare M = 1;
      #end
    #declare M = (cos(pi*(M+1))+1)/2; // slow-fast-slow
    
    #declare M1 = clock*8-3;      // zoom on mite (2.25 to 3 )
    #if (M1<0)
      #declare M1 = 0;
      #end
    #if (M1>1)
      #declare M1 = 1;
      #end
    #declare M1 = sqrt(M1);
    
    #declare M2 = clock*2.4-1.2;   // pan around front (3 to 5.5 )
    #if (M2<0)
      #declare M2 = 0;
      #end
    #if (M2>1)
      #declare M2 = 1;
      #end
    #declare M2 = (cos(pi*(M2+1))+1)/2; // slow-fast-slow
        
    #declare M3 = clock*12-11;          // into mite (5.5 to 6 )
    #if (M3<0)
      #declare M3 = 0;
      #end
    #if (M3>1)
      #declare M3 = 1;
      #end  
      
    camera {
      angle 20-17*M1
      location <43-12*M-13*M2-15*M3, 6, 0>
      rotate <0,(-6-36*M+42*M2),0>
      translate <0,-5.86*M2,-6.25*M2>  
      look_at <0, 3-2.86*M1, 0>+M1*<2,0,-6.25>
      aperture 0.75-0.5*M2
      focal_point <0,0,0>+M1*Miteloc
      blur_samples 12
      }
        
    sky_sphere {
      pigment { gradient y
        color_map { 
          [0 color rgb <1,0.97,0.93>]
          [0.125 color rgb <1,0.97,0.93>*0.375]
          }
        }
      }
    
    light_source {
      <2000, 6000, 6000>
      color rgb 1
      area_light
      <300,0,0>, <0,80,-80>,3,3
      } 
      
    light_source {
      <3000, -6000, -6000>
      color rgb <1,0.97,0.93>*0.5
      shadowless
      } 
      
    light_source {
      <3000, 1000, -2000>
      color rgb <1,0.97,0.93>*0.25
      shadowless
      }
    
    #declare Fabric = normal { checker
  normal { gradient x scale 0.2 scallop_wave }
  normal { gradient z scale 0.2 scallop_wave }
  }
    
    plane {y,0    // chair surface
      texture {
        pigment { color rgb <1,0.97,0.93> }
        finish { ambient 0.2 diffuse 0.8 }
        normal { average
          normal_map {
            [ 0.25 quilted ]
            [ 0.25 Fabric ]
            [ 1 bumps 0.1 scale 0.01 ]
            }
          scale 5
          }
        }
      translate <-112,-0.125,-84.75>
      }
      
    height_field {
      tga "fabric.tga"
      scale <133.333,0.25,100>
      texture {
        pigment { color rgb <1,0.97,0.93> }
        finish { ambient 0.15 diffuse 0.8 }
        normal { bumps 0.1 scale 0.01 }
        }
      translate <-112,-0.125,-84.75>
      }
      
      
    // Flea motions
    
    #declare F1 = clock*24-1;       // flea rebounding(0.25 to 0.5)
    #if (F1<0)
      #declare F1 = 0;
      #end
    #if (F1>1)
      #declare F1 = 1;
      #end
    #declare F1 = sqrt(1-F1);
    
    #declare F2 = clock*48;         // flea landing (0 to 0.25)
    #if (F2>2)
      #declare F2 = 2;
      #end
    #declare F2 = (2-F2)*2;
    
    #declare FF2 = F2*F2;
    #if (FF2>1)
      #declare FF2 = 1;
      #end
      
    #declare F3 = (F2-1)*3;        // flea landing into crouch (0.125 to 0.25)
    #if (F3<0)
      #declare F3 = 0;
      #end
    
    #declare FF3 = F3;
    #if (FF3>1)
      #declare FF3 = 1;
      #end
      
    #declare F4 = clock*24-6;      // flea crouching (1.5 to 1.75)
    #if (F4<0)
      #declare F4 = 0;
      #end
    #if (F4>1)
      #declare F4 = 1;
      #end 
    #declare F4 = sqrt(F4);
    
    #declare F5 = clock*48-14;     // flea springing off (1.75 to 2)
    #if (F5<0)
      #declare F5 = 0;
      #end
    #declare F5 = F5*2;
    
    #declare FF5 = F5*F5;
    #if (FF5>1)
      #declare FF5 = 1;
      #end
      
    #declare F6 = (F5-1)*3;      // flea jumping (1.875 tp 2)
    #if (F6<0)
      #declare F6 = 0;
      #end
    
    #declare FF6 = F6;
    #if (FF6>1)
      #declare FF6 = 1;
      #end
    
    object { Flea(-F4+FF5+FF6-F1+FF2+FF3)
      scale 100
      rotate y*90
      translate <0,3-1.05*(F4+F1)+1.05*(FF5+FF2)+1.3*(F6+F3),0.65*(F3-F6)>  
      }
    
    object { Mite(clock*15)
      translate Miteloc
      }
    
    #break  // end Scene 4

  #case (5) //***************** Scene 5 ***********************
  // 4 seconds, frames 313 to 408 
  // Enter the mite - open from dark (0.25s)
  // Cells and microbes floating in murky innards
  // Zoom twd cell with microbe attached
  
    sky_sphere {
      pigment { gradient y
        color_map { 
          [0.3 color rgb <0.14,0.09,0.03>] 
          [0.6 color rgb <0.74,0.71,0.38>]
          }
        scale 2
        translate y*-1
        }
      }

    light_source {
      <3000, 3000, -1000>
      color rgb 1
      }
      
    fog {
      fog_type 1
      distance 70
      color rgbft <0.74,0.71,0.38,0.2,0.5>
      }  
      
    camera {
      angle 50
      location <0,0,-10>
      look_at <0,0,0>
      }
    
    #declare C1 = 0;   // cells
    #while (C1<20)
      #declare C2 = seed(C1*10);
      object { Cell(C1,clock*3)
        rotate <rand(C2)*720*clock,rand(C2)*720*clock,0>
        translate <(rand(C2)-0.5)*20+(rand(C2)-0.5)*5*clock,
          (rand(C2)-0.5)*20+(rand(C2)-0.5)*5*clock,rand(C2)*20+(rand(C2)-0.5)*5*clock>
        }
      #declare C1 = C1 + 1;
      #end // end while loop 
      
    #declare C1 = 0;       // microbes
    #while (C1<20)
      #declare C2 = seed(C1*154);
      object { Microbe(C1,clock*5)
        rotate <rand(C2)*180*clock,rand(C2)*180*clock,0>
        translate <(rand(C2)-0.5)*20+(rand(C2)-0.5)*8*clock,
          (rand(C2)-0.5)*20+(rand(C2)-0.5)*8*clock,rand(C2)*20+(rand(C2)-0.5)*8*clock>
        }
      #declare C1 = C1 + 1;
      #end // end while loop
    
    union { // microbe attached to cell
      object { Cell(10,clock*4)}
      object {Microbe(1,clock*5)
        translate x*0.75
        }
      rotate z*(70-70*clock)
      rotate y*(-120+120*clock)
      translate z*-9.75*clock
      }  // end union
    
    
    #declare K = clock*32;   // fade-in from black, first 0.25 sec
    #if (K>1)
      #declare K = 1;
      #end
    
    #if (K<1)
      plane {z,0  // fade in from black
        texture {
          pigment { wood
            color_map {
              [(K*0.5) color rgbt <0.27,0.27,0.216,1>]
              [(K*0.55) color rgb <0.27,0.27,0.216>]
              }
            scale 2
            }
          finish { ambient 1 diffuse 0 }
          }
        translate -z*9
        no_shadow
        hollow
        }
      #end // end fade-in

    #break  // end Scene 5
    
  #case (6) //***************** Scene 6 ***********************
  // 3 seconds, frames 409 to 480 
  // Inside cell - move over to side of nucleus (1s)
  // Zoom twd protein chains attached to nucleus (2s)
  
    global_settings { max_trace_level 6 }
    
    sky_sphere {
      pigment { gradient y
        color_map { 
          [0.3 color rgb <0.34,0.26,0.17>] 
          [0.6 color rgb <0.68,0.66,0.41>]
          }
        scale 2
        translate y*-1
        }
      }

    light_source {
      <3000000, 3000000, -1000000>
      color rgb 1
      }
      
    #declare F1 = clock*3;
    #if (F1>1)
      #declare F1 = 1;
      #end
    #declare F1 = sqrt(F1); // fast-slow 
      
    #declare F2 = clock*3-1;
    #if (F2<0)
      #declare F2 = 0;
      #end
    #if (F2>1)
      #declare F2 = 1;
      #end 
      
    #declare F3 = clock*3-2;
    #if (F3<0)
      #declare F3 = 0;
      #end
    
    camera {
      angle 50-20*F1-20*F2
      location <217*F1-88.75*F3,0.81*F3,-250+125*F1+125*F3>
      look_at <125*F1+3*F2-0.297*F3,1.2*F3,0.519*F3>
      }
    
    #declare C = 0;
    #declare S = seed(5);
    
    union { // microbe attached to cell
      object { Nucleus((clock+1)*3) scale 1000}
      object {Microbe(1,clock*5)
        scale 1000
        translate x*750
        }
      object { Protein (12,clock*2)  // will zoom in on this one...
        scale 0.5
        translate x*125
        }
      #while (C<7)
        object { Protein (int(10*rand(S)*3+5),clock*2+rand(S))
          scale 0.5
          translate x*125
          rotate <360*rand(S),8*(rand(S)-0.5),8*(rand(S)-0.5)>
          }
        #declare C = C + 1;
        #end  // end while loop
      
      }  // end union
    
    #break  // end Scene 6
    
  #case (7) //***************** Scene 7 ***********************
  // 1.5 seconds, frames 481 to 516 
  // Inside atom - zoom twd nucleus 
  
    camera {
      angle 10
      location <0,0,-2>
      look_at <0,0,0>
      }
    
    sky_sphere {
      pigment { gradient y
        color_map { 
          [0.995 color rgb <0.08,0.09,0.09>] 
          [1 color rgb <0.6,0.69,0.73>]
          }
        rotate x*90
        }
      } 
          
    fog {
      fog_type 1
      distance 200
      color rgbt <0.6,0.69,0.73, 0.4>
      }  
           
    sphere {0, 1 // nucleus
      translate <0,0,1000-1000*sqrt(sqrt(clock))>
      }
      
    
    
    #break  // end Scene 7   
    
  #case (8) //***************** Scene 8 ***********************
  // 3 seconds, frames 517 to 588 
  // Closing title and credit 
  
    camera {
      angle 50
      location <0,0,-2>
      look_at <0,0,0>
      }
      
    #declare K = clock*3;    // motion for title
      
    #declare K1 = sqrt(K);
    #declare K2 = K;
    #if (K2>1)
      #declare K2 = 1;
      #end 
      
    #declare N = clock*3-1;  // motion for credit
    #if (N<0)
      #declare N = 0;
      #end 
      
    #declare N1 = sqrt(N);
    #declare N2 = N;
    #if (N2>1)
      #declare N2 = 1;
      #end
    
    #if (clock<0.5)
      text { ttf "Tt0768m_.ttf" "MICROCOSMS" 100, 0
        texture { 
          pigment { planar
            color_map {
              [0 color rgbt 1]
              [1 color rgb 1] 
              }
            scale 60-50*K2
            rotate x*90
            #if(K1<1)
              translate z*120-120*K1
            #else
              translate z*50-50*K1
              #end
            }
          finish { ambient 1 }
          }
        scale <1,2,1>
        translate <-3.625,0,7>
        no_shadow
        }
      #end 
        
    text { ttf "Tt0768m_.ttf" " Neil Alexander 1998" 100, 0
        texture { 
          pigment { planar
            color_map {
              [0 color rgbt 1]
              [1 color rgb 1] 
              }
            scale 60-55*N2
            rotate x*90
            #if(N1<1)
              translate z*180-180*N1
            #else
              translate z*15-15*N1
              #end
            }
          finish { ambient 1 }
          }
        scale 0.625
        translate <-3.562,-0.5,7>
        no_shadow
        }
    
    #break  // end Scene 8
    

#end  // end switch scene
