#macro Position(starts,ends,locations,time) // Returns a vector at the specified locations
                                            // if 'time' is within the specified time periods.
                                            // Otherwise returns 0.
  #if (time < starts[0])
    (0*locations[0])
  #end
  #declare a = dimension_size(locations,1);
  #declare b = 0;
  #while (b < a)
    #if ((time >= starts[b]) & (time < ends[b]))
      (locations[b])
    #end
    #if (b < (a-1))
      #if ((time >= ends[b]) & (time < starts[b+1]))
        (0*locations[0])
      #end         
    #end
    #declare b=b+1;
  #end
  #if (time >= ends[a-1])
    (0*locations[0])     
  #end
#end

#macro Linear(times,locations,time) // Returns a vector that is linearly interpolated between
                                    // the given locations, according to the value of time.
                                    // Returns zero if time is outside the given bounds.
  #if (time < times[0])
    (0*locations[0])
  #end
  #declare a = dimension_size(locations,1);
  #declare b = 0;
  #while (b < (a-1))
    #if ((time >= times[b]) & (time < times[b+1]))
      #declare t0 = (time - times[b])/(times[b+1] - times[b]);
      (locations[b] + t0*(locations[b+1]-locations[b]))
    #end
    #declare b=b+1;
  #end
  #if (time >= times[a-1])
    (0*locations[0])
  #end
#end

#macro Bezier2(times,locations,look_ats,time) // Returns a vector that is interpolated by use of 
                                             // a Bezier curve.
  #if (time < times[0])
    (0*locations[0])
  #end
  #declare a = dimension_size(locations,1);
  #declare look_froms = array[a]
  #declare b = 0;
  #while (b < (a-1))
    #declare look_froms[b+1] = 2*locations[b+1] - look_ats[b+1];
    #if ((time >= times[b]) & (time < times[b+1]))
      #declare t0 = (time - times[b])/(times[b+1] - times[b]);
      ((1-t0)*(1-t0)*(1-t0)*locations[b] + 3*t0*(1-t0)*(1-t0)*look_ats[b] + 3*t0*t0*(1-t0)*look_froms[b+1] + t0*t0*t0*locations[b+1])
    #end
    #declare b=b+1;
  #end
  #if (time >= times[a-1])
    (0*locations[0])
  #end
#end

#macro Bezier(times,locations,time) // Returns a vector that is interpolated by use of 
                                             // a Bezier curve.
  #if (time < times[0])
    (0*locations[0])
  #end                           
  #declare a = dimension_size(locations,1);
  #declare look_ats = locations
  #declare look_froms = array[a]
  #declare b = 0;
  #while (b < (a-1))
    #if (b > 0)
      #declare look_ats[b] = locations[b] + ((locations[b+1] - locations[b-1])/6);
    #end                                                                          
    #if (b < (a-2))
      #declare look_ats[b+1] = locations[b+1] + ((locations[b+2] - locations[b])/6);
    #end
    #declare look_froms[b+1] = 2*locations[b+1] - look_ats[b+1];
    #if ((time >= times[b]) & (time < times[b+1]))
      #declare t0 = (time - times[b])/(times[b+1] - times[b]);
      ((1-t0)*(1-t0)*(1-t0)*locations[b] + 3*t0*(1-t0)*(1-t0)*look_ats[b] + 3*t0*t0*(1-t0)*look_froms[b+1] + t0*t0*t0*locations[b+1])
    #end
    #declare b=b+1;
  #end
  #if (time >= times[a-1])
    (0*locations[0])
  #end
#end

#macro LinearDeriv(times,locations,time) // Returns the directional derivative for linear interpolation
  #if (time < times[0])
    (0*locations[0])
  #end
  #declare a = dimension_size(locations,1);
  #declare b = 0;
  #while (b < (a-1))
    #if ((time >= times[b]) & (time < times[b+1]))
      (locations[b+1]-locations[b])
    #end
    #declare b=b+1;
  #end
  #if (time >= times[a-1])
    (0*locations[0])
  #end
#end

#macro BezierDeriv2(times,locations,look_ats,time) // Returns directional derivative for interpolation
                                                  // by Bezier curves
  #if (time < times[0])
    (0*locations[0])
  #end
  #declare a = dimension_size(locations,1);
  #declare look_froms = array[a]
  #declare b = 0;
  #while (b < (a-1))
    #declare look_froms[b+1] = 2*locations[b+1] - look_ats[b+1];
    #if ((time >= times[b]) & (time < times[b+1]))
      #declare t0 = (time - times[b])/(times[b+1] - times[b]);
      (-3*(1-t0)*(1-t0)*locations[b] + 3*(t0-1)*(3*t0-1)*look_ats[b] + 3*t0*(2-3*t0)*look_froms[b+1] + 3*t0*t0*locations[b+1])
    #end
    #declare b=b+1;
  #end
  #if (time >= times[a-1])
    (0*locations[0])
  #end
#end
                           
#macro BezierDeriv(times,locations,time) // Returns directional derivative for interpolation
                                                  // by Bezier curves
  #if (time < times[0])
    (0*locations[0])
  #end
  #declare a = dimension_size(locations,1);
  #declare look_ats = locations
  #declare look_froms = array[a]
  #declare b = 0;
  #while (b < (a-1))
    #if (b > 0)
      #declare look_ats[b] = locations[b] + ((locations[b+1] - locations[b-1])/6);
    #end                                                                          
    #if (b < (a-2))
      #declare look_ats[b+1] = locations[b+1] + ((locations[b+2] - locations[b])/6);
    #end
    #declare look_froms[b+1] = 2*locations[b+1] - look_ats[b+1];
    #if ((time >= times[b]) & (time < times[b+1]))
      #declare t0 = (time - times[b])/(times[b+1] - times[b]);
      (-3*(1-t0)*(1-t0)*locations[b] + 3*(t0-1)*(3*t0-1)*look_ats[b] + 3*t0*(2-3*t0)*look_froms[b+1] + 3*t0*t0*locations[b+1])
    #end
    #declare b=b+1;
  #end
  #if (time >= times[a-1])
    (0*locations[0])
  #end
#end 

#macro Bezier2ndDeriv2(times,locations,look_ats,time) // Returns second derivative for interpolation
                                                     // by Bezier curves
  #if (time < times[0])
    (0*locations[0])
  #end
  #declare a = dimension_size(locations,1);
  #declare look_froms = array[a]
  #declare b = 0;
  #while (b < (a-1))
    #declare look_froms[b+1] = 2*locations[b+1] - look_ats[b+1];
    #if ((time >= times[b]) & (time < times[b+1]))
      #declare t0 = (time - times[b])/(times[b+1] - times[b]);
      ((6-6*t0)*locations[b] + (18*t0-12)*look_ats[b] + (6-18*t0)*look_froms[b+1] + 6*t0*locations[b+1])
    #end
    #declare b=b+1;
  #end
  #if (time >= times[a-1])
    (0*locations[0])
  #end
#end
                           
#macro Bezier2ndDeriv(times,locations,time) // Returns second derivative for interpolation
                                                     // by Bezier curves
  #if (time < times[0])
    (0*locations[0])
  #end    
  #declare a = dimension_size(locations,1);
  #declare look_ats = locations
  #declare look_froms = array[a]
  #declare b = 0;
  #while (b < (a-1))
    #if (b > 0)
      #declare look_ats[b] = locations[b] + ((locations[b+1] - locations[b-1])/6);
    #end                                                                          
    #if (b < (a-2))
      #declare look_ats[b+1] = locations[b+1] + ((locations[b+2] - locations[b])/6);
    #end
    #declare look_froms[b+1] = 2*locations[b+1] - look_ats[b+1];
    #if ((time >= times[b]) & (time < times[b+1]))
      #declare t0 = (time - times[b])/(times[b+1] - times[b]);
      ((6-6*t0)*locations[b] + (18*t0-12)*look_ats[b] + (6-18*t0)*look_froms[b+1] + 6*t0*locations[b+1])
    #end
    #declare b=b+1;
  #end
  #if (time >= times[a-1])
    (0*locations[0])
  #end
#end
                           
#macro align(xt,yt,zt) // Returns a matrix that rotates from the coordinate axes to the
                       // given set of axes, which should be perpendicular.
                       // If <0,0,0> is provided for one axis, this axis is calculated from
                       // the other two axes.
                       // If only xt is non-zero, and it does not point straight up or down,
                       // then yt is deemed to be horizontal, and zt in a generally up direction.
                       // Something similar occurs for other axes.
  #local xtemp = xt;
  #local ytemp = yt;
  #local ztemp = zt;
  #if ((vlength(yt) = 0) & (vlength(zt) = 0) & ((xt.x != 0) | (xt.y != 0)))
    #declare yt = vcross(z,xt);
    #declare zt = vcross(xt,yt);
  #end
  #if ((vlength(zt) = 0) & (vlength(xt) = 0) & ((yt.y != 0) | (yt.x != 0)))
    #declare xt = vcross(z,yt);
    #declare zt = vcross(xt,yt);
  #end
  #if (vlength(xt) = 0)
    #declare xt = vcross(yt,zt);
    #declare yt = vcross(zt,xt); 
  #end
  #if (vlength(yt) = 0)
    #declare yt = vcross(zt,xt);
    #declare zt = vcross(xt,yt);
  #end
  #if (vlength(zt) = 0)
    #declare zt = vcross(xt,yt);
    #declare xt = vcross(yt,zt);
  #end
  #if ((vlength(xt) = 0) | (vlength(yt) = 0) | (vlength(zt) = 0))
    #declare xt = x;
    #declare yt = y;
    #declare zt = z;
  #end
  #declare xt = xt/vlength(xt);
  #declare yt = yt/vlength(yt);
  #declare zt = zt/vlength(zt);
  <xt.x,xt.y,xt.z,
   yt.x,yt.y,yt.z,
   zt.x,zt.y,zt.z,
   0,0,0>
  #declare xt = xtemp;
  #declare yt = ytemp;
  #declare zt = ztemp;
#end

#macro orbit(xt,yt,zt,time) // Returns a vector that orbits in the (yt - zt) plane.
                            // Zero vectors are treated as above.
                            // Vectors are not normalised, however.
                            // 1 unit of time is 1 revolution.
  #if ((vlength(yt) = 0) & (vlength(zt) = 0) & ((xt.x != 0) | (xt.y != 0)))
    #declare yt = vcross(z,xt);
    #declare zt = vcross(xt,yt);
  #end
  #if (vlength(xt) = 0)
    #declare xt = vcross(yt,zt);
    #declare yt = vcross(zt,xt); 
  #end
  #if (vlength(yt) = 0)
    #declare yt = vcross(zt,xt);
    #declare zt = vcross(xt,yt);
  #end
  #if (vlength(zt) = 0)
    #declare zt = vcross(xt,yt);
    #declare xt = vcross(yt,zt);
  #end
  (cos(2*pi*time)*yt + sin(2*pi*time)*zt)
#end

#macro ripple(start,position,height,time)
  #if ((time >= start) & (time < start+2))
    [
      wood
      translate position
      ramp_wave
      frequency 0.005
      normal_map {
        [0.0 wood height*(start+2 - time)/2 frequency 0.05 sine_wave phase ((start+2-time)/2)*10]
        [(time - start)/2 bozo 0.0]
        [1.0 bozo 0.0]
      }  
    ]  
  #end
#end   

#macro TestObject(bez,bezder,bez2ndder,acc_scale,obj_scale)
  union {
    sphere
    {
      <0, 0, 0>
      0.75
      texture {
        pigment {rgbt <1, 1-mod(step,1), mod(step,1), 0>}    
        finish {ambient 0.3}
      }
      scale obj_scale
    }
    cylinder
    {
      0*x,  3*x,  0.25
      texture {
        pigment {rgbt <1, 0, 0, 0>}
        finish {ambient 0.3}
      }
      scale obj_scale
      matrix align(bezder,<0,0,0>,<0,0,0>)
    }
    cylinder
    {
      0*y,  3*y,  0.25
      texture {
        pigment {rgbt <0, 1, 0, 0>}
        finish {ambient 0.3}
      }
      scale<1,
      #if (vlength(bez2ndder) > 0)
        vlength(bez2ndder)*acc_scale,
      #else
        0.000000001,
      #end
      1>
      scale obj_scale
      matrix align(<0,0,0>,bez2ndder,<0,0,0>)
    }
    cylinder
    {
      0*z,  3*z,  0.25
      texture {
        pigment {rgbt <0, 0, 1, 0>}
        finish {ambient 0.3}
      }
      scale obj_scale
    }
    cone
    {
      4*x,  0.0,
      3*x, 0.5
      texture {
        pigment {rgbt <1, 0, 0, 0>}
        finish {ambient 0.3}
      }
      scale obj_scale
      matrix align(bezder,<0,0,0>,<0,0,0>)
    }
    cone
    {
      4*y,  0.0,
      3*y, 0.5
      texture {
        pigment {rgbt <0, 1, 0, 0>}
        finish {ambient 0.3}
      }
      scale<1,
      #if (vlength(bez2ndder) > 0)
        vlength(bez2ndder)*acc_scale,
      #else
        0.000000001,
      #end
      1>
      scale obj_scale
      matrix align(<0,0,0>,bez2ndder,<0,0,0>)
    }
    cone
    {
      4*z,  0.0,
      3*z, 0.5
      texture {
        pigment {rgbt <0, 0, 1, 0>}
        finish {ambient 0.3}
      }
      scale obj_scale
    }
    translate bez
  }
#end

#macro TestLinear(times,locations,steps,obj_scale)
  #local step = times[0] + 0.000000001;
  #local k = dimension_size(times,1);
  #while (step < times[k-1])
    TestObject((Linear(times,locations,step)),(LinearDeriv(times,locations,step)),vcross(z,vnormalize((LinearDeriv(times,locations,step)))),1,obj_scale)
    #local step = step + (times[k-1] - times[0])/steps; 
  #end
#end

#macro TestBezier(times,locations,steps,acc_scale,obj_scale)
  #local step = times[0] + 0.000000001;
  #local k = dimension_size(times,1);
  #while (step < times[k-1])
    TestObject((Bezier(times,locations,step)),(BezierDeriv(times,locations,step)),(Bezier2ndDeriv(times,locations,step)),acc_scale,obj_scale)
    #local step = step + (times[k-1] - times[0])/steps; 
  #end
#end

#macro TestBezier2(times,locations,look_ats,steps,acc_scale,obj_scale)
  #local step = times[0] + 0.000000001;
  #local k = dimension_size(times,1);
  #while (step < times[k-1])
    TestObject((Bezier2(times,locations,look_ats,step)),(BezierDeriv2(times,locations,look_ats,step)),(Bezier2ndDeriv2(times,locations,look_ats,step)),acc_scale,obj_scale)
    #local step = step + (times[k-1] - times[0])/steps; 
  #end
#end

