/*
 *  This program copyright 2000 Mark Wagner
 *
 * This file contains all functions that are not linked to a VCL component
 */

//---------------------------------------------------------------------------
#include <vcl.h>
#include <stdio.h>
#include <stdlib.h>
#pragma hdrstop
#include "gal_main.h"
#include "internal.h"
#include "globals.h"
//---------------------------------------------------------------------------
#pragma package(smart_init)

HBITMAP ShipIconBitmap = NULL;
HINSTANCE AppInst;
HDC IconDC = NULL;

/* Status file format:
 *  int FrameType
 *  Frame data
 *
 *  Frame data:
 *   Frame type 1: normal frame
 *    int score
 *    int high_score
 *    int Level_num
 *    int Num_Comp_Ships
 *    Comp ship data
 *     int x
 *     int y
 *     int type
 *     int status
 *    Player ship data
 *     int x
 *     int y
 *     int status
 *    int Num_Missiles
 *     Missile data
 *     int x
 *     int y
 *     int type
 *    Status can be:
 *      1: Normal facing
 *      2: Dive phase 1 (up and right)
 *      3: Dive phase 1 (up and left)
 *      4: Dive phase 2 (right)
 *      5: Dive phase 2 (left)
 *      6: Dive phase 3 (down and right)
 *      7: Dive phase 3 (down and left)
 *      8: Dive phase 4 (down)
 *      9-12: Explosion frame
 *   Frame type 2:
 *    Level change
 *   Frame type 3:
 *     Game over
 */

bool WriteStatus(char *Filename)
{
        FILE *outfile;
        outfile = fopen(Filename, "a");
        fprintf(outfile,"\t#case(%d)\n", TheGame.total_time);
        /* Output the ships */
        for(int count = 0; count < TheGame.num_ships;count++)
        {
                fprintf(outfile, "\t\tobject{Ships[%d][%d] translate <%d,0,%d>*feet}\n", TheGame.cships[count].type, TheGame.cships[count].status, TheGame.cships[count].x, TheGame.cships[count].y);
        }
        /* Output the missiles */
        for(int count = 0; count < TheGame.num_missiles; count++)
        {
                fprintf(outfile, "\t\tobject{Missile[%d] translate <%d,0,%d>*feet}\n", (TheGame.missiles[count].dir+2), TheGame.missiles[count].x, TheGame.missiles[count].y);
        }
        /* Output the player */
        fprintf(outfile, "\t\tobject{Ships[0][%d] translate <%d,0,%d>*feet}\n", TheGame.player.status, TheGame.player.x, TheGame.player.y);
        /* Output the game status */
        fprintf(outfile, "\t\t#declare Score = %d;\n", TheGame.score);
        fprintf(outfile, "\t\t#declare Level = %d;\n", TheGame.level);
        fprintf(outfile, "\t\t#declare HighScore = %d;\n", HighScore);
        fprintf(outfile, "\t\t#declare Status = %d;\n", TheGame.status);
        fprintf(outfile, "\t#break\n");
        fclose(outfile);
        return true;
}

bool InitIcons(HDC FormDC)
{
        /* Create a DC for storing the icon bitmap */
        IconDC = CreateCompatibleDC(FormDC);
        if(NULL == IconDC)
                return false;
        /* Load the icon bitmap */
        ShipIconBitmap = LoadBitmap(AppInst, "SHIPICONS");
        if(NULL == ShipIconBitmap)
        {
                show_win_error_message("Error loading bitmap", 0);
                DeleteDC(IconDC);
                return false;
        }
        if(NULL == SelectObject(IconDC, ShipIconBitmap))
        {
                DeleteObject(ShipIconBitmap);
                DeleteDC(IconDC);
                return false;
        }
        return true;
}

void DeleteIcons()
{
        if(ShipIconBitmap != NULL)
                DeleteObject(ShipIconBitmap);
        if(IconDC != NULL)
                DeleteDC(IconDC);
}

void Init_Game()
{
        TheGame.score = 0;
        TheGame.level = 1;
        TheGame.status = PLAYING;
        TheGame.stat_time = 0;
        TheGame.total_time = 0;
        TheGame.player.x = 144;
        TheGame.player.y = 165;
        TheGame.player.status = 1;
        TheGame.player.lives = 3;
        TheGame.player.next_life_score = 5000;
        Initialize_Ships();
        /* Let the games begin! */
        Form1->Timer1->Enabled = true;
}

void Initialize_Ships()
{
        TheGame.num_missiles = 0;
        TheGame.num_ships = 24;
        TheGame.move_direction = 1;
        for(int count = 0; count < 3; count++)
        {
                /* Initialize comp ships */
                for(int count2 = 0; count2 < 8; count2++)
                {
                        TheGame.cships[count2+8*count].type = count+1;
                        TheGame.cships[count2+8*count].status = (rand() % 3 + 1);
                        TheGame.cships[count2+8*count].y = count*(SHIP_HEIGHT + 3)+8;
                        TheGame.cships[count2+8*count].x = count2*(SHIP_WIDTH + 4)+50;
                }
        }
}

void Human_Fire()
{
        if(TheGame.status != PLAYING)
                return;
        for(int count = 0; count < TheGame.num_missiles;count++)
        {
                if(TheGame.missiles[count].dir == -1) /* Human already fired */
                        return;
        }
        TheGame.missiles[TheGame.num_missiles].dir = -1;
        TheGame.missiles[TheGame.num_missiles].x = TheGame.player.x+16;
        TheGame.missiles[TheGame.num_missiles].y = TheGame.player.y;
        TheGame.num_missiles++;
}

void Destroy_Missile(int missile_num)
{
        TheGame.num_missiles--;
        TheGame.missiles[missile_num].x = TheGame.missiles[TheGame.num_missiles].x;
        TheGame.missiles[missile_num].y = TheGame.missiles[TheGame.num_missiles].y;
        TheGame.missiles[missile_num].dir = TheGame.missiles[TheGame.num_missiles].dir;
}

void Update_Missiles()
{
        for(int count = 0;count < TheGame.num_missiles;count++)
        {
                TheGame.missiles[count].y += MISSILE_MOVE_SPEED*TheGame.missiles[count].dir;
                /* Collision detection -- human missile*/
                if(-1 == TheGame.missiles[count].dir)
                {
                        for(int count2 = 0;count2 < TheGame.num_ships;count2++)
                        {
                                if(TheGame.cships[count2].status > 8) /* Can't shoot a ship twice */
                                        continue;
                                if(TheGame.missiles[count].x > TheGame.cships[count2].x && TheGame.missiles[count].x < TheGame.cships[count2].x + SHIP_WIDTH)
                                {
                                        if(TheGame.missiles[count].y > TheGame.cships[count2].y && TheGame.missiles[count].y < TheGame.cships[count2].y + SHIP_HEIGHT)
                                        {
                                                Destroy_Missile(count);
                                                TheGame.cships[count2].status = 9;
                                                TheGame.score += ((4 - TheGame.cships[count2].type) * 50);
                                                continue;
                                        }
                                }
                        }
                }
                /* Collision detection -- comp. missile */
                else
                {
                        if(TheGame.status == PLAYING && TheGame.missiles[count].x > TheGame.player.x && TheGame.missiles[count].x < TheGame.player.x + SHIP_WIDTH)
                        {
                                if(TheGame.missiles[count].y > TheGame.player.y && TheGame.missiles[count].y < TheGame.player.y + SHIP_HEIGHT)
                                {
                                        Destroy_Missile(count);
                                        TheGame.player.status = 7;
                                        TheGame.status = DYING;
                                        TheGame.stat_time = 0;
                                        continue;
                                }
                        }
                }
                /* Off screen */
                if(TheGame.missiles[count].y > BOARD_HEIGHT || TheGame.missiles[count].y < 0)
                {
                        Destroy_Missile(count);
                }
        }
}

void Update_Computer()
{
        int form_min = BOARD_WIDTH;
        int form_max = 0;
        if(TheGame.status == NEXTLEVEL)
                return;
        /* Find formation edges */
        for(int count = 0; count < TheGame.num_ships; count++)
        {
                if((TheGame.cships[count].x + SHIP_WIDTH) > form_max)
                        form_max = TheGame.cships[count].x + SHIP_WIDTH;
                if(TheGame.cships[count].x < form_min)
                        form_min = TheGame.cships[count].x;
        }
        /* If about to hit edge of board, bounce */
        if((form_min < 5 && TheGame.move_direction == -1) || (form_max > (BOARD_WIDTH-5) && TheGame.move_direction == 1))
        {
                TheGame.move_direction *= -1;
        }
        /* Update ship positions and statuses */
        for(int count = 0; count < TheGame.num_ships; count++)
        {
                /* Normal status */
                if(TheGame.cships[count].status <= 3)
                {
                        TheGame.cships[count].x += COMP_MOVE_SPEED * TheGame.move_direction;
                        TheGame.cships[count].status = ((TheGame.cships[count].status + 1) % 3) + 1;
                        /* Check to drop bomb -- rate independant of number of ships */
                        if(TheGame.status == PLAYING && TheGame.num_missiles < 40 && !(rand() % (17 * TheGame.num_ships)))
                        {
                                TheGame.missiles[TheGame.num_missiles].x = TheGame.cships[count].x + (SHIP_WIDTH/2);
                                TheGame.missiles[TheGame.num_missiles].y = TheGame.cships[count].y + SHIP_HEIGHT;
                                TheGame.missiles[TheGame.num_missiles].dir = 1;
                                TheGame.num_missiles++;
                        }
                }
                /* Ship exploding */
                if(TheGame.cships[count].status > 8)
                {
                        TheGame.cships[count].status ++;
                        if(TheGame.cships[count].status > 12)
                        {
                                TheGame.num_ships--;
                                TheGame.cships[count].x = TheGame.cships[TheGame.num_ships].x;
                                TheGame.cships[count].y = TheGame.cships[TheGame.num_ships].y;
                                TheGame.cships[count].type = TheGame.cships[TheGame.num_ships].type;
                                TheGame.cships[count].status = TheGame.cships[TheGame.num_ships].status;
                        }
                }
        }
}

void Update_Player()
{
        /* Check status -- if exploding, update explosion and return */
        if(TheGame.player.status >= 7/* Explosion 1 */ && TheGame.player.status < 12)
        {
                TheGame.player.status++;
        }
        if(TheGame.status == PLAYING || TheGame.status == NEXTLEVEL)
        {
                if(left_down && TheGame.player.x > 0)
                        TheGame.player.x -= PLAYER_MOVE_SPEED;
                if(right_down && (TheGame.player.x + SHIP_WIDTH) < BOARD_WIDTH)
                        TheGame.player.x += PLAYER_MOVE_SPEED;
        }
}

void DrawGame(HDC FormDC)
{
        /* Clear screen */
        BitBlt(FormDC, 0, 0, BOARD_WIDTH, BOARD_HEIGHT, IconDC, 0, 0, WHITENESS);
        /* Draw missiles */
        for(int count = 0; count < TheGame.num_missiles; count++)
        {
                BitBlt(FormDC, TheGame.missiles[count].x, TheGame.missiles[count].y, MISSILE_WIDTH, MISSILE_HEIGHT, IconDC, SHIP_WIDTH*3-1, 0, SRCCOPY);
        }
        /* Draw human ship */
        BitBlt(FormDC,TheGame.player.x, TheGame.player.y, SHIP_WIDTH,SHIP_HEIGHT,IconDC,(TheGame.player.status-1)*SHIP_WIDTH,0,SRCCOPY);
        /* Draw computer ships */
        for(int count = 0; count < TheGame.num_ships; count++)
        {
                BitBlt(FormDC,TheGame.cships[count].x, TheGame.cships[count].y, SHIP_WIDTH,SHIP_HEIGHT, IconDC, (TheGame.cships[count].status-1)*SHIP_WIDTH, TheGame.cships[count].type*SHIP_HEIGHT, SRCCOPY);
        }
        /* Draw score, highscore, ships_left */
}

/* Update game status */
void Update_Status()
{
        TheGame.total_time++;
        if(TheGame.status == PLAYING)
        {
                if(TheGame.score > TheGame.player.next_life_score)
                {
                        TheGame.player.lives++;
                        TheGame.player.next_life_score += 5000;
                }
                if(TheGame.num_ships == 0)
                {
                        TheGame.status = NEXTLEVEL;
                        TheGame.stat_time = 0;
                }
        }
        else if(TheGame.status == DYING)
        {
                TheGame.stat_time++;
                if(TheGame.stat_time == 1)
                {
                        TheGame.player.lives--;
                }
                if(TheGame.stat_time == 30)
                {
                        if(TheGame.player.lives == 0)
                        {
                                TheGame.status = GAMEOVER;
                        }
                        else
                        {
                                TheGame.status = PLAYING;
                                TheGame.player.status = 1;
                        }
                }
        }
        else if(TheGame.status == NEXTLEVEL)
        {
                TheGame.stat_time++;
                if(TheGame.stat_time == 1)
                {
                        TheGame.level++;
                        Initialize_Ships();
                }
                if(TheGame.stat_time == 30)
                {
                        TheGame.status = PLAYING;
                }
        }
        else if(TheGame.status == GAMEOVER)
        {
                /* End the game */
                Form1->Timer1->Enabled = false;
                /* Clear the screen */

                /* Show the GameOver message */

                /* Update the high score */
                if(TheGame.score > HighScore)
                        HighScore = TheGame.score;
        }
        else
        {
//                show_error_msg("Unknown game status: %d", TheGame.status);
        }
}


int show_win_error_message(char *message_title,UINT message_style)
{
    LPSTR MsgBuffer;
    DWORD ErrorCode = GetLastError();
    FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER|FORMAT_MESSAGE_FROM_SYSTEM,NULL,ErrorCode,0,(LPTSTR)&MsgBuffer,128,NULL);
    return MessageBox(NULL, MsgBuffer, message_title,message_style|MB_APPLMODAL);
}

void show_error_msg(int err_number, char *message_string, char *message_title)
{
    char *error_message = NULL;
    char *error_buffer = NULL;
    char *temp_ptr = NULL;

    if(err_number == 0)
    {
    }
    else
    {
        if(message_string != NULL)
        {
            // Remove any trailing '\n' from the error string
            error_buffer = new char[strlen(strerror(err_number))+1];
            strcpy(error_buffer, strerror(err_number));
            temp_ptr = strrchr(error_buffer, '\n');
            if((temp_ptr != NULL) && ('\0' == *(temp_ptr+1)))
            {
                *temp_ptr = '\0';
            }
            //Allocate a message buffer
            error_message = new char[strlen(message_string)+strlen(error_buffer)+1];
            sprintf(error_message, message_string, error_buffer);
        }
        else
        {
            // Remove any trailing '\n' from the error string
            error_buffer = new char[strlen(strerror(err_number))+1];
            strcpy(error_buffer, strerror(err_number));
            temp_ptr = strrchr(error_buffer, '\n');
            if((temp_ptr != NULL) && ('\0' == *(temp_ptr+1)))
            {
                *temp_ptr = '\0';
            }
            //Allocate a message buffer
            error_message = new char[strlen(message_string)+strlen(error_buffer)+1];
            sprintf(error_message, "Error %d (%s) occured", err_number, strerror(err_number));
        }
    }
    if(NULL == message_title)
    {
        strcpy(message_title, "Error");
    }
    MessageBox (NULL, error_message, message_title, MB_ICONWARNING);
    if(error_message != NULL)
        delete [] error_message;
    if(error_buffer != NULL)
        delete [] error_buffer;
}



