#include "colors.inc"
#include "metals.inc"
#include "textures.inc"

#declare R = seed(54);

light_source {
   <17.4, 19.0, 5.80>
   colour White*0.15
}

light_source {
   <-17.4, 19.0, 5.80>
   colour White*0.15
}

light_source {
   <-17.4, 19.0, -5.80>
   colour White*0.15
}

light_source {
   <17.4, 19.0, -5.80>
   colour White*0.15
}

light_source {
   <5.80, 19.0, -5.80>
   colour White*0.15
}

light_source {
   <-5.80, 19.0, -5.80>
   colour White*0.15
}

light_source {
   <5.80, 19.0, 5.80>
   colour White*0.15
}

light_source {
   <-5.80, 19.0, 5.80>
   colour White*0.15
}

camera {
   //location <13.0, 13.0, -3.00> // "in box" conveyor view
   //look_at <20.0, 8.00, 2.50>
   location <-13.0, 4.00, -8.0> // "out box" conveyor and connecting tube view
   look_at <-13.0, 2.00, 5.00>
   //location <3.00, 10.0, -10.0>   // window view
   //look_at <3.00, 6.00, 10.0>
}

// texture for rubber objects
#declare Rubber = texture {
   pigment { colour rgb < 0.35, 0.35, 0.35 > }
   finish {
      ambient 0.3
      specular 0.5
      roughness 0.01
   }
   normal {
      bumps 1.0
      scale 0.01
   }
   scale 5
}

// texture for painted metal (chipped paint)
#declare Chipped_Paint = texture {
   pigment {
      spotted
      colour_map {
         [0.00 colour rgbt <1, 1, 1, 1>]
         [0.10 colour rgbt <1, 1, 1, 1>]
         [0.11 colour Red ]
         [1.00 colour Red ]
      }
   }
   finish {
      metallic 0.5
      ambient 0.3
      phong 0.5
      phong_size 0.2
   }
   scale <0.05, 0.40, 1>    // the z scale controls the pattern of scratches
}

// texture for concrete objects
#declare Concrete = texture {
   pigment {
      agate
      turbulence 2.0
      colour_map {
         [0.00 colour rgb <0.35, 0.35, 0.35> ]
         [0.10 colour rgb <0.35, 0.35, 0.35> ]
         [0.20 colour rgb <0.50, 0.50, 0.50> ]
         [0.60 colour rgb <0.50, 0.50, 0.50> ]
         [0.70 colour rgb <0.65, 0.65, 0.65> ]
         [0.75 colour rgb <0.65, 0.65, 0.65> ]
         [0.80 colour rgb <0.50, 0.50, 0.50> ]
         [1.00 colour rgb <0.50, 0.50, 0.50> ]
      }
   }
   normal {
      granite
      turbulence 2.0
   }
   scale 1.0
   finish { 
      reflection 0.0
      diffuse 1.0
   }
}

// texture for steel objects
#declare Steel = texture {
   pigment { colour rgb <0.90, 0.90, 1.00 > } 
   finish {
      ambient 0.4
      diffuse 0.1
      reflection 0.05
      specular 0.1
      roughness 10
   }
}

// bright white texture
#declare Whitewashed = texture {
   pigment { colour White }
   finish {
      ambient 1.0
      diffuse 0.0
      reflection 0.0
   }
}

// Creates a regular prism object
// Parameters:
//    Sides:  The number of sides of the base polygon
//    Height1 and Height2:  The two y-coordinates that form the ends of the prism object
//    Radius:  The distance from the centre of the prism to its corners
//    Rotate:  Rotation vector for the prism
//    Translate:  Translation vector for the prism
#macro Regular_Prism( Sides, Height1, Height2, Radius, Rotate, Translate )
   prism {
      linear_spline
      Height1, Height2, Sides + 1,
      // the first points are placed by a while loop            
         #declare Angle = 0.0;
         #while ( Angle < 2*pi - 0.00001 )   // for some reason, just < 2*pi doesn't work properly
            <Radius * cos( Angle ), Radius * sin( Angle )>,
            #declare Angle = Angle + 2*pi / Sides;
         #end
      <Radius, 0.0>  // the closing point ( not in loop due to the lack of end comma )                                            
      rotate Rotate
      translate Translate
   }
#end

// Handle for the door
#declare Handle = union {
   cylinder { <0.00, 0.00, 0.20>, <0.00, 0.00, 0.50>, 0.15 }   // cylinder connecting to door
   box { <-1.10, -0.20, 0.00>, <0.25, 0.20, 0.20> }   // body of handle
   // end of handle
   difference {
      cylinder { <0, -0.20, 0>, <0, 0.20, 0>, 0.40
         scale <0.75, 1.00, 1.00>
         translate <-1.10, 0.00, 0.40>
      }
      cylinder { <0, -0.21, 0>, <0, 0.21, 0>, 0.30
         scale <1.00, 1.00, 2/3>
         translate <-1.10, 0.00, 0.40>
      }
      plane { -x, 1.10 }
      plane { -z, -0.30 }
   } 
}    

// indentation for door handle
#declare Indentation = union {
   // left side (quarter circle and smoothing torus)
   intersection {
      union {
         cylinder { <0.0, 0.0, 0.0>, <0.0, 0.0, 0.5>, 1.5 }
         torus { 1.5, 0.5
            rotate x*90
         }
      }
      box { <-2.0, -2.0, 0.0>, <0.0, 0.0, 0.5> }   // take bottom left section of union
   }
   // right side (scaled quarter circle and smoothing torus)
   intersection {
      union {
         cylinder { <0.0, 0.0, 0.0>, <0.0, 0.0, 0.5>, 1.5
            scale <0.50, 1.00, 1.00>
         }
         torus { 1.5, 0.5
            rotate x*90
            scale <0.50, 1.00, 1.00>
         }
      }
      box { <0.0, -2.0, 0.0>, <1.25, 0.0, 0.5> }   // take bottom right section of union
   }
   // smooth out the top (anything extra in -z is not cut out - the object is to be used a difference where it won't matter)
   union {
      sphere { <-1.5, 0.0, 0.0>, 0.5 } // left sphere
      sphere { <0.0, 0.0, 0.0>, 0.5    // right sphere
         scale <0.50, 1.00, 1.00>
         translate <0.75, 0.0, 0.0>
      }
      cylinder { <-1.5, 0.0, 0.0>, <0.75, 0.0, 0.0>, 0.5 }  // connecting cylinder
   }
}

// A 10x17x2 steel door.  A corner of the door is on the origin, and the door is contained in the first octant.
// The door has no bottom section, since that is already part of the wall.
#declare Door = union {
   difference {
      // body of door
      union {
         box { <1.00, 0.00, 0.20>, <9.00, 16.0, 1.80> }  // central section
         box { <0.00, 16.0, 0.00>, <10.0, 17.0, 2.00> }  // top section
         box { <0.00, 0.00, 0.00>, <1.00, 16.0, 2.00> }  // left section
         box { <9.00, 0.00, 0.00>, <10.0, 16.0, 2.00> }  // right section
         // smooth out outside edges ( inside edges remain rough as this is the door-wall interface
         sphere { <0.00, 0.00, 0.20>, 0.20 }
         sphere { <0.00, 0.00, 1.80>, 0.20 }
         cylinder { <0.00, 0.00, 0.20>, <0.00, 17.0, 0.20>, 0.20 }
         cylinder { <0.00, 0.00, 1.80>, <0.00, 17.0, 1.80>, 0.20 }
         sphere { <0.00, 17.0, 0.20>, 0.20 }
         sphere { <0.00, 17.0, 1.80>, 0.20 }
         cylinder { <0.00, 17.0, 0.20>, <10.0, 17.0, 0.20>, 0.20 }
         cylinder { <0.00, 17.0, 1.80>, <10.0, 17.0, 1.80>, 0.20 }
         sphere { <10.0, 17.0, 0.20>, 0.20 }
         sphere { <10.0, 17.0, 1.80>, 0.20 }
         cylinder { <10.0, 17.0, 0.20>, <10.0, 0.00, 0.20>, 0.20 }
         cylinder { <10.0, 17.0, 1.80>, <10.0, 0.00, 1.80>, 0.20 }
         sphere { <10.0, 0.00, 0.20>, 0.20 }
         sphere { <10.0, 0.00, 1.80>, 0.20 }
         // bolts around the edges
         // while loop for bolts on vertical edges
            #declare Y = 0.5;
            #while ( Y < 17.0 )
               Regular_Prism ( 6, -1.20, 1.20, 0.20, <90, 0.0, 0.0>, <0.50, Y, 1.00> )
               Regular_Prism ( 6, -1.20, 1.20, 0.20, <90, 0.0, 0.0>, <9.50, Y, 1.00> )
               #declare Y = Y + 1;
            #end
         // while loop for horizontal edges
            #declare X = 1.5;
            #while ( X < 9.0 )
               Regular_Prism ( 6, -1.20, 1.20, 0.20, <90, 0.0, 0.0>, <X, 16.5, 1.00> )
               #declare X = X + 1;
            #end
      }
      object { Indentation
         scale 0.75
         translate <3.0, 8.5, 0.19>
      }
   }
   object { Handle
      scale 0.75
      translate <3.0, 8.4, 0.19>
      texture { Silver_Texture }
      texture { Chipped_Paint scale <1.0, 1.0, rand(R)> }
   }
   // this box is to eliminate a texture problem due to coincident surfaces
   box { <1.00, 7.00, 0.565>, <4.00, 10.0, 1.00> }
   texture { Steel }
}
// A steel section of 20x20x2 wall.  A corner of the wall is on the origin, and the wall is contained in the first octant.
#declare Wall_Piece = union {
   box { <1.00, 1.00, 0.20>, <19.0, 19.0, 1.80> }  // central section
   box { <0.00, 0.00, 0.00>, <20.0, 1.00, 2.00> }  // bottom section
   box { <0.00, 19.0, 0.00>, <20.0, 20.0, 2.00> }  // top section
   box { <0.00, 1.00, 0.00>, <1.00, 19.0, 2.00> }  // left section 
   box { <19.0, 1.00, 0.00>, <20.0, 19.0, 2.00> }  // right section
   // smooth out cuts
   sphere { <1.00, 1.00, 0.20>, 0.20 }
   sphere { <1.00, 1.00, 1.80>, 0.20 }
   cylinder { <1.00, 1.00, 0.20>, <1.00, 19.0, 0.20>, 0.20 }
   cylinder { <1.00, 1.00, 1.80>, <1.00, 19.0, 1.80>, 0.20 }
   sphere { <1.00, 19.0, 0.20>, 0.20 }
   sphere { <1.00, 19.0, 1.80>, 0.20 }
   cylinder { <1.00, 19.0, 0.20>, <19.0, 19.0, 0.20>, 0.20 }
   cylinder { <1.00, 19.0, 1.80>, <19.0, 19.0, 1.80>, 0.20 }
   sphere { <19.0, 19.0, 0.20>, 0.20 }
   sphere { <19.0, 19.0, 1.80>, 0.20 }
   cylinder { <19.0, 19.0, 0.20>, <19.0, 1.00, 0.20>, 0.20 }
   cylinder { <19.0, 19.0, 1.80>, <19.0, 1.00, 1.80>, 0.20 }
   sphere { <19.0, 1.00, 0.20>, 0.20 }
   sphere { <19.0, 1.00, 1.80>, 0.20 }
   cylinder { <19.00, 1.00, 0.20>, <1.00, 1.00, 0.20>, 0.20 }
   cylinder { <19.00, 1.00, 1.80>, <1.00, 1.00, 1.80>, 0.20 }
   // bolts around the edges
   // while loop for bolts on vertical edges
      #declare Y = 0.5;
      #while ( Y < 20.0 )
         Regular_Prism ( 6, -1.20, 1.20, 0.20, <90, 0.0, 0.0>, <0.50, Y, 1.00> )
         Regular_Prism ( 6, -1.20, 1.20, 0.20, <90, 0.0, 0.0>, <19.50, Y, 1.00> )
         #declare Y = Y + 1;
      #end
   // while loop for horizontal edges
      #declare X = 1.5;
      #while ( X < 19.0 )
         Regular_Prism ( 6, -1.20, 1.20, 0.20, <90, 0.0, 0.0>, <X, 0.50, 1.00> )
         Regular_Prism ( 6, -1.20, 1.20, 0.20, <90, 0.0, 0.0>, <X, 19.5, 1.00> )
         #declare X = X + 1;
      #end
   texture { Steel }
}

// Shaft for conveyor belt, face of wall is at x=0
#declare Shaft = difference {
   union {
      // shaft box
      box { <-0.20, -3.00, -3.50>, <15.0, 3.00, 3.50> }
      // smoothing for wall
      sphere { <0.00, -3.00, -3.50>, 0.20 }                       // bottom near
      cylinder { <0.00, -3.00, -3.50>, <0.00, 3.00, -3.50>, 0.20 }
      sphere { <0.00, 3.00, -3.50>, 0.20 }                        // top near
      cylinder { <0.00, 3.00, -3.50>, <0.00, 3.00, 3.50>, 0.20 }
      sphere { <0.00, 3.00, 3.50>, 0.20 }                         // top far
      cylinder { <0.00, -3.00, 3.50>, <0.00, 3.00, 3.50>, 0.20 }
      sphere { <0.00, -3.00, 3.50>, 0.20 }                        // bottom far
      cylinder { <0.00, -3.00, -3.50>, <0.00, -3.00, 3.50>, 0.20 }
      // while loop for vertical bolts
      #declare Y = -2.50;
      #while ( Y < 3.00 )
         Regular_Prism ( 6, 0.00, 0.20, 0.20, <0.00, 0.00, 90.0>, <-0.20, Y, -3.00> )
         Regular_Prism ( 6, 0.00, 0.20, 0.20, <0.00, 0.00, 90.0>, <-0.20, Y, 3.00> )
         #declare Y = Y + 1;
      #end
      // while loop for depth (z) bolts
      #declare Z = -2.00;
      #while ( Z < 2.50 )
         Regular_Prism ( 6, 0.00, 0.20, 0.20, <0.00, 0.00, 90.0>, <-0.20, -2.50, Z> )
         Regular_Prism ( 6, 0.00, 0.20, 0.20, <0.00, 0.00, 90.0>, <-0.20, 2.50, Z> )
         #declare Z = Z + 1;
      #end
   }
   // cut out centre
   box { <-0.30, -2.00, -2.50>, <15.1, 2.00, 2.50 > }
   texture { Steel }
}
      
#declare Room = union {
   object { Wall_Piece translate <-20.0, 0.0, 10.0> } // front left
   object { Wall_Piece translate <0.0, 0.0, 10.0> }   // front right
   difference {                                                    
      object { Wall_Piece rotate y*90 translate <20.0, 0.0, 10.0> } // right
      box { <19.9, 8.50, 0.00>, <22.1, 12.5, 5.00> } // hole for "in box" conveyor shaft
   }
   object { Wall_Piece translate <0.0, 0.0, -12.0> }  // back right
   object { Wall_Piece translate <-20.0, 0.0, -12.0> }   // back left
   difference {
      object { Wall_Piece rotate -y*90 translate <-20.0, 0.0, -10.0> }  // left
      box { <-22.1, 0.00, 0.00>, <-19.5, 4.00, 5.00> } // hole for "out box" conveyor belt
   }
   object { Door translate <5.00, 1.00, 10.0> } // door
   object { Shaft translate <20.0, 10.5, 2.50> } // "in box" conveyor shaft
   object { Shaft rotate y*180 translate <-20.0, 2.00, 2.50> }
   box { <-22.0, -2.0, -12.0>, <22.0, 0.0, 12.0> } // floor
   box { <-22.0, 20.0, -12.0>, <22.0, 22.0, 12.0> } // ceiling
   texture { Concrete }
}

// Creates an animated conveyor belt
// Parameters:
//    Speed:  angular velocity of the end of the conveyor belt, in radians per clock
//    Translate:  translation of the belt
//    Rotate:  rotation of the belt
#macro Conveyor_Belt ( Speed, Rotate, Translate )
union {   
   // the belt
   union {
      // rounded portion
      difference {
         union {
            cylinder { <0.0, 0.0, -2.0>, <0.0, 0.0, 2.0>, 1.0 }   // starting cylinder
            // loop creates raised sections at invervals of pi/4
            #declare Pos = 0.0;
            #while ( Pos < 2*pi )
               difference {
                  cylinder { <0.0, 0.0, -2.0>, <0.0, 0.0, 2.0>, 1.05 }  // slightly bigger cylinder
                  plane { < sin( Pos ), -cos( Pos ), 0.0>, 0 }
                  plane { < -sin( Pos + pi/4 ), cos( Pos + pi/4 ), 0.0>, 0 }
               }
               #declare Pos = Pos + pi/2;
            #end
            // texturing, etc.
            texture { Rubber }
            rotate z*Speed*180/pi*clock    // animation
            clipped_by {  box { <-1.05, -1.05, -2.0>, <0.0, 1.05, 2.0> } }
            bounded_by { clipped_by }
         }
         plane { -x, 0 }                                       // cut off half
         cylinder { <0.0, 0.0, -2.1>, <0.0, 0.0, 2.1>, 0.75 }  // cut out inside section
         bounded_by { box { <-1.05, -1.05, -2.0>, <0.0, 1.05, 2.0> } }
      }
      // the top section
      difference {
         box { <0.0, 0.75, -2.0>, <100.0, 1.05, 2.0> }
         // loop cuts out sections at intervals of pi/4 units
         #declare Pos = 0.0;
         #while ( Pos < 100.0 )
            box { < Pos, 1.0, -2.1>, < Pos + pi/4, 1.06, 2.1> }
            #declare Pos = Pos + pi/2;
         #end
         // texturing, etc.
         texture { Rubber }
         translate <-Speed*clock, 0.0, 0.0>      // animation
         clipped_by { box { <0.0, 0.75, -2.0>, <14.0, 1.05, 2.0> } }
         bounded_by { clipped_by }
      }
      // the bottom section
      difference {
         box { <-100.0, -1.05, -2.0>, <14.0, -0.75, 2.0> } // bottom
         // loop cuts out sections at intervals of pi/4 units
         #declare Pos = pi/4 - 32*pi;
         #while ( Pos < 14.0 )
            box { <Pos, -1.06, -2.1>, <Pos + pi/4, -1.0, 2.1> }
            #declare Pos = Pos + pi/2;
         #end
         // texturing, etc.
         texture { Rubber }
         translate <Speed*clock, 0.0, 0.0>    // animation
         clipped_by { box { <0.0, -1.05, -2.0>, <14.0, -0.75, 2.0> } }
         bounded_by { clipped_by }
      }
   }
   
   // the driving wheels
   union {
      // wheels
      difference {
         cylinder { <0.0, 0.0, -1.9>, <0.0, 0.0, 1.9>, 0.75 }  // long cylinder
         cylinder { <0.0, 0.0, -1.5>, <0.0, 0.0, 1.5>, 0.80 }  // cut out middle
         cylinder { <0.0, 0.0, -2.0>, <0.0, 0.0, -1.7>, 0.50 }  // cut off a little at ends
         cylinder { <0.0, 0.0, 1.7>, <0.0, 0.0, 2.0>, 0.50 }
         cylinder { <0.0, 0.0, -2.0>, <0.0, 0.0, 2.0>, 0.10 }  // hole of axle
         bounded_by { box { <-0.75, -0.75, -1.9>, <0.75, 0.75, 1.9> } }
      }
      // axle
      cylinder { <0.0, 0.0, -1.8>, <0.0, 0.0, 1.8>, 0.10 }
      // bolts
      difference {
         prism {
            linear_spline
            -1.8, 1.8, 7,
            // the first six points are placed by the while loop            
            #declare Angle = 0.0;
            #while ( Angle < 2*pi - 0.00001 )   // for some reason, just < 2*pi doesn't work properly
               <0.20 * cos( Angle ), 0.20 * sin( Angle )>,
               #declare Angle = Angle + pi/3;
            #end
            // the closing point ( not in loop due to the lack of end comma )
            <0.20, 0.0>                                              
            rotate x*90    // rotate to fit with the conveyor belt
         }
         cylinder { <0.0, 0.0, -1.7>, <0.0, 0.0, 1.7>, 0.21 }  // cut out middle
         cylinder { <0.0, 0.0, -2.0>, <0.0, 0.0, 2.0>, 0.10 }  // hole of axle
         rotate z*(Speed*180/pi*clock - 180)
         bounded_by { box { <-0.20, -0.20, -1.8>, <0.20, 0.20, 1.8> } }
      }
      texture { Silver_Texture rotate z*(Speed*180/pi*clock - 180) }   // rotates with the wheel                           
      texture { Chipped_Paint scale <1.0, 1.0, rand(R)> rotate z*(Speed*180/pi*clock - 180) }  // rotates with the wheel, the rand(R) should give different scratch patterns
   }
   rotate Rotate
   translate Translate
}
#end 

// Creates a flat slab with rounded corners ( radius Width/4 ), with base at y=0 before translation
// Parameters:
//    Length:  The longer side of the block, along x-axis ( must be > Width )
//    Width:  The shorter side of the block, along z-axis ( must be < Length )
//    Translate:  Translation of the Block
#macro Block_Piece ( Length, Width, Translate )
   union {
      box { <-Length/2, 0.00, -Width/4>, <Length/2, 0.005, Width/4> } // central slab
      box { <-Length/2 + Width/4, 0.00, Width/4>, <Length/2 - Width/4, 0.005, Width/2> } // side slab
      box { <-Length/2 + Width/4, 0.00, -Width/2>, <Length/2 - Width/4, 0.005, -Width/4> } // side slab
      // corners
      cylinder { <-Length/2 + Width/4, 0.00, -Width/4>, <-Length/2 + Width/4, 0.005, -Width/4>, Width/4 }
      cylinder { <-Length/2 + Width/4, 0.00, Width/4>, <-Length/2 + Width/4, 0.005, Width/4>, Width/4 }
      cylinder { <Length/2 - Width/4, 0.00, -Width/4>, <Length/2 - Width/4, 0.005, -Width/4>, Width/4 }
      cylinder { <Length/2 - Width/4, 0.00, Width/4>, <Length/2 - Width/4, 0.005, Width/4>, Width/4 }
      translate Translate
   }
#end

// A block of height 1.00, with base 3.00 by 2.00, and top 1.50 by 1.00
#declare Block = union {
   // while loop places pieces
   #declare Height = 0.00;
   #while ( Height < 1.00 )
      #declare X = 3.00 - Height * ( 1.50 / 0.995 );
      #declare Y = 2.00 - Height * ( 1.00 / 0.995 );
      Block_Piece ( X, Y, <0.00, Height, 0.00> )
      #declare Height = Height + 0.005;
   #end
}    

// Lead Block, see "Block" for dimensions, translate vector defines centre of base
#declare Lead_Block = difference {
   object { Block }
   text { ttf "c:\windows\fonts\times.ttf" "Pb" 0.1, 0
      scale 0.75
      rotate x*90
      translate <-0.375, 1.001, -0.25>
   }
   texture { T_Silver_3B }
   bounded_by { box { <-1.50, 0.00, -0.75>, <1.50, 1.00, 0.75> } }
}

// Gold Block, see "Block" for dimensions, translate vector defines centre of base
#declare Gold_Block =  difference {
   object { Block }
   text { ttf "c:\windows\fonts\times.ttf" "Au" 0.1, 0
      scale 0.75
      rotate x*90
      translate <-0.45, 1.001, -0.25>
   }
   texture { T_Copper_5C }
   bounded_by { box { <-1.50, 0.00, -0.75>, <1.50, 1.00, 0.75> } }
}

// The transformation machine, base extends to -1 unit, height to 10 units
#declare Machine = union {
   // body
   difference {
      cylinder { <-9.00, 5.00, 0.00>, <9.00, 5.00, 0.00>, 5.00 }  // body
      cylinder { <-9.10, 5.00, 0.00>, <9.10, 5.00, 0.00>, 4.00 }  // hollow out body
      box { <-3.00, 5.00 - 5.00 * sin(0.4), -6.00>, <3.00, 5.00 + 5.00 * sin(0.4), 6.00> }   // cut for windows
   }
   // windows
   difference {
      cylinder { <-3.10, 5.00, 0.00>, <3.10, 5.00, 0.00>, 4.75 }
      cylinder { <-3.20, 5.00, 0.00>, <3.20, 5.00, 0.00>, 4.25 }
      texture { Glass }
   }
   // raised section by windows
   difference {
      union {
         intersection {
            union {
               cylinder { <-4.00, 5.00, 0.00>, <4.00, 5.00, 0.00>, 5.20 }  // raised area around windows
               // smoothing
               torus { 5.00, 0.20
                  rotate z*90
                  translate <-4.00, 5.00, 0.00>
               }
               torus { 5.00, 0.20
                  rotate z*90
                  translate <4.00, 5.00, 0.00>
               }
            }               
            box { <-4.20, 5.00 - 5.00 * sin(0.6), -6.00>, <4.20, 5.00 + 5.00 * sin(0.6), 6.00> } // take section around windows
         }
         // more smoothing
         // -y, +z
         sphere { <-4.00, 5.00 - 5.00 * sin(0.6), cos(0.6)>, 0.20 }
         cylinder { <-4.00, 5.00 - 5.00 * sin(0.6), cos(0.6)>, <4.00, 5.00 - 5.00 * sin(0.6), cos(0.6)>, 0.20 }
         sphere { <4.00, 5.00 - 5.00 * sin(0.6), cos(0.6)>, 0.20 }         
         // +y, +z         
         sphere { <-4.00, 5.00 + 5.00 * sin(0.6), cos(0.6)>, 0.20 }
         cylinder { <-4.00, 5.00 + 5.00 * sin(0.6), cos(0.6)>, <4.00, 5.00 - 5.00 * sin(0.6), cos(0.6)>, 0.20 }
         sphere { <4.00, 5.00 + 5.00 * sin(0.6), cos(0.6)>, 0.20 }
         // -y, -z
         sphere { <-4.00, 5.00 - 5.00 * sin(0.6), -cos(0.6)>, 0.20 }
         cylinder { <-4.00, 5.00 - 5.00 * sin(0.6), -cos(0.6)>, <4.00, 5.00 - 5.00 * sin(0.6), cos(0.6)>, 0.20 }
         sphere { <4.00, 5.00 - 5.00 * sin(0.6), -cos(0.6)>, 0.20 }
         // +y, -z
         sphere { <-4.00, 5.00 + 5.00 * sin(0.6), -cos(0.6)>, 0.20 }
         cylinder { <-4.00, 5.00 + 5.00 * sin(0.6), -cos(0.6)>, <4.00, 5.00 - 5.00 * sin(0.6), cos(0.6)>, 0.20 }
         sphere { <4.00, 5.00 + 5.00 * sin(0.6), -cos(0.6)>, 0.20 }
      }
      cylinder { <-4.10, 5.00, 0.00>, <4.10, 5.00, 0.00>, 4.80 }  // hollow out centre
      box { <-3.00, 5.00 - 5.00 * sin(0.4), -6.00>, <3.00, 5.00 + 5.00 * sin(0.4), 6.00> }   // cut for windows
      bounded_by { cylinder { <-4.20, 5.00, 0.00>, <4.20, 5.00, 0.00>, 5.20 } }
   }
   // bolts around windows
   difference {
      union {
         // while loop for vertical edges
         #declare Theta = -0.50;
         #while ( Theta < 0.51 )
            Regular_Prism( 6, -5.40, 5.40, 0.20, <90 + Theta * 180 / pi, 0.00, 0.00>, <-3.50, 5.00, 0.00> )
            Regular_Prism( 6, -5.40, 5.40, 0.20, <90 + Theta * 180 / pi, 0.00, 0.00>, <3.50, 5.00, 0.00> )
            #declare Theta = Theta + 0.20;
         #end
         // while loop for horizontal edges
         #declare X = -2.50;
         #while ( X < 2.51 )
            Regular_Prism ( 6, -5.40, 5.40, 0.20, <90 - 0.50 * 180 / pi, 0.00, 0.00>, <X, 5.00, 0.00> )
            Regular_Prism ( 6, -5.40, 5.40, 0.20, <90 + 0.50 * 180 / pi, 0.00, 0.00>, <X, 5.00, 0.00> )
            # declare X = X + 1.00;
         #end
      }
      cylinder { <-4.00, 5.00, 0.00>, <4.00, 5.00, 0.00>, 4.50 }
   }      
   // rounded ends on cylinder
   // left end (with hole for connector tube)
   difference {
      union {
         cylinder { <-10.0, 5.00, 0.00>, <-9.00, 5.00, 0.00>, 4.00 }
         torus { 4.00, 1.00
            rotate z*90
            translate <-9.00, 5.00, 0.00>
         }
      }
      cylinder { <-10.1, 5.00, 0.00>, <-8.90, 5.00, 0.00>, 0.80 }
   }
   // right end
   cylinder { < 9.00, 5.00, 0.00>, <10.0, 5.00, 0.00>, 4.00 }
   torus { 4.00, 1.00
      rotate z*90
      translate <9.00, 5.00, 0.00>
   }
   // raised strips with bolts on them
   difference {
      union {
         // raised strips
         difference {
            cylinder { <-7.50, 5.00, 0.00>, <7.50, 5.00, 0.00>, 5.20 }  // raised cylinder
            cylinder { <-6.50, 5.00, 0.00>, <6.50, 5.00, 0.00>, 5.21 }  // cut out middle
         }
         // smooting
         torus { 5.00, 0.20
            rotate z*90
            translate <-7.50, 5.00, 0.00>
         }
         torus { 5.00, 0.20
            rotate z*90
            translate <-6.50, 5.00, 0.00>
         }
         torus { 5.00, 0.20
            rotate z*90
            translate <6.50, 5.00, 0.00>
         }
         torus { 5.00, 0.20            
            rotate z*90
            translate <7.50, 5.00, 0.00>
         }
         // while loop to place bolts
         #declare Theta = 0.00;
         #while ( Theta < 180 )
            Regular_Prism ( 6, -5.40, 5.40, 0.20, < 90 + Theta, 0.00, 0.00>, <-7.00, 5.00, 0.00> )
            Regular_Prism ( 6, -5.40, 5.40, 0.20, < 90 + Theta, 0.00, 0.00>, <7.00, 5.00, 0.00> )
            #declare Theta = Theta + 12;
         #end
      }
      // hollow out middle
      cylinder { <-7.51, 5.00, 0.00>, <7.51, 5.00, 0.00>, 4.00 }
      bounded_by { cylinder { <-7.50, 5.00, 0.00>, <7.50, 5.00, 0.00>, 5.40 } }
   }
   // "in box"
   difference {
      union {
         // main box
         box { <10.0, 2.50, -3.00>, <14.0, 7.50, 3.00> }
         // raised sections and smooting
         box { <10.0, 6.50, -3.20>, <14.0, 7.50, -3.00> }      // front upper
         cylinder { <10.0, 6.50, -3.00>, <14.0, 6.50, -3.00>, 0.20 }
         box { <10.0, 2.50, -3.20>, <14.0, 3.50, -3.00> }      // front lower
         cylinder { <10.0, 3.50, -3.00>, <14.0, 3.50, -3.00>, 0.20 }
         box { <13.0, 3.50, -3.20>, <14.0, 6.50, -3.00> }      // front right
         cylinder { <13.0, 3.50, -3.00>, <13.0, 6.50, -3.00>, 0.20 }
         box { <10.0, 6.50, 3.00>, <14.0, 7.50, 3.20> }        // back upper
         cylinder { <10.0, 6.50, 3.00>, <14.0, 6.50, 3.00>, 0.20 }
         box { <10.0, 2.50, 3.00>, <14.0, 3.50, 3.20> }        // back lower
         cylinder { <10.0, 3.50, 3.00>, <14.0, 3.50, 3.00>, 0.20 }
         box { <13.0, 3.50, 3.00>, <14.0, 6.50, 3.20> }        // back right      
         cylinder { <13.0, 3.50, 3.00>, <13.0, 6.50, 3.00>, 0.20 }
         box { <10.0, 2.30, -3.00>, <14.0, 2.50, -2.00> }      // bottom near
         cylinder { <10.0, 2.50, -3.00>, <14.0, 2.50, -3.00>, 0.20 }
         box { <10.0, 2.30, 2.00>, <14.0, 2.50, 3.00> }        // bottom far
         cylinder { <10.0, 2.50, 3.00>, <14.0, 2.50, 3.00>, 0.20 }
         box { <13.0, 2.30, -2.00>, <14.0, 2.50, 2.00> }       // bottom right
         cylinder { <13.0, 2.50, -2.00>, <13.0, 2.50, 2.00>, 0.20 }
         box { <14.0, 2.50, -3.00>, <14.2, 7.50, -2.00> }      // right near
         cylinder { <14.0, 2.50, -2.00>, <14.0, 7.50, -2.00>, 0.20 }
         cylinder { <14.0, 2.50, -3.00>, <14.0, 7.50, -3.00>, 0.20 }
         sphere { <14.0, 2.50, -3.00>, 0.20 }
         box { <14.0, 2.50, 2.00>, <14.2, 7.50, 3.00> }        // right far
         cylinder { <14.0, 2.50, 2.00>, <14.0, 7.50, 2.00>, 0.20 }
         cylinder { <14.0, 2.50, 3.00>, <14.0, 7.50, 3.00>, 0.20 }
         sphere { <14.0, 2.50, 3.00>, 0.20 }
         box { <14.0, 2.50, -2.00>, <14.2, 3.50, 2.00> }       // right lower
         cylinder { <14.0, 3.50, -2.00>, <14.0, 3.50, 2.00>, 0.20 }
         cylinder { <14.0, 2.50, -3.00>, <14.0, 2.50, 3.00>, 0.20 }
         box { <14.0, 6.50, -2.00>, <14.2, 7.50, 2.00> }       // right upper
         cylinder { <14.0, 6.50, -2.00>, <14.0, 6.50, 2.00>, 0.20 }
         //smooth out top
         cylinder { <10.0, 7.50, -3.00>, <14.0, 7.50, -3.00>, 0.20 }
         sphere { <14.0, 7.50, -3.00>, 0.20 }
         cylinder { <14.0, 7.50, -3.00>, <14.0, 7.50, 3.00>, 0.20 }
         sphere { <14.0, 7.50, 3.00>, 0.20 }
         cylinder { <10.0, 7.50, 3.00>, <14.0, 7.50, 3.00>, 0.20}
         box { <10.0, 7.50, -3.00>, <14.0, 7.70, -2.20> }
         box { <10.0, 7.50, 2.20>, <14.0, 7.70, 3.00> }
         box { <13.2, 7.50, -2.20>, <14.0, 7.70, 2.20> }
         cylinder { <10.0, 7.50, -2.20>, <14.0, 7.50, -2.20>, 0.20 }
         cylinder { <10.0, 7.50, 2.20>, <14.0, 7.50, 2.20>, 0.20 }
         cylinder { <13.2, 7.50, -2.20>, <13.2, 7.50, 2.20>, 0.20 }
         // bolts on box
         // while loop for horizontal bolts
         #declare X = 10.5;
         #while ( X < 14.0 )
            Regular_Prism( 6, -3.40, 3.40, 0.20, <90, 0, 0>, <X, 3.00, 0.00> )// lower (front/back)
            Regular_Prism( 6, -3.40, 3.40, 0.20, <90, 0, 0>, <X, 7.00, 0.00> )// upper (front/back)
            Regular_Prism( 6, 2.10, 2.30, 0.20, <0, 0, 0>, <X, 0.00, -2.50> ) // near  (bottom)
            Regular_Prism( 6, 2.10, 2.30, 0.20, <0, 0, 0>, <X, 0.00, 2.50> )  // far   (bottom)
            #declare X = X + 1;
         #end
         // while loop for depth (Z) bolts
         #declare Z = -2.50;
         #while ( Z < 3.00 )
            Regular_Prism( 6, 0.00, 0.20, 0.20, <0, 0, -90>, <14.2, 3.00, Z> )// lower (right)
            Regular_Prism( 6, 0.00, 0.20, 0.20, <0, 0, -90>, <14.2, 7.00, Z> )// upper (right)
            Regular_Prism( 6, 2.10, 2.30, 0.20, <0, 0, 0>, <13.5, 0.00, Z> )  // right (bottom)
            #declare Z = Z + 1;
         #end
         // while loop for vertical bolts
         #declare Y = 4.00;
         #while ( Y < 6.50 )
            Regular_Prism( 6, -3.40, 3.40, 0.20, <90, 0, 0>, <13.5, Y, 0.00> )// right (front/back)
            Regular_Prism( 6, 0.00, 0.20, 0.20, <0, 0, -90>, <14.2, Y, -2.50> )// near (right)
            Regular_Prism( 6, 0.00, 0.20, 0.20, <0, 0, -90>, <14.2, Y, 2.50> )// far   (right)
            #declare Y = Y + 1;
         #end
      }
      box { <10.0, 3.50, -2.00>, <13.0, 7.51, 2.00> }
      bounded_by { box { <10.0, 2.10, -3.40>, <14.4, 7.70, 3.40> } }
   }
   // base for chamber
   union {
      box { <-4.00, -0.80, -3.00>, <4.00, 1.00, 3.00> }
      box { <-5.00, -1.00, -4.00>, <5.00, -0.80, 4.00> }
      // while loop for horizontal bolts
      #declare X = -4.50;
      #while ( X < 5.00 )
         Regular_Prism( 6, -0.80, -0.60, 0.20, <0, 0, 0>, <X, 0, -3.50> )
         Regular_Prism( 6, -0.80, -0.60, 0.20, <0, 0, 0>, <X, 0, 3.50> )
         #declare X = X + 1;
      #end
      // while loop for depth (Z) bolts
      #declare Z = -3.50;
      #while ( Z < 4.00 )
         Regular_Prism( 6, -0.80, -0.60, 0.20, <0, 0, 0>, <-4.50, 0, Z> )
         Regular_Prism( 6, -0.80, -0.60, 0.20, <0, 0, 0>, <4.50, 0, Z> )
         #declare Z = Z + 1;
      #end
   }
   // base for "in box"
   union {
      cylinder { <12.0, -0.80, 0.00>, <12.0, 2.50, 0.00>, 0.77 }
      cylinder { <12.0, -1.00, 0.00>, <12.0, -0.80, 0.00>, 1.77 }
      // while loop to place bolts in a circle
      #declare Theta = 0;
      #while ( Theta < 2*pi )
         Regular_Prism( 6, -0.80, -0.60, 0.20, <0, 0, 0>, <12.0 + 1.27 * sin( Theta ), 0, 1.27 * cos( Theta )> )
         #declare Theta = Theta + pi/4;
      #end
   }
   // glass connector tube
   difference {
      union {
         cylinder { <-11.0, 5.00, 0.00>, <-10.0, 5.00, 0.00>, 1.10 }
         cylinder { <-15.0, 5.00, 0.00>, <-11.0, 5.00, 0.00>, 1.00
            texture { Glass }
         }
         cylinder { <-16.0, 5.00, 0.00>, <-15.0, 5.00, 0.00>, 1.10 }
      }
      cylinder { <-16.1, 5.00, 0.00>, <-9.90, 5.00, 0.00>, 0.80
         pigment { colour rgbt <1, 1, 1, 1> }
      }
   }
   // "out box"
   difference {
      union {
         // main box
         box { <-20.0, 3.00, -3.00>, <-16.0, 7.00, 3.00> }
         // raised sections and smoothing
         sphere { <-20.0, 7.00, -3.00>, 0.20 }                         // left near
         cylinder { <-20.0, 3.00, -3.00>, <-20.0, 7.00, -3.00>, 0.20 } // down
         box { <-20.0, 3.00, -3.20>, <-19.0, 7.00, -3.00> }            // front
         cylinder { <-19.0, 3.00, -3.00>, <-19.0, 7.00, -3.00>, 0.20 }
         box { <-20.2, 3.00, -3.00>, <-20.0, 7.00, -2.00> }            // side
         cylinder { <-20.0, 3.00, -2.00>, <-20.0, 7.00, -2.00>, 0.20 }
         cylinder { <-20.0, 7.00, -3.00>, <-16.0, 7.00, -3.00>, 0.20 } // right
         box { <-20.0, 6.00, -3.20>, <-16.0, 7.00, -3.00> }            // front upper
         cylinder { <-20.0, 6.00, -3.00>, <-16.0, 6.00, -3.00>, 0.20 }
         box { <-20.0, 3.00, -3.20>, <-16.0, 4.00, -3.00> }            // front lower
         cylinder { <-20.0, 4.00, -3.00>, <-16.0, 4.00, -3.00>, 0.20 }
         box { <-20.0, 7.00, -3.00>, <-16.0, 7.20, -2.00> }            // top
         cylinder { <-20.0, 7.00, -2.00>, <-16.0, 7.00, -2.00>, 0.20 }
         sphere { <-16.0, 7.00, -3.00>, 0.20 }                         // right near
         cylinder { <-16.0, 3.00, -3.00>, <-16.0, 7.00, -3.00>, 0.20 } // down
         box { <-17.0, 3.00, -3.20>, <-16.0, 7.00, -3.00> }            // front
         cylinder { <-17.0, 3.00, -3.00>, <-17.0, 7.00, -3.00>, 0.20 }
         box { <-16.0, 3.00, -3.00>, <-15.8, 7.00, -2.00> }            // side
         cylinder { <-16.0, 3.00, -2.00>, <-16.0, 7.00, -2.00>, 0.20 }
         cylinder { <-16.0, 7.00, -3.00>, <-16.0, 7.00, 3.00>, 0.20 }  // back
         box { <-16.0, 6.00, -3.00>, <-15.8, 7.00, 3.00> }             // side upper
         cylinder { <-16.0, 6.00, -3.00>, <-16.0, 6.00, 3.00>, 0.20 }
         box { <-16.0, 3.00, -3.00>, <-15.8, 4.00, 3.00> }             // side lower
         cylinder { <-16.0, 4.00, -3.00>, <-16.0, 4.00, 3.00>, 0.20 }
         box { <-16.0, 7.00, -3.00>, <-17.0, 7.20, 3.00> }             // top
         cylinder { <-17.0, 7.00, -3.00>, <-17.0, 7.00, 3.00>, 0.20 }
         sphere { <-16.0, 7.00, 3.00>, 0.20 }                          // right far
         cylinder { <-16.0, 3.00, 3.00>, <-16.0, 7.00, 3.00>, 0.20 }   // down
         box { <-17.0, 3.00, 3.00>, <-16.0, 7.00, 3.20> }              // back
         cylinder { <-17.0, 3.00, 3.00>, <-17.0, 7.00, 3.00>, 0.20 }
         box { <-16.0, 3.00, 2.00>, <-15.8, 7.00, 3.00> }              // side
         cylinder { <-16.0, 3.00, 2.00>, <-16.0, 7.00, 2.00>, 0.20 }
         cylinder { <-20.0, 7.00, 3.00>, <-16.0, 7.00, 3.00>, 0.20 }   // left
         box { <-20.0, 6.00, 3.00>, <-16.0, 7.00, 3.20> }              // back upper
         cylinder { <-20.0, 6.00, 3.00>, <-16.0, 7.00, 3.00>, 0.20 }
         box { <-20.0, 3.00, 3.00>, <-16.0, 4.00, 3.20> }              // back lower
         cylinder { <-20.0, 4.00, 3.00>, <-16.0, 4.00, 3.00>, 0.20 }
         box { <-20.0, 7.00, 2.00>, <-16.0, 7.20, 3.00> }              // top
         cylinder { <-20.0, 7.00, 2.00>, <-16.0, 7.00, 2.00>, 0.20 }
         sphere { <-20.0, 7.00, 3.00>, 0.20 }                          // left far
         cylinder { <-20.0, 3.00, 3.00>, <-20.0, 7.00, 3.00>, 0.20 }   // down
         box { <-20.0, 3.00, 3.00>, <-19.0, 7.00, 3.20> }              // back
         cylinder { <-19.0, 3.00, 3.00>, <-19.0, 7.00, 3.00>, 0.20 }
         box { <-20.2, 3.00, 2.00>, <-20.0, 7.00, 3.00> }              // side
         cylinder { <-20.0, 3.00, 2.00>, <-20.0, 7.00, 2.00>, 0.20 }
         cylinder { <-20.0, 7.00, -3.00>, <-20.0, 7.00, 3.00>, 0.20 }  // forward
         box { <-20.2, 6.00, -3.00>, <-20.0, 7.00, 3.00> }             // side upper
         cylinder { <-20.0, 6.00, -3.00>, <-20.0, 6.00, 3.00>, 0.20 }
         box { <-20.2, 3.00, -3.00>, <-20.0, 4.00, 3.00> }             // side lower
         cylinder { <-20.0, 4.00, -3.00>, <-20.0, 4.00, 3.00>, 0.20 }
         box { <-20.0, 7.00, -3.00>, <-19.0, 7.20, 3.00> }             // top
         cylinder { <-19.0, 7.00, -3.00>, <-19.0, 7.00, 3.00>, 0.20 }
         // bolts
         // while loop for horizontal bolts
         #declare X = -19.5;
         #while ( X < -16.0 )
            Regular_Prism( 6, -3.40, 3.40, 0.20, <90, 0, 0>, <X, 3.50, 0.00> )// lower (front/back)
            Regular_Prism( 6, -3.40, 3.40, 0.20, <90, 0, 0>, <X, 6.50, 0.00> )// upper (front/back)
            Regular_Prism( 6, 7.20, 7.40, 0.20, <0, 0, 0>, <X, 0.00, -2.50> ) // near  (top)
            Regular_Prism( 6, 7.20, 7.40, 0.20, <0, 0, 0>, <X, 0.00, 2.50> )  // far   (top)
            #declare X = X + 1;
         #end
         // while loop for depth (Z) bolts
         #declare Z = -2.50;
         #while ( Z < 3.00 )
            Regular_Prism( 6, -2.40, 2.40, 0.20, <0, 0, -90>, <-18.0, 3.50, Z> )// lower (left/right)
            Regular_Prism( 6, -2.40, 2.40, 0.20, <0, 0, -90>, <-18.0, 6.50, Z> )// upper (left/right)
            Regular_Prism( 6, 7.20, 7.40, 0.20, <0, 0, 0>, <-16.5, 0.00, Z> )   // right (top)
            Regular_Prism( 6, 7.20, 7.40, 0.20, <0, 0, 0>, <-19.5, 0.00, Z> )   // left (top)
            #declare Z = Z + 1;
         #end
         // while loop for vertical bolts
         #declare Y = 4.50;
         #while ( Y < 6.00 )
            Regular_Prism( 6, -3.40, 3.40, 0.20, <90, 0, 0>, <-16.5, Y, 0.00> ) // right (front/back)
            Regular_Prism( 6, -3.40, 3.40, 0.20, <90, 0, 0>, <-19.5, Y, 0.00> ) // left (front/back)
            Regular_Prism( 6, -2.40, 2.40, 0.20, <0, 0, -90>, <-18.0, Y, -2.50> )// near (left/right)
            Regular_Prism( 6, -2.40, 2.40, 0.20, <0, 0, -90>, <-18.0, Y, 2.50> ) // far   (left/right)
            #declare Y = Y + 1;
         #end         
      }
      // spot for block
      /*object { Gold_Block
         rotate y*-90
         translate <-18.0, 5.00, 0.00>
      }*/
      cylinder { <-18.0, 5.00, 0.00>, <-15.9, 5.00, 0.00>, 0.80 } // hole for connector tube
      bounded_by { box { <-20.4 3.00, -3.40>, <-15.6, 7.40, 3.40> } }
   }
   // "out box" / ceiling connector
   union {
      cylinder { <-18.0, 7.00, 0.00>, <-18.0, 18.8, 0.00>, 0.77 }
      cylinder { <-18.0, 18.8, 0.00>, <-18.0, 19.0, 0.00>, 1.77 }
      // while loop to place bolts in a circle
      #declare Theta = 0;
      #while ( Theta < 2*pi )
         Regular_Prism( 6, 18.6, 18.8, 0.20, <0, 0, 0>, <-18.0 + 1.27 * sin( Theta ), 0, 1.27 * cos( Theta )> )
         #declare Theta = Theta + pi/4;
      #end
   }
   // block lowering mechanism
   union {
      difference {
         union {
            // base of "out box"
            box { <-20.0, 2.80, -3.00>, <-16.0, 3.00, 3.00> }
            sphere { <-20.0, 3.00, -3.00>, 0.20 }                        // left near
            cylinder { <-20.0, 3.00, -3.00>, <-16.0, 3.00, -3.00>, 0.20 }
            sphere { <-16.0, 3.00, -3.00>, 0.20 }                        // right near
            cylinder { <-16.0, 3.00, -3.00>, <-16.0, 3.00, 3.00>, 0.20 }
            sphere { <-16.0, 3.00, 3.00>, 0.20 }                         // right far
            cylinder { <-20.0, 3.00, 3.00>, <-16.0, 3.00, 3.00>, 0.20 }
            sphere { <-20.0, 3.00, 3.00>, 0.20 }                         // left far
            cylinder { <-20.0, 3.00, -3.00>, <-20.0, 3.00, 3.00>, 0.20 }
            // top cylinder section
            cylinder { <-18.0, 1.30, 0.00>, <-18.0, 2.80, 0.00>, 0.50 }
         }
         plane { -y, -3.00 }
         // translate for animation stage 1
         translate <0.00, -1.50*clock, 0.00>
      }
      // bottom cylinder section and box (smoothed)
      cylinder { <-18.0, 1.00, 0.00>, <-18.0, 1.30, 0.00>, 0.70 }
      box { <-20.0, -1.00, -2.20>, <-16.0, 0.80, 2.20> } // expanded z
      box { <-20.2, -1.00, -2.00>, <-15.8, 0.80, 2.00> } // expanded x
      box { <-20.0, 0.80, -2.00>, <-16.0, 1.00, 2.00> }  // expanded y
      sphere { <-20.0, 0.80, -2.00>, 0.20 }                         // left near
      cylinder { <-20.0, -1.00, -2.00>, <-20.0, 0.80, -2.00>, 0.20 }
      cylinder { <-20.0, 0.80, -2.00>, <-16.0, 0.80, -2.00>, 0.20 }
      sphere { <-16.0, 0.80, -2.00>, 0.20 }                         // right near
      cylinder { <-16.0, -1.00, -2.00>, <-16.0, 0.80, -2.00>, 0.20 }
      cylinder { <-16.0, 0.80, -2.00>, <-16.0, 0.80, 2.00>, 0.20 }
      sphere { <-16.0, 0.80, 2.00>, 0.20 }                          // right far
      cylinder { <-16.0, -1.00, 2.00>, <-16.0, 0.80, 2.00>, 0.20 }
      cylinder { <-20.0, 0.80, 2.00>, <-16.0, 0.80, 2.00>, 0.20 }
      sphere { <-20.0, 0.80, 2.00>, 0.20 }                          // left far
      cylinder { <-20.0, -1.00, 2.00>, <-20.0, 0.80, 2.00>, 0.20 }
      cylinder { <-20.0, 0.80, -2.00>, <-20.0, 0.80, 2.00>, 0.20 }
   }
   // mechanism to push block onto conveyor
   union {
      difference {
         union { 
            box { <-15.5, 1.70, -1.80>, <-15.3, 2.80, 1.80> }
            sphere { <-15.3, 1.70, -1.80>, 0.20 }                        // bottom near
            cylinder { <-15.3, 1.70, -1.80>, <-15.3, 2.80, -1.80>, 0.20 }
            sphere { <-15.3, 2.80, -1.80>, 0.20 }                        // top near
            cylinder { <-15.3, 2.80, -1.80>, <-15.3, 2.80, 1.80>, 0.20 }
            sphere { <-15.3, 2.80, 1.80>, 0.20 }                         // top far
            cylinder { <-15.3, 1.70, 1.80>, <-15.3, 2.80, 1.80>, 0.20 }
            sphere { <-15.3, 1.70, 1.80>, 0.20 }                         // bottom far
            cylinder { <-15.3, 1.70, -1.80>, <-15.3, 1.70, 1.80>, 0.20 }
         }
         plane { -x, 15.3 }
         // translation for animation stage 2/3
         translate -x*0
      }
      // smallest cylinder section
      cylinder { <-15.3, 2.25, 0.00>, <-13.3, 2.25, 0.00>, 0.50
         // translation for animation stage 2/3
         translate -x*0
      }
      // middle cylinder section
      cylinder { <-15.3, 2.25, 0.00>, <-13.3, 2.25, 0.00>, 0.60
         // translation for animation stage 2
         translate -x*0
      }
      // right cylinder and box (smoothed)
      cylinder { <-15.3, 2.25, 0.00>, <-14.7, 2.25, 0.00>, 0.70 }
      box { <-14.7, -1.00, -2.00>, <-12.8, 3.00, 2.00> } // expanded x
      box { <-14.5, 3.00, -2.00>, <-13.0, 3.20, 2.00> }  // expanded y
      box { <-14.5, -1.00, -2.20>, <-13.0, 3.00, 2.20> } // expanded z
      sphere { <-14.5, 3.00, -2.00>, 0.20 }                         // left near
      cylinder { <-14.5, -1.00, -2.00>, <-14.5, 3.00, -2.00>, 0.20 }
      cylinder { <-14.5, 3.00, -2.00>, <-13.0, 3.00, -2.00>, 0.20 }
      sphere { <-13.0, 3.00, -2.00>, 0.20 }                         // right near
      cylinder { <-13.0, -1.00, -2.00>, <-13.0, 3.00, -2.00>, 0.20 }
      cylinder { <-13.0, 3.00, -2.00>, <-13.0, 3.00, 2.00>, 0.20 }
      sphere { <-13.0, 3.00, 2.00>, 0.20 }                          // right far
      cylinder { <-13.0, -1.00, 2.00>, <-13.0, 3.00, 2.00>, 0.20 }
      cylinder { <-14.5, 3.00, 2.00>, <-13.0, 3.00, 2.00>, 0.20 }
      sphere { <-14.5, 3.00, 2.00>, 0.20 }                          // left far
      cylinder { <-14.5, -1.00, 2.00>, <-14.5, 3.00, 2.00>, 0.20 }
      cylinder { <-14.5, 3.00, -2.00>, <-14.5, 3.00, 2.00>, 0.20 }
   }
}
      
object { Room }

object { Machine
   texture { T_Copper_3A}
   translate <3.00, 1.00, 2.50>
}

Conveyor_Belt ( pi, <0, 0, 0>, <16.0, 10.0, 2.50> ) // in belt

Conveyor_Belt ( pi, <180, 180, 0>, <-18.0, 1.45, 2.50> ) // out belt

object { Gold_Block
   rotate y*-90
   translate <-15.0, 4.00 - 1.50*clock, 2.50>
}                                                                          