//	Persistence of Vision Ray Tracer Scene Description File
//	File: Terrain.pov
//	Authors: Virtu Halttunen, Steve Jackson
// 	Copyright 2002

#include "Macros.mcr"

global_settings {
	max_trace_level 7
}

#declare MasterClock = Timer( clock, 0.15, 0.85, true );

#declare CityStartsDyingAt = 0.05;				// The MasterClock value when city starts deteriorating
#declare CityGoneAt = 0.45;						// The MasterClock value when the city is completely gone (0.3)
#declare WinterStartsAt = 0.17;					// The MasterClock value when we start blending to winter
#declare WinterCompleteAt = 0.33; 			// The MasterClock value when transition to winter is complete
#declare SnowBlanketCompleteAt = 0.33;	// The WinterClock value when snow covers all the grassland
#declare DeadTreeStartsSwingingAt = 0;		// MasterClock value
#declare DeadTreeStartsFallingAt = 0.1;		// MasterClock value. Tree stops swinging and starts tumbling over
#declare DeadTreeFallCompleteAt = 0.25;	// MasterClock value
#declare GlacierEntersAt = 0.2;					// The MasterClock value when the glacier first crawls out from the left
#declare GlacierVanishesAt = 0.65;			// The MasterClock value when the glacier melts completely and exits the scene
#declare GlacierParksAt = 0.55;					// GlacierClock value
#declare GlacierLeavesAt = 0.775;				// GlacierClock value
#declare SmallTreesStartAt = 0.2;				// MasterClock value
#declare SmallTreesStopAt = 0.55;				// MasterClock value
#declare FirstMorphStartsAt = 0.45;				// MasterClock value for the first terrain morph 
#declare FirstMorphDoneAt = 0.675;			// MasterClock value
#declare SpringStartsAt = 0.55;					// MasterClock value (0.6)
#declare SpringCompleteAt = 0.7;				// MasterClock value
#declare SummerInFullBloomAt = 0.8;			// MasterClock value
#declare SecondMorphStartsAt = 0.725;			// MasterClock value for the second terrain morph (0.75)
#declare SecondMorphDoneAt = 0.9525;		// MasterClock value
#declare LiveTreeStartsAppearingAt = 0.875;	// MasterClock value
#declare LiveTreeDoneAt = 0.95;				// MasterClock value
#declare NightStartsAt = 0.25;					// MasterClock value
#declare NightEndsAt = 0.6;						// MasterClock value

#declare IntroClock = Timer( clock, 0, 0.15, true );
#if( IntroClock < 1 )
	#include "Intro.inc"
#end

#declare OutroClock = Timer( clock, 0.85, 1, true );
#if( OutroClock > 0 )
	#include "Outro.inc"
#end

#if( MasterClock < SpringStartsAt )
	#declare WinterClock = Timer( MasterClock, WinterStartsAt, WinterCompleteAt, true );
	#declare SnowBlanketClock = Timer( WinterClock, 0, SnowBlanketCompleteAt, true );
	#if( SnowBlanketClock >= 1 )
		#declare SnowLevelClock = Timer( WinterClock, SnowBlanketCompleteAt, 0.6, true );
		#declare SnowLevelClock = Sinefy(  SnowLevelClock  );
		#declare heightShiftLimit = 0.1;
		#declare heightShift=0.1*SnowLevelClock;
	#end
#else
	#declare WinterClock = Timer( MasterClock, SpringStartsAt, SpringCompleteAt, true );
	#declare SnowBlanketClock = 1-Timer( WinterClock, 0, SnowBlanketCompleteAt, true );
	#declare WinterClock = 1-WinterClock;
	#if( SnowBlanketClock >= 1 )
		#declare SnowLevelClock = Timer( WinterClock, SnowBlanketCompleteAt, 0.6, true );
		#declare SnowLevelClock = Sinefy(  SnowLevelClock  );
		#declare heightShiftLimit = 0.1;
		#declare heightShift=0.1*SnowLevelClock;
	#end
#end	

#declare NightLevel = 0;
#declare NightClock = Timer( MasterClock, NightStartsAt, NightEndsAt, false );
#switch( NightClock )
	#range( 0, 0.33 )
		#declare NightLevel = Sinefy( Timer( NightClock, 0, 0.33, true ) );
	#break
	#range( 0.33, 0.66 )
		#declare NightLevel = 1;
	#break
	#range( 0.66, 1 )
		#declare NightLevel = 1-Sinefy( Timer( NightClock, 0.66, 1, true ) );
	#break
#end


// Set a color of the background (sky)
//background { color red 0.1 green 0.3 blue 0.8 }
sphere {
	<0,0,0>, 1000
	hollow
	texture {
		pigment { 
			gradient y
			color_map {
				[ 0 rgbt <0.1,0.3,0.8,NightLevel> ]
				[ 1-NightLevel*2 rgbt <0,0,0.5, 0.5> ]
				[ 2-NightLevel*2 rgbt <1,1,1,1> ]
			 }
			scale 1500
			translate y*-250
		}
		finish { ambient 1 diffuse 0 }
	}
}

#declare galaxy_seed = 11;
#declare galaxy_intensity = 0.9;
#declare galaxy_bgstars = false;
#declare galaxy_bgnebula = false;
#declare galaxy_object_name = "";
#declare galaxy_rotate = <-33,-50,12>;
#if( NightLevel > 0 )
	#include "galaxymod.inc"
#end

light_source
{
  <-20, 40, -20>
  color <1,1,0.98>
}

light_source {
	<20,30,-30>
	color rgb 0.3
	shadowless
}

// 2997 was nice...
#declare TerrainOne = pigment { MakeTerrainRoot(4805, 5 ) scale <-1,1,1> }		// Original terrain
#declare TerrainTwo = pigment { MakeTerrainRoot( 353, 5 ) }								// New terrain
								
//manual values for making a lake in 353.2
#declare heightShiftLimit = 0.1275;
#declare heightShift = -0.05;
#declare TerrainThree = pigment { MakeTerrainRoot( 353, 2 ) }							// New terrain, eroded and with a lake

#declare PatternFuncOne = function { pigment { TerrainOne } }
#declare PatternFuncTwo = function { pigment { TerrainTwo } }

#declare MaxHeight = 30;
#declare TreeRand = seed( 2 );
#declare TreeScale = 0.05;
#declare TreesClock = Timer( MasterClock, SmallTreesStartAt, SmallTreesStopAt, false );


#if( WinterClock > 0 )
	#declare Winter = true;
#end

#declare WaterClock = 0.5*sin( clock*pi*2 )+0.5;
#include "Textures.inc"

#declare LandClock = 0;
#if( MasterClock < FirstMorphDoneAt )
	#declare LandClock = Timer( MasterClock, FirstMorphStartsAt, FirstMorphDoneAt, true );
	#declare LandClock = Sinefy( LandClock );
#end
#if( MasterClock >= FirstMorphDoneAt )
	#undef PatternFuncOne
	#undef PatternFuncTwo
	#declare PatternFuncOne = function { pigment { TerrainTwo } }	// New terrains to work on
	#declare PatternFuncTwo = function { pigment { TerrainThree } }
	#declare LandClock = Timer( MasterClock, SecondMorphStartsAt, SecondMorphDoneAt, true );
	#declare LandClock = Sinefy( LandClock );	
#end

#declare Landscape =
isosurface {
	//normal on	
	function {
		y - ((1-LandClock)*(PatternFuncOne(x, y, z).y)) - LandClock*(PatternFuncTwo(x, y, z).y)
	}
	threshold 0.001
	accuracy 0.0001
	max_gradient  2.0
	contained_by { 	box { <-18, -5, -15>, <18, 10, 15>	}	}

	texture{ TestiTex } 
//	texture {
//		pigment { rgb <0,1,0> }
//	}

	scale <1,3,1>
}

#declare BlackAndWhiteScape = 
isosurface {
function{ y }
contained_by { box { <-18, -15, -5>, <18, 15, 15>	}	}
//Sorry there's no better way to do this yet.
//Comment out the texture you don't want to test.
texture{
pigment {
//PatternRoot
//PatternMount
PatternRange
//PatternFour
}}}


//comment out whichever you don't want.
object {Landscape}
//object {BlackAndWhiteScape}

#declare Glacier = box { <-1,-1000,-1000>, <1,-999,-999> } // To be on the safe side
#declare GlacierClock = Timer( MasterClock, GlacierEntersAt, GlacierVanishesAt, false);
#if( GlacierClock > 0)
	#include "glacier.inc"
	object{ Glacier }
#end

#if( TreesClock > 0 )
	Treepulate(  Landscape, Glacier,<-18,0,-5>, <18,1.5,15>, 5000 )
#end
#declare CloudDensity = 0.5-0.45*NightLevel;
#declare CloudIntensity = 0.8-0.79*NightLevel; // 0.8
#declare CoudPatternScale = <10,6,1>;
#declare CloudPatternTrans = <-7,0,-2>*clock;
object { Cloud( <4, 2, 2>, CloudPatternTrans )  scale <4,2,2> translate <0,8, 8+15> }

// The dead tree
#declare DeadTreeRot = 0;
#declare DeadTreeSwingClock = Timer(  MasterClock, DeadTreeStartsSwingingAt, DeadTreeStartsFallingAt, true );
#if( DeadTreeSwingClock < 1 )
	#declare DeadTreeRot = 12*sin( DeadTreeSwingClock*2*pi );
#end
#declare DeadTreeToppleClock = Timer( MasterClock, DeadTreeStartsFallingAt, DeadTreeFallCompleteAt, true );
//#declare DeadTreeToppleClock = pow( DeadTreeToppleClock, 2.5 );
#if( DeadTreeToppleClock > 0 )
	#declare DeadTreeRot = 90*DeadTreeToppleClock;
#end
#debug concat( "\nDead tree rotation = ", str( DeadTreeRot, 0, 0 ) )
#if( DeadTreeToppleClock < 1 )
	#include "DeadTree.inc"
	object { DeadTree scale 0.0025 rotate z*-DeadTreeRot translate <-0.071,1.69,-4.8>}
#end

// The live tree
#declare TreeClock = Timer(  MasterClock, LiveTreeStartsAppearingAt, LiveTreeDoneAt, true );
#declare TreeClock = Sinefy( TreeClock );

#debug concat( "TreeClock: ", str( TreeClock, 0, 3 ) )
#if( TreeClock > 0 )
	#include "LiveTree.inc"
	object { LiveTree scale 0.0025 translate <0.0425,1.6975,-4.8> }
#end

// The city
#declare CityClock = Timer( MasterClock, CityStartsDyingAt, CityGoneAt, true );
#if( CityClock < 1 )
	#include "TheCity.inc"
#end

#declare WaterTable=
box { <-15,-1,-5> <15,0.355,5> material { Water } }
#if( MasterClock >= SecondMorphStartsAt )
	object { WaterTable }
#end

#debug concat( "\nMaster clock: ", str( MasterClock, 0, 3 ) )

fog {
	fog_type 2	//ground_fog
	up <0,0.5,0>
	fog_alt 0.075
	distance 0.08
	rgbft <0.757008, 0.850164, 1.000000, 0.6+0.35*NightLevel, 0.4+0.55*NightLevel>	// Day
//	rgbft <0.757008, 0.850164, 1.000000, 0.55, 0.45>*<0.75,0.75,0.75,1,1>	// Night
	turbulence 0.15
}


camera
{
	up y
	right x*16/9
  	location  <0.0 , 1.77 ,-4.9>
	look_at   <0.0 , 1.2 , 0.0>
	// Render at 500x281, or something similar
}
