//////////////////////////////////////////////////////////////////////////
// CameraMotion.inc  1.0  KEA  Copyright(c) 2003, K.E. Ayers Designs    //
//////////////////////////////////////////////////////////////////////////


//////////////////////////////////////////////////////////////////////////
// TO DO:                                                               //
// ==================================================================== //
//                                                                      //
// 1)   There are a couple of discontinuities in the camera's motion    //
//      around the time Ball Boy charges the west wall.                 //
//////////////////////////////////////////////////////////////////////////


//////////////////////////////////////////////////////////////////////////
// Computes the eye position and look at position based on the          //
// animation clock.  This file computes the following variables:        //
//                                                                      //
//  EyePoint:       A vector defining the camera's location.            //
//  LookAtPoint:    A vector defining the point at which the camera is  //
//                  looking.                                            //
//  LightPoint:     A vector defining the light's location.             //
//  FadeAmount:     A floating point value representing the amount by   //
//                  which the scene is to be faded:                     //
//                      0=no fade, the scene is at full brightness      //
//                      1=full fade, the scene is completely dark       //
//                                                                      //
// Here are the scene identifiers:                                      //
//                                                                      //
//  SCN_Intro                                                           //
//  SCN_FadingIn                                                        //
//  SCN_CameraPansUp                                                    //
//  SCN_RollsToEndOfHall                                                //
//  SCN_TurnsToFaceNorth                                                //
//  SCN_RollsIntoRoom                                                   //
//  SCN_TurnsToFaceEast                                                 //
//  SCN_LooksAtEastWall                                                 //
//  SCN_LooksDetermined                                                 //
//  SCN_ChargesEastWall                                                 //
//  SCN_CollidesWithEastWall                                            //
//  SCN_RollsBackFromEastWall                                           //
//  SCN_EastWallStarsAppear                                             //
//  SCN_TurnsToFaceWest                                                 //
//  SCN_LooksFrustrated                                                 //
//  SCN_ChargesWestWall                                                 //
//  SCN_CollidesWithWestWall                                            //
//  SCN_RollsBackFromWestWall                                           //
//  SCN_WestWallStarsAppear                                             //
//  SCN_TurnsToFaceCenter                                               //
//  SCN_LooksSad                                                        //
//  SCN_RollsToCenterOfRoom                                             //
//  SCN_TurnsToFaceNorthWall                                            //
//  SCN_LooksUp                                                         //
//  SCN_SeesTopOfWall                                                   //
//  SCN_Leaps                                                           //
//  SCN_LeapsHigher                                                     //
//  SCN_LeapsOverWall                                                   //
//  SCN_EscapeLeap1                                                     //
//  SCN_EscapeLeap2                                                     //
//  SCN_EscapeLeap3                                                     //
//  SCN_EscapeLeap4                                                     //
// The fade-out starts two seconds from the end.                        //
//  SCN_FadingOut                                                       //
//////////////////////////////////////////////////////////////////////////

#ifndef (_CameraMotion_Inc_)
#declare _CameraMotion_Inc_ = true;

#include "SceneConstants.inc"
#include "SceneTiming.inc"


// The light's position probably won't change.        
#declare LightPoint =
<
    WestWallPos + PenWallHgt * 2,
    PenWallHgt * 4,
    SouthWallPos - PenWallHgt * 5
>;


//////////////////////////////////////////////////////////////////////////
// Here's the pen's general layout, with respect to the number of turns //
// along Ball Boy's various paths:                                      //
//                                                                      //
//                              +Z (N)                                  //
//                                ^                                     //
//                                |                                     //
//            XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX                //
//            X                                        X                //
//            X                                        X                //
//            X                                        X                //
//            X                                        X                //
//            X                                        X                //
//            X                                        X                //
//            X                 (Room)                 X                //
//            X     |                            |     X                //
//       -X<--X .5t-+<--------- 3 turns -------->+-    X-->+X           //
//       (W)  X     ^                            |     X  (E)           //
//            X     |                                  X                //
//            X     |                                  X                //
//            X     |                                  X                //
//            X  1.5 turns                             X                //
//            X     |                                  X                //
//            X     |                                  X                //
//            X     |                                  X                //
//            X    -+-.5t XXXXXXXXXXXXXXXXXXXXXXXXXXXXXX                //
//            X     |                                  X                //
//            X  .5 turn                               X                //
//            X     |          (Hall)                  X                //
//            X     v                            |     X                //
//            X .5t-+<--------- 3 turns -------->+-.5t X                //
//            X     |           (LA1)            |     X                //
//            X                                (LA2*)  X                //
//            X                                        X                //
//            X                                        X                //
//            XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX                //
//                                |                                     //
//          (EP2)                 v                                     //
//                              -Z (S)                                  //
//                                                                      //
//                              (EP1)                                   //
//                                                                      //
//                                                                      //
//////////////////////////////////////////////////////////////////////////

// Here's where the camera starts.
#local CameraStart      = < 0,
                            PenWallHgt * 0.5,
                            SouthWallPos - PenWallHgt * 4 >;
#local LookStart        = < 0, PenWallHgt * 0.5, SouthWallPos >;

// Here's where the camera pans to for the opening scene.
#local CameraPan1       = < WestWallPos + PenWallHgt * 0,
                            PenWallHgt * 3.5,
                            SouthWallPos - PenWallHgt * 3 >;
#local LookPan1         = < -PenWallHgt * 0.55,
                            BallBoyRad,
                            BodyPosition.z >;

// Here's where the camera moves to as Ball Boy turns and
// rolls into the room. The camera will continue to track
// the character's position.
#local CameraSECorner   = < EastWallPos,
                            PenWallHgt * 3.5,
                            SouthWallPos - PenWallHgt * 2 >;
#local LookSWCorner     = < WestWallPos + WestMarkOfs,
                            BallBoyRad,
                            SouthWallPos + SouthMarkOfs >;

// Here's the camera position as it hovers over the center
// of the "hall".
#local CameraCenterHall = < RoomCenterXOfs,
                            PenWallHgt * 3.5,
                            SouthWallPos - PenWallHgt * 3 >;

// Here's where we'll move the camera to before he starts
// his charge for the west wall.
#local CameraEastWall   = < EastWallPos + PenWallHgt,
                            PenWallHgt * 3.5,
                            SouthWallPos + SouthMarkOfs >;


// This is the camera position while the character is along the west
// wall in the room.
#local LookWestWall     = < WestWallPos + WestMarkOfs,
                            BallBoyRad,
                            NorthWallPos - NorthMarkOfs >;

// This is where the camera is looking when the character is
// in the center of the room.
#local LookCenterRoom   = < RoomCenterXOfs,
                            BallBoyRad,
                            NorthWallPos - NorthMarkOfs >; 
                                                        
// Here's where the camera ends up.
#local CameraEnd        = < (RoomCenterXOfs + EastWallPos) / 2,
                            PenWallHgt * 3.5,
                            SouthWallPos + SouthMarkOfs >;
#local LookEnd          = < RoomCenterXOfs,
                            BallBoyRad,
                            NorthWallPos >; 

//////////////////////////////////////////////////////////////////////////


#switch (ActiveScene)
    #range (SCN_Intro, SCN_FadingIn)
        #declare EyePoint       = CameraStart;        
        #declare LookAtPoint    = LookStart;
        #declare FadeAmount     = ((ActiveScene = SCN_Intro)
                                        ? 1 : (1 - SceneClock));
    #break

    #case (SCN_CameraPansUp)
        #declare EyePoint       = PanVector(CameraStart, CameraPan1, SceneClock);        
        #declare LookAtPoint    = PanVector(LookStart, LookPan1, SceneClock);
    #break
    
    #case (SCN_RollsToEndOfHall)
        #declare EyePoint       = CameraPan1;        
        #declare LookAtPoint    = PanVector(LookPan1, LookSWCorner, SceneClock);
    #break
    
    #range (SCN_TurnsToFaceNorth, SCN_LooksDetermined)
        // As the character turns and moves into the room, the
        // camera will move along the southwall to the center.
        #local mvClk = MultiSceneClock(SCN_TurnsToFaceNorth,
                                       SCN_ChargesEastWall);
        #declare EyePoint       = PanVector(CameraPan1, CameraCenterHall, mvClk);        
        #declare LookAtPoint    = BodyPosition;
    #break
    
    #range (SCN_ChargesEastWall, SCN_RollsBackFromEastWall)
        #declare EyePoint       = CameraCenterHall;
        #declare LookAtPoint    = BodyPosition;
    #break

    #case (SCN_EastWallStarsAppear)
        #declare EyePoint       = PanVector(CameraCenterHall, CameraEastWall, SceneClock);
        #declare LookAtPoint    = BodyPosition;
    #break
    
    #range (SCN_TurnsToFaceWest, SCN_LooksFrustrated)
        #declare EyePoint       = CameraEastWall;
        #declare LookAtPoint    = BodyPosition;
    #break

    #case (SCN_ChargesWestWall)
        #declare EyePoint       = PanVector(CameraEastWall, CameraEnd, SceneClock);
        #declare LookAtPoint    = BodyPosition;
    #break

    #range (SCN_CollidesWithWestWall, SCN_TurnsToFaceCenter)
        #declare EyePoint       = CameraEnd;
        #declare LookAtPoint    = BodyPosition;
    #break
    
    #case (SCN_LooksSad)
        #declare EyePoint       = CameraEnd;
        #declare LookAtPoint    = LookWestWall;
    #break
    
    #case (SCN_RollsToCenterOfRoom)
        #declare EyePoint       = CameraEnd;
        #declare LookAtPoint    = PanVector(LookWestWall, LookCenterRoom, SceneClock);
    #break

    #range(SCN_TurnsToFaceNorthWall, SCN_Leaps)        
        #declare EyePoint       = CameraEnd;
        #declare LookAtPoint    = LookCenterRoom;
    #break

    #case (SCN_LeapsHigher)
        #declare EyePoint       = CameraEnd;
        #declare LookAtPoint    = PanVector(LookCenterRoom, LookEnd, SceneClock);
    #break
    
    #else
        // The camera is at its final location.
        #declare EyePoint       = CameraEnd;        
        #declare LookAtPoint    = LookEnd;

        // The scene fades at the end.
        #if (AnimationClock >= SCN_FadingOut)
            #local FadeLen  = AnimationLength - SCN_FadingOut;
            #local FadeProg = AnimationClock  - SCN_FadingOut;

            #if (FadeLen > 0)
                #declare FadeAmount = FadeProg / FadeLen;
            #end
        #end

#end    // #switch (ActiveScene)


#end    // #ifndef (_CameraMotion_Inc_)
