//////////////////////////////////////////////////////////////////////////
// EyeMotion.inc  1.0  KEA  Copyright(c) 2003, K.E. Ayers Designs       //
//////////////////////////////////////////////////////////////////////////
// Controls the motion of our character's eyes, including the blink     //
// action. It outputs the following global values:                      //
//                                                                      //
// LeftEyePosition          A vector that specifies the position of     //
//                          the character's left eye, with respect to   //
//                          the center of the body.                     //
// LeftEyeRotate            A vector that specifies the Y-axis          //
//                          (usually) rotation of the character's left  //
//                          eye.                                        //
// LeftEyeScale             A vector that shapes the left eye.          //
// LeftEyebrowPosition      A vector that specifies the position of the //
//                          character's left eyebrow, relative to the   //
//                          body's position.                            //
// LeftBlinkRotate          A vector that specifies the Z-axis rotation //
//                          for the left eye's "eyelid".                //
// RightEyePosition         A vector that specifies the position of     //
//                          the character's right eye, with respect to  //
//                          the center of the body.                     //
// RightEyeRotate           A vector that specifies the Y-axis          //
//                          (usually) rotation of the character's right //
//                          eye.                                        //
// RightEyeScale            A vector that shapes the right eye.         //
// RightEyebrowPosition     A vector that specifies the position of the //
//                          character's right eyebrow.                  //
// RightBlinkRotate         A vector that specifies the Z-axis rotation //
//                          for the left eye's "eyelid".                //
// EyeRotate                A vector that specifies the Z-axis rotation //
//                          (usually) for the pair of eyes. Typically   //
//                          used to give the illusion of the character  //
//                          looking up or down.                         //
//////////////////////////////////////////////////////////////////////////

#ifndef (_EyeMotion_Inc_)
#declare _EyeMotion_Inc_ = true;

#include "SceneConstants.inc"
#include "SceneTiming.inc"
#include "BodyMotion.inc"


// The horizontal position of the eye relative to the body's centerline.
#local EyeXOfs = BallBoyRad * 1.1;
// The vertical position of the eye relative to the body's centerline.
#local EyeYOfs = BallBoyRad * 0.8;
// The separation between the eyes.
#local EyeSep = BallBoyRad * 0.7;
// The default vertical position for the eyebrow, relative to
// the eye's centerline.
#local EyebrowYOfs = 0;


//////////////////////////////////////////////////////////////////////////
// NOTE:    Ultimately, the eye motion parameters will be calculated    //
//          based on the current animation clock.                       //
//////////////////////////////////////////////////////////////////////////

#local EyeLidRestAngle      = -20;
#local EyeLidClosedAngle    = -90;


// The left eye's starting position.
#local LEXOfs = EyeXOfs;
#local LEYOfs = EyeYOfs;
#local LEZOfs = EyeSep / 2;

// The left eye's starting rotation.
#local LEXRot = 0;
#local LEYRot = 0;
#local LEZRot = 0;

// The left eye's shape scale factors.
#local LEXScl = 1;
#local LEYScl = 1;
#local LEZScl = 1;

// The left eye's blink (eyelid) rotation.
#local LLXRot = 0;
#local LLYRot = 0;
#local LLZRot = EyeLidRestAngle;

// The left eyebrow's position relative to the eye's centerline.
#local LBXOfs = 0;
#local LBYOfs = EyebrowYOfs;
#local LBZOfs = 0;


// The right eye's starting position.
#local REXOfs = EyeXOfs;
#local REYOfs = EyeYOfs;
#local REZOfs = -EyeSep / 2;

// The right eye's starting rotation.
#local REXRot = 0;
#local REYRot = 0;
#local REZRot = 0;

// The right eye's shape scale factors.
#local REXScl = 1;
#local REYScl = 1;
#local REZScl = 1;

// The right eye's blink (eyelid) rotation.
#local RLXRot = 0;
#local RLYRot = 0;
#local RLZRot = EyeLidRestAngle;

// The right eyebrow's position relative to the eye's centerline.
#local RBXOfs = 0;
#local RBYOfs = EyebrowYOfs;
#local RBZOfs = 0;

// The rotation for the pair of eyes, relative to the body's center.
#local EyeXRot = 0;
#local EyeYRot = 0;
#local EyeZRot = 0;

// This will be set to false if a scene is doing something with
// the eyes other thqan the default blink. In that case, the
// scene will set the eye parameters accordingly.
#local isBlinking       = true;
#local LookAtCameraYRot = -45;
#local LookAtCameraZRot =  15;

// Control the eye motions during specific scenes.
#switch (ActiveScene)
    #case (SCN_CameraPansUp)
        #local LEYRot = LookAtCameraYRot;
        #local LEZRot = LookAtCameraZRot;
        #local REYRot = LEYRot;
        #local REZRot = LEZRot;
    #break

    #case (SCN_RollsToEndOfHall)
        #if (SceneClock < 0.5)
            #local LEYRot = LookAtCameraYRot * (1 - SceneClock * 2);
            #local LEZRot = LookAtCameraZRot * (1 - SceneClock * 2);
            #local REYRot = LEYRot;
            #local REZRot = LEZRot;
        #end
    #break
            
    #case (SCN_LooksDetermined)
    #case (SCN_LooksFrustrated)
        #local isBlinking = false;
        #local LLZRot = EyeLidRestAngle
                            + (EyeLidClosedAngle
                                - EyeLidRestAngle * 0.8)
                                    * SceneClock;
        #local RLZRot = LLZRot;
    #break
    
    #case (SCN_ChargesEastWall)
    #case (SCN_ChargesEastWall)
        #local isBlinking = false;
        #local LLZRot = EyeLidRestAngle
                            + (EyeLidClosedAngle
                                - EyeLidRestAngle * 0.8);
        #local RLZRot = LLZRot;
    #break
            
#end    // #switch (ActiveScene)

#if (isBlinking)
    // FOR NOW: We'll just have Ball Boy blink every two seconds.
    #local BlinkTime        = 2;
    #local BlinkClock       = mod(AnimationClock, BlinkTime) / BlinkTime;
    #local BlinkCloseStart  = 0.6;
    #local BlinkCloseEnd    = 0.8; 
    #local BlinkOpenStart   = 0.9;
    #local BlinkOpenEnd     = 1.0;
    
    #switch (BlinkClock)
        #range (BlinkCloseStart, BlinkCloseEnd)
            #local BlinkAmt =
                (BlinkClock - BlinkCloseStart)
                    / (BlinkCloseEnd - BlinkCloseStart);
        #break

        #range (BlinkCloseEnd, BlinkOpenStart)
            #local BlinkAmt = 1;
        #break
                
        #range (BlinkOpenStart, BlinkOpenEnd)
            #local BlinkAmt =
                1 - ((BlinkClock - BlinkOpenStart)
                        / (BlinkOpenEnd - BlinkOpenStart));
        #break
        
        #else
            #local BlinkAmt = 0;
    #end

    #local LLZRot = EyeLidRestAngle
                        + (EyeLidClosedAngle - EyeLidRestAngle)
                                * BlinkAmt;
    #local RLZRot = LLZRot;

#end    // #if (isBlinking)


// Assign the motion values for use by other files.
#declare LeftEyePosition        = < LEXOfs, LEYOfs, LEZOfs >;
#declare LeftEyeRotate          = < LEXRot, LEYRot, LEZRot >;
#declare LeftEyeScale           = < LEXScl, LEYScl, LEZScl >;
#declare LeftBlinkRotate        = < LLXRot, LLYRot, LLZRot >;
#declare LeftEyebrowPosition    = < LBXOfs, LBYOfs, LBZOfs >;

#declare RightEyePosition       = < REXOfs, REYOfs, REZOfs >;
#declare RightEyeRotate         = < REXRot, REYRot, REZRot >;
#declare RightEyeScale          = < REXScl, REYScl, REZScl >;
#declare RightBlinkRotate       = < RLXRot, RLYRot, RLZRot >;
#declare RightEyebrowPosition   = < RBXOfs, RBYOfs, RBZOfs >;

#declare EyeRotate              = < EyeXRot, EyeYRot, EyeZRot >;


#end    // #ifndef (_EyeMotion_Inc_)

