// Persistence of Vision Ray Tracer Scene Description File
// File: look_at.inc
// Vers: 3.5
// Desc: Macro to use Look_At-Functionality with objects 
// Date: 01.04.2002 (dd.mm.yy)
// Auth: Tim Nikias Wenclawiak 
// Last Update: 24.09.2003 (dd.mm.yy)

//Updates
// 24.09.2003 - Fixed a "Divide by Zero" Warning
// 23.09.2003 - Added Look_At_Raised Macro
// 18.07.2003 - Modified algorithm to a more compact version

#declare _look_at_inc_tnw=1;

/* General Macro Description

The Look_At Macros assume an object sitting at the origin,
facing -z with y being the sky_vector.
The Macros, along with any other data, will return a
Rotation-Vector which will rotate the object to directly
face the Target-Position.

Examples:
cone{<0,0,0>,2 <0,0,-2>,0 //A cone pointing towards -z
 pigment{rgb 1}
 rotate Look_At(<5,8,20>) //rotates the cone to look at the given position
 }

cone{<0,1,0>,2 <0,1,-2>,0 //A cone pointing towards -z, but one unit above the origin
 pigment{rgb 1}
 rotate Look_At_Raised(<5,8,20>,1) //rotates the cone around <0,0,0> to look at the given position
 }

*/

//Homepage:
// www.digitaltwilight.de/no_lights
//Email:
// Tim.Nikias@gmx.de

//Look_At
//Parameters:
// 1. Target-Position
//Returns:
// 1. Rotation-Vector
//Notes:
// Assumes object at <0,0,0> facing -z, y=sky_vector
#macro Look_At(Target)
 #if (Target.y!=0)
  #local X_Rot_Float = degrees(atan2(Target.y,vlength(Target*<1,0,1>))); #else
  #local X_Rot_Float = 0; #end
 #if (Target.x!=0)
  #local Y_Rot_Float = degrees(atan2(-Target.x,-Target.z)); #else
  #if (Target.z<0)
   #local Y_Rot_Float = 0; #else
   #local Y_Rot_Float =180; #end
  #end
 #local Result=<X_Rot_Float, Y_Rot_Float, 0>;
 (Result)
#end

//Look_At_Raised
//Parameters:
// 1. Target-Position
// 2. Height above Origin of Object
//Returns:
// 1. Rotation-Vector
//Notes:
// Assumes object at <0,0,0>, facing -z, y=sky_vector, but
// unlike the original Look_At, this macro assumes that the
// position which is to face the target is y*Height above
// <0,0,0> of the object.
#macro Look_At_Raised(Target,Raised_Height)
 //Turn
 // Requires mapping to y-plane to just check for Y-Rotation,
 // and then some tweaking on the results for actual position
 // of focal-point
 #local _dist = vlength(Target*<1,0,1>);
 #if (-Target.x<0)
  #local _NegPos_Mod = -1; #else
  #local _NegPos_Mod =  1; #end
 #local _y_angle = degrees(acos(sqrt(_dist*_dist-pow(Target.x,2))/_dist))*_NegPos_Mod;
 #if (Target.z>0) #local _y_angle = -_y_angle+180; #end
 //Raise
 // Requires adjustment of focal-point to get a plane on which
 // to calculate the X-Rotation
 #local _newFocal = vrotate(Target,-y*_y_angle);
 #local _dist = vlength(_newFocal*<0,1,1>);
 //Hypotenuse
 #local _x_angle_hyp = degrees(acos(sqrt(_dist*_dist-pow(_newFocal.y,2))/_dist));
 //Interior-Angle on triangle
 #local _x_angle_int = degrees(acos(sqrt(_dist*_dist-Raised_Height*Raised_Height)/_dist));
 //Prepare Return-Vector and Return
 #local Result = <_x_angle_hyp-_x_angle_int,_y_angle,0>;
 (Result)
#end
