// Persistence of Vision Ray Tracer Scene Description File
// File: mmmm.inc
// Vers: 3.5
// Desc: Mesh-Modifying-Macros
// Date: 28.03.02 (dd.mm.yy)
// Auth: Tim Nikias Wenclawiak
// Latest Update: 13.09.04 (dd.mm.yy)

// 13.09.04 - Reversed node-saving from Mirror_Mesh_Dat to retain old nodes in same
//            indices on the new nodes, and fixed the node-mirroring
//          - Introduced override-switch to Mirror_Mesh_Dat to override old behaviour
//            of checking for center-axis (instead of simply changing it, I wanted to
//            keep the old behaviour for older scene files)
// 19.09.03 - Reversed _axis and _md in Find_Min/Max, were wrong compared to Help

//Description:
// This include supplies some basic macros for the generation, handling and
// alteration of meshes. It does not model an actual mesh, but requires
// 2D-Arrays of vectors/positions. Assuming that a 4x4 array describes a
// rectangular mesh with 16 nodes, with [0][0] being one edge, [4][0] another,
// [0][4] the third and [4][4] the last edge.
// Simply put, its a set of macros that are designed for rectangular and
// abstract meshes, "abstract" meaning that the macros don't use an actual
// object-mesh, but only the nodes that would be put inside the triangle-
// statements.

//Should be overwritten, but are "defaulted":
//  _Normal_Addition_Macro(X,Z,Val)
//  _Uneven_Calc_Macro(X,Z,Val)
// Both are meant to be declared by the User. The implementation should be based upon
// the idea that X and Z are values from 0 to 1, which represent a value going from
// the [0][0] edge of a 2D-Array to its maximum. "Val" is an identifier that should
// be #declared within the macro, making use of POV's Parameter-Returning, and should
// also range from 0 to 1.

//_Normal_Addition_Macro
// Two macros may be used to deform the mesh based on its own or other given normals.
// These are Add_Normals and Add_Given_Normals. As described above, the macro may be
// used to deform the mesh e.g. only while X-Values are below .5, which would result in
// a deformation only affecting one side of the mesh.

//_Uneven_Calc_Macro
// Uneven_Scale_Mesh_Dat, Uneven_Rotate_Mesh_Dat and Uneven_Translate_Mesh_Dat
// make use of this macro. The idea is similiar to the of the _Normal_Addition_Macro:
// The translation/scale is applied based upon the X and Z values, resulting in the
// desired, User-Programmed uneven scaling/translation/rotation.

//Homepage:
// www.nolights.de
//Email:
// Tim.Nikias(@)nolights.de

//Other files check this variable to see if the macros were already included
#declare _mmm_inc_tnw=1;

/****  VNormalize()  ****/
#macro VNormalize(_Vec)
 //Create proper return-value
 #if (vlength(_Vec))
  #local _Ret=vnormalize(_Vec); #else
  #local _Ret=<0,0,0>; #end
 //Return:
 _Ret
#end

//Macro for normal additions
#macro _Normal_Addition_Macro(X,Z,Val) #declare Val=(X-Z); #end
//Macro for uneven-calculations
#macro _Uneven_Calc_Macro(X,Z,Val) #declare Val=1; #end


/************************************************************************** Mesh-Creation *************************************************/

/*************/
/**Make_Mesh**/
/*************/
//Give: 2D-Array (mesh-field)
//Return: Mesh
#macro Make_Mesh(_data_array)
 #local _dX=dimension_size(_data_array,1);
 #local _dZ=dimension_size(_data_array,2);
 mesh{
 //Mesh with triangles
  #local A=0;
  #while (A<_dX-1)
   #local B=0;
   #while (B<_dZ-1)
    //Shortest connection
    #if (
        vlength(_data_array[A][B]-_data_array[A+1][B+1]) >=
        vlength(_data_array[A][B+1]-_data_array[A+1][B]) )
     #if (vlength(_data_array[A][B+1]-_data_array[A+1][B])!=0)
      #if (
          (vlength(_data_array[A][B+1]-_data_array[A][B])!=0) &
          (vlength(_data_array[A+1][B]-_data_array[A][B])!=0))
       triangle{_data_array[ A ][B+1],_data_array[A+1][ B ],_data_array[ A ][ B ]}
      #end
      #if (
          (vlength(_data_array[A][B+1]-_data_array[A+1][B+1])!=0) &
          (vlength(_data_array[A+1][B]-_data_array[A+1][B+1])!=0))
       triangle{_data_array[ A ][B+1],_data_array[A+1][ B ],_data_array[A+1][B+1]}
      #end
     #end
    #else
     #if (vlength(_data_array[A][B]-_data_array[A+1][B+1])!=0)
      #if (
          (vlength(_data_array[ A ][ B ]-_data_array[A][B+1])!=0) &
          (vlength(_data_array[A+1][B+1]-_data_array[A][B+1])!=0))
        triangle{_data_array[A+1][B+1],_data_array[ A ][ B ],_data_array[ A ][B+1]}
      #end
      #if (
          (vlength(_data_array[ A ][ B ]-_data_array[A+1][B])!=0) &
          (vlength(_data_array[A+1][B+1]-_data_array[A+1][B])!=0))
       triangle{_data_array[A+1][B+1],_data_array[ A ][ B ],_data_array[A+1][ B ]}
      #end
     #end
    #end
   #local B=B+1;
   #end
  #local A=A+1;
  #end
 }
#end

/******************/
/**Make_Triangles**/
/******************/
//Give: 2D-Array (mesh-field)
//Return: Triangles
#macro Make_Triangles(_data_array)
 #local _dX=dimension_size(_data_array,1);
 #local _dZ=dimension_size(_data_array,2);
 //Mesh with triangles
  #local A=0;
  #while (A<_dX-1)
   #local B=0;
   #while (B<_dZ-1)
    //Shortest connection
    #if (
        vlength(_data_array[A][B]-_data_array[A+1][B+1]) >=
        vlength(_data_array[A][B+1]-_data_array[A+1][B]) )
     #if (vlength(_data_array[A][B+1]-_data_array[A+1][B])!=0)
      #if (
          (vlength(_data_array[A][B+1]-_data_array[A][B])!=0) &
          (vlength(_data_array[A+1][B]-_data_array[A][B])!=0))
       triangle{_data_array[ A ][B+1],_data_array[A+1][ B ],_data_array[ A ][ B ]}
      #end
      #if (
          (vlength(_data_array[A][B+1]-_data_array[A+1][B+1])!=0) &
          (vlength(_data_array[A+1][B]-_data_array[A+1][B+1])!=0))
       triangle{_data_array[ A ][B+1],_data_array[A+1][ B ],_data_array[A+1][B+1]}
      #end
     #end
    #else
     #if (vlength(_data_array[A][B]-_data_array[A+1][B+1])!=0)
      #if (
          (vlength(_data_array[ A ][ B ]-_data_array[A][B+1])!=0) &
          (vlength(_data_array[A+1][B+1]-_data_array[A][B+1])!=0))
        triangle{_data_array[A+1][B+1],_data_array[ A ][ B ],_data_array[ A ][B+1]}
      #end
      #if (
          (vlength(_data_array[ A ][ B ]-_data_array[A+1][B])!=0) &
          (vlength(_data_array[A+1][B+1]-_data_array[A+1][B])!=0))
       triangle{_data_array[A+1][B+1],_data_array[ A ][ B ],_data_array[A+1][ B ]}
      #end
     #end
    #end
   #local B=B+1;
   #end
  #local A=A+1;
  #end
#end

/********************/
/**Make_UVTriangles**/
/********************/
//Give: 2D-Array (mesh-field)
//Return: Triangles with UV-Data
#macro Make_UVTriangles(_data_array)
 #local _dX=dimension_size(_data_array,1);
 #local _dZ=dimension_size(_data_array,2);
 //Mesh with triangles
  #local A=0;
  #while (A<_dX-1)
   #local B=0;
   #while (B<_dZ-1)
    //Shortest connection
    #if (
        vlength(_data_array[A][B]-_data_array[A+1][B+1]) >=
        vlength(_data_array[A][B+1]-_data_array[A+1][B]) )
     #if (vlength(_data_array[A][B+1]-_data_array[A+1][B])!=0)
      #if (
          (vlength(_data_array[A][B+1]-_data_array[A][B])!=0) &
          (vlength(_data_array[A+1][B]-_data_array[A][B])!=0))
       triangle{_data_array[ A ][B+1],_data_array[A+1][ B ],_data_array[ A ][ B ]
        uv_vectors <A/(_dX-1),(B+1)/(_dZ-1)>,<(A+1)/(_dX-1),B/(_dZ-1)>,<A/(_dX-1),B/(_dZ-1)>
       }
      #end
      #if (
          (vlength(_data_array[A][B+1]-_data_array[A+1][B+1])!=0) &
          (vlength(_data_array[A+1][B]-_data_array[A+1][B+1])!=0))
       triangle{_data_array[ A ][B+1],_data_array[A+1][ B ],_data_array[A+1][B+1]
        uv_vectors <A/(_dX-1),(B+1)/(_dZ-1)>,<(A+1)/(_dX-1),B/(_dZ-1)>,<(A+1)/(_dX-1),(B+1)/(_dZ-1)>
       }
      #end
     #end
    #else
     #if (vlength(_data_array[A][B]-_data_array[A+1][B+1])!=0)
      #if (
          (vlength(_data_array[ A ][ B ]-_data_array[A][B+1])!=0) &
          (vlength(_data_array[A+1][B+1]-_data_array[A][B+1])!=0))
        triangle{_data_array[A+1][B+1],_data_array[ A ][ B ],_data_array[ A ][B+1]
        uv_vectors <(A+1)/(_dX-1),(B+1)/(_dZ-1)>,<A/(_dX-1),B/(_dZ-1)>,<A/(_dX-1),(B+1)/(_dZ-1)>
        }
      #end
      #if (
          (vlength(_data_array[ A ][ B ]-_data_array[A+1][B])!=0) &
          (vlength(_data_array[A+1][B+1]-_data_array[A+1][B])!=0))
       triangle{_data_array[A+1][B+1],_data_array[ A ][ B ],_data_array[A+1][ B ]
        uv_vectors <(A+1)/(_dX-1),(B+1)/(_dZ-1)>,<A/(_dX-1),B/(_dZ-1)>,<(A+1)/(_dX-1),B/(_dZ-1)>
       }
      #end
     #end
    #end
   #local B=B+1;
   #end
  #local A=A+1;
  #end
#end

/***************/
/**Make_UVMesh**/
/***************/
//Give: 2D-Array (mesh-field)
//Return: Mesh with UV-Data
#macro Make_UVMesh(_data_array)
 #local _dX=dimension_size(_data_array,1);
 #local _dZ=dimension_size(_data_array,2);
 mesh{
 //Mesh with triangles
  #local A=0;
  #while (A<_dX-1)
   #local B=0;
   #while (B<_dZ-1)
    //Shortest connection
    #if (
        vlength(_data_array[A][B]-_data_array[A+1][B+1]) >=
        vlength(_data_array[A][B+1]-_data_array[A+1][B]) )
     #if (vlength(_data_array[A][B+1]-_data_array[A+1][B])!=0)
      #if (
          (vlength(_data_array[A][B+1]-_data_array[A][B])!=0) &
          (vlength(_data_array[A+1][B]-_data_array[A][B])!=0))
       triangle{_data_array[ A ][B+1],_data_array[A+1][ B ],_data_array[ A ][ B ]
        uv_vectors <A/(_dX-1),(B+1)/(_dZ-1)>,<(A+1)/(_dX-1),B/(_dZ-1)>,<A/(_dX-1),B/(_dZ-1)>
       }
      #end
      #if (
          (vlength(_data_array[A][B+1]-_data_array[A+1][B+1])!=0) &
          (vlength(_data_array[A+1][B]-_data_array[A+1][B+1])!=0))
       triangle{_data_array[ A ][B+1],_data_array[A+1][ B ],_data_array[A+1][B+1]
        uv_vectors <A/(_dX-1),(B+1)/(_dZ-1)>,<(A+1)/(_dX-1),B/(_dZ-1)>,<(A+1)/(_dX-1),(B+1)/(_dZ-1)>
       }
      #end
     #end
    #else
     #if (vlength(_data_array[A][B]-_data_array[A+1][B+1])!=0)
      #if (
          (vlength(_data_array[ A ][ B ]-_data_array[A][B+1])!=0) &
          (vlength(_data_array[A+1][B+1]-_data_array[A][B+1])!=0))
        triangle{_data_array[A+1][B+1],_data_array[ A ][ B ],_data_array[ A ][B+1]
        uv_vectors <(A+1)/(_dX-1),(B+1)/(_dZ-1)>,<A/(_dX-1),B/(_dZ-1)>,<A/(_dX-1),(B+1)/(_dZ-1)>
        }
      #end
      #if (
          (vlength(_data_array[ A ][ B ]-_data_array[A+1][B])!=0) &
          (vlength(_data_array[A+1][B+1]-_data_array[A+1][B])!=0))
       triangle{_data_array[A+1][B+1],_data_array[ A ][ B ],_data_array[A+1][ B ]
        uv_vectors <(A+1)/(_dX-1),(B+1)/(_dZ-1)>,<A/(_dX-1),B/(_dZ-1)>,<(A+1)/(_dX-1),B/(_dZ-1)>
       }
      #end
     #end
    #end
   #local B=B+1;
   #end
  #local A=A+1;
  #end
 }
#end

/********************/
/**Make_Smooth_Mesh**/
/********************/
//Give: 2D-Array (mesh-field), 2D-Array (normal-field)
//Return: Mesh made of smooth-triangles
#macro Make_Smooth_Mesh(_data_array,_normal_array)
 #local _dX=dimension_size(_data_array,1);
 #local _dZ=dimension_size(_data_array,2);
 //Mesh with triangles
 mesh{
 #local A=0;
 #while (A<_dX-1)
  #local B=0;
  #while (B<_dZ-1)
    //Check for shortest connection
    #if (vlength(_data_array[A][B]-_data_array[A+1][B+1]) >= vlength(_data_array[A][B+1]-_data_array[A+1][B]))
     //Only if mid-section is longer than 0, we may place triangles
     #if (vlength(_data_array[A][B+1]-_data_array[A+1][B])!=0)
      //Check other sides for each triangle
      #if (
          (vlength(_data_array[A][B+1]-_data_array[A][B])!=0) &
          (vlength(_data_array[A+1][B]-_data_array[A][B])!=0)
          )
       smooth_triangle{
        _data_array[ A ][B+1],_normal_array[ A ][B+1],
        _data_array[A+1][ B ],_normal_array[A+1][ B ],
        _data_array[ A ][ B ],_normal_array[ A ][ B ]
        }
      #end//End of Side-Length-Check for triangle
      #if (
          (vlength(_data_array[A][B+1]-_data_array[A+1][B+1])!=0) &
          (vlength(_data_array[A+1][B]-_data_array[A+1][B+1])!=0))
       smooth_triangle{
        _data_array[ A ][B+1],_normal_array[ A ][B+1],
        _data_array[A+1][ B ],_normal_array[A+1][ B ],
        _data_array[A+1][B+1],_normal_array[A+1][B+1]
        }
      #end //End of Side-Length-Check for triangle
     #end //End of mid-Length-Check for both triangles
    #else
     //Only if mid-section is longer than 0, we may place triangles
     #if (vlength(_data_array[A][B]-_data_array[A+1][B+1])!=0)
      //Check other sides for each triangle
      #if (
          (vlength(_data_array[ A ][ B ]-_data_array[A][B+1])!=0) &
          (vlength(_data_array[A+1][B+1]-_data_array[A][B+1])!=0))
       smooth_triangle{
         _data_array[A+1][B+1],_normal_array[A+1][B+1],
         _data_array[ A ][ B ],_normal_array[ A ][ B ],
         _data_array[ A ][B+1],_normal_array[ A ][B+1]
         }
      #end
      //Check other sides for each triangle
      #if (
          (vlength(_data_array[ A ][ B ]-_data_array[A+1][B])!=0) &
          (vlength(_data_array[A+1][B+1]-_data_array[A+1][B])!=0))
       smooth_triangle{
         _data_array[A+1][B+1],_normal_array[A+1][B+1],
         _data_array[ A ][ B ],_normal_array[ A ][ B ],
         _data_array[A+1][ B ],_normal_array[A+1][ B ]
         }
      #end //End of Side-Length-Check for triangle
     #end //End of mid-Length-Check for both triangles
    #end //End of If for Shortest-Connection-Check
  #local B=B+1;
  #end
 #local A=A+1;
 #end
 }
#end

/*************************/
/**Make_Smooth_Triangles**/
/*************************/
//Give: 2D-Array (mesh-field), 2D-Array (normal-field)
//Return: smooth-triangles
#macro Make_Smooth_Triangles(_data_array,_normal_array)
 #local _dX=dimension_size(_data_array,1);
 #local _dZ=dimension_size(_data_array,2);
 #local A=0;
 #while (A<_dX-1)
  #local B=0;
  #while (B<_dZ-1)
    //Check for shortest connection
    #if (vlength(_data_array[A][B]-_data_array[A+1][B+1]) >= vlength(_data_array[A][B+1]-_data_array[A+1][B]))
     //Only if mid-section is longer than 0, we may place triangles
     #if (vlength(_data_array[A][B+1]-_data_array[A+1][B])!=0)
      //Check other sides for each triangle
      #if (
          (vlength(_data_array[A][B+1]-_data_array[A][B])!=0) &
          (vlength(_data_array[A+1][B]-_data_array[A][B])!=0)
          )
       smooth_triangle{
        _data_array[ A ][B+1],_normal_array[ A ][B+1],
        _data_array[A+1][ B ],_normal_array[A+1][ B ],
        _data_array[ A ][ B ],_normal_array[ A ][ B ]
        }
      #end//End of Side-Length-Check for triangle
      #if (
          (vlength(_data_array[A][B+1]-_data_array[A+1][B+1])!=0) &
          (vlength(_data_array[A+1][B]-_data_array[A+1][B+1])!=0))
       smooth_triangle{
        _data_array[ A ][B+1],_normal_array[ A ][B+1],
        _data_array[A+1][ B ],_normal_array[A+1][ B ],
        _data_array[A+1][B+1],_normal_array[A+1][B+1]
        }
      #end //End of Side-Length-Check for triangle
     #end //End of mid-Length-Check for both triangles
    #else
     //Only if mid-section is longer than 0, we may place triangles
     #if (vlength(_data_array[A][B]-_data_array[A+1][B+1])!=0)
      //Check other sides for each triangle
      #if (
          (vlength(_data_array[ A ][ B ]-_data_array[A][B+1])!=0) &
          (vlength(_data_array[A+1][B+1]-_data_array[A][B+1])!=0))
       smooth_triangle{
         _data_array[A+1][B+1],_normal_array[A+1][B+1],
         _data_array[ A ][ B ],_normal_array[ A ][ B ],
         _data_array[ A ][B+1],_normal_array[ A ][B+1]
         }
      #end
      //Check other sides for each triangle
      #if (
          (vlength(_data_array[ A ][ B ]-_data_array[A+1][B])!=0) &
          (vlength(_data_array[A+1][B+1]-_data_array[A+1][B])!=0))
       smooth_triangle{
         _data_array[A+1][B+1],_normal_array[A+1][B+1],
         _data_array[ A ][ B ],_normal_array[ A ][ B ],
         _data_array[A+1][ B ],_normal_array[A+1][ B ]
         }
      #end //End of Side-Length-Check for triangle
     #end //End of mid-Length-Check for both triangles
    #end //End of If for Shortest-Connection-Check
  #local B=B+1;
  #end
 #local A=A+1;
 #end
#end

/***************************/
/**Make_Smooth_UVTriangles**/
/***************************/
//Give: 2D-Array (mesh-field), 2D-Array (normal-field)
//Return: smooth-triangles with UV-Data
#macro Make_Smooth_UVTriangles(_data_array,_normal_array)
 #local _dX=dimension_size(_data_array,1);
 #local _dZ=dimension_size(_data_array,2);
 #local A=0;
 #while (A<_dX-1)
  #local B=0;
  #while (B<_dZ-1)
    //Check for shortest connection
    #if (vlength(_data_array[A][B]-_data_array[A+1][B+1]) >= vlength(_data_array[A][B+1]-_data_array[A+1][B]))
     //Only if mid-section is longer than 0, we may place triangles
     #if (vlength(_data_array[A][B+1]-_data_array[A+1][B])!=0)
      //Check other sides for each triangle
      #if (
          (vlength(_data_array[A][B+1]-_data_array[A][B])!=0) &
          (vlength(_data_array[A+1][B]-_data_array[A][B])!=0)
          )
       smooth_triangle{
        _data_array[ A ][B+1],_normal_array[ A ][B+1],
        _data_array[A+1][ B ],_normal_array[A+1][ B ],
        _data_array[ A ][ B ],_normal_array[ A ][ B ]
        uv_vectors <A/(_dX-1),(B+1)/(_dZ-1)>,<(A+1)/(_dX-1),B/(_dZ-1)>,<A/(_dX-1),B/(_dZ-1)>
        }
      #end//End of Side-Length-Check for triangle
      #if (
          (vlength(_data_array[A][B+1]-_data_array[A+1][B+1])!=0) &
          (vlength(_data_array[A+1][B]-_data_array[A+1][B+1])!=0))
       smooth_triangle{
        _data_array[ A ][B+1],_normal_array[ A ][B+1],
        _data_array[A+1][ B ],_normal_array[A+1][ B ],
        _data_array[A+1][B+1],_normal_array[A+1][B+1]
        uv_vectors <A/(_dX-1),(B+1)/(_dZ-1)>,<(A+1)/(_dX-1),B/(_dZ-1)>,<(A+1)/(_dX-1),(B+1)/(_dZ-1)>
        }
      #end //End of Side-Length-Check for triangle
     #end //End of mid-Length-Check for both triangles
    #else
     //Only if mid-section is longer than 0, we may place triangles
     #if (vlength(_data_array[A][B]-_data_array[A+1][B+1])!=0)
      //Check other sides for each triangle
      #if (
          (vlength(_data_array[ A ][ B ]-_data_array[A][B+1])!=0) &
          (vlength(_data_array[A+1][B+1]-_data_array[A][B+1])!=0))
       smooth_triangle{
         _data_array[A+1][B+1],_normal_array[A+1][B+1],
         _data_array[ A ][ B ],_normal_array[ A ][ B ],
         _data_array[ A ][B+1],_normal_array[ A ][B+1]
        uv_vectors <(A+1)/(_dX-1),(B+1)/(_dZ-1)>,<A/(_dX-1),B/(_dZ-1)>,<A/(_dX-1),(B+1)/(_dZ-1)>
         }
      #end
      //Check other sides for each triangle
      #if (
          (vlength(_data_array[ A ][ B ]-_data_array[A+1][B])!=0) &
          (vlength(_data_array[A+1][B+1]-_data_array[A+1][B])!=0))
       smooth_triangle{
         _data_array[A+1][B+1],_normal_array[A+1][B+1],
         _data_array[ A ][ B ],_normal_array[ A ][ B ],
         _data_array[A+1][ B ],_normal_array[A+1][ B ]
        uv_vectors <(A+1)/(_dX-1),(B+1)/(_dZ-1)>,<A/(_dX-1),B/(_dZ-1)>,<(A+1)/(_dX-1),B/(_dZ-1)>
         }
      #end //End of Side-Length-Check for triangle
     #end //End of mid-Length-Check for both triangles
    #end //End of If for Shortest-Connection-Check
  #local B=B+1;
  #end
 #local A=A+1;
 #end
#end

/**********************/
/**Make_Smooth_UVMesh**/
/**********************/
//Give: 2D-Array (mesh-field), 2D-Array (normal-field)
//Return: smooth-mesh with UV-Data
#macro Make_Smooth_UVMesh(_data_array,_normal_array)
 #local _dX=dimension_size(_data_array,1);
 #local _dZ=dimension_size(_data_array,2);
 mesh{
 #local A=0;
 #while (A<_dX-1)
  #local B=0;
  #while (B<_dZ-1)
    //Check for shortest connection
    #if (vlength(_data_array[A][B]-_data_array[A+1][B+1]) >= vlength(_data_array[A][B+1]-_data_array[A+1][B]))
     //Only if mid-section is longer than 0, we may place triangles
     #if (vlength(_data_array[A][B+1]-_data_array[A+1][B])!=0)
      //Check other sides for each triangle
      #if (
          (vlength(_data_array[A][B+1]-_data_array[A][B])!=0) &
          (vlength(_data_array[A+1][B]-_data_array[A][B])!=0)
          )
       smooth_triangle{
        _data_array[ A ][B+1],_normal_array[ A ][B+1],
        _data_array[A+1][ B ],_normal_array[A+1][ B ],
        _data_array[ A ][ B ],_normal_array[ A ][ B ]
        uv_vectors <A/(_dX-1),(B+1)/(_dZ-1)>,<(A+1)/(_dX-1),B/(_dZ-1)>,<A/(_dX-1),B/(_dZ-1)>
        }
      #end//End of Side-Length-Check for triangle
      #if (
          (vlength(_data_array[A][B+1]-_data_array[A+1][B+1])!=0) &
          (vlength(_data_array[A+1][B]-_data_array[A+1][B+1])!=0))
       smooth_triangle{
        _data_array[ A ][B+1],_normal_array[ A ][B+1],
        _data_array[A+1][ B ],_normal_array[A+1][ B ],
        _data_array[A+1][B+1],_normal_array[A+1][B+1]
        uv_vectors <A/(_dX-1),(B+1)/(_dZ-1)>,<(A+1)/(_dX-1),B/(_dZ-1)>,<(A+1)/(_dX-1),(B+1)/(_dZ-1)>
        }
      #end //End of Side-Length-Check for triangle
     #end //End of mid-Length-Check for both triangles
    #else
     //Only if mid-section is longer than 0, we may place triangles
     #if (vlength(_data_array[A][B]-_data_array[A+1][B+1])!=0)
      //Check other sides for each triangle
      #if (
          (vlength(_data_array[ A ][ B ]-_data_array[A][B+1])!=0) &
          (vlength(_data_array[A+1][B+1]-_data_array[A][B+1])!=0))
       smooth_triangle{
         _data_array[A+1][B+1],_normal_array[A+1][B+1],
         _data_array[ A ][ B ],_normal_array[ A ][ B ],
         _data_array[ A ][B+1],_normal_array[ A ][B+1]
        uv_vectors <(A+1)/(_dX-1),(B+1)/(_dZ-1)>,<A/(_dX-1),B/(_dZ-1)>,<A/(_dX-1),(B+1)/(_dZ-1)>
         }
      #end
      //Check other sides for each triangle
      #if (
          (vlength(_data_array[ A ][ B ]-_data_array[A+1][B])!=0) &
          (vlength(_data_array[A+1][B+1]-_data_array[A+1][B])!=0))
       smooth_triangle{
         _data_array[A+1][B+1],_normal_array[A+1][B+1],
         _data_array[ A ][ B ],_normal_array[ A ][ B ],
         _data_array[A+1][ B ],_normal_array[A+1][ B ]
        uv_vectors <(A+1)/(_dX-1),(B+1)/(_dZ-1)>,<A/(_dX-1),B/(_dZ-1)>,<(A+1)/(_dX-1),B/(_dZ-1)>
         }
      #end //End of Side-Length-Check for triangle
     #end //End of mid-Length-Check for both triangles
    #end //End of If for Shortest-Connection-Check
  #local B=B+1;
  #end
 #local A=A+1;
 #end
 }
#end

/************************************************************************** Field-Changes *************************************************/

/*****************/
/*Mirror_Mesh_Dat*/
/*****************/
//Give: 2D-Array (mesh-field), mirror-vector (x,y,z : along axis)
//-Enlargens mesh if not on axis
#macro Mirror_Mesh_Dat(Node_Field,_MVec,_Override)
 #local _X=dimension_size(Node_Field,1);
 #local _Z=dimension_size(Node_Field,2);
 #local _MirVec=<abs(_MVec.x),abs(_MVec.y),abs(_MVec.z)>;
 #local Offset = on;
 //Check if we are directly on axis, if not...
 #if (_MirVec.x=1)
  #if (Node_Field[0][0].x!=0)
   #local Offset=off;
   #end 
 #end
 #if (_MirVec.y=1)
  #if (Node_Field[0][0].y!=0)
   #local Offset=off;
  #end 
 #end
 #if (_MirVec.z=1)
  #if (Node_Field[0][0].z!=0)
   #local Offset=off;
  #end 
 #end
 #if (_Override=1)
   #local Offset=on;
 #end
 #if (_Override=2)
   #local Offset=off;
 #end
 
 #if (Offset)
  #local _NF=array[_X*2-1][_Z]
  #local A=0;
   #while (A<_X)
   #local B=0;
   #while (B<_Z)
    #local _NF[A][B]=Node_Field[A][B];
    #if (A<_X-1)
      #local _NF[A+_X][B]=Node_Field[_X-A-2][B]*(<1,1,1>-2*_MirVec);
    #end
    #local B=B+1;
   #end
   #local A=A+1;
  #end
 #else
  #local _NF=array[_X*2][_Z]
  #local A=0;
  #while (A<_X)
   #local B=0;
   #while (B<_Z)
    #local _NF[A][B]=Node_Field[A][B];
    #local _NF[A+_X][B]=Node_Field[_X-A-1][B]*(<1,1,1>-2*_MirVec);
    #local B=B+1;
   #end
   #local A=A+1;
  #end
 #end 
 //Return
 #declare Node_Field=_NF
#end

/***************/
/*Trim_Mesh_Dat*/
/***************/
//Give: 2D-Array (Mesh-Field) and trim-vector (x/-x/z/-z)
//Outmost line, as given by trim-vector, is removed
//Note: integer amount of pos/neg value describes amount of removal
#macro Trim_Mesh_Dat(Node_Field,_t_side)
 #local _X=dimension_size(Node_Field,1);
 #local _Z=dimension_size(Node_Field,2);
 #if (_t_side.x<0)
  #local _nf=array[_X+_t_side.x][_Z]
  //Move through entire Data
  #local A=0; #while (A<_X+_t_side.x)
  #local B=0; #while (B<_Z)
     #local _nf[A][B]=Node_Field[A-_t_side.x][B];
  #local B=B+1; #end
  #local A=A+1; #end
 #end
 #if (_t_side.x>0)
  #local _nf=array[_X-_t_side.x][_Z]
  //Move through entire Data
  #local A=0; #while (A<_X-_t_side.x)
  #local B=0; #while (B<_Z)
     #local _nf[A][B]=Node_Field[A][B];
  #local B=B+1; #end
  #local A=A+1; #end
 #end
 #if (_t_side.z<0)
  #local _nf=array[_X][_Z+_t_side.z]
  //Move through entire Data
  #local A=0; #while (A<_X)
  #local B=0; #while (B<_Z+_t_side.z)
     #local _nf[A][B]=Node_Field[A][B+t_side.z];
  #local B=B+1; #end
  #local A=A+1; #end
 #end
 #if (_t_side.z>0)
  #local _nf=array[_X][_Z-_t_side.z]
  //Move through entire Data
  #local A=0; #while (A<_X)
  #local B=0; #while (B<_Z-_t_side.z)
     #local _nf[A][B]=Node_Field[A][B];
  #local B=B+1; #end
  #local A=A+1; #end
 #end
//Change and return
#declare Node_Field=_nf 
#end




/************************************************************************** Mesh-Transforms *************************************************/

/********************/
/*Translate_Mesh_Dat*/
/********************/
//Given 2D-Array (Mesh-field) and translation, given array is altered
#macro Translate_Mesh_Dat(Node_Field,_MVec)
 #local _X=dimension_size(Node_Field,1);
 #local _Z=dimension_size(Node_Field,2);
 //Move through entire Data
 #local A=0;
 #while (A<_X)
  #local B=0;
  #while (B<_Z)
    #declare Node_Field[A][B]=Node_Field[A][B]+_MVec;
  #local B=B+1;
  #end
 #local A=A+1;
 #end
#end

/********************/
/*Rotate_Mesh_Dat*/
/********************/
//Given 2D-Array (Mesh-field) and rotation, given array is altered
#macro Rotate_Mesh_Dat(Node_Field,_RVec)
 #local _X=dimension_size(Node_Field,1);
 #local _Z=dimension_size(Node_Field,2);
 //Move through entire Data
 #local A=0;
 #while (A<_X)
  #local B=0;
  #while (B<_Z)
    #declare Node_Field[A][B]=vrotate(Node_Field[A][B],_RVec);
  #local B=B+1;
  #end
 #local A=A+1;
 #end
#end

/****************/
/*Scale_Mesh_Dat*/
/****************/
//Given 2D-Array (Mesh-field) and scalement, given array is altered
#macro Scale_Mesh_Dat(Node_Field,_SVec)
 #local _X=dimension_size(Node_Field,1);
 #local _Z=dimension_size(Node_Field,2);
 //Move through entire Data
 #local A=0;
 #while (A<_X)
  #local B=0;
  #while (B<_Z)
    #declare Node_Field[A][B]=Node_Field[A][B]*_SVec;
  #local B=B+1;
  #end
 #local A=A+1;
 #end
#end

/***************************/
/*Uneven_Translate_Mesh_Dat*/
/***************************/
//Given 2D-Array (Mesh-field) and translation, given array is altered
#macro Uneven_Translate_Mesh_Dat(Node_Field,_MVec)
 #local _X=dimension_size(Node_Field,1);
 #local _Z=dimension_size(Node_Field,2);
 //Move through entire Data
 #local A=0;
 #while (A<_X)
  #local B=0;
  #while (B<_Z)
    #local Val=0;
    _Uneven_Calc_Macro(A/(_X-1),B/(_Z-1),Val)
    #declare Node_Field[A][B]=Node_Field[A][B]+_MVec*Val;
  #local B=B+1;
  #end
 #local A=A+1;
 #end
#end

/***********************/
/*Uneven_Scale_Mesh_Dat*/
/***********************/
//Give: 2D-Array (Mesh-Field) and plane-vector
//Result: Scalement using _Uneven_Calc_Macro on plane
#macro Uneven_Scale_Mesh_Dat(Node_Field,_vec)
 #local _X=dimension_size(Node_Field,1);
 #local _Z=dimension_size(Node_Field,2);
  //Move through entire Data
  #local A=0; #while (A<_X)
  #local B=0; #while (B<_Z)
     #local Val=0;
     _Uneven_Calc_Macro(A/(_X-1),B/(_Z-1),Val)
     #declare Node_Field[A][B]=Node_Field[A][B]*((<1,1,1>-_vec)+Val*_vec);
  #local B=B+1; #end
  #local A=A+1; #end
#end

/************************/
/*Uneven_Rotate_Mesh_Dat*/
/************************/
//Given 2D-Array (Mesh-field) and rotation, given array is altered
#macro Uneven_Rotate_Mesh_Dat(Node_Field,_RVec)
 #local _X=dimension_size(Node_Field,1);
 #local _Z=dimension_size(Node_Field,2);
 //Move through entire Data
 #local A=0; #while (A<_X)
 #local B=0; #while (B<_Z)
     #local Val=0;
     _Uneven_Calc_Macro(A/(_X-1),B/(_Z-1),Val)
     #declare Node_Field[A][B]=vrotate(Node_Field[A][B],_RVec*Val);
 #local B=B+1; #end
 #local A=A+1; #end
#end

/************************************************************************** Pure-Normals-Related *************************************************/

/****************/
/**Make_Normals**/
/****************/
//Given 2D-Array (mesh-field), returns 2D-Array with normals for the mesh
#macro Make_Normals(_dat_array)
 #local _X=dimension_size(_dat_array,1);
 #local _Z=dimension_size(_dat_array,2);
 #local Normal_Array=array[_X][_Z]
 //Initialize Mesh
 #local A=0;
 #while (A<_X)
  #local B=0;
  #while (B<_Z)
   #local Normal_Array[A][B]=<0,0,0>; 
  #local B=B+1;
  #end
 #local A=A+1;
 #end
 //Generate and add all normals
 #local A=0;
 #while (A<_X-1)
  #local B=0;
  #while (B<_Z-1)
   #local Mid_Sec = (_dat_array[A][B]+_dat_array[A+1][B]+_dat_array[A][B+1]+_dat_array[A+1][B+1])/4;
   #local Nor_1=vcross(_dat_array[A+1][B+1]-Mid_Sec,_dat_array[A][B+1]-Mid_Sec);
   #local Nor_2=vcross(_dat_array[A][B+1]-Mid_Sec,_dat_array[A][B]-Mid_Sec);
   #local Nor_3=vcross(_dat_array[A][B]-Mid_Sec,_dat_array[A+1][B]-Mid_Sec);
   #local Nor_4=vcross(_dat_array[A+1][B]-Mid_Sec,_dat_array[A+1][B+1]-Mid_Sec);
   #local Normal_Array[A][B]=Normal_Array[A][B]+Nor_2+Nor_3;
   #local Normal_Array[A+1][B]=Normal_Array[A+1][B]+Nor_3+Nor_4;
   #local Normal_Array[A][B+1]=Normal_Array[A][B+1]+Nor_1+Nor_2;
   #local Normal_Array[A+1][B+1]=Normal_Array[A+1][B+1]+Nor_4+Nor_1;
  #local B=B+1;
  #end 
 #local A=A+1;
 #end
 //Normalize the entire data
 #local A=0;
 #while (A<_X)
  #local B=0;
  #while (B<_Z)
   #local Normal_Array[A][B]=VNormalize(Normal_Array[A][B]); 
   #if (vlength(Normal_Array[A][B])=0)
    #debug concat("Warning! Index: ",str(A,1,0),",",str(B,1,0)," is <0,0,0>!\n")
   #end
  #local B=B+1;
  #end
 #local A=A+1;
 #end
 //Return
 Normal_Array
#end

//Macro for addition
//#macro _Normal_Addition_Macro(X,Z) (X-Z) #end

/***************/
/**Add_Normals**/
/***************/
//Given 2D-Array (Mesh-Field), normals are calculated and added with macro
#macro Add_Normals(_dat_array,_amount)
 #local _X=dimension_size(_dat_array,1);
 #local _Z=dimension_size(_dat_array,2);
 #local _n_data=Make_Normals(_dat_array)
 #local A=0;
 #while (A<_X)
  #local B=0;
  #while (B<_Z)
   #declare Val=0;
   _Normal_Addition_Macro( A/(_X-1), B/(_Z-1) , Val)
   #declare _dat_array[A][B]=_dat_array[A][B]+_n_data[A][B]*Val*_amount; 
  #local B=B+1;
  #end
 #local A=A+1;
 #end
 
#end

/*********************/
/**Add_Given_Normals**/
/*********************/
//Given 2D-Array (Mesh-Field), normals are calculated and added with macro
#macro Add_Given_Normals(_dat_array,_normal_array,_amount)
 #local _X=dimension_size(_dat_array,1);
 #local _Z=dimension_size(_dat_array,2);
 #local A=0;
 #while (A<_X)
  #local B=0;
  #while (B<_Z)
   #declare Val=0;
   _Normal_Addition_Macro( A/(_X-1), B/(_Z-1) , Val)
   #declare _dat_array[A][B]=_dat_array[A][B]+_normal_array[A][B]*(Val*_amount); 
  #local B=B+1;
  #end
 #local A=A+1;
 #end
#end


/****************/
/*Invert_Normals*/
/****************/
//Given normal-data, this file will invert the normals,
//from pointing outwards to inwards and vice versa.
#macro Invert_Normals(_normal_dat)
 #local _X=dimension_size(_normal_dat,1);
 #local _Z=dimension_size(_normal_dat,2);
 #local A=0; #while (A<_X)
  #local B=0; #while (B<_Z)
   #declare _normal_dat[A][B]=-_normal_dat[A][B];
  #local B=B+1; #end 
 #local A=A+1; #end 
#end

/***************/
/*Check_Normals*/
/***************/
//Checks if normals are unit-length and makes em if they're not
#macro Check_Normals(_nor_data)
 #local _A=dimension_size(_nor_data,1);
 #local _B=dimension_size(_nor_data,2);
 #local A=0;
 #while (A<_A)
  #local B=0;
  #while (B<_B)
   #if (vlength(_nor_data[A][B])!=1)
    #declare _nor_data[A][B]=VNormalize(_nor_data[A][B]);
    #debug concat(" Error at: ",str(A,1,0),","str(B,1,0),"\n")   
   #end  
  #local B=B+1;
  #end 
 #local A=A+1;
 #end
#end

/***********************/
/*Average_2Mesh_Normals*/
/***********************/
//Give: Normal-data #1 and side (x or z), Normal-Data #2 and side (x or z)
//This Macro will average the given sides of the 2D-Normal-Data
//Useful to make two adjacent meshes fit smoothly
#macro Average_2Mesh_Normals(_nd1,_s1,_nd2,_s2)
 #local _X1=dimension_size(_nd1,1);
 #local _Z1=dimension_size(_nd1,2);
 #local _X2=dimension_size(_nd2,1);
 #local _Z2=dimension_size(_nd2,2);
 #if (_s1.x!=0)
  #declare Max_A=_X1;
 #end
 #if (_s1.z!=0)
  #declare Max_A=_Z1;
 #end
 #local A=0; #while (A<Max_A)
  #if (_s1.x= 1) #local _1_Vec=_nd1[A][_Z1-1]; #end
  #if (_s1.x=-1) #local _1_Vec=_nd1[A][0]; #end
  #if (_s1.z= 1) #local _1_Vec=_nd1[_X1-1][A]; #end
  #if (_s1.z=-1) #local _1_Vec=_nd1[0][A]; #end
  #if (_s2.x= 1) #local _2_Vec=_nd2[A][_Z2-1]; #end
  #if (_s2.x=-1) #local _2_Vec=_nd2[A][0]; #end
  #if (_s2.z= 1) #local _2_Vec=_nd2[_X2-1][A]; #end
  #if (_s2.z=-1) #local _2_Vec=_nd2[0][A]; #end
  #local Mid=VNormalize(_1_Vec+_2_Vec);
  #if (_s1.x= 1) #declare _nd1[A][_Z1-1]=Mid; #end
  #if (_s1.x=-1) #declare _nd1[A][0]=Mid; #end
  #if (_s1.z= 1) #declare _nd1[_X1-1][A]=Mid; #end
  #if (_s1.z=-1) #declare _nd1[0][A]=Mid; #end
  #if (_s2.x= 1) #declare _nd2[A][_Z2-1]=Mid; #end
  #if (_s2.x=-1) #declare _nd2[A][0]=Mid; #end
  #if (_s2.z= 1) #declare _nd2[_X2-1][A]=Mid; #end
  #if (_s2.z=-1) #declare _nd2[0][A]=Mid; #end
 #local A=A+1; #end
#end

/*********************/
/*Average_Rim_Normals*/
/*********************/
//Given normal-data and side (x or z), this macro
//will average the sides of the 2D-Normal-Data
//-Useful to smoothen adjacent sides on meshes that close on itself
#macro Average_Rim_Normals(_normal_dat,_side)
 #local _X=dimension_size(_normal_dat,1);
 #local _Z=dimension_size(_normal_dat,2);
 #if (_side.x!=0)
  #local A=0;
  #while (A<_Z)
   #local Mid=vnormalize(_normal_dat[0][A]+_normal_dat[_X-1][A]);
   #declare _normal_dat[  0 ][A]=Mid;
   #declare _normal_dat[_X-1][A]=Mid;
  #local A=A+1;
  #end 
 #end
 #if (_side.z!=0)
  #local A=0;
  #while (A<_X)
   #local Mid=vnormalize(_normal_dat[A][0]+_normal_dat[A][_Z-1]);
   #declare _normal_dat[A][  0 ]=Mid;
   #declare _normal_dat[A][_Z-1]=Mid;
  #local A=A+1;
  #end 
 #end
#end

/******************************************** CSG *************************************************/

/*************/
/**Make_Grid**/
/*************/
//Given a 2D-Array (mesh-field) and a thickness-float, this returns a bunch of cylinders and spheres
#macro Make_Grid(_dt,_thickness)
 #local _X=dimension_size(_dt,1);
 #local _Z=dimension_size(_dt,2);
 //Mesh with triangles
  #local A=0;
  #while (A<_X)
   #local B=0;
   #while (B<_Z)
  sphere{_dt[A][B],_thickness.x}
  #if (A<_X-1)
  #if (vlength(_dt[A][B]-_dt[A+1][B])!=0)
   cylinder{_dt[A][B],_dt[A+1][B],_thickness.x}
  #end #end
  #if (B<_Z-1)
  #if (vlength(_dt[A][B]-_dt[A][B+1])!=0)
   cylinder{_dt[A][B],_dt[A][B+1],_thickness.x}
  #end #end
  #if (A<_X-1 & B<_Z-1)
  #if (vlength(_dt[A+1][B]-_dt[A+1][B+1])!=0)
   cylinder{_dt[A+1][B],_dt[A+1][B+1],_thickness.x}
  #end #end
  #if (A<_X-1 & B<_Z-1)
  #if (vlength(_dt[A][B+1]-_dt[A+1][B+1])!=0)
   cylinder{_dt[A][B+1],_dt[A+1][B+1],_thickness.x}
  #end #end
  #if (A<_X-1 & B<_Z-1)
    //Shortest connection
    #if (
        vlength(_dt[A][B]-_dt[A+1][B+1]) >=
        vlength(_dt[A][B+1]-_dt[A+1][B]) )
     #if (vlength(_dt[A][B+1]-_dt[A+1][B])!=0)
      cylinder{_dt[A][B+1],_dt[A+1][B],_thickness.x}
     #end
    #else
     #if (vlength(_dt[A][B]-_dt[A+1][B+1])!=0)
     cylinder{_dt[A][B],_dt[A+1][B+1],_thickness.x}
     #end
    #end
  #end 
   #local B=B+1;
   #end
  #local A=A+1;
  #end
#end

/**************/
/*Show_Normals*/
/**************/
//Given mesh-data and normal-data, this file will place cones according to mesh and normals
//The last parameter (vector) defines <Base_Radius,Cap_Radius,Length)
#macro Show_Normals(_mesh_dat,_normal_dat,_cony)
 #local _X=dimension_size(_mesh_dat,1);
 #local _Z=dimension_size(_mesh_dat,2);
 #local A=0; #while (A<_X)
  #local B=0; #while (B<_Z)
   #if (vlength(_normal_dat[A][B])!=0)
    cone{_mesh_dat[A][B],_cony.x _mesh_dat[A][B]+_normal_dat[A][B]*_cony.z,_cony.y}
   #else
    #debug " Didn't work for values "
    #debug concat("(A=",str(A,1,0),") and ")
    #debug concat("(B=",str(B,1,0),") because of <0,0,0>-Issue!\n")
   #end
  #local B=B+1; #end 
 #local A=A+1; #end 
#end

/************/
/*Show_Nodes*/
/************/
//Given mesh-data and normal-data, this file
//will place cones according to mesh and normals
//The Vector defines <Base_Radius,Cap_Radius,Length)
#macro Show_Nodes(_mesh_dat,_sphery)
 #local _X=dimension_size(_mesh_dat,1);
 #local _Z=dimension_size(_mesh_dat,2);
 #local A=0; #while (A<_X)
  #local B=0; #while (B<_Z)
    sphere{_mesh_dat[A][B],_sphery.x}
  #local B=B+1; #end 
 #local A=A+1; #end 
#end

/************************************************************************** Output *************************************************/

/**************************/
/**Write_Smooth_Triangles**/
/**************************/
//Given a 2D-Array (mesh-field),normal-field, and a filename this writes a smooth-mesh-object to file
#macro Write_Smooth_Triangles(_data_array,_normal_array,_filename)
#fopen _O _filename write
 #local _dX=dimension_size(_data_array,1);
 #local _dZ=dimension_size(_data_array,2);
 #local A=0;
 #while (A<_dX-1)
  #local B=0;
  #while (B<_dZ-1)
    //Check for shortest connection
    #if (vlength(_data_array[A][B]-_data_array[A+1][B+1]) >= vlength(_data_array[A][B+1]-_data_array[A+1][B]))
     //Only if mid-section is longer than 0, we may place triangles
     #if (vlength(_data_array[A][B+1]-_data_array[A+1][B])!=0)
      //Check other sides for each triangle
      #if (
          (vlength(_data_array[A][B+1]-_data_array[A][B])!=0) &
          (vlength(_data_array[A+1][B]-_data_array[A][B])!=0)
          )
       #write (_O, "smooth_triangle{\n")
       #write (_O, " ",_data_array[ A ][B+1],",",_normal_array[ A ][B+1],"\n")
       #write (_O, " ",_data_array[A+1][ B ],",",_normal_array[A+1][ B ],"\n")
       #write (_O, " ",_data_array[ A ][ B ],",",_normal_array[ A ][ B ],"\n }\n")
      #end//End of Side-Length-Check for triangle
      #if (
          (vlength(_data_array[A][B+1]-_data_array[A+1][B+1])!=0) &
          (vlength(_data_array[A+1][B]-_data_array[A+1][B+1])!=0))
       #write (_O, "smooth_triangle{\n")
       #write (_O, " ",_data_array[ A ][B+1],",",_normal_array[ A ][B+1],"\n")
       #write (_O, " ",_data_array[A+1][ B ],",",_normal_array[A+1][ B ],"\n")
       #write (_O, " ",_data_array[A+1][B+1],",",_normal_array[A+1][B+1],"\n }\n")
      #end //End of Side-Length-Check for triangle
     #end //End of mid-Length-Check for both triangles
    #else
     //Only if mid-section is longer than 0, we may place triangles
     #if (vlength(_data_array[A][B]-_data_array[A+1][B+1])!=0)
      //Check other sides for each triangle
      #if (
          (vlength(_data_array[ A ][ B ]-_data_array[A][B+1])!=0) &
          (vlength(_data_array[A+1][B+1]-_data_array[A][B+1])!=0))
       #write (_O, "smooth_triangle{\n")
       #write (_O, " ",_data_array[A+1][B+1],",",_normal_array[A+1][B+1],"\n")
       #write (_O, " ",_data_array[ A ][ B ],",",_normal_array[ A ][ B ],"\n")
       #write (_O, " ",_data_array[ A ][B+1],",",_normal_array[ A ][B+1],"\n }\n")
      #end
      //Check other sides for each triangle
      #if (
          (vlength(_data_array[ A ][ B ]-_data_array[A+1][B])!=0) &
          (vlength(_data_array[A+1][B+1]-_data_array[A+1][B])!=0))
       #write (_O, "smooth_triangle{\n")
       #write (_O, " ",_data_array[A+1][B+1],",",_normal_array[A+1][B+1],"\n")
       #write (_O, " ",_data_array[ A ][ B ],",",_normal_array[ A ][ B ],"\n")
       #write (_O, " ",_data_array[A+1][ B ],",",_normal_array[A+1][ B ],"\n }\n")
      #end //End of Side-Length-Check for triangle
     #end //End of mid-Length-Check for both triangles
    #end //End of If for Shortest-Connection-Check
  #local B=B+1;
  #end
 #local A=A+1;
 #end
 #fclose _O
#end

/*********************/
/**Write_Smooth_Mesh**/
/*********************/
//Given a 2D-Array (mesh-field),normal-field, and a filename this writes a smooth-mesh-object to file
#macro Write_Smooth_Mesh(_data_array,_normal_array,_filename)
#fopen _O _filename write
 #local _dX=dimension_size(_data_array,1);
 #local _dZ=dimension_size(_data_array,2);
 #write (_O, "mesh{\n")
 #local A=0;
 #while (A<_dX-1)
  #local B=0;
  #while (B<_dZ-1)
    //Check for shortest connection
    #if (vlength(_data_array[A][B]-_data_array[A+1][B+1]) >= vlength(_data_array[A][B+1]-_data_array[A+1][B]))
     //Only if mid-section is longer than 0, we may place triangles
     #if (vlength(_data_array[A][B+1]-_data_array[A+1][B])!=0)
      //Check other sides for each triangle
      #if (
          (vlength(_data_array[A][B+1]-_data_array[A][B])!=0) &
          (vlength(_data_array[A+1][B]-_data_array[A][B])!=0)
          )
       #write (_O, " smooth_triangle{\n")
       #write (_O, "  ",_data_array[ A ][B+1],",",_normal_array[ A ][B+1],"\n")
       #write (_O, "  ",_data_array[A+1][ B ],",",_normal_array[A+1][ B ],"\n")
       #write (_O, "  ",_data_array[ A ][ B ],",",_normal_array[ A ][ B ],"\n }\n")
      #end//End of Side-Length-Check for triangle
      #if (
          (vlength(_data_array[A][B+1]-_data_array[A+1][B+1])!=0) &
          (vlength(_data_array[A+1][B]-_data_array[A+1][B+1])!=0))
       #write (_O, " smooth_triangle{\n")
       #write (_O, "  ",_data_array[ A ][B+1],",",_normal_array[ A ][B+1],"\n")
       #write (_O, "  ",_data_array[A+1][ B ],",",_normal_array[A+1][ B ],"\n")
       #write (_O, "  ",_data_array[A+1][B+1],",",_normal_array[A+1][B+1],"\n }\n")
      #end //End of Side-Length-Check for triangle
     #end //End of mid-Length-Check for both triangles
    #else
     //Only if mid-section is longer than 0, we may place triangles
     #if (vlength(_data_array[A][B]-_data_array[A+1][B+1])!=0)
      //Check other sides for each triangle
      #if (
          (vlength(_data_array[ A ][ B ]-_data_array[A][B+1])!=0) &
          (vlength(_data_array[A+1][B+1]-_data_array[A][B+1])!=0))
       #write (_O, " smooth_triangle{\n")
       #write (_O, "  ",_data_array[A+1][B+1],",",_normal_array[A+1][B+1],"\n")
       #write (_O, "  ",_data_array[ A ][ B ],",",_normal_array[ A ][ B ],"\n")
       #write (_O, "  ",_data_array[ A ][B+1],",",_normal_array[ A ][B+1],"\n }\n")
      #end
      //Check other sides for each triangle
      #if (
          (vlength(_data_array[ A ][ B ]-_data_array[A+1][B])!=0) &
          (vlength(_data_array[A+1][B+1]-_data_array[A+1][B])!=0))
       #write (_O, " smooth_triangle{\n")
       #write (_O, "  ",_data_array[A+1][B+1],",",_normal_array[A+1][B+1],"\n")
       #write (_O, "  ",_data_array[ A ][ B ],",",_normal_array[ A ][ B ],"\n")
       #write (_O, "  ",_data_array[A+1][ B ],",",_normal_array[A+1][ B ],"\n }\n")
      #end //End of Side-Length-Check for triangle
     #end //End of mid-Length-Check for both triangles
    #end //End of If for Shortest-Connection-Check
  #local B=B+1;
  #end
 #local A=A+1;
 #end
 #write (_O, "}\n")
 #fclose _O
#end

/*******************/
/**Write_Triangles**/
/*******************/
//Given a 2D-Array (mesh-field),normal-field, and a filename this writes a smooth-mesh-object to file
#macro Write_Triangles(_data_array,_filename)
#fopen _O _filename write
 #local _dX=dimension_size(_data_array,1);
 #local _dZ=dimension_size(_data_array,2);
 #local A=0;
 #while (A<_dX-1)
  #local B=0;
  #while (B<_dZ-1)
    //Check for shortest connection
    #if (vlength(_data_array[A][B]-_data_array[A+1][B+1]) >= vlength(_data_array[A][B+1]-_data_array[A+1][B]))
     //Only if mid-section is longer than 0, we may place triangles
     #if (vlength(_data_array[A][B+1]-_data_array[A+1][B])!=0)
      //Check other sides for each triangle
      #if (
          (vlength(_data_array[A][B+1]-_data_array[A][B])!=0) &
          (vlength(_data_array[A+1][B]-_data_array[A][B])!=0)
          )
       #write (_O, "triangle{\n")
       #write (_O, " ",_data_array[ A ][B+1],",\n")
       #write (_O, " ",_data_array[A+1][ B ],",\n")
       #write (_O, " ",_data_array[ A ][ B ],",\n }\n")
      #end//End of Side-Length-Check for triangle
      #if (
          (vlength(_data_array[A][B+1]-_data_array[A+1][B+1])!=0) &
          (vlength(_data_array[A+1][B]-_data_array[A+1][B+1])!=0))
       #write (_O, "triangle{\n")
       #write (_O, " ",_data_array[ A ][B+1],",\n")
       #write (_O, " ",_data_array[A+1][ B ],",\n")
       #write (_O, " ",_data_array[A+1][B+1],",\n }\n")
      #end //End of Side-Length-Check for triangle
     #end //End of mid-Length-Check for both triangles
    #else
     //Only if mid-section is longer than 0, we may place triangles
     #if (vlength(_data_array[A][B]-_data_array[A+1][B+1])!=0)
      //Check other sides for each triangle
      #if (
          (vlength(_data_array[ A ][ B ]-_data_array[A][B+1])!=0) &
          (vlength(_data_array[A+1][B+1]-_data_array[A][B+1])!=0))
       #write (_O, "triangle{\n")
       #write (_O, " ",_data_array[A+1][B+1],",\n")
       #write (_O, " ",_data_array[ A ][ B ],",\n")
       #write (_O, " ",_data_array[ A ][B+1],",\n }\n")
      #end
      //Check other sides for each triangle
      #if (
          (vlength(_data_array[ A ][ B ]-_data_array[A+1][B])!=0) &
          (vlength(_data_array[A+1][B+1]-_data_array[A+1][B])!=0))
       #write (_O, "triangle{\n")
       #write (_O, " ",_data_array[A+1][B+1],",\n")
       #write (_O, " ",_data_array[ A ][ B ],",\n")
       #write (_O, " ",_data_array[A+1][ B ],",\n }\n")
      #end //End of Side-Length-Check for triangle
     #end //End of mid-Length-Check for both triangles
    #end //End of If for Shortest-Connection-Check
  #local B=B+1;
  #end
 #local A=A+1;
 #end
 #fclose _O
#end

/**************/
/**Write_Mesh**/
/**************/
//Given a 2D-Array (mesh-field),normal-field, and a filename this writes a smooth-mesh-object to file
#macro Write_Mesh(_data_array,_filename)
#fopen _O _filename write
 #local _dX=dimension_size(_data_array,1);
 #local _dZ=dimension_size(_data_array,2);
 #write (_O, "mesh{\n")
 #local A=0;
 #while (A<_dX-1)
  #local B=0;
  #while (B<_dZ-1)
    //Check for shortest connection
    #if (vlength(_data_array[A][B]-_data_array[A+1][B+1]) >= vlength(_data_array[A][B+1]-_data_array[A+1][B]))
     //Only if mid-section is longer than 0, we may place triangles
     #if (vlength(_data_array[A][B+1]-_data_array[A+1][B])!=0)
      //Check other sides for each triangle
      #if (
          (vlength(_data_array[A][B+1]-_data_array[A][B])!=0) &
          (vlength(_data_array[A+1][B]-_data_array[A][B])!=0)
          )
       #write (_O, " triangle{\n")
       #write (_O, "  ",_data_array[ A ][B+1],",\n")
       #write (_O, "  ",_data_array[A+1][ B ],",\n")
       #write (_O, "  ",_data_array[ A ][ B ],",\n }\n")
      #end//End of Side-Length-Check for triangle
      #if (
          (vlength(_data_array[A][B+1]-_data_array[A+1][B+1])!=0) &
          (vlength(_data_array[A+1][B]-_data_array[A+1][B+1])!=0))
       #write (_O, " triangle{\n")
       #write (_O, "  ",_data_array[ A ][B+1],",\n")
       #write (_O, "  ",_data_array[A+1][ B ],",\n")
       #write (_O, "  ",_data_array[A+1][B+1],",\n }\n")
      #end //End of Side-Length-Check for triangle
     #end //End of mid-Length-Check for both triangles
    #else
     //Only if mid-section is longer than 0, we may place triangles
     #if (vlength(_data_array[A][B]-_data_array[A+1][B+1])!=0)
      //Check other sides for each triangle
      #if (
          (vlength(_data_array[ A ][ B ]-_data_array[A][B+1])!=0) &
          (vlength(_data_array[A+1][B+1]-_data_array[A][B+1])!=0))
       #write (_O, " triangle{\n")
       #write (_O, "  ",_data_array[A+1][B+1],",\n")
       #write (_O, "  ",_data_array[ A ][ B ],",\n")
       #write (_O, "  ",_data_array[ A ][B+1],",\n }\n")
      #end
      
      //Check other sides for each triangle
      #if (
          (vlength(_data_array[ A ][ B ]-_data_array[A+1][B])!=0) &
          (vlength(_data_array[A+1][B+1]-_data_array[A+1][B])!=0))
       #write (_O, " triangle{\n")
       #write (_O, "  ",_data_array[A+1][B+1],",\n")
       #write (_O, "  ",_data_array[ A ][ B ],",\n")
       #write (_O, "  ",_data_array[A+1][ B ],",\n }\n")
      #end //End of Side-Length-Check for triangle
     #end //End of mid-Length-Check for both triangles
    #end //End of If for Shortest-Connection-Check
  #local B=B+1;
  #end
 #local A=A+1;
 #end
 #write (_O, "}\n")
 #fclose _O
#end

/*********************/
/**Write_UVTriangles**/
/*********************/
//Given a 2D-Array (mesh-field),normal-field, and a filename this writes a smooth-mesh-object to file
#macro Write_UVTriangles(_data_array,_filename)
#fopen _O _filename write
 #local _dX=dimension_size(_data_array,1);
 #local _dZ=dimension_size(_data_array,2);
 #local A=0;
 #while (A<_dX-1)
  #local B=0;
  #while (B<_dZ-1)
    //Check for shortest connection
    #if (vlength(_data_array[A][B]-_data_array[A+1][B+1]) >= vlength(_data_array[A][B+1]-_data_array[A+1][B]))
     //Only if mid-section is longer than 0, we may place triangles
     #if (vlength(_data_array[A][B+1]-_data_array[A+1][B])!=0)
      //Check other sides for each triangle
      #if (
          (vlength(_data_array[A][B+1]-_data_array[A][B])!=0) &
          (vlength(_data_array[A+1][B]-_data_array[A][B])!=0)
          )
       #write (_O, "triangle{\n")
       #write (_O, " ",_data_array[ A ][B+1],",\n")
       #write (_O, " ",_data_array[A+1][ B ],",\n")
       #write (_O, " ",_data_array[ A ][ B ],",\n")
       #write (_O, " uv_vectors ",<A/(_dX-1),(B+1)/(_dZ-1)>,",",<(A+1)/(_dX-1),B/(_dZ-1)>,",",<A/(_dX-1),B/(_dZ-1)>,"\n")
       #write (_O, " }\n")
      #end//End of Side-Length-Check for triangle
      #if (
          (vlength(_data_array[A][B+1]-_data_array[A+1][B+1])!=0) &
          (vlength(_data_array[A+1][B]-_data_array[A+1][B+1])!=0))
       #write (_O, "triangle{\n")
       #write (_O, " ",_data_array[ A ][B+1],",\n")
       #write (_O, " ",_data_array[A+1][ B ],",\n")
       #write (_O, " ",_data_array[A+1][B+1],",\n")
       #write (_O, " uv_vectors ",<A/(_dX-1),(B+1)/(_dZ-1)>,",",<(A+1)/(_dX-1),B/(_dZ-1)>,",",<(A+1)/(_dX-1),(B+1)/(_dZ-1)>,"\n")
       #write (_O, " }\n")
      #end //End of Side-Length-Check for triangle
     #end //End of mid-Length-Check for both triangles
    #else
     //Only if mid-section is longer than 0, we may place triangles
     #if (vlength(_data_array[A][B]-_data_array[A+1][B+1])!=0)
      //Check other sides for each triangle
      #if (
          (vlength(_data_array[ A ][ B ]-_data_array[A][B+1])!=0) &
          (vlength(_data_array[A+1][B+1]-_data_array[A][B+1])!=0))
       #write (_O, "triangle{\n")
       #write (_O, " ",_data_array[A+1][B+1],",\n")
       #write (_O, " ",_data_array[ A ][ B ],",\n")
       #write (_O, " ",_data_array[ A ][B+1],",\n")
       #write (_O, " uv_vectors ",<(A+1)/(_dX-1),(B+1)/(_dZ-1)>,",",<A/(_dX-1),B/(_dZ-1)>,",",<A/(_dX-1),(B+1)/(_dZ-1)>,"\n")
       #write (_O, " }\n")
      #end
      //Check other sides for each triangle
      #if (
          (vlength(_data_array[ A ][ B ]-_data_array[A+1][B])!=0) &
          (vlength(_data_array[A+1][B+1]-_data_array[A+1][B])!=0))
       #write (_O, "triangle{\n")
       #write (_O, " ",_data_array[A+1][B+1],",\n")
       #write (_O, " ",_data_array[ A ][ B ],",\n")
       #write (_O, " ",_data_array[A+1][ B ],",\n")
       #write (_O, " uv_vectors ",<(A+1)/(_dX-1),(B+1)/(_dZ-1)>,",",<A/(_dX-1),B/(_dZ-1)>,",",<(A+1)/(_dX-1),B/(_dZ-1)>,"\n")
       #write (_O, " }\n")
      #end //End of Side-Length-Check for triangle
     #end //End of mid-Length-Check for both triangles
    #end //End of If for Shortest-Connection-Check
  #local B=B+1;
  #end
 #local A=A+1;
 #end
 #fclose _O
#end

/****************/
/**Write_UVMesh**/
/****************/
//Given a 2D-Array (mesh-field),normal-field, and a filename this writes a smooth-mesh-object to file
#macro Write_UVTriangles(_data_array,_filename)
#fopen _O _filename write
 #local _dX=dimension_size(_data_array,1);
 #local _dZ=dimension_size(_data_array,2);
 #write (_O "mesh{\n")
 #local A=0;
 #while (A<_dX-1)
  #local B=0;
  #while (B<_dZ-1)
    //Check for shortest connection
    #if (vlength(_data_array[A][B]-_data_array[A+1][B+1]) >= vlength(_data_array[A][B+1]-_data_array[A+1][B]))
     //Only if mid-section is longer than 0, we may place triangles
     #if (vlength(_data_array[A][B+1]-_data_array[A+1][B])!=0)
      //Check other sides for each triangle
      #if (
          (vlength(_data_array[A][B+1]-_data_array[A][B])!=0) &
          (vlength(_data_array[A+1][B]-_data_array[A][B])!=0)
          )
       #write (_O, " triangle{\n")
       #write (_O, "  ",_data_array[ A ][B+1],",\n")
       #write (_O, "  ",_data_array[A+1][ B ],",\n")
       #write (_O, "  ",_data_array[ A ][ B ],",\n")
       #write (_O, "  uv_vectors ",<A/(_dX-1),(B+1)/(_dZ-1)>,",",<(A+1)/(_dX-1),B/(_dZ-1)>,",",<A/(_dX-1),B/(_dZ-1)>,"\n")
       #write (_O, "  }\n")
      #end//End of Side-Length-Check for triangle
      #if (
          (vlength(_data_array[A][B+1]-_data_array[A+1][B+1])!=0) &
          (vlength(_data_array[A+1][B]-_data_array[A+1][B+1])!=0))
       #write (_O, " triangle{\n")
       #write (_O, "  ",_data_array[ A ][B+1],",\n")
       #write (_O, "  ",_data_array[A+1][ B ],",\n")
       #write (_O, "  ",_data_array[A+1][B+1],",\n")
       #write (_O, "  uv_vectors ",<A/(_dX-1),(B+1)/(_dZ-1)>,",",<(A+1)/(_dX-1),B/(_dZ-1)>,",",<(A+1)/(_dX-1),(B+1)/(_dZ-1)>,"\n")
       #write (_O, "  }\n")
      #end //End of Side-Length-Check for triangle
     #end //End of mid-Length-Check for both triangles
    #else
     //Only if mid-section is longer than 0, we may place triangles
     #if (vlength(_data_array[A][B]-_data_array[A+1][B+1])!=0)
      //Check other sides for each triangle
      #if (
          (vlength(_data_array[ A ][ B ]-_data_array[A][B+1])!=0) &
          (vlength(_data_array[A+1][B+1]-_data_array[A][B+1])!=0))
       #write (_O, " triangle{\n")
       #write (_O, "  ",_data_array[A+1][B+1],",\n")
       #write (_O, "  ",_data_array[ A ][ B ],",\n")
       #write (_O, "  ",_data_array[ A ][B+1],",\n")
       #write (_O, "  uv_vectors ",<(A+1)/(_dX-1),(B+1)/(_dZ-1)>,",",<A/(_dX-1),B/(_dZ-1)>,",",<A/(_dX-1),(B+1)/(_dZ-1)>,"\n")
       #write (_O, "  }\n")
      #end
      //Check other sides for each triangle
      #if (
          (vlength(_data_array[ A ][ B ]-_data_array[A+1][B])!=0) &
          (vlength(_data_array[A+1][B+1]-_data_array[A+1][B])!=0))
       #write (_O, " triangle{\n")
       #write (_O, "  ",_data_array[A+1][B+1],",\n")
       #write (_O, "  ",_data_array[ A ][ B ],",\n")
       #write (_O, "  ",_data_array[A+1][ B ],",\n")
       #write (_O, "  uv_vectors ",<(A+1)/(_dX-1),(B+1)/(_dZ-1)>,",",<A/(_dX-1),B/(_dZ-1)>,",",<(A+1)/(_dX-1),B/(_dZ-1)>,"\n")
       #write (_O, "  }\n")
      #end //End of Side-Length-Check for triangle
     #end //End of mid-Length-Check for both triangles
    #end //End of If for Shortest-Connection-Check
  #local B=B+1;
  #end
 #local A=A+1;
 #end
 #write (_O "}\n")
 #fclose _O
#end

/****************************/
/**Write_Smooth_UVTriangles**/
/****************************/
//Given a 2D-Array (mesh-field),normal-field, and a filename this writes a smooth-mesh-object to file
#macro Write_Smooth_UVTriangles(_data_array,_normal_array,_filename)
#fopen _O _filename write
 #local _dX=dimension_size(_data_array,1);
 #local _dZ=dimension_size(_data_array,2);
 #local A=0;
 #while (A<_dX-1)
  #local B=0;
  #while (B<_dZ-1)
    //Check for shortest connection
    #if (vlength(_data_array[A][B]-_data_array[A+1][B+1]) >= vlength(_data_array[A][B+1]-_data_array[A+1][B]))
     //Only if mid-section is longer than 0, we may place triangles
     #if (vlength(_data_array[A][B+1]-_data_array[A+1][B])!=0)
      //Check other sides for each triangle
      #if (
          (vlength(_data_array[A][B+1]-_data_array[A][B])!=0) &
          (vlength(_data_array[A+1][B]-_data_array[A][B])!=0)
          )
       #write (_O, "smooth_triangle{\n")
       #write (_O, " ",_data_array[ A ][B+1],",",_normal_array[ A ][B+1],"\n")
       #write (_O, " ",_data_array[A+1][ B ],",",_normal_array[A+1][ B ],"\n")
       #write (_O, " ",_data_array[ A ][ B ],",",_normal_array[ A ][ B ],"\n")
       #write (_O, " uv_vectors ",<A/(_dX-1),(B+1)/(_dZ-1)>,",",<(A+1)/(_dX-1),B/(_dZ-1)>,",",<A/(_dX-1),B/(_dZ-1)>,"\n")
       #write (_O, " }\n")
      #end//End of Side-Length-Check for triangle
      #if (
          (vlength(_data_array[A][B+1]-_data_array[A+1][B+1])!=0) &
          (vlength(_data_array[A+1][B]-_data_array[A+1][B+1])!=0))
       #write (_O, "smooth_triangle{\n")
       #write (_O, " ",_data_array[ A ][B+1],",",_normal_array[ A ][B+1],"\n")
       #write (_O, " ",_data_array[A+1][ B ],",",_normal_array[A+1][ B ],"\n")
       #write (_O, " ",_data_array[A+1][B+1],",",_normal_array[A+1][B+1],"\n")
       #write (_O, " uv_vectors ",<A/(_dX-1),(B+1)/(_dZ-1)>,",",<(A+1)/(_dX-1),B/(_dZ-1)>,",",<(A+1)/(_dX-1),(B+1)/(_dZ-1)>,"\n")
       #write (_O, " }\n")
      #end //End of Side-Length-Check for triangle
     #end //End of mid-Length-Check for both triangles
    #else
     //Only if mid-section is longer than 0, we may place triangles
     #if (vlength(_data_array[A][B]-_data_array[A+1][B+1])!=0)
      //Check other sides for each triangle
      #if (
          (vlength(_data_array[ A ][ B ]-_data_array[A][B+1])!=0) &
          (vlength(_data_array[A+1][B+1]-_data_array[A][B+1])!=0))
       #write (_O, "smooth_triangle{\n")
       #write (_O, " ",_data_array[A+1][B+1],",",_normal_array[A+1][B+1],"\n")
       #write (_O, " ",_data_array[ A ][ B ],",",_normal_array[ A ][ B ],"\n")
       #write (_O, " ",_data_array[ A ][B+1],",",_normal_array[ A ][B+1],"\n")
       #write (_O, " uv_vectors ",<(A+1)/(_dX-1),(B+1)/(_dZ-1)>,",",<A/(_dX-1),B/(_dZ-1)>,",",<A/(_dX-1),(B+1)/(_dZ-1)>,"\n")
       #write (_O, " }\n")
      #end
      //Check other sides for each triangle
      #if (
          (vlength(_data_array[ A ][ B ]-_data_array[A+1][B])!=0) &
          (vlength(_data_array[A+1][B+1]-_data_array[A+1][B])!=0))
       #write (_O, "smooth_triangle{\n")
       #write (_O, " ",_data_array[A+1][B+1],",",_normal_array[A+1][B+1],"\n")
       #write (_O, " ",_data_array[ A ][ B ],",",_normal_array[ A ][ B ],"\n")
       #write (_O, " ",_data_array[A+1][ B ],",",_normal_array[A+1][ B ],"\n")
       #write (_O, " uv_vectors ",<(A+1)/(_dX-1),(B+1)/(_dZ-1)>,",",<A/(_dX-1),B/(_dZ-1)>,",",<(A+1)/(_dX-1),B/(_dZ-1)>,"\n")
       #write (_O, " }\n")
      #end //End of Side-Length-Check for triangle
     #end //End of mid-Length-Check for both triangles
    #end //End of If for Shortest-Connection-Check
  #local B=B+1;
  #end
 #local A=A+1;
 #end
 #fclose _O
#end

/***********************/
/**Write_Smooth_UVMesh**/
/***********************/
//Given a 2D-Array (mesh-field),normal-field, and a filename this writes a smooth-mesh-object to file
#macro Write_Smooth_UVMesh(_data_array,_normal_array,_filename)
#fopen _O _filename write
 #local _dX=dimension_size(_data_array,1);
 #local _dZ=dimension_size(_data_array,2);
 #write (_O "mesh{\n")
 #local A=0;
 #while (A<_dX-1)
  #local B=0;
  #while (B<_dZ-1)
    //Check for shortest connection
    #if (vlength(_data_array[A][B]-_data_array[A+1][B+1]) >= vlength(_data_array[A][B+1]-_data_array[A+1][B]))
     //Only if mid-section is longer than 0, we may place triangles
     #if (vlength(_data_array[A][B+1]-_data_array[A+1][B])!=0)
      //Check other sides for each triangle
      #if (
          (vlength(_data_array[A][B+1]-_data_array[A][B])!=0) &
          (vlength(_data_array[A+1][B]-_data_array[A][B])!=0)
          )
       #write (_O, " smooth_triangle{\n")
       #write (_O, "  ",_data_array[ A ][B+1],",",_normal_array[ A ][B+1],"\n")
       #write (_O, "  ",_data_array[A+1][ B ],",",_normal_array[A+1][ B ],"\n")
       #write (_O, "  ",_data_array[ A ][ B ],",",_normal_array[ A ][ B ],"\n")
       #write (_O, "  uv_vectors ",<A/(_dX-1),(B+1)/(_dZ-1)>,",",<(A+1)/(_dX-1),B/(_dZ-1)>,",",<A/(_dX-1),B/(_dZ-1)>,"\n")
       #write (_O, "  }\n")
      #end//End of Side-Length-Check for triangle
      #if (
          (vlength(_data_array[A][B+1]-_data_array[A+1][B+1])!=0) &
          (vlength(_data_array[A+1][B]-_data_array[A+1][B+1])!=0))
       #write (_O, " smooth_triangle{\n")
       #write (_O, "  ",_data_array[ A ][B+1],",",_normal_array[ A ][B+1],"\n")
       #write (_O, "  ",_data_array[A+1][ B ],",",_normal_array[A+1][ B ],"\n")
       #write (_O, "  ",_data_array[A+1][B+1],",",_normal_array[A+1][B+1],"\n")
       #write (_O, "  uv_vectors ",<A/(_dX-1),(B+1)/(_dZ-1)>,",",<(A+1)/(_dX-1),B/(_dZ-1)>,",",<(A+1)/(_dX-1),(B+1)/(_dZ-1)>,"\n")
       #write (_O, "  }\n")
      #end //End of Side-Length-Check for triangle
     #end //End of mid-Length-Check for both triangles
    #else
     //Only if mid-section is longer than 0, we may place triangles
     #if (vlength(_data_array[A][B]-_data_array[A+1][B+1])!=0)
      //Check other sides for each triangle
      #if (
          (vlength(_data_array[ A ][ B ]-_data_array[A][B+1])!=0) &
          (vlength(_data_array[A+1][B+1]-_data_array[A][B+1])!=0))
       #write (_O, " smooth_triangle{\n")
       #write (_O, "  ",_data_array[A+1][B+1],",",_normal_array[A+1][B+1],"\n")
       #write (_O, "  ",_data_array[ A ][ B ],",",_normal_array[ A ][ B ],"\n")
       #write (_O, "  ",_data_array[ A ][B+1],",",_normal_array[ A ][B+1],"\n")
       #write (_O, "  uv_vectors ",<(A+1)/(_dX-1),(B+1)/(_dZ-1)>,",",<A/(_dX-1),B/(_dZ-1)>,",",<A/(_dX-1),(B+1)/(_dZ-1)>,"\n")
       #write (_O, "  }\n")
      #end
      //Check other sides for each triangle
      #if (
          (vlength(_data_array[ A ][ B ]-_data_array[A+1][B])!=0) &
          (vlength(_data_array[A+1][B+1]-_data_array[A+1][B])!=0))
       #write (_O, " smooth_triangle{\n")
       #write (_O, "  ",_data_array[A+1][B+1],",",_normal_array[A+1][B+1],"\n")
       #write (_O, "  ",_data_array[ A ][ B ],",",_normal_array[ A ][ B ],"\n")
       #write (_O, "  ",_data_array[A+1][ B ],",",_normal_array[A+1][ B ],"\n")
       #write (_O, "  uv_vectors ",<(A+1)/(_dX-1),(B+1)/(_dZ-1)>,",",<A/(_dX-1),B/(_dZ-1)>,",",<(A+1)/(_dX-1),B/(_dZ-1)>,"\n")
       #write (_O, "  }\n")
      #end //End of Side-Length-Check for triangle
     #end //End of mid-Length-Check for both triangles
    #end //End of If for Shortest-Connection-Check
  #local B=B+1;
  #end
 #local A=A+1;
 #end
 #write (_O "}\n")
 #fclose _O
#end


/************************************************************************** Mesh-Checks *************************************************/
/************/
/**Find_Min**/
/************/
//Give: axis and 2D-Array (mesh-field)
//Result: returns float with mininmum axis-value
#macro Find_Min(_md,_axis)
 #local _dX=dimension_size(_md,1);
 #local _dZ=dimension_size(_md,2);
 #local _v=0;
 #if (_axis.x=1)
  #local A=0; #while (A<_dX)
   #local B=0; #while (B<_dZ)
    #if (_md[A][B].x<_v) #local _v=_md[A][B].x; #end      
   #local B=B+1; #end
  #local A=A+1; #end
 #end
 #if (_axis.y=1)
  #local A=0; #while (A<_dX)
   #local B=0; #while (B<_dZ)
    #if (_md[A][B].y<_v) #local _v=_md[A][B].y; #end      
   #local B=B+1; #end
  #local A=A+1; #end
 #end
 #if (_axis.z=1)
  #local A=0; #while (A<_dX)
   #local B=0; #while (B<_dZ)
    #if (_md[A][B].z<_v) #local _v=_md[A][B].z; #end      
   #local B=B+1; #end
  #local A=A+1; #end
 #end
 //Return
 _v
#end

//Find_Max
//Give: axis (using x,y,z) and Field-Array
//Result: float filled with highest axis-value
#macro Find_Max(_md,_axis)
 #local _dX=dimension_size(_md,1);
 #local _dZ=dimension_size(_md,2);
 #local _v=0;
 #if (_axis.x=1)
  #local A=0; #while (A<_dX)
   #local B=0; #while (B<_dZ)
    #if (_md[A][B].x>_v) #local _v=_md[A][B].x; #end      
   #local B=B+1; #end
  #local A=A+1; #end
 #end
 #if (_axis.y=1)
  #local A=0; #while (A<_dX)
   #local B=0; #while (B<_dZ)
    #if (_md[A][B].y>_v) #local _v=_md[A][B].y; #end      
   #local B=B+1; #end
  #local A=A+1; #end
 #end
 #if (_axis.z=1)
  #local A=0; #while (A<_dX)
   #local B=0; #while (B<_dZ)
    #if (_md[A][B].z>_v) #local _v=_md[A][B].z; #end      
   #local B=B+1; #end
  #local A=A+1; #end
 #end
 //Return
 _v
#end