// Persistence of Vision Ray Tracer Scene Description File
// File: x_over.inc
// Vers: 3.5
// Desc: Macros that are cross-overs from several other includes
// Date: 03.28.02 (mm.dd.yy)
// Auth: Tim Nikias Wenclawiak
// Last Update: 04.03.04 (mm.dd.yy)

// Requirements:
//  bsplines.inc
//  look_at.inc
//  vnormalize.inc

#declare _x_over_inc_tnw=1; //For other includes to check if already included

#ifndef (_bsplines_inc_tnw) #include "bsplines.inc" #end
#ifndef (_look_at_inc_tnw) #include "look_at.inc" #end
#ifndef (_vnormalize_inc_tnw) #include "vnormalize.inc" #end

/******************/
/*Connect_BSplines*/
/******************/
//Function:
// Creates Mesh from Spline_1 to Spline_2
//Parameters:
// 1. BSpline_1
// 2. BSpline_2
// 3. X-Detail (along BSpline_1)
// 4. Z_Detail (span between BSpline_1 and BSpline_2)
//Returns:
// Mesh-Array
//Notes:
// None
#macro Connect_BSplines(_spl_1,_spl_2,_dX,_dZ)
 #local _dat_array=array[_dX][_dZ]
 #local Z=0;
 #while (Z<_dZ)
  #local _T_Pos = BSpline_Pos(_spl_1,Z/(_dZ-1));
  #local _S_Pos = BSpline_Pos(_spl_2,Z/(_dZ-1));
  #local X=0;
  #while (X<_dX)
   //X and Z coordinates
   #local _dat_array[X][Z]=_T_Pos-(_T_Pos-_S_Pos)*(X/(_dX-1));
  #local X=X+1;
  #end
 #local Z=Z+1;
 #end
 //Return the array
 _dat_array
#end

/***************/
/*BSpline_Lathe*/
/***************/
//Function:
// Builds Lathe of BSpline
//Parameters:
// 1. BSpline
// 2. Steps along BSpline (X)
// 3. Steps for Lathe (Z)
//Returns:
// Mesh-Array
//Notes:
// Lathe is rotated around y-Axis. First and last line on
// first dimensions are the same (use Average_Rim_Normals with x)
#macro BSpline_Lathe(_bspline,_bsp_steps,_lathe_steps)
 //Initialize
 #local _array=array[_bsp_steps][_lathe_steps]
 #local A=0;
 #while (A<_bsp_steps)
  #local B=0;
  #while (B<_lathe_steps)
   #if (B!=_lathe_steps-1)
    #local _array[A][B]=vrotate(BSpline_Pos(_bspline,A/(_bsp_steps-1)),y*360*B/(_lathe_steps-1));
   #else
    #local _array[A][B]=_array[A][0];
   #end
  #local B=B+1;
  #end
 #local A=A+1;
 #end
 //Return the array
 _array 
#end


/***************/
/*BSpline_Sweep*/
/***************/
//Function:
// Sweeps one Bspline along another
//Parameters:
// 1. BSpline_1
// 2. Intervals on BSpline_1 (X)
// 3. BSpline_2
// 4. Intervals on BSpline_2 (Z)
// 5. Switch to adjust BSpline_2 along normal of BSpline_1
//Returns:
// Mesh-Array
//Notes:
// None
#macro BSpline_Sweep(_1_spline,_1D,_2_spline,_2D,_switch)
 #local _spl_dat=array[_1D]
 #local _relocate=BSpline_Pos(_1_spline,0);
 #local A=0;
 #while (A<_1D)
  #local _spl_dat[A]=BSpline_Pos(_1_spline,A/(_1D-1))-_relocate;
 #local A=A+1;
 #end
 
 #local mesh_dat=array[_1D][_2D]
 #local A=0;
 #while (A<_1D)
  #local B=0;
  #while (B<_2D)
   #if (_switch)
    #local mesh_dat[A][B]=vrotate(_spl_dat[A],Look_At(BSpline_Normal(_2_spline,B/(_2D-1) )))+BSpline_Pos(_2_spline,B/(_2D-1));
   #else
    #local mesh_dat[A][B]=_spl_dat[A]+BSpline_Pos(_2_spline,B/(_2D-1));
   #end
  #local B=B+1;
  #end
 #local A=A+1;
 #end
 
 //Return mesh-data
 mesh_dat
#end

/**********************/
/*Con_2Mesh_by_BSpline*/
/**********************/
//Function:
// Connects edge of two meshes via BSpline_Curved Segment
//Parameters:
// 1. Mesh-Array-#1
// 2. Side on Array-#1
// 3. Length of Direction-Node for BSpline on Array-#1
// 4. Mesh-Array-#2
// 5. Side on Array-#2
// 6. Length of Direction-Node for BSpline on Array-#2
// 7. Intervals on BSpline
//Returns:
// Mesh-Array, fitting onto edge od Array-#1 and -#2
//Notes:
// None
#macro Con_2Mesh_by_Bspline(_field1,_side1,_node_len1,_field2,_side2,_node_len2,_i)
 #local _X1=dimension_size(_field1,1);
 #local _Z1=dimension_size(_field1,2);
 #local _X2=dimension_size(_field2,1);
 #local _Z2=dimension_size(_field2,2);
 
 #if (_side1.x= 1) #local A_Max=_X1; #local _A1=_Z1-1; #end
 #if (_side1.x=-1) #local A_Max=_X1; #local _A1= 0 ; #end
 #if (_side1.z= 1) #local A_Max=_Z1; #local _A1=_X1-1; #end
 #if (_side1.z=-1) #local A_Max=_Z1; #local _A1= 0 ; #end

 #if (_side2.x= 1) #local _B2=_Z2-1; #end
 #if (_side2.x=-1) #local _B2= 0 ; #end
 #if (_side2.z= 1) #local _B2=_X2-1; #end
 #if (_side2.z=-1) #local _B2= 0 ; #end
 
 //Prepare the new field
 #local _field_o=array[_i][A_Max]
 
 //Going through the rim of the first mesh
 #local A=0;
 #while (A<A_Max)
  
  #local _spl=array[4]
   #if (_side1.x=1 | _side1.x=-1)
    #local _spl[0]=_field1[A][_A1];
   #end
   #if (_side1.z=1 | _side1.z=-1)
    #local _spl[0]=_field1[_A1][A];
   #end

   #if (_side2.x=1 | _side2.x=-1)
    #local _spl[2]=_field2[A][_B2];
   #end
   #if (_side2.z=1 | _side2.z=-1)
    #local _spl[2]=_field2[_B2][A];
   #end

   #if (_side1.x= 1) #local _spl[1]=VNormalize(_field1[A][_Z1-1]-_field1[A][_Z1-2])*_node_len1; #end
   #if (_side1.x=-1) #local _spl[1]=VNormalize(_field1[A][0]-_field1[A][1])*_node_len1; #end
   #if (_side1.z= 1) #local _spl[1]=VNormalize(_field1[_X1-1][A]-_field1[_X1-2][A])*_node_len1; #end
   #if (_side1.z=-1) #local _spl[1]=VNormalize(_field1[0][A]-_field1[1][A])*_node_len1; #end
    
   #if (_side2.x= 1) #local _spl[3]=VNormalize(_field2[A][_Z2-1]-_field2[A][_Z2-2])*_node_len2; #end
   #if (_side2.x=-1) #local _spl[3]=VNormalize(_field2[A][0]-_field2[A][1])*_node_len2; #end
   #if (_side2.z= 1) #local _spl[3]=VNormalize(_field2[_X2-1][A]-_field2[_X2-2][A])*_node_len2; #end
   #if (_side2.z=-1) #local _spl[3]=VNormalize(_field2[0][A]-_field2[1][A])*_node_len2; #end

  //Going over the intervals
  #local B=0;
  #while (B<_i)
   #local _nm=B/(_i-1);

   #if ( B =0) #local _field_o[B][A]=_spl[0]; #end
   #if (_nm=1) #local _field_o[B][A]=_spl[2]; #end
   #if (_nm>0 & _nm<1)
    #local _field_o[B][A]=BSpline_Pos(_spl,B/(_i-1));
   #end

  #local B=B+1;
  #end
 #local A=A+1;
 #end

//Return mesh
_field_o
 
#end

/****************/
/**BSpline2Tube**/
/****************/
//Function:
// Generates Tube from BSpline
//Parameters:
// 1. BSpline
// 2. Radius
// 3. Circle-Detail
// 4. Length-Detail
// 5. Name (for files and variables)
//Returns:
// 1. Tube_Nodes = Array of nodes (MMM)
// 2. Tube_BSpline_Name = String of Name of BSpline
// 3. Tube_Linear_Name = String of Name of Linear representation
// 4. File with BSpline and Linear using above names
//Notes:
// Makes use of temporary file "temp.tmp", may be deleted afterwards
#macro BSpline2Tube(_BSpline,_Radius,_XDet,_ZDet,_Name)

//First, we convert the Spline to one with even-distributed intervals
BSpline2Linear_Ext(
  //Input-Spline
  _BSpline,
  //Detail for Calculations
  _ZDet*10,_ZDet,
  //Names for files and arrays
  concat(_Name,".spl"),concat(_Name,"_BSpline"),concat(_Name,"_Linear"))

#include concat(_Name,".spl")

//Hand the actual arrays to this Include-File
#fopen _temp "temp.tmp" write
#write (_temp, concat("#declare _thisTube_BSpline=",_Name,"_BSpline;\n"))
#write (_temp, concat("#declare _thisTube_Linear=",_Name,"_Linear;\n"))
#fclose _temp
#include "temp.tmp"
#fopen _temp "temp.tmp" write
#fclose _temp

#declare Tube_BSpline_Name = concat(_Name,"_BSpline");
#declare Tube_Linear_Name = concat(_Name,"_Linear");
#declare Tube_Nodes = array[_XDet][_ZDet];

#local Last_Up = <0,1,0>;
//Creates the basic Tube
#local C=0;
#while (C<_ZDet)
  //Basic Transformation Data
  #local Actual_Center = V3(_thisTube_Linear[C]);
  #local New_Direction = vnormalize(BSpline_Dir(_thisTube_BSpline,_thisTube_Linear[C].t));
  #local Up = vnormalize(vcross(vcross(New_Direction,Last_Up), New_Direction));
  #local Right = vnormalize(vcross(New_Direction,Up));
  #local _X=-Right;
  #local _Y=Up;
  #local _Z=-New_Direction;
  #local _C=Actual_Center;
  #local Matrix = transform{matrix<_X.x,_X.y,_X.z,_Y.x,_Y.y,_Y.z,_Z.x,_Z.y,_Z.z,_C.x,_C.y,_C.z>}
  #local Matrix_Func=function{transform{Matrix}};
  #local Circling = 0;
  #while (Circling < _XDet)

    //Build the Circle
    #local Local_Pos = vrotate(y*_Radius,z*360*(Circling/(_XDet-1)));
    #local Local_Pos = Matrix_Func(Local_Pos.x,Local_Pos.y,Local_Pos.z);
    //Apply to Aray
    #declare Tube_Nodes[Circling][C] = Local_Pos;

    #local Circling = Circling + 1;
  #end
  #undef Matrix_Func
  #undef Matrix
#local C=C+1;
#end

#end