package com.firestick.graphics.format.bvh;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import com.firestick.debug.Debug;
import com.firestick.graphics.format.AbstractFormatFile;

public class BvhMotionFileBase extends AbstractFormatFile   {

	// when true we are reading motion files
	private boolean readingMotionLines = false;

	private StringBuffer header = new StringBuffer();

	private List<MotionLine> motionLines = new ArrayList<MotionLine>();

	private List<Integer> deltaedElements;
	private String deltaeSource;

	public BvhMotionFileBase(File file) throws IOException {
		super(file);
		readFile();
	}
	
	@Override
	protected String getDelimiters() {
		return "\t ";
	}
	
	public void debug () {
		try {
			Debug.debugNewLine();
			Debug.debugInfo("BvhMotionFile : " + getFileName());
			Debug.debugInfo("Number of frames : " + motionLines.size());
			List list = getDeltaedElements();
			Debug.debugInfo("Deltaes : from " + deltaeSource);
			String msg = "";
			for (int idx=0;idx<list.size();idx++) {
				if (idx!=0) {
					msg = msg + ",";
				}
				msg = msg + list.get(idx);
			}
			Debug.debugInfo(msg);
			} catch (Exception ex) {
				throw new RuntimeException(ex);
			}
		
	}

	@Override
	public void lineEvent(List<String> line, long lineNumber) {
		if (readingMotionLines) {
			motionLines.add(new MotionLine(line));
		} else {
			String firstWord = line.get(0);
			if (!"MOTION".equals(firstWord) && !"Frames:".equals(firstWord)
					&& !"Frame".equals(firstWord)) {
				header.append(delimitLine(line));
				header.append("\n");
			}

			if ("Frame".equals(firstWord)) {
				readingMotionLines = true;
			}
		}
	}

	public void write(File file) throws IOException {

		Writer writer = null;

		try {
			writer = new FileWriter(file);
			writer.write(header.toString());
			writer.write("MOTION\n");
			writer.write("Frames:     ");
			writer.write(new Integer(motionLines.size()).toString());
			writer.write("\n");
			writer.write("Frame Time: 0.033333\n");
			for (int idx = 0; idx < motionLines.size(); idx++) {
				writer.write(motionLines.get(idx).toString());
				writer.write("\n");
			}
		} finally {
			writer.close();
		}
	}
	
	private String delimitLine(List<String> line) {
		StringBuffer delimited = new StringBuffer();
		for (int idx = 0; idx < line.size(); idx++) {
			if (idx > 0) {
				delimited.append(" ");
			}
			delimited.append(line.get(idx));
		}
		return delimited.toString();
	}


	protected MotionLine getMotionLine(int idx) {
		if (idx>=motionLines.size()) {
			throw new IndexOutOfBoundsException("BVH file " + getFileName() + " motionline " + idx + " requested but only " + motionLines.size() + " available");
		}
		return motionLines.get(idx);
	}
	
	protected MotionLine getMotionLine (int idx, int scaleToFrames) {
		if (scaleToFrames > 0) {
			double frameScale = (double)motionLines.size() /(double)scaleToFrames ;
			int scaledIdx = (int)((double)idx * frameScale);
			return getMotionLine(scaledIdx);

		} else {
			return getMotionLine (idx);
		}
	}
	

	protected int getMotionSize() {
		return motionLines.size();
	}
	
	protected int getMotionSize (int scaleToFrames) {
		if (scaleToFrames > 0) {
			return scaleToFrames;
		} else {
			return getMotionSize();
		}
		
	}
	
	protected void setMotionLine (int idx, MotionLine motionLine) {
		motionLines.set(idx, motionLine);
	}
	
	protected void addMotionLine (MotionLine motionLine) {
		motionLines.add(motionLine);
	}
	
	
	protected List<Integer> getDeltaedElements() throws IOException {
		if (deltaedElements == null) {
			File deltaeFile = new File (getFileName().substring(0, getFileName().indexOf(".")) + ".dlt");
			//System.out.println ("#debug looking for deltae - " + deltaeFile.getName());
			if (deltaeFile.exists()) {
				deltaedElements = new ArrayList<Integer>();
				deltaeSource = "from file : "  + deltaeFile.getName();
				BvhMotionDeltaedFile dfile = new BvhMotionDeltaedFile(deltaeFile); 
				Iterator<Integer> iter = dfile.getDeltaes();
				while (iter.hasNext()) {
					Integer delement = iter.next();
					deltaedElements.add(delement);
				}
			} else {
				deltaeSource = "calculated";
				calculateDeltae();
			}
			
		}
		return deltaedElements;
	
	}

	private void calculateDeltae () {
		deltaedElements = new ArrayList<Integer>();
		for (int idxElement=0;idxElement<motionLines.get(0).getElementSize();idxElement++) {
			if (doesElementDelta(idxElement)) {
				deltaedElements.add(idxElement);
			}
		}

	}
	
	private boolean doesElementDelta (int idxElement) {
		double firstValue = getMotionLine(0).getElement(idxElement);
		for (int idx=1;idx<getMotionSize();idx++) {
			if (getMotionLine(idx).getElement(idxElement) != firstValue) {
				return true;
			}
		}
		return false;
	}

	
	// ***********************************************************************************************
	protected class MotionLine implements Cloneable {

		private List<Double> list = new ArrayList<Double>();
		
		public MotionLine(List<String> lineAsString) {
			for (int idx = 0; idx < lineAsString.size(); idx++) {
				list.add(new Double(lineAsString.get(idx)));
			}
		}
		

		private MotionLine () {
			
		}

		public List<Integer> calculateChangingElements(MotionLine motionLine) {
			List<Integer> list = new ArrayList<Integer>();
			for (int idx=0;idx<getElementSize();idx++) {
				if (getElement(idx) != motionLine.getElement(idx)) {
					list.add(idx);
				}
			}
			return list;
		}
		
		public MotionLine calculateInterpolationIncrements(MotionLine toThis, int from, int to) {
			MotionLine result = new MotionLine();
			int numberOfFrames = to - from;
			for (int idx = 0; idx <getElementSize(); idx++) {
				result.list.add((toThis.getElement(idx) - getElement(idx))/numberOfFrames);
			}
			return result;
		}
		
		@Override
		protected Object clone() {
			try {
				MotionLine copy = new MotionLine();
				for (int idx=0;idx<list.size();idx++) {
					copy.list.add(new Double(list.get(idx)));
				}
				return copy;
			} catch (Exception ex) {
				throw new RuntimeException(ex);
			}
		}
		
	
		public void addSpecificElements (MotionLine addThis, List<Integer>specificElements) {
			for (int idx=0; idx<specificElements.size();idx++) {
				int idxElement = specificElements.get(idx);
				list.set(idxElement, list.get(idxElement) + addThis.getElement(idxElement));
			}
		}
		
		// adds the difference between to motion lines used to add relative movements to existing pose
		// rather than replacing a position in its entirity. 
		public void addDelta(MotionLine addThis, MotionLine deltaReference) {
			for (int idx = 0; idx < list.size(); idx++) {
				list.set(idx, list.get(idx)
						+ (addThis.getElement(idx) - deltaReference
								.getElement(idx)));
			}
		}

		public void setElement (int idx, double value) {
			list.set(idx, value);
		}
		
		public void replaceElements (MotionLine withThis, List<Integer> elements) {
			for (int idx=0; idx<elements.size(); idx++) {
				list.set(elements.get(idx), withThis.getElement(elements.get(idx)));
			}
		}
		
		public String toString() {
			StringBuffer buffer = new StringBuffer();
			for (int idx = 0; idx < list.size(); idx++) {
				if (idx > 0) {
					buffer.append(" ");
				}
				buffer.append(list.get(idx));
			}
			return buffer.toString();
		}

		public Double getElement(int idx) {
			if (idx > list.size()) {
				throw new IndexOutOfBoundsException("A motion line index is out of bounds in '" + getFileName() + "' requesteded " + idx + " but size is " + list.size());
			}
			return list.get(idx);
		}

		public int getElementSize() {
			return list.size();
		}
	
	}
	
	
}
