// This POV-Ray include file contains a definition
// for a book, with a nice cover, title, and paper.
//
// - The front and back covers of the book are 
//   superellipsoids, nearly boxes.  The text on
//   the front cover is constructed by subtracting
//   the letters from the cover object, and then
//   inserting an inner layer of gold.
//
// - The paper of the book is just a box with
//   the texture a squeezed up y gradient.
//
// - The binding of the book is a tricky bit with
//   composed of a thinner superellipsoid for the
//   spine, and long thin tubes, halved, connecting
//   the spine to the covers.
//


// include our scene constants
#include "school1.inc"
#include "golds.inc"


// declare shape for the book cover

#declare BookCoverBasicShape = superellipsoid { 
	<BookCoverEWRoundness, BookCoverNSRoundness>
	translate <0,1,0>
	scale 0.5
	scale <BookCoverLength, BookCoverThickness, BookCoverWidth>
}

#declare BookTitleTextBasicShape = text {
	ttf "tt0574m_.ttf",
	"Geography",
	BookCoverThickness*2, <0,0,0>
	scale 1.18
	translate <-2.06,0,0>
}

#declare BookTitleCoverPhoto = box {
	<0,0,0>,<1,1,BookCoverThickness>
	texture {
	  pigment {	
		image_map {
		    tga "coverphoto.tga"
		    once
		}
	  }
	  finish { ambient BasicAmbient*1.05 diffuse 0.95 }

	}
	rotate <90,0,0>
	translate <-0.5,0,-0.5>
	scale <BookCoverWidth * 0.6,1,BookCoverWidth * 0.6>
}

#declare BookTitleCoverPhotoHole = box {
	<0,0,0>,<1,1,BookCoverThickness*2>
	rotate <90,0,0>
	translate <-0.5,0,-0.5>
	scale <BookCoverWidth * 0.6,1,BookCoverWidth * 0.6>
}

#declare BookTitleCoverPhotoBorderHole = difference {
	object { BookTitleCoverPhotoHole scale <1.16,1,1.16> }
	object { BookTitleCoverPhotoHole scale <1.1, 1,1.1> }
}

#declare BookCoverFrontShape = difference {
	object { BookCoverBasicShape }
	object { BookTitleTextBasicShape 
		rotate <90,0,0>
		translate <0,BookCoverThickness*1.1,0>
		rotate <0,-90,0>
		translate <-(BookCoverLength/5),0,0>
	}
	object { BookTitleCoverPhotoBorderHole 
		 rotate <0,-90,0>
	 	 translate <BookCoverLength/5, BookCoverThickness*1.1, 0>
	}
	object { BookTitleCoverPhotoHole
		 rotate <0,-90,0>
	 	 translate <BookCoverLength/5, BookCoverThickness*1.1, 0>
	}
}

#declare BookCoverFrontGold = object {
	BookCoverBasicShape
	scale 0.89
	texture { 
		pigment { P_Gold2 } 
		finish { F_MetalC ambient (BasicAmbient+0.056) } 
	}
}
	

#declare BookCoverBasicTexture = texture {
	pigment {
		granite 
		color_map {
			[0.0 color DarkGreen]
			[0.4 color rgb<0.181,0.297,0.181>]
			[0.6 color DarkGreen]
			[1.0 color DarkGreen]
		}
		turbulence 1.5
		octaves 2
		scale 19
	}
	normal {
//		wrinkles 1.5
		quilted 0.73 control0 0.53 control1 0.8
		turbulence 1.2
		octaves 3
		scale 0.35
	}
	finish {
		ambient BasicAmbient
		diffuse BasicDiffuse
		specular 0.79
		roughness 0.002
	}
}


#declare BookSpineBasicShape = superellipsoid {
	<BookSpineEWRoundness, BookSpineNSRoundness >
	translate <0,1,0>
	scale 0.5
	scale <BookCoverLength, BookSpineThickness, BookOverallThickness>
	rotate <90,0,0>
	translate <0,0, -(BookCoverWidth/2 + BookHingeWidth) + BookHingeThickness >
}


#declare BookHingeBasicShape = intersection {
    difference {
	cylinder { 
	  <-(BookCoverLength/2 - BasicFudge/2), 0, 0>, 
	  < (BookCoverLength/2 - BasicFudge/2), 0, 0>,
	  BookHingeWidth/2
	}
	cylinder {
	  <-BookCoverLength, 0, 0>, <BookCoverLength, 0, 0>,
	  (BookHingeWidth/2 - BookHingeThickness)
	}
    }
    box {
	<-BookCoverLength/2, -(BookHingeWidth), -(BookHingeWidth)>,
	< BookCoverLength/2,   0,                 BookHingeWidth >
    }
}

#declare BookCoverFinalFrontShape = union {
	object { BookCoverFrontShape texture { BookCoverBasicTexture } }
	object { BookCoverFrontGold }
	object { BookTitleCoverPhoto
		 rotate <0,-90,0>
		 translate <BookCoverLength/5, BookCoverThickness*1.1, 0>
	}
}

#declare BookCoverBasicTest = union {
	object { 
	  BookCoverFinalFrontShape 
	  translate <0, BookOverallThickness - BookCoverThickness/2, -BasicFudge/2> 
	}
	object { 
	  BookSpineBasicShape 
	  translate <0, BookOverallThickness/2, 0>
	  texture { BookCoverBasicTexture scale 0.7 rotate <12,19,40> }
	}
	object { 
	  BookCoverBasicShape 
	  translate <0, 0, -BasicFudge/2>
  	  texture { BookCoverBasicTexture rotate <3,9,12> }
	}
	object {
	  BookHingeBasicShape
  	  texture { BookCoverBasicTexture scale 0.7  }
	  translate <0, BookOverallThickness + BookCoverThickness/2, -(BookCoverWidth/2 + BookHingeWidth/2) + BookHingeThickness/2> 
	}
	object {
	  BookHingeBasicShape
	  scale <1,-1,1>
	  translate <0, 0, -(BookCoverWidth/2 + BookHingeWidth/2) + BookHingeThickness/2> 
  	  texture { BookCoverBasicTexture scale 0.7 }
	}
}

	


#declare BookPagesBasicShape = box {
	<0,0,0>,
	<BookCoverLength - 2*BookCoverExtension, 
	 BookOverallThickness - BookCoverThickness,
	 (BookCoverWidth + BookHingeWidth/2) - BookCoverExtension>
}

#declare PaperWhite = color red 0.96 green 0.91 blue 0.88
#declare BookPagesSidesTexture = texture {
	pigment {
		gradient y
		color_map {
			[0.0 0.75 color PaperWhite color PaperWhite]
			[0.75 0.9 color PaperWhite color (PaperWhite*0.6) ]
			[1.0 color (PaperWhite*0.6)]
		}
		scale 0.06
	}
	normal {
		gradient y, 0.69
		scale 0.06
	}
	finish {
		ambient BasicAmbient diffuse 0.45
		specular 0.55 roughness 0.1
	}
}

#declare BookPagesFinal = object {
	BookPagesBasicShape
	texture { BookPagesSidesTexture }
}

#declare BookTest = union {
	object { 
	  BookCoverFinalFrontShape 
	  translate <0, BookOverallThickness - BookCoverThickness/2, 0> 
	}
	object { 
	  BookSpineBasicShape 
	  translate <0, BookOverallThickness/2, -BasicFudge/2>
	  texture { BookCoverBasicTexture scale 0.7 rotate <12,19,40> }
	}
	object { 
	  BookCoverBasicShape 
	  translate <0, 0, 0>
  	  texture { BookCoverBasicTexture rotate <3,9,12> }
	}
	object {
	  BookHingeBasicShape
  	  texture { BookCoverBasicTexture scale 0.7  }
	  translate <0, 
		BookOverallThickness + BookCoverThickness/2 - BasicFudge, 
	       -(BookCoverWidth + BookHingeWidth)/2 + BookHingeThickness * 0.7 > 
	}
	object {
	  BookHingeBasicShape
	  scale <1,-1,1>
	  translate <0, BasicFudge, 
		-(BookCoverWidth + BookHingeWidth)/2 + BookHingeThickness * 0.7> 
  	  texture { BookCoverBasicTexture scale 0.7 }
	}
	object {
	  BookPagesFinal
	  translate < -(BookCoverLength - BookCoverExtension)/2,
		      BookCoverThickness/2,
		      -(BookCoverWidth/2 + BookHingeWidth/2) + BasicFudge/2>
	}
}




