// This POV-Ray include file contains a  
// globe of a fictious world, in a nice wooden stand.
//
// All units in this file are in inches.
// 
//    - The world is a sphere with very complex layered textures
//      controlled with a layered texture map.  The globe is 12
//	inches in radius.
//
//    - The meridian ring of the globe is made of bronze, with
//	little indentations every five degrees.  The meridian ring
//	is 13 inches in radius and .3 of an inch wide.
//    - There is a bronze support rod running throught the globe
//	at an angle of 24 degrees.  The rod is 1/5 of an inch thick.
//    - Around the globe is a wooden rail, situated so that its
//	top coincides with the globe's diameter, and about 3/4 of
//      an inch thick.  The inner radius of the rail is 12.7 inches
//	and the outer radius is 14 inches.
//
//    - Holding up the globe are three wooden legs.  The legs are
//	each chopped ellipsoids, possibly with some toruses for
//	decoration.  Each leg is 25 inches long.  Each leg is 1/2
//	inch in radius.
//    - The legs are attached to a circular wooden base, 1 inch
//	thick.
//    - The wooden base is suspended off the floor by cylindrical
//	wooden legs, 6 inches long, and about 1/2 inch in radius.
//    - Each leg is ends with a little cylindrical brass wheel
//      about .25 inches thick and .75 inches in radius.

// include our scene constants
#include "school1.inc"



// declare textures for the globe

#declare GlobeDefaultFinish = finish {
       ambient BasicAmbient
       diffuse 1
       // specular 0.004
       // roughness 0.0015
     }

#declare LandTexture = texture
{
   pigment {
	granite
	color_map {
		[0.0 0.5 color DarkGreen color Green]
		[0.5 0.6 color DarkTan color Brown]
		[0.6 0.91 color Brown   color DarkBrown]
		[0.90 0.91 color DarkBrown color DarkBrown]
		[0.91 0.92 color DarkBrown color White]
		[0.92 1.0  color White color White]
	}
	turbulence 1.17
	octaves 3
	scale 1.6
   }
   finish { GlobeDefaultFinish }
}

#declare WaterTexture = texture
{
   pigment {
      color rgb<0.2,0.66,0.85>
   }
   finish { GlobeDefaultFinish }
}

#declare TemperateGlobeTexture =    texture 
{
	bozo
	texture_map {
	   [0.0 LandTexture]
	   [0.47 LandTexture]
	   [0.47 WaterTexture]
	   [1.0 WaterTexture]
	}
	turbulence 0.885
	octaves 4
	scale 0.434
}

#declare PolarGlobeTexture = texture {
	pigment { color White }
	finish { GlobeDefaultFinish }
}


#declare GlobeContentTexture = texture {
	gradient y
	texture_map {
	   [0.0 TemperateGlobeTexture]
	   [0.89 TemperateGlobeTexture]
	   [0.891 PolarGlobeTexture]
	   [1.0  PolarGlobeTexture]
	}
	turbulence 0.18
	// lambda 1.39
	// omega 0.7
	octaves 4
   }

#declare GlobeMarkingsTexture = texture {
  pigment { color Gray40 }
        finish {
           ambient BasicAmbient
           diffuse BasicDiffuse
           specular 0.32
           roughness 0.012
        }
}

#declare GlobeTexture = texture {
  material_map {
    gif "grid.gif"
    map_type 1
    texture { GlobeMarkingsTexture }
    texture { GlobeContentTexture } 
  }
}


// declare the globe itself at the origin

#declare GlobeBody = sphere
{
   <0,0,0>,1
   texture {
     GlobeTexture
   }
      
   scale GlobeRadius
   rotate GlobeRotation1
   rotate GlobeRotation2
   // move the globe to its eventual center position relative to stand
   translate GlobePosition
}


// declare the bronze ring that surrounds
#declare GlobeRingTexture = texture { 
  pigment { P_Brass4 } finish { F_MetalB ambient BasicAmbient reflection 0.13 }
}

#declare GlobeRingBasicShape = torus { 
GlobeRingMajorRadius,GlobeRingMinorRadius 
}

#declare GlobeRingFullShape = difference {
  object { GlobeRingBasicShape }
  union {
    cylinder { <0,-2,0>,<0,2,0>, GlobeRingEmptyPad }
    cylinder {   <0,GlobeRingThickness/2,0> ,<0,2,0>, GlobeRadius*2 }
    cylinder {   <0,-(GlobeRingThickness/2),0>, <0,-2,0>, GlobeRadius*2 }
  }
}

// if you want to put little tick marks in the ring, do
// it here:
// #declare GlobeTickMarkShape0 = box {
//  <-GlobeRingThickness/3,-GlobeRingThickness*3.75,0>,
//  < GlobeRingThickness/3, GlobeRingThickness*3.75, GlobeRingMinorRadius*1.5>
//  translate <0,0, GlobeRingEmptyPad>
// }
#declare GlobeTickMarkShape = cylinder {
  <0,0,0>,<0,0,GlobeRingMinorRadius * 1.1>, GlobeRingThickness * .18
  scale <1,12,1>
  translate <0,0, GlobeRingEmptyPad>
}

#declare GlobeRingFinalShape = union {
  object { GlobeRingFullShape }
  // 72 little boxes
  object { GlobeTickMarkShape rotate <0,0,0> }
  // object { GlobeTickMarkShape rotate <0,5,0> }
  object { GlobeTickMarkShape rotate <0,10,0> }
  // object { GlobeTickMarkShape rotate <0,15,0> }
  object { GlobeTickMarkShape rotate <0,20,0> }
  // object { GlobeTickMarkShape rotate <0,25,0> }
  object { GlobeTickMarkShape rotate <0,30,0> }
  // object { GlobeTickMarkShape rotate <0,35,0> }
  object { GlobeTickMarkShape rotate <0,40,0> }
  // object { GlobeTickMarkShape rotate <0,45,0> }
  object { GlobeTickMarkShape rotate <0,50,0> }
  // object { GlobeTickMarkShape rotate <0,55,0> }
  object { GlobeTickMarkShape rotate <0,60,0> }
  // object { GlobeTickMarkShape rotate <0,65,0> }
  object { GlobeTickMarkShape rotate <0,70,0> }
  // object { GlobeTickMarkShape rotate <0,75,0> }
  object { GlobeTickMarkShape rotate <0,80,0> }
  // object { GlobeTickMarkShape rotate <0,85,0> }
  object { GlobeTickMarkShape rotate <0,90,0> }
  // object { GlobeTickMarkShape rotate <0,95,0> }
  object { GlobeTickMarkShape rotate <0,100,0> }
  // object { GlobeTickMarkShape rotate <0,105,0> }
  object { GlobeTickMarkShape rotate <0,110,0> }
  // object { GlobeTickMarkShape rotate <0,115,0> }
  object { GlobeTickMarkShape rotate <0,120,0> }
  // object { GlobeTickMarkShape rotate <0,125,0> }
  object { GlobeTickMarkShape rotate <0,130,0> }
  // object { GlobeTickMarkShape rotate <0,135,0> }
  object { GlobeTickMarkShape rotate <0,140,0> }
  // object { GlobeTickMarkShape rotate <0,145,0> }
  object { GlobeTickMarkShape rotate <0,150,0> }
  // object { GlobeTickMarkShape rotate <0,155,0> }
  object { GlobeTickMarkShape rotate <0,160,0> }
  // object { GlobeTickMarkShape rotate <0,165,0> }
  object { GlobeTickMarkShape rotate <0,170,0> }
  // object { GlobeTickMarkShape rotate <0,175,0> }
  object { GlobeTickMarkShape rotate <0,180,0> }
  // object { GlobeTickMarkShape rotate <0,185,0> }
  object { GlobeTickMarkShape rotate <0,190,0> }
  // object { GlobeTickMarkShape rotate <0,195,0> }
  object { GlobeTickMarkShape rotate <0,200,0> }
  // object { GlobeTickMarkShape rotate <0,205,0> }
  object { GlobeTickMarkShape rotate <0,210,0> }
  // object { GlobeTickMarkShape rotate <0,215,0> }
  object { GlobeTickMarkShape rotate <0,220,0> }
  // object { GlobeTickMarkShape rotate <0,225,0> }
  object { GlobeTickMarkShape rotate <0,230,0> }
  // object { GlobeTickMarkShape rotate <0,235,0> }
  object { GlobeTickMarkShape rotate <0,240,0> }
//   object { GlobeTickMarkShape rotate <0,245,0> }
     object { GlobeTickMarkShape rotate <0,250,0> }
//   object { GlobeTickMarkShape rotate <0,255,0> }
   object { GlobeTickMarkShape rotate <0,260,0> }
//   object { GlobeTickMarkShape rotate <0,265,0> }
     object { GlobeTickMarkShape rotate <0,270,0> }
//   object { GlobeTickMarkShape rotate <0,275,0> }
   object { GlobeTickMarkShape rotate <0,280,0> }
//   object { GlobeTickMarkShape rotate <0,285,0> }
     object { GlobeTickMarkShape rotate <0,290,0> }
//   object { GlobeTickMarkShape rotate <0,295,0> }
   object { GlobeTickMarkShape rotate <0,300,0> }
//   object { GlobeTickMarkShape rotate <0,305,0> }
     object { GlobeTickMarkShape rotate <0,310,0> }
//   object { GlobeTickMarkShape rotate <0,315,0> }
   object { GlobeTickMarkShape rotate <0,320,0> }
//   object { GlobeTickMarkShape rotate <0,325,0> }
     object { GlobeTickMarkShape rotate <0,330,0> }
//   object { GlobeTickMarkShape rotate <0,335,0> }
  object { GlobeTickMarkShape rotate <0,340,0> }
//   object { GlobeTickMarkShape rotate <0,345,0> }
     object { GlobeTickMarkShape rotate <0,350,0> }
//   object { GlobeTickMarkShape rotate <0,355,0> }
}


#declare GlobeRing = object {
  GlobeRingFinalShape
  texture { 
    GlobeRingTexture
  }
  rotate <90,0,0>
  rotate GlobeRotation2
  translate GlobePosition
}


// the support rod through the globe

#declare GlobeRodBasicShape = cylinder {
  <0,-(GlobeRingMajorRadius+GlobeRingMinorRadius),0>,
  <0, (GlobeRingMajorRadius+GlobeRingMinorRadius),0>,
  GlobeRodRadius
}

#declare GlobeRodTexture = texture { T_Brass_5D }

#declare GlobeRod = object {
  GlobeRodBasicShape
  texture {
    GlobeRodTexture
  }
  rotate GlobeRotation2
  translate GlobePosition
}


// The Globe with its ring and rod!
#declare GlobeWithRingObject = union {
  object { GlobeBody }
  object { GlobeRod }
  object { GlobeRing }
}


// the wooden rim of the globe
#declare grr = GlobeRimRadius
#declare grh = GlobeRimHeight

#declare GlobeRimBasicShape  = lathe {
  quadratic_spline
	13,
	<0,          -(8*grh)/8>,
        <grr - 0.05, -(8*grh)/8>,
	<grr - 0.0,  -(8*grh)/8>,
	<grr - 0.08, -(7*grh)/8>,
	<grr - 0.2 , -(6*grh)/8>,	
	<grr - 0.44 ,-(5*grh)/8>,
	<grr - 0.55 ,-(4*grh)/8>,
	<grr - 0.6 , -(3*grh)/8>,
	<grr - 0.61, -(2*grh)/8>,
	<grr - 0.65 ,-(1*grh)/8>,
	<grr - 0.89, -(0*grh)/8>,
	<grr - 1,    -(0*grh)/8>,
        <0,          -(0*grh)/8>
}

#declare GlobeRimFinalShape = difference {
  intersection {
    object { GlobeRimBasicShape }
    cylinder { <0,-grh,0>,<0,0,0>,grr }
  }
  cylinder { <0,-(grh*2),0>,<0,grh,0>, GlobeRingEmptyPad }
}

// this is basically T_Wood8 from the standard Povray3 textures
// with a different finish and scaled up a little for a bigger grain. 
#declare GlobeStandWoodTexture =
    texture { 
      pigment { P_WoodGrain8A  color_map { M_Wood8A } }
      finish { ambient BasicAmbient diffuse BasicDiffuse 
	       specular 0.1 roughness 0.01
      }
      rotate <4,0,0>
      scale 5
    } 
    texture { 
      pigment { P_WoodGrain8B  color_map { M_Wood8B } }
      finish { ambient BasicAmbient diffuse BasicDiffuse
	       specular 0.23 roughness 0.035
      }
      rotate <4,0,0>
      scale 5
    }

#declare GlobeRim = object {
      GlobeRimFinalShape
      texture { GlobeStandWoodTexture rotate <0,0,39> }
      translate GlobePosition
}

// legs for the globe rim

#declare GlobeRimLegBasicShape = sphere {
  <0,-(GlobeRimLegRadius/4),0>, GlobeRimLegRadius
  scale <1,(GlobeRimLegLength/GlobeRimLegRadius)*1.5,1>
}

#declare GlobeRimLegChoppedShape = intersection {
  object { GlobeRimLegBasicShape }
  cylinder { <0,-GlobeRimLegLength,0>,<0,0,0>,3 }
}

// note that this shape is still not translated to its final height or 
// position, nor does it have a texture yet!
#declare GlobeRimLegFinalShape = union {
  object { GlobeRimLegChoppedShape }
  torus { GlobeRimLegRadius,GlobeRimLegDecorationMinorRadius
	  translate <0,-(GlobeRimLegLength*0.42),0>
  }
  torus { GlobeRimLegRadius,GlobeRimLegDecorationMinorRadius
	  translate <0,-(GlobeRimLegLength*0.5),0>
  }
  torus { GlobeRimLegRadius,GlobeRimLegDecorationMinorRadius
	  translate <0,-(GlobeRimLegLength*0.58),0>
  }
}

#declare grldist = ((GlobeRimRadius - 1) - (0.7 * GlobeRimLegRadius))

// here are the three legs with textures and position set
#declare GlobeRimLeg1 = object {
  GlobeRimLegFinalShape
  texture { GlobeStandWoodTexture scale <1,1.6,1> rotate <40,0,0> }
  translate <0 * grldist,GlobeCenterHeight - BasicFudge, -1 * grldist>
}
#declare GlobeRimLeg2 = object {
  GlobeRimLegFinalShape
  texture { GlobeStandWoodTexture scale <1,1.7,1.3> rotate <30,0,2> }
  translate <0.87 * grldist,GlobeCenterHeight - BasicFudge, 0.5 * grldist>
}
#declare GlobeRimLeg3 = object {
  GlobeRimLegFinalShape
  texture { GlobeStandWoodTexture scale <1,1.3,1.1> rotate <21,1,0> }
  translate <-0.87 * grldist,GlobeCenterHeight - BasicFudge, 0.5 * grldist>
}



// The little base of the globestand - keep this simple!
#declare GlobeStandBaseShape = cylinder {
  <0,0,0>,<0,GlobeStandBaseThickness,0>, GlobeStandBaseRadius
}

#declare GlobeStandBase = object {
  GlobeStandBaseShape
  texture { GlobeStandWoodTexture scale <1,1.2,1.6> rotate <0,0,15> }
  translate <0,GlobeStandBaseHeight,0>
}


// A little foot for the globe, with little wheel -- keep this simple!

#declare GlobeStandFootWoodShape = cylinder {
  <0,-(GlobeStandFootLength),0>,<0,0,0>, GlobeStandFootRadius
}

#declare GlobeStandFootWoodFinal = object {
  GlobeStandFootWoodShape
  texture { GlobeStandWoodTexture }
  translate <0,GlobeStandBaseHeight,0>
}

#declare GlobeStandFootWheelShape = cylinder {
  <0,GlobeStandFootWheelRadius,-0.1>,
  <0,GlobeStandFootWheelRadius, 0.1>,
  GlobeStandFootWheelRadius
}

#declare GlobeStandFootWheelFinal = object {
  GlobeStandFootWheelShape
  texture { pigment { P_Brass4 } finish { F_MetalD ambient BasicAmbient } }
}

#declare GlobeStandFootFinal = union {
  object { GlobeStandFootWoodFinal }
  object { GlobeStandFootWheelFinal }
}

#declare GlobeStandFoot1 = object {
  GlobeStandFootFinal
  translate <0 * grldist, 0, -1 * grldist>
}
#declare GlobeStandFoot2 = object {
  GlobeStandFootFinal
  translate <0.86 * grldist, 0, 0.5 * grldist>
}
#declare GlobeStandFoot3 = object {
  GlobeStandFootFinal
  translate <-0.86 * grldist, 0, 0.5 * grldist>
}

#declare GlobeStandFinalObject = union {
  object { GlobeRim }
  object { GlobeRimLeg1 }
  object { GlobeRimLeg2 }
  object { GlobeRimLeg3 }
  object { GlobeStandBase }
  object { GlobeStandFoot1 }
  object { GlobeStandFoot2 }
  object { GlobeStandFoot3 }
}

// finally, the full definition of the globe itself, with all textures
// and correctly positioned and rotated on the Y axis.  It may need to be 
// moved on the X or Z axes.

#declare GlobeFinalObject = union {
  object { GlobeWithRingObject }
  object { GlobeStandFinalObject }
  rotate <0,29,0>
}


