// Persistence of Vision Ray Tracer Scene Description File
// File: Clarence.inc
// Vers: 3
// Desc: Clarence, the Armadillo (stuffed)
// Date: 8/7/1997
// Auth: Steve Sloan II

#declare hip_angle = 45     // Rotation angle of the upper rear leg
#declare knee_angle = 70    // Rotation angle of the lower rear leg
#declare fore_angle = 40    // Rotation angle of the foreleg

#declare steps = 20         // # of cones used to create the tail
#declare sr = 0.075         // Radius of the base of the tail
#declare er = 0             // Radius of the tip of the tail
#declare yr = 0.2           // Radius of curvature of the tail in the y direction
#declare zr = 0.3           // Radius of curvature of the tail in the z direction

#declare armadillo_color = pigment { color Gray50 }

#declare StrengthVal = 1.0  // (+ or -) strength of component's radiating density
#declare RadiusVal   = 1.0  // (0 < RadiusVal) outer sphere of influence on other components

// Shell patterns
#declare scale_pattern =
pigment {
    brick
    color Black, color Gray50
    mortar 0.25
    scale 0.005
}

#declare band_pigment =
pigment {
    radial
    pigment_map {
        [0.0  color Black ]
        [0.05 color Black ]
        [0.05 armadillo_color ]
        [0.95 armadillo_color ]
        [0.95 color Black ]
        [1.0  color Black ]
    }
    frequency 50
    rotate 90*x
    scale .002
}

#declare armor_radial =
pigment{
    radial
    pigment_map {
        [0   armadillo_color]
        [0.9 armadillo_color]
        [0.9 color Black]
        [1.0 color Black]
    }
    frequency 60
    rotate 90*x
}

#declare armor_pattern =
pigment{
    gradient z
    pigment_map {
        [0   armor_radial]
        [0.9 armor_radial]
        [0.9 color Black]
        [1.0 color Black]
    }
    scale <1, 1, 0.05>
}

#declare band_pattern =
pigment {
    gradient z
    pigment_map {
        [0.0  color Black ]
        [0.05 color Black ]
        [0.05 band_pigment ]
        [0.95 band_pigment ]
        [0.95 color Black ]
        [1.0  color Black ]
    }
    scale <1, 1, 0.5 / 7>
    translate 0.25*z
}

#declare shell_pattern =
pigment {
    gradient z
    pigment_map {
        [0.0  armor_pattern ]
        [0.25 armor_pattern ]
        [0.25 band_pattern ]
        [0.75 band_pattern ]
        [0.75 armor_pattern ]
        [1.0  armor_pattern ]
    }
    translate -0.5*z
    scale <1, 1, 0.55>
}

// The bands that allow an armadillo to roll into a ball
#declare band_shape =
union {
    #declare BandCount = 0
    #while (BandCount < 7)
        cone {
            0*z, 0.975,
            -0.6875*z/7, 0.95
            translate -BandCount*0.6875*z/7
        }
        #declare BandCount = BandCount+1 // increment our counter
    #end
    translate 0.34375*z
}

// Sphere with room for the "bands" that allow an armadillo to roll into a ball
#declare body_sphere =
difference {
    sphere { 0, 1 }
    box { <-10, -10, -0.34375>, <10, 10, 0.34375> }
}

// The basic shape of the shell
#declare outer_shell =
union {
    intersection {
        union {
            object { body_sphere }
            object { band_shape }
            scale <0.25, 0.3, 0.4>
        }
        plane { -y, 0 }
    }
    union {
        object { body_sphere }
        object { band_shape }
        scale <0.25, 0.15, 0.4>
    }
}

// The actual shell
#declare body_shell =
difference {
    object {
        outer_shell
    }
    union {
        object {
            outer_shell
            scale 0.95
        }
        cylinder {
            -10*z,  10*z,  1
            scale <0.125, 0.35, 1>
            translate -0.15*y
        }
    }
    pigment { shell_pattern }
}

// The true body, inside the shell
#declare inner_body =
sphere {
    0, 1
    scale <0.2, 0.175, 0.375>
    translate 0.075*y
    pigment { scale_pattern }
}

// Clarence's body
#declare body =
union {
    object { body_shell }
    object { inner_body }
}

#declare rear_leg =
blob {
    threshold 0.6

    // Upper rear leg
   cylinder {
        0, -0.175*y, 0.075, StrengthVal
        rotate -hip_angle*x
    }

    // Lower rear leg
    cylinder {
        -0.025*y, -0.19*y, 0.075, StrengthVal
        scale <1, 1, 0.5>
        rotate knee_angle*x
        translate vrotate(-0.2*y, -hip_angle*x)
    }

    // Rear toes
    cylinder {
        -0.025*y, -0.05*y, 0.025, StrengthVal
        rotate (90 - knee_angle)*x
        translate <0.03, -0.2, 0>
        rotate knee_angle*x
        translate vrotate(-0.2*y, -hip_angle*x)
    }
    cylinder {
        -0.025*y, -0.05*y, 0.025, StrengthVal
        rotate (90 - knee_angle)*x
        translate -0.2*y
        rotate knee_angle*x
        translate vrotate(-0.2*y, -hip_angle*x)
    }
    cylinder {
        -0.025*y, -0.05*y, 0.025, StrengthVal
        rotate (90 - knee_angle)*x
        translate <-0.03, -0.2, 0>
        rotate knee_angle*x
        translate vrotate(-0.2*y, -hip_angle*x)
    }
    pigment { scale_pattern }
}

#declare front_leg =
blob {
    threshold 0.6

    // Foreleg
    cylinder {
        -0.025*y, -0.19*y, 0.075, StrengthVal
        scale <1, 1, 0.5>
        rotate fore_angle*x
    }

    // Front toes
    cylinder {
        -0.025*y, -0.05*y, 0.025, StrengthVal
        rotate (90 - fore_angle)*x
        translate <0.03, -0.2, 0>
        rotate fore_angle*x
    }
    cylinder {
        -0.025*y, -0.05*y, 0.025, StrengthVal
        rotate (90 - fore_angle)*x
        translate -0.2*y
        rotate fore_angle*x
    }
    cylinder {
        -0.025*y, -0.05*y, 0.025, StrengthVal
        rotate (90 - fore_angle)*x
        translate <-0.03, -0.2, 0>
        rotate fore_angle*x
    }
    pigment { scale_pattern }
}

#declare ear =
difference {
    intersection {
        sphere { 0, 1 scale <0.04, 0.075, 0.03> pigment { scale_pattern } }
        plane { z, 0 }
    }
    sphere { 0, 1 scale <0.04, 0.075, 0.02> pigment { color Gray60 } }
    translate 0.07*y
}

#declare ears =
union {
    object {
        ear
        translate 0.1*y
        rotate 20*z
    }
    object {
        ear
        translate 0.1*y
        rotate -20*z
    }
    rotate 180*y
    rotate -80*x
}

#declare eyes =
union {
    sphere {
        0, 1
        scale 0.015
        texture {
            pigment { color Black }
            finish { phong 1 }
        }
        translate < 0.085, -0.05, 0>
        rotate 10*y
    }
    sphere {
        0, 1
        scale 0.015
        texture {
            pigment { color Black }
            finish { phong 1 }
        }
        translate <-0.085, -0.05, 0>
        rotate -10*y
    }
}

#declare head =
union {
    sphere {
        0, 0.1
        pigment { scale_pattern }
    }
    cone {
        0, 0.1,
        -0.2*y, 0.05
        pigment { scale_pattern }
    }
    sphere {
        0, 1
        scale <0.05, 0.075, 0.05>
        translate -0.2*y
        pigment { scale_pattern }
    }
    object { ears }
    object { eyes }
}

#declare tail_texture =
pigment{
    radial
    pigment_map {
        [0   armadillo_color]
        [0.9 armadillo_color]
        [0.9 color Black]
        [1.0 color Black]
    }
    frequency 10
    rotate 90*x
}

#declare tail =
union {
    #declare curr_count = 0
    #while (curr_count < steps)
        #declare currt = curr_count / steps
        #declare nextt = (curr_count + 1) / steps
        #declare curr_angle = 270 - 180*currt
        #declare next_angle = 270 - 180*nextt
        #declare y1 = yr*sin(radians(curr_angle))
        #declare z1 = zr*cos(radians(curr_angle))
        #declare y2 = yr*sin(radians(next_angle))
        #declare z2 = zr*cos(radians(next_angle))
        #declare curr_radius = sr + currt*(er - sr)
        #declare next_radius = sr + nextt*(er - sr)
        #declare rot_angle = degrees(atan2(y1 - y2, z1 - z2))
        #declare cone_length = vlength(<0, y2, z2> - <0, y1, z1>)

        cone {
            0, curr_radius
            1.2*cone_length*z, next_radius*0.9

            pigment { tail_texture }

            rotate (180 - rot_angle)*x
            translate <0, y1, z1>
        }
        #declare curr_count = curr_count+1
    #end

    translate yr*y
}

#declare armadillo =
union {
    object { rear_leg translate < 0.075, 0, 0.15> }
    object { rear_leg translate <-0.075, 0, 0.15> }
    object { front_leg translate < 0.075, -0.05, -0.15> }
    object { front_leg translate <-0.075, -0.05, -0.15> }
    object { body }
    object { head rotate 45*x translate <0, 0.1, -0.35> }
    object { tail rotate 180*y translate <0, 0.1, 0.325> }
    translate 0.225*y
    scale 20
}

