
// Persistence of Vision Ray Tracer Scene Description File
// File: 8088.pov
// Vers: 3
// Desc: 8088 motherboard & cards
// Date: 01-30-1998
// Auth: Morgan Hamish Fraser
//


#version 3

#include "colors.inc"

#declare QuickColour=1
#declare SimpleShapes=1
#declare ViewMode=3
//#declare ShowFDDISABoard=1
//#declare UseComputerCase=1

// really good mode
#declare QuickColour=0
#declare SimpleShapes=0
#declare ViewMode=4
#declare ShowFDDISABoard=1
#declare UseComputerCase=1


global_settings
{
  assumed_gamma 1.0
  ambient_light rgb<1,1,1>
  max_trace_level 20
}

// ----------------------------------------
// lower left corner chips
#if (ViewMode=0)
camera
    {
      location  <2.0, 20, -30.0>
      direction 1.5*z
      right     4/3*x
      look_at   <20.0, 0.0,  20.0>
    }
#end
// angle view of board
#if (ViewMode=1)
    camera
    {
      location  <20.0, 60, -150.0>*1.5
      direction 1.5*z
      right     4/3*x
      look_at   <150.0, 3.0,  100.0>
    }
#end
// overhead view
#if (ViewMode=2)
    camera
    {
      location  <150.0, 151, 90>
      direction 1.5*z
      right     4/3*x
      angle     90
      look_at   <150.0, 0.0,  100.0>
    }
#end
// overhead view - vary position
#if (ViewMode=3)
    #declare ViewPoint=<50,0,50>
    #declare ViewPoint=<220,0,50>   // by CPU
    #declare ViewPoint=<175,-150,60>
//    #declare ViewPoint=<135,0,65>   // centre lower
//    #declare ViewPoint=<135,-150,65>   // over AMD
    camera
    {
      location  ViewPoint+<0,200,0>
      direction 1.5*z
      right     4/3*x
      look_at   ViewPoint
    }
#end
// View from Voltage regulator
#if (ViewMode=4)
    camera
    {
      location  <300,20,100>
      direction 1.5*z
      right     4/3*x
      angle     60
      look_at   <0,0,0>
    }
#end
// View from bottom right
#if (ViewMode=5)
    camera
    {
      location  <320,20,-20>
      direction 1.5*z
      right     4/3*x
      look_at   <80,0,190>
    }
#end

// completely variable mode
#if (ViewMode=6)
   camera
    {
      location  <200,20,-80>
      direction 1.5*z
      right     4/3*x
      look_at   <250,0,50>
    }
#end
// FDD Board
#if (ViewMode=7)
   camera
    {
      location  <150,20,-150>
      direction 1.5*z
      right     4/3*x
      look_at   <170,50,80>
    }
#end

sky_sphere
{
  pigment
  {
    gradient y
    color_map { [0.0 color <.2,0,0>] [1.0 color rgb <.3,.3,.3>] }
  }
}

light_source
{
  0*x // light's position (translated below)
  color red 1.0  green 1.0  blue 1.0  // light's color
  #ifdef (UseComputerCase)
      translate <-15, 140, -15>
      //translate <300, 140, -15>
  #else
      translate <350-15, 500, -15>
  #end
}

// ----------------------------------------
//plane { y, -1 pigment {color rgb <0.7,0.5,0.3>}}

#declare T_ChipBody=texture{
    #if (QuickColour=1)
        pigment{
            color  rgb<.04,.04,.08>
        }
    #else
        pigment{
            color  rgb<.04,.04,.08>
        }
        finish{
            specular .7
            roughness .003
            metallic
            reflection .15
        }
    #end
}

#declare T_ChipText=texture{
    pigment{
        color rgb<.74,.74,.74>
    }
}

#declare T_GreenBoard=texture{
    #if (QuickColour=1)
        pigment{
            color rgb<.05,.2,.05>
        }
    #else
        pigment{
            crackle
            //color rgb<.025,.1,.025>
            color_map{
                [0.0 rgb<0.025,0.1,0.025>]
                [1.0 rgb<0.015,0.02,0.015>]
            }
            scale 0.5
        }
        normal{
            crackle
            scale 5
        }
        finish{
            specular .2
            roughness .002
            reflection 0.15
        }
    #end
}

#declare T_ClearBoard=texture{
    pigment{
        color rgb<.6,.6,.6>
    }
}

#declare T_GoldWire=texture{
    #if (QuickColour=1)
        pigment{
            color rgb<.8,.8,.3>
        }
    #else
        pigment{
            color rgb<.8,.8,.3>
        }
        finish{
            specular 0.2
            roughness 0.01
            metallic
            reflection 0.7
        }
    #end
}

#declare T_CapacitorYellow=texture{
    pigment{
        color rgb<.6,.6,.1>
    }
}

#declare T_Circuits=texture{
    #if (QuickColour=1)
        pigment{
            color rgb<.3,.9,.3>
        }
    #else
        pigment{
            granite
            color_map{
                [0.0 rgb<.3,.9,.3>]
                [1.0 rgb<.4,.4,.3>]
            }
            turbulence .3
            scale .05
        }
        finish{
            specular .3
            roughness .04
            metallic
            reflection .25
        }
    #end
}

#declare T_RedPart=texture{
    pigment{
        color rgb<.8,.2,.2>
    }
}

#declare T_Steel=texture{
    #if (QuickColour=1)
        pigment{
            color rgb<.4,.4,.4>
        }
    #else
        pigment{
            color rgb<.4,.4,.4>
        }
        finish{
            specular 0.7
            roughness .03
            metallic
            reflection 0.3
        }
    #end
}

#declare T_DipSwitchBlueBox=texture{
    #if (QuickColour=1)
        pigment{
            color rgb<.1,.1,.6>
        }
    #else
        pigment{
            color rgb<.1,.1,.6>
        }
        finish{
            specular 0.1
            roughness .005
            reflection <0.1,0.1,0.25>
        }
    #end
}

#declare T_DipSwitchWhite=texture{
    pigment{
        color rgb<.8,.8,.8>
    }
}

#declare T_YellowBlock=texture{
    pigment{
        color rgb<.8,.8,.1>
    }
}

#declare T_GreenBlock=texture{
    #if (QuickColour=1)
        pigment{
            color rgb<.2,0.8,.2>
        }
    #else
        pigment{
            granite
            turbulence .4
            color_map{
                [0.4 rgb <.2,0.55,.43>]
                [1.0 rgb <.05,.1,.2>]
            }
            scale 1
        }
        finish{
            diffuse .5
            specular 0.45
            roughness .02
        }
    #end
}

#declare T_ResistorGreen=texture{
    pigment{
        color rgb<.5,1.0,.5>
    }
}

#declare T_ChipWhiteBody=texture{
    pigment{
        color rgb<0.8,0.8,0.8>
    }
}


#debug "Done Textures\n\n"

// origin is centre of chip
// bottom of chip where pins connect to motherboard
#declare Chip_74LS04=object{
    union{
        // body
        difference{
            box{<-19/2,3,-2.5>,<19/2,1,2.5>} //2 19 5
            cylinder{<19/2,3.01,0>,<19/2,2,0>,1.5/2}
            texture{T_ChipBody}
        }
        // text
        union{
            text {
                ttf "mic000.TTF",
                "74LS04",
                .05, 0
                scale <2.2,1.5,1>
                rotate<0,180,0>
                rotate<-90,0,0>
                translate<6,3,-0.5>
            }
            text {
                ttf "mic000.TTF",
                "PC",
                .05, 0
                scale <2.2,1.5,1>
                rotate<0,180,0>
                rotate<-90,0,0>
                translate<-3.5,3,-0.5>
            }
            text {
                ttf "mic000.TTF",
                "8237",
                .05, 0
                scale <1.6,1.6,1>
                rotate<0,180,0>
                rotate<-90,0,0>
                translate<5,3,2.4>
            }
            text {
                ttf "mic000.TTF",
                "SINGAPORE",
                .05, 0
                scale <1.2,1.2,1>
                rotate<0,180,0>
                rotate<-90,0,0>
                translate<0,3,2.4>
            }
            texture{T_ChipText}
        }
        // legs
        #declare Pins_Count=7
        #declare Pins_Height=2
        #declare Pins_ChipWidth=5
        #declare Pins_Length=15
        #declare Pins_IncludeSoderPoint=1
        #include "ChipPins.inc"

    }
}
#declare Chip_DM74LS175N=union{
    difference{
        union{
            box{<-9.5,1,-3>,<9.5,4,3>}
            box{<7.5,4,-3>,<8.7,4.01,3>
                texture{T_ChipText}
            }
        }
        cylinder{<9.5,3,0>,<9.5,4.02,0>,1}
        cylinder{<8.5,3.75,2>,<8.5,4.02,2>,0.8}
        texture{T_ChipBody}
     }
     // legs
     #declare Counter=0
     #while (Counter<6)
        object{ChipLeg translate<-13/2+(13/5)*Counter,2.5,-3>}
        object{SoderPoint translate<-13/2+(13/5)*Counter,0,-4>}
        object{ChipLeg rotate<0,180,0> translate<-13/2+(13/5)*Counter,2.5,3>}
        object{SoderPoint translate<-13/2+(13/5)*Counter,0,4>}
        #declare Counter=Counter+1
     #end
}

// origin is base, dead centre
#declare Chip_PE_21712=object{
    box{<-10,0.25,-5>,<10,6.5,5>}
    texture{T_ChipBody}
}

#debug "Chip_IP8255A_5\n"
#declare Chip_IP8255A_5=union{
    difference{
        box{<-52/2,1,-13/2>,<52/2,4.5,13/2>}
        box{<52/2-5,4,-3.5/2>,<52.01/2,4.51,3.5/2>}
        cylinder{<-52/2+13,4.25,0>,<-52/2+13,4.51,0>,3/2}
        cylinder{<52/2-2,4.25,13/2-2>,<52/2-2,4.51,13/2-2>,1/2}
        cylinder{<52/2-3,3.75,0>,<52/2-3,4.49,0>,1.25/2}
    }
    // chip legs
    #declare Counter=0
    #while (Counter<20)
        object{ChipLeg translate<-48/2+48/19*Counter,2.5,-13/2>}
        object{SoderPoint translate<-48/2+48/19*Counter,0,-13/2-1>}
        object{ChipLeg rotate<0,180,0> translate<-48/2+48/19*Counter,2.5,13/2>}
        object{SoderPoint translate<-48/2+48/19*Counter,0,13/2+1>}
        #declare Counter=Counter+1
    #end
    texture{T_ChipBody}
}

#debug "Chip_D8237AC_5\n"
#declare Chip_D8237AC_5=union{
    difference{
        box{<-25,1,-6>,<25,5,6>}
        cylinder{<25,0.9,0>,<25,5.01,0>,3.5/2}
        cylinder{<-25+39,4.75,0>,<-25+39,5.01,0>,3/2}
        cylinder{<-25+11,4.75,0>,<-25+11,5.01,0>,3/2}
    }
    #declare Counter=0
    #while (Counter<20)
        object{ChipLeg translate<-48/2+48/19*Counter,2.5,-6>}
        object{SoderPoint translate<-48/2+48/19*Counter,0,-6-1>}
        object{ChipLeg rotate<0,180,0> translate<-48/2+48/19*Counter,2.5,6>}
        object{SoderPoint translate<-48/2+48/19*Counter,0,6+1>}
        #declare Counter=Counter+1
    #end
    texture{T_ChipBody}
}

#debug "Chip_Q5500303\n"
#declare Chip_Q5500303=union{
    box{<-21/2,2.5,-3>,<21/2,3,3> texture{T_ChipBody}}
    box{<-21/2,0.5,-3>,<21/2,2.5,3> texture{T_ChipWhiteBody}}

    #declare Counter=0
    #while (Counter<8)
        object{ChipLeg translate<-18/2+18/7*Counter,2,-3>}
        object{SoderPoint translate<-18/2+18/7*Counter,0,-3-1>}
        object{ChipLeg rotate<0,180,0> translate<-18/2+18/7*Counter,2,3>}
        object{SoderPoint translate<-18/2+18/7*Counter,0,3+1>}
        #declare Counter=Counter+1
    #end

}

#debug "Chip_SN74LS158N\n"
#declare Chip_SN74LS158N=union{
    difference{
        box{<-19/2,1,-5/2>,<19/2,3.5,5/2>}
        cylinder{<-19/2+1.75,3.0,0>,<-19/2+1.75,3.51,0>,1.25/2}
        box{<19/2-2,3.0,-1.5/2>,<19.01/2,3.51,1.5/2>}
    }
    #declare Counter=0
    #while (Counter<6)
        object{ChipLeg translate<-13/2+13/5*Counter,2,-2.5>}
        object{SoderPoint translate<-13/2+13/5*Counter,0,-2.5-1>}
        object{ChipLeg rotate<0,180,0> translate<-13/2+13/5*Counter,2,2.5>}
        object{SoderPoint translate<-13/2+13/5*Counter,0,2.5+1>}
        #declare Counter=Counter+1
    #end
    texture{T_ChipBody}
}

#declare ChipHolder_Intel808X=union{
    difference{
        box{<-25,0.5,-9>,<25,4,9>}
        box{<-25.01,3.5,-8>,<25.01,4.01,8>}
        box{<-25.01,0.49,-5>,<25.01,1,5>}
        #declare Counter=0
        #while (Counter<20)
            box{<-48.5/2+48.5/19*Counter-0.5,4.02,8.01>,<-48.5/2+48.5/19*Counter+0.5,2,6>}
            box{<-48.5/2+48.5/19*Counter-0.5,4.02,-8.01>,<-48.5/2+48.5/19*Counter+0.5,2,-6>}
            #declare Counter=Counter+1
        #end
    }
    texture{T_ChipBody}
}

#debug "Chip_Intel8088\n"
#declare Chip_Intel8088=union{
    object{ChipHolder_Intel808X}
    difference{
        box{<-26,5,-13/2>,<26,9,13/2>}
        box{<20,8,-2>,<26.01,9.01,2>}
        cylinder{<-13,8.5,0>,<-13,9.01,0>,3/2}
    }
    // pins
    #declare Counter=0
    #while (Counter<20)
        object{ChipLeg translate<-48.5/2+48.5/19*Counter,7,-13/2>}
        object{ChipLeg rotate<0,180,0> translate<-48.5/2+48.5/19*Counter,7,13/2>}
        #declare Counter=Counter+1
    #end

    union{
        text{
            ttf "Arial.ttf","i",.05,0
            scale <5/0.6,5/0.6,1> rotate<0,180,0> rotate<-90,0,0>
            translate<26-16,9,0>
        }
        text{
            ttf "Arial.ttf","P8088",.05,0
            scale <1.5/0.6,1.5/0.6,1> rotate<0,180,0> rotate<-90,0,0>
            translate<26-18,9,6.5-8>
        }
        text{
            ttf "Arial.ttf","1309001D",.05,0
            scale <1.5/0.6,1.5/0.6,1> rotate<0,180,0> rotate<-90,0,0>
            translate<26-18,9,6.5-5.5>
        }
        text{
            ttf "Arial.ttf","Intel  '78  '81",.05,0
            scale <1.5/0.6,1.5/0.6,1> rotate<0,180,0> rotate<-90,0,0>
            translate<26-18,9,6.5-3.25>
        }
        texture{T_ChipText}
    }
    texture{T_ChipBody}
}

#debug "Chip_Intel8087\n"
#declare Chip_Intel8087=object{ChipHolder_Intel808X}

#debug "Chip_D8259AC\n"
#declare Chip_D8259AC=union{
    difference{
        box{<-18,1,-6>,<18,5,6>}
        cylinder{<18.5,0.99,0>,<18.5,5.01,0>,2.5/2}
    }
    //pins
    #declare Pins_Count=14
    #declare Pins_Height=3
    #declare Pins_ChipWidth=12
    #declare Pins_Length=32
    #declare Pins_IncludeSoderPoint=1
    #include "ChipPins.inc"

    union{
        text{
            ttf "Arial.ttf","D8259AC",.05,0
            scale <2.5/0.6/1.75,2.5/0.6,1> rotate<0,180,0> rotate<-90,0,0>
            translate<18-9,5,6-3>
        }
        text{
            ttf "Arial.ttf","8250E9",.05,0
            scale <1.5/0.6,1.5/0.6,1> rotate<0,180,0> rotate<-90,0,0>
            translate<18-17,5,6-7>
        }
        text{
            ttf "Arial.ttf","JAPAN",.05,0
            scale <1/0.6,1/0.6,1> rotate<0,180,0> rotate<-90,0,0>
            translate<18-10,5,6-7>
        }
        text{
            ttf "Arial.ttf","NEC",.05,0
            scale <1.5/0.6,1.5/0.6,1> rotate<0,180,0> rotate<-90,0,0>
            translate<18-10,5,6-8.5>
        }
        texture{T_ChipText}
    }
    texture{T_ChipBody}
}

#debug "Chip_DM74LS373N\n"
#declare Chip_DM74LS373N=union{
    difference{
        union{
            box{<-13,0.5,-3>,<13,3.5,3>}
            box{<13-1,3.4,-3>,<13-2.25,3.55,3>
                texture{T_ChipText}
            }
        }
        cylinder{<13,3.51,0>,<13,2.75,0>,1/2}
        cylinder{<-13+2,3.51,0>,<-13+2,3.25>,2/2}
        sphere{<13-1.5,3.5,3-1.5>,1/2}
    }
    #declare Pins_Count=10
    #declare Pins_Height=1.5
    #declare Pins_ChipWidth=6
    #declare Pins_Length=22
    #declare Pins_IncludeSoderPoint=1
    #include "ChipPins.inc"

    // text part
    union{
        #declare FontHeight=0.1
        text{
            ttf "Arial","DM74LS373N",0.05,0
            scale<1.25/0.75,1.25/0.75,1>
            rotate<90,0,0>
            rotate<0,180,0>
            translate<13-7.5,0,3-1>
        }
        text{
            ttf "Arial","SB8306*",0.05,0
            scale<1.25/0.75,1.25/0.75,1>
            rotate<90,0,0>
            rotate<0,180,0>
            translate<13-11,0,3-3.1>
        }
        translate<0,3.5,0>
        texture{T_ChipText}
    }
    texture{T_ChipBody}
}

#debug "Chip_SN74LS245N\n"
#declare Chip_SN74LS245N=union{
    difference{
        box{<-25/2,0.5,-3>,<25/2,3.5,3>}
        cylinder{<-25/2+2,3.0,0>,<-25/2+2,3.51,0>,0.5}
        box{<25/2-2,2.75,-1>,<25.01/2,3.511,1>}
        cylinder{<25/2-1,3.3,0>,<25/2-1,3.52,0>,2.5/2}
    }
    union{
        #declare TextHeight=0.1
        #declare TextSize=1.5/0.75
        text {
            ttf "arial.TTF","SN74LS245N",
            TextHeight, 0 scale <TextSize,TextSize,1>
            rotate<90,180,0> translate<25/2-6,0,3-1.7>
        }
        text {
            ttf "arial.TTF","MALAYSIA",
            TextHeight, 0 scale <TextSize*0.75,TextSize,1>
            rotate<90,180,0> translate<25/2-4,0,3-4>
        }
        #declare TextSize=1.2/0.75
        text {
            ttf "arial.TTF","2318",
            TextHeight, 0 scale <TextSize*0.85,TextSize,1>
            rotate<90,180,0> translate<25/2-16,0,3-4>
        }
        texture{T_ChipText}
        translate<0,3.5,0>
    }
    #declare Pins_Count=8
    #declare Pins_Height=1.5
    #declare Pins_ChipWidth=6
    #declare Pins_Length=18
    #declare Pins_IncludeSoderPoint=1
    #include "ChipPins.inc"
    texture{T_ChipBody}
}

#debug "Chip_AM8253\n"
#declare Chip_AM8253=union{
    difference{
        box{<-16,1,-6>,<16,4.5,6>}
        box{<16-4,4.0,-3.5/2>,<16.01,4.51,3.5/2>}
    }

    // text
    union{
        text{ ttf "Arial.ttf","AM8253-5PC",0.02,<0.5,0,0>
            scale<1.0/.75,1.5/0.75,1>
            rotate<90,0,0>
            rotate<0,180,0>
            translate<16-14,0,-6+3>
        }
        text{ ttf "Arial.ttf","P8253-5",0.02,<0.3,0,0>
            scale<1.0/.75,1.5/0.75,1>
            rotate<90,0,0>
            rotate<0,180,0>
            translate<16-14,0,-6+5.5>
        }
        text{ ttf "Arial.ttf","8313DP",0.02,<0.3,0,0>
            scale<1.0/.75,1.5/0.75,1>
            rotate<90,0,0>
            rotate<0,180,0>
            translate<16-14,0,-6+8>
        }
        text{ ttf "Arial.ttf","1980 AMD",0.02,<0.5,0,0>
            scale<1.0/.75,1.5/0.75,1>
            rotate<90,0,0>
            rotate<0,180,0>
            translate<16-16,0,-6+10.5>
        }

        // neato AMD symbol
        prism{
            linear_spline
            0,0.01
            3+4+3+4+7+7,
            <16-4,-6+1.5>,<16-12,-6+1.5>,<16-12,-6+10>,
            <16-10,-6+8>,<16-10,-6+3.5>,<16-6,-6+3.5>,<16-4,-6+1.5>,
            <16-5.25,-6+2.0>,<16-11.5,-6+2.0>,<16-11.5,-6+8.5>,
            <16-10.5,-6+7.75>,<16-10.5,-6+3.0>,<16-6.5,-6+3.0>,<16-5.25,-6+2.0>,
            <16-3.5,-6+10.0>,<16-3.5,-6+6.0>,<16-6.0,-6+4.00>,<16-6.0,-6+8.0>,<16-10.0,-6+8.0>,<16-7.5,-6+10.0>,<16-3.5,-6+10>,
            <16-4.0,-6+ 9.5>,<16-4.0,-6+6.5>,<16-5.5,-6+4.75>,<16-5.5,-6+8.5>,<16- 9.0,-6+8.5>,<16-7.5,-6+ 9.5>,<16-4.0,-6+9.5>
        }

        translate<0,4.5,0>
        texture{T_ChipText}
    }

    #declare Pins_Count=12
    #declare Pins_Height=2
    #declare Pins_ChipWidth=12
    #declare Pins_Length=28
    #declare Pins_IncludeSoderPoint=1
    #include "ChipPins.inc"

    texture{T_ChipBody}
}

#declare ChipHolder_Wide12Pin=union{
    difference{
        box{<-15,0.5,-17/2>,<15,5,17/2>}
        box{<-15.01,4.5,-8>,<15.01,5.01,8>}
        box{<-15.01,0.49,-5>,<15.01,1.5,5>}
    }

    texture{T_ChipBody}
}

#declare Chip_1501476=union{
    difference{
        box{<-31/2,5,-13/2>,<31/2,9,13/2>}
    }

    #declare Pins_Count=12
    #declare Pins_Height=7
    #declare Pins_ChipWidth=13
    #declare Pins_Length=28
    #declare Pins_IncludeSoderPoint=0
    #include "ChipPins.inc"

    object{ChipHolder_Wide12Pin}

    texture{T_ChipBody}
}

#declare Chip_8243ED=union{
    difference{
        box{<-23/2,0.4,-5>,<23/2,5,5>}
        box{<-23.01/2,4.5,-4.5>,<23.01/2,5.01,4.5>}
        cylinder{<23/2,0.25,0>,<23/2,5.02,0>,0.5}
    }
    union{
        #declare Pins_Count=9
        #declare Pins_Height=4
        #declare Pins_ChipWidth=7
        #declare Pins_Length=20
        #declare Pins_IncludeSoderPoint=0
        #include "ChipPins.inc"
        translate<0,7-4,0>
    }
    difference{
        union{
            box{<-22.5/2,5,-3.5>,<22.5/2,7,3.5>}
            box{<-22.5/2,7.001,-3.5>,<22.5/2,7.75,3.5> texture{T_ChipWhiteBody}}
            box{<-22.5/2,7.751,-3.5>,<22.5/2,9.5,3.5>}
        }
        cylinder{<22.5/2,5-0.01,0>,<22.5/2,9.51,0>,0.5}
    }
    texture{T_ChipBody}
}

#debug "Chip_Q5547008\n"
#declare Chip_Q5547008=union{
    box{<-21/2,2.5,-3>,<21/2,3,3> texture{T_ChipBody}}
    box{<-21/2,0.5,-3>,<21/2,2.5,3> texture{T_ChipWhiteBody}}

    union{
        #declare TextHeight=0.1
        #declare TextSize=2.1/0.75
        text {
            ttf "arial.TTF","Q5547008",
            TextHeight, 0 scale <TextSize*0.9,TextSize,1>
            rotate<90,180,0> translate<21/2-5,0,3-1>
        }
        #declare TextSize=1.4/0.75
        text {
            ttf "arial.TTF","316A472",
            TextHeight, 0 scale <TextSize*0.7,TextSize,1>
            rotate<90,180,0> translate<21/2-5,0,-3+1.5>
        }
        text {
            ttf "arial.TTF","8251",
            TextHeight, 0 scale <TextSize*0.7,TextSize,1>
            rotate<90,180,0> translate<21/2-15,0,-3+1.5>
        }
        translate<0,3,0>        
        texture{T_ChipText}
    }
    #declare Counter=0
    #while (Counter<8)
        object{ChipLeg translate<-18/2+18/7*Counter,2,-3>}
        object{SoderPoint translate<-18/2+18/7*Counter,0,-3-1>}
        object{ChipLeg rotate<0,180,0> translate<-18/2+18/7*Counter,2,3>}
        object{SoderPoint translate<-18/2+18/7*Counter,0,3+1>}
        #declare Counter=Counter+1
    #end
}

#debug "Chip_Q5582008\n"
#declare Chip_Q5582008=union{
    box{<-21/2,2.5,-3>,<21/2,3,3> texture{T_ChipBody}}
    box{<-21/2,0.5,-3>,<21/2,2.5,3> texture{T_ChipWhiteBody}}

    union{
        #declare TextHeight=0.1
        #declare TextSize=2.1/0.75
        text {
            ttf "arial.TTF","Q5582008",
            TextHeight, 0 scale <TextSize*0.9,TextSize,1>
            rotate<90,180,0> translate<21/2-5,0,3-1>
        }
        #declare TextSize=1.4/0.75
        text {
            ttf "arial.TTF","316A822",
            TextHeight, 0 scale <TextSize*0.7,TextSize,1>
            rotate<90,180,0> translate<21/2-5,0,-3+1.5>
        }
        text {
            ttf "arial.TTF","8251",
            TextHeight, 0 scale <TextSize*0.7,TextSize,1>
            rotate<90,180,0> translate<21/2-15,0,-3+1.5>
        }
        translate<0,3,0>        
        texture{T_ChipText}
    }
    #declare Counter=0
    #while (Counter<8)
        object{ChipLeg translate<-18/2+18/7*Counter,2,-3>}
        object{SoderPoint translate<-18/2+18/7*Counter,0,-3-1>}
        object{ChipLeg rotate<0,180,0> translate<-18/2+18/7*Counter,2,3>}
        object{SoderPoint translate<-18/2+18/7*Counter,0,3+1>}
        #declare Counter=Counter+1
    #end
}

#declare Chip_74LS10=object{Chip_74LS04} //*** change later
#declare Chip_74LS08=object{Chip_74LS04} //*** change later
#declare Chip_74LS74=object{Chip_74LS04} //*** change later
#declare Chip_DM74S74N=object{Chip_DM74LS175N} // change later
#declare Chip_SN74LS04N=object{Chip_74LS04} // *** change later
#declare Chip_74S00=object{Chip_74LS04} // *** change later
#declare Chip_SN74LS125AN=object{Chip_74LS04} // *** change later
#declare Chip_SN74S280N=object{Chip_74LS04} // *** change later
#declare Chip_74LS74=object{Chip_74LS04}
#declare Chip_DM74LS138N=object{Chip_DM74LS175N}
#declare Chip_DM74LS02N=object{Chip_DM74LS175N}
#declare Chip_DM74S08N=object{Chip_DM74LS175N}
#declare Chip_DM74LS244N=object{Chip_DM74LS373N}
#declare Chip_SN74LS322AN=object{Chip_SN74LS245N}
#declare Chip_74LS670=object{Chip_74LS04}
#declare Chip_MK36A71N_4=object{Chip_1501476}
#declare Chip_MK36A70N_4=object{Chip_1501476}
#declare Chip_XE5000021=object{Chip_1501476}
#declare Chip_MK36A68N_4=object{Chip_1501476}

// origin is mid left side (top), bottom=board connection
#debug "ISA_Slot\n"
#declare ISA_Slot=object{
    union{
        difference{
            box{<0,6,4.5>,<85,15,-4.5>}
            box{<2,5,-1>,<83,15.01,1>}
            #declare Counter=0
            #while (Counter<32)
                box{<3+77/31*Counter-0.5,11,-3.5>,<3+77/31*Counter+0.5,15.01,3.5>}
                #declare Counter=Counter+1
            #end
        }
        difference{
            box{<1,6,4>,<84,1.25,-4>}
            #declare Counter=0
            #while (Counter<32)
                box{<3+77/31*Counter-0.5,3.75,-4.01>,<3+77/31*Counter+0.5,5.75,-3.5>}
                box{<3+77/31*Counter-0.5,3.75,4.01>,<3+77/31*Counter+0.5,5.75,3.5>}
                #declare Counter=Counter+1
            #end
        }

        // connector pins
        union{
            #declare Counter=0
            #while (Counter<32)
                cylinder{<3+77/31*Counter,0,-3>,<3+77/31*Counter,3,-3>,0.25}
                object{SoderPoint translate <3+77/31*Counter,0,-3>}
                cylinder{<3+77/31*Counter,0,3>,<3+77/31*Counter,3,3>,0.25}
                object{SoderPoint translate <3+77/31*Counter,0,3>}
                #declare Counter=Counter+1
            #end
            texture{T_Steel}
        }
    }

    texture{T_ChipBody}
}

// origin is base, centre
#debug "Capacitor_BlobYellow\n"
#declare Capacitor_BlobYellow=object{
    union{
        sphere{<0,4,0>,2}
        sphere{<0,2,0>,1.6}
    }
    texture{T_CapacitorYellow}
}

// origin - dead centre, connection to Motherboard
// bottom space of 2
// total height of 4
// total width of 5
// total depth of 2 (or a bit less)
#debug "Capacitor_K5M_473\n"
#declare Capacitor_K5M_473=union{
    blob{
        threshold .2
        sphere{<-1.5,3,0>,1.5,1}
        sphere{<-1.5,4,0>,1.5,1}
        sphere{<0,3,0>,1.3,1}
        sphere{<0,4,0>,1.3,2}
        sphere{<1.5,3,0>,1.5,1}
        sphere{<1.5,4,0>,1.5,1}
        cylinder{<-2,0,0>,<-2,2,0>,.5,4}
        cylinder{<2,0,0>,<2,2,0>,.5,1}
    }
    texture{T_CapacitorYellow}
}

// origin - center, base
#debug "RedBarrel\n"
#declare RedBarrel=union{
    cylinder{<0,2,-5>,<0,2,5>,1.75}
    cylinder{<0,2,-5.5>,<0,2,-2>,2}
    cylinder{<0,2,5.5>,<0,2,2>,2}
    object{SoderPoint translate<0,0,-10>}
    union{
        #declare Wire_Width=.4
        #declare Wire_StartSeg=<0,0,-10>
        #declare Wire_EndSeg=<0,2,-10>
        #include "Wire.inc"
        #declare Wire_EndSeg=<0,2,10>
        #include "Wire.inc"
        #declare Wire_EndSeg=<0,0,10>
        #include "Wire.inc"
        texture{T_Steel}
    }
    object{SoderPoint translate<0,0,10>}
    texture{T_RedPart}
}

#debug "DipSwitchBox\n"
#declare DipSwitchBox=union{
    difference{
        union{
            box{<-20/2,0.5,-9/2>,<20/2,7,9/2>}
            box{<-21/2,0.5,-8/2>,<21/2,7.01,8/2>}
            cylinder{<-20/2,0.5,-8/2>,<-20/2,7.001,-8/2>,0.5}
            cylinder{<-20/2,0.5,8/2>,<-20/2,7.001,8/2>,0.5}
            cylinder{<20/2,0.5,-8/2>,<20/2,7.001,-8/2>,0.5}
            cylinder{<20/2,0.5,8/2>,<20/2,7.001,8/2>,0.5}
        }
        box{<21.01/2,4,-3>,<20/2,5,3>}
        box{<-21.01/2,4,-3>,<-20/2,5,3>}
        #declare Counter=0
        #while (Counter<8)
            box{<-18/2+18/7*Counter-1.5/2,7.01,-3>,
                <-18/2+18/7*Counter+1.5/2,4,3>}
            #declare Counter=Counter+1
        #end
    }
    
    union{
        #declare TextHeight=0.1
        #declare TextSize=1/0.75
        #declare Counter=1
        #while (Counter<=8)
            text {
                ttf "arial.TTF",str(Counter,0,0),
                TextHeight, 0 scale <TextSize*0.9,TextSize,1>
                rotate<90,180,0> translate<18/2+0.3-18/7*(Counter-1),7.1,-4+0.5>
            }
            #declare Counter=Counter+1
        #end
        texture{T_ChipText}
    }
    
    texture{T_DipSwitchBlueBox}
}

#debug "DipSwitch\n"
#declare DipSwitch=object{
    prism{
        linear_spline
        -1.5/2,
        1.5/2,
        13,
        <0,3>,<-2,3>,<-2,-3>,<0,-3>,<1,-1.5>,<0.25,-1.5>,
        <0.4,-1>,<0,0>,<.25,1.75>,<0,2>,<0.5,2.5>,<0.5,2.75>,<0,3>
        rotate<0,0,90>
    }
    texture{T_DipSwitchWhite}
}

#debug "DipSwitchBlock2\n"
#declare DipSwitchBlock1=union{
    object{DipSwitchBox}
    object{DipSwitch translate<-18/2+18/7*0,7.0,0>}
    object{DipSwitch translate<-18/2+18/7*1,7.0,0>}
    object{DipSwitch translate<-18/2+18/7*2,7.0,0>}
    object{DipSwitch translate<-18/2+18/7*3,7.0,0>}
    object{DipSwitch rotate<0,180,0> translate<-18/2+18/7*4,7.0,0>}
    object{DipSwitch rotate<0,180,0> translate<-18/2+18/7*5,7.0,0>}
    object{DipSwitch translate<-18/2+18/7*6,7.0,0>}
    object{DipSwitch rotate<0,180,0> translate<-18/2+18/7*7,7.0,0>}
}

#debug "DipSwitchBlock2\n"
#declare DipSwitchBlock2=union{
    object{DipSwitchBox}
    object{DipSwitch rotate<0,180,0> translate<-18/2+18/7*0,7.0,0>}
    object{DipSwitch translate<-18/2+18/7*1,7.0,0>}
    object{DipSwitch translate<-18/2+18/7*2,7.0,0>}
    object{DipSwitch translate<-18/2+18/7*3,7.0,0>}
    object{DipSwitch translate<-18/2+18/7*4,7.0,0>}
    object{DipSwitch translate<-18/2+18/7*5,7.0,0>}
    object{DipSwitch rotate<0,180,0> translate<-18/2+18/7*6,7.0,0>}
    object{DipSwitch translate<-18/2+18/7*7,7.0,0>}
}

// origin is centre of chip
// bottom of chip where pins connect to motherboard
#declare Chip_Holder1=union{
    difference{
        box{<-10,0.5,-5>,<10,5,5>}
        box{<-10.1,5.01,-4.5>,<10.1,4.2,4.5>}
    }
    texture{T_ChipBody}
}

#declare Chip_M5K4164NP_20=union{
    box{<-9.5,1,-3>,<9.5,4,3>}
    #declare Counter=0
    #while (Counter<6)
        object{ChipLeg translate<-13/2+13/5*Counter,3,-3>}
        object{ChipLeg rotate<0,180,0> translate<-13/2+13/5*Counter,3,3>}
        #declare Counter=Counter+1
    #end
    texture{T_ChipBody}
}

// voltage regulator?? Yellow brick
#declare Chip_YellowBlock=union{
    difference{
        box{<-9.5/2,7,-19/2>,<9.5/2,11,19/2>}
        sphere{<9.5/2-2,11,19/2-2>,1/2}
    }
    box{<-9.0/2,0.25,-18.5/2>,<9.0/2,7,18.5/2>}
    texture{T_YellowBlock}
}

#debug "TallGreenBlock\n"
#declare TallGreenBlock=union{
    box{<-6,0.5,-2.5/2>,<6,8,2.5/2>}
    box{<-6,1,2.5/2>,<6,8.1,2.5/2+0.1>
        texture{T_ChipBody}
    }
    union{
        text{
            ttf "Arial.ttf","1021-02",0.1,0
            scale<1.25/0.75,1.75/0.75,1>
            translate<-6+3,8-2,0>
        }
        text{
            ttf "Arial.ttf","1212",0.1,0
            scale<1.25/0.75,1.75/0.75,1>
            translate<-6+1,8-5,0>
        }
        text{
            ttf "Arial.ttf","8307",0.1,0
            scale<1.25/0.75,1.75/0.75,1>
            translate<6-4.5,8-5,0>
        }
        text{
            ttf "Arial.ttf","4",0.1,0
            scale<1.25/0.75,1.75/0.75,1>
            translate<-6+1,8-7.25,0>
        }
        text{
            ttf "Arial.ttf","2",0.1,0
            scale<1.25/0.75,1.75/0.75,1>
            translate<-6+7,8-7.25,0>
        }
        text{
            ttf "Arial.ttf","1",0.1,0
            scale<1.25/0.75,1.75/0.75,1>
            translate<-6+11,8-7.25,0>
        }
        translate<0,0,-2.5/2-0.1>
        texture{T_ChipText}
    }
    texture{T_GreenBlock}
    //scale 30
    //rotate<90,0,0>
}

#declare Chip_MK4564N_25=object{Chip_M5K4164NP_20} // **** change later

#declare Chip_RAM1=union{
    object{Chip_Holder1}
    object{Chip_M5K4164NP_20 translate<0,5,0>}
}

#declare Chip_RAM2=union{
    object{Chip_Holder1}
    object{Chip_MK4564N_25 translate<0,5,0>}
}

#declare JumperBlock_P4=union{
    box{<-4.5/2,0,-4.5/2>,<4.5/2,2,4.5/2>}
    union{
        box{<-0.5,1.5,-0.5>,<0.5,8,0.5> translate<-2.5/2,0,1>}
        box{<-0.5,1.5,-0.5>,<0.5,8,0.5> translate<2.5/2,0,1>}
        box{<-0.5,1.5,-0.5>,<0.5,8,0.5> translate<-2.5/2,0,3.5>}
        box{<-0.5,1.5,-0.5>,<0.5,8,0.5> translate<2.5/2,0,3.5>}
        translate<0,0,-4.5/2>
        texture{T_GoldWire}
    }
    texture{T_ChipBody}
}

#debug "***Resistors\n"
#declare Resistor_R6=union{
    sphere{<-2,1,0>,1}
    sphere{<2,1,0>,1}
    cylinder{<-2,1,0>,<2,1,0>,1.5/2}
    union{
        #declare Wire_Width=.4
        #declare Wire_StartSeg=<-12.5/2,0,0>
        #declare Wire_EndSeg=<-12/2,1,0>
        #include "Wire.inc"
        #declare Wire_EndSeg=<12/2,1,0>
        #include "Wire.inc"
        #declare Wire_EndSeg=<12.5/2,0,0>
        #include "Wire.inc"
        texture{T_Steel}
    }
    object{SoderPoint translate<-12.5/2,0,0>}
    object{SoderPoint translate<12.5/2,0,0>}
    texture{T_ResistorGreen}
}

#declare Resistor_R10=union{
    sphere{<-4+2.5/2,2.5/2,0>,2.5/2}
    sphere{<4-2.5/2,2.5/2,0>,2.5/2}
    cylinder{<-4+2.5/2,2.5/2,0>,<4-2.5/2,2.5/2,0>,2.0/2}
    union{
        #declare Wire_Width=.4
        #declare Wire_StartSeg=<-12.5/2,0,0>
        #declare Wire_EndSeg=<-12/2,1,0>
        #include "Wire.inc"
        #declare Wire_EndSeg=<12/2,1,0>
        #include "Wire.inc"
        #declare Wire_EndSeg=<12.5/2,0,0>
        #include "Wire.inc"
        texture{T_Steel}
    }
    object{SoderPoint translate<-12.5/2,0,0>}
    object{SoderPoint translate<12.5/2,0,0>}
    texture{T_ResistorGreen}
}

#declare Resistor_R9=object{Resistor_R6}
#declare Resistor_R8=object{Resistor_R6}
#declare Resistor_R7=object{Resistor_R6}

// origin is middle, outside edge of circuit board, circuit board top
#debug "KeyboardConnector\n"
#declare KeyboardConnector=union{
    box{<-2.5,0,-11>,<0,22,11>}
    cylinder{<-17,12,0>,<-2,12,0>,16.5/2}
    box{<-17.001,0,-17/2>,<-2,9,17/2>}
    cylinder{<-17.5,18,0>,<-16,18,0>,3/2}
    cylinder{<-17.5,4.5,-2.5>,<-16,4.5,-2.5>,2/2}
    cylinder{<-17.5,4.5,2.5>,<-16,4.5,2.5>,2/2}
    texture{T_ChipBody}
}

// fallen over metal object near bottom of board
// origin is centre of electrical connectors on left (CPU) side
#debug "Oscillator\n"
#declare Oscillator=union{
    box{<-16,0.25,-7/2>,<-3.5,4,7/2.0>}
    cylinder{<-16.01,3.75/2+0.25,-7/2>,<-3.5,3.75/2+0.25,-7/2>,3.75/2}
    cylinder{<-16.01,3.75/2+0.25,7/2>,<-3.5,3.75/2+0.25,7/2>,3.75/2}

    // black portion
    union{
        box{<-3.75,0.25,-7/2>,<-3.0,4,7/2.0>}
        cylinder{<-3.75,3.75/2+0.25,-7/2>,<-3.01,3.75/2+0.25,-7/2>,3.75/2}
        cylinder{<-3.75,3.75/2+0.25,7/2>,<-3.01,3.75/2+0.25,7/2>,3.75/2}
        texture{T_ChipBody}
    }
    difference{
        union{
            box{<-4.2,0,-7/2>,<-2.74,4.25,7/2.0>}
            cylinder{<-4.21,3.75/2+0.25,-7/2>,<-2.75,3.75/2+0.25,-7/2>,4.25/2}
            cylinder{<-4.21,3.75/2+0.25,7/2>,<-2.75,3.75/2+0.25,7/2>,4.25/2}
        }
        box{<-3.75,0.25,-7/2>,<-2.0,4,7/2.0>}
        cylinder{<-3.75,3.75/2+0.25,-7/2>,<-2.01,3.75/2+0.25,-7/2>,3.75/2}
        cylinder{<-3.75,3.75/2+0.25,7/2>,<-2.01,3.75/2+0.25,7/2>,3.75/2}
    }

    #declare Wire_Width=.35
    #declare Wire_StartSeg=<0,0,2.5>
    object{SoderPoint translate Wire_StartSeg}
    #declare Wire_EndSeg=<0,0.25,2.5>
    #include "Wire.inc"
    #declare Wire_EndSeg=<-3,2.5,2.5>
    #include "Wire.inc"

    #declare Wire_Width=.35
    #declare Wire_StartSeg=<0,0,-2.5>
    object{SoderPoint translate Wire_StartSeg}
    #declare Wire_EndSeg=<0,0.25,-2.5>
    #include "Wire.inc"
    #declare Wire_EndSeg=<-3,2.5,-2.5>
    #include "Wire.inc"

    // warped wire on top
    #declare Wire_Width=.35
    #declare Wire_StartSeg=<-10,0,10>
    object{SoderPoint translate Wire_StartSeg}
    #declare Wire_EndSeg=<-12,3.5,7>
    #include "Wire.inc"
    #declare Wire_EndSeg=<-10,4.5,3>
    #include "Wire.inc"
    #declare Wire_EndSeg=<-10.5,4.25,0>
    object{SoderPoint scale 2 translate Wire_EndSeg}
    #include "Wire.inc"
    #declare Wire_EndSeg=<-10.25,4.5,-6>
    #include "Wire.inc"
    #declare Wire_EndSeg=<-10.0,0,-6.5>
    #include "Wire.inc"
    object{SoderPoint translate Wire_EndSeg}

     texture{T_Steel}
}

#declare PowerConnector=union{
    difference{
        prism{
            linear_spline
            -46/2,46/2,
            19,
            <0,0>,<0,1>,<0.25,1>,<0.25,8>,<0,8>,<0,9.5>,<10,9>,<11,8>,
            <12.5,7>,<12.5,6>,<4,6>,<4,4>,<1.5,4>,<1.5,1>,<4,1>,
            <4,-0.25>,<4,-1>,<3,0>,<0,0>
            rotate<0,0,90>
        }
        #declare Counter=0
        #while (Counter<12)
            box{<-0.5,3,5>,<0.5,14,7>
                translate<-42/2+(42/11)*Counter,0,0>
            }
            #declare Counter=Counter+1
        #end
    }

    #declare Counter=0
    #while (Counter<12)
        union{
            box{<-0.4,1,1>,<0.4,4,4>}
            #if ((Counter!=5) & (Counter!=8))
                box{<0.5,11,7>,<2.5/2,14.5,8>}
                box{<-0.5,13.5,7>,<0.5,14.5,8>}
                box{<-2.5/2,11,7>,<-0.5,14.5,8>}
            #end
            translate<-42/2+(42/11)*Counter,0,0>
        }
        #declare Counter=Counter+1
    #end

    union{
        #declare Counter=0
        #while (Counter<12)
            box{<-1,3,-0.5>,<1,13,0.5>
                translate<-42/2+(42/11)*Counter,0,4.5>
            }
            #declare Counter=Counter+1
        #end
        texture{T_Steel}
    }
    texture{T_DipSwitchWhite}
}

#declare CircuitWidth=0.4

#declare Motherboard_Circuits=union{
    #declare Wire_Width=CircuitWidth
    #declare Wire_StartSeg=<7.5,0,29>
    #declare Wire_EndSeg=<7.5,0,55>
    #include "Wire.inc"
    #declare Wire_EndSeg=<10,0,57>
    #include "Wire.inc"
    #declare Wire_EndSeg=<10,0,65>
    #include "Wire.inc"

    cylinder{<8.5,0,24>,<8.5,0,55>,CircuitWidth/2}
    cylinder{<11,0,34.5>,<11,0,15>,CircuitWidth/2}
    cylinder{<11,0,15>,<12,0,14>,CircuitWidth/2}
    cylinder{<12,0,14>,<12,0,5>,CircuitWidth/2}
    cylinder{<12,0,5>,<14,0,3.5>,CircuitWidth/2}
    cylinder{<14,0,3.5>,<17,0,3.5>,CircuitWidth/2}
    cylinder{<17,0,3.5>,<17.5,0,5.5>,CircuitWidth/2}
    #declare Wire_StartSeg=<10,0,14>
    #declare Wire_EndSeg=<10,0,51>
    #include "wire.inc"

    #declare Wire_StartSeg=<17.5,0,12>
    #declare Wire_EndSeg=<17.5,0,15>
    #include "wire.inc"
    #declare Wire_EndSeg=<16,0,16>
    #include "wire.inc"
    #declare Wire_EndSeg=<16,0,53>
    #include "wire.inc"

    #declare Wire_StartSeg=<15,0,30.5>
    #declare Wire_EndSeg=<14,0,30.5>
    #include "wire.inc"
    #declare Wire_EndSeg=<14,0,24>
    #include "wire.inc"

    #declare Wire_StartSeg=<19,0,10>
    #declare Wire_EndSeg=<19,0,27>
    #include "wire.inc"
    #declare Wire_EndSeg=<20,0,28>
    #include "wire.inc"
    #declare Wire_EndSeg=<20,0,31>
    #include "wire.inc"

    #declare Wire_StartSeg=<8.5,0,57.5>
    #declare Wire_EndSeg=<4,0,57.5>
    #include "wire.inc"
    #declare Wire_EndSeg=<2.5,0,59>
    #include "wire.inc"
    #declare Wire_EndSeg=<2.5,0,115>
    #include "wire.inc"
    #declare Wire_EndSeg=<3.25,0,116>
    #include "wire.inc"
    #declare Wire_EndSeg=<5,0,116>
    #include "wire.inc"
    #declare Wire_EndSeg=<9,0,119>
    #include "wire.inc"
    #declare Wire_EndSeg=<9,0,124>
    #include "wire.inc"
    #declare Wire_EndSeg=<6,0,128>
    #include "wire.inc"
    #declare Wire_EndSeg=<4.25,0,128>
    #include "wire.inc"
    #declare Wire_EndSeg=<2.5,0,130>
    #include "wire.inc"
    #declare Wire_EndSeg=<2.5,0,187.5>
    #include "wire.inc"
    #declare Wire_EndSeg=<5,0,190>
    #include "wire.inc"
    #declare Wire_EndSeg=<7.5,0,190>
    #include "wire.inc"

    #declare Wire_StartSeg=<304-41,0,100>
    object{SoderPoint translate Wire_StartSeg}
    #declare Wire_EndSeg=<304-41,0,128>
    #include "wire.inc"
    #declare Wire_EndSeg=<304-40.5,0,129>
    #include "wire.inc"
    #declare Wire_EndSeg=<304-40.5,0,131>
    #include "wire.inc"

    #declare Wire_StartSeg=<304-43,0,70>
    #declare Wire_EndSeg=<304-43,0,98>
    #include "wire.inc"
    #declare Wire_EndSeg=<304-42,0,99>
    #include "wire.inc"
    #declare Wire_EndSeg=<304-42,0,107>
    #include "wire.inc"

    #declare Wire_StartSeg=<304-47,0,96.5>
    object{SoderPoint translate Wire_StartSeg}
    #declare Wire_EndSeg=<304-47,0,98>
    #include "wire.inc"
    #declare Wire_EndSeg=<304-45,0,100>
    #include "wire.inc"
    #declare Wire_EndSeg=<304-45,0,107>
    #include "wire.inc"

    #declare Wire_StartSeg=<304-46,0,73>
    object{SoderPoint translate Wire_StartSeg}
    #declare Wire_EndSeg=<304-46,0,79>
    #include "wire.inc"

    #declare Wire_StartSeg=<304-47,0,70>
    #declare Wire_EndSeg=<304-47,0,85>
    #include "wire.inc"

    #declare Wire_StartSeg=<304-49,0,90>
    #declare Wire_EndSeg=<304-49,0,98>
    #include "wire.inc"
    #declare Wire_EndSeg=<304-47,0,101>
    #include "wire.inc"
    #declare Wire_EndSeg=<304-47,0,107>
    #include "wire.inc"

    #declare Wire_StartSeg=<304-49,0,107>
    #declare Wire_EndSeg=<304-49,0,100.5>
    #include "wire.inc"
    #declare Wire_EndSeg=<304-49.5,0,100>
    #include "wire.inc"
    #declare Wire_EndSeg=<304-49.5,0,94.25>
    #include "wire.inc"
    #declare Wire_EndSeg=<304-50.5,0,94.25>
    #include "wire.inc"
    #declare Wire_EndSeg=<304-51,0,93.5>
    #include "wire.inc"
    #declare Wire_EndSeg=<304-51,0,89>
    #include "wire.inc"

    #declare Wire_StartSeg=<304-50,0,107>
    #declare Wire_EndSeg=<304-50,0,95>
    #include "wire.inc"
    #declare Wire_EndSeg=<304-53,0,95>
    #include "wire.inc"
    #declare Wire_EndSeg=<304-52.5,0,94.5>
    #include "wire.inc"
    #declare Wire_EndSeg=<304-52.5,0,89>
    #include "wire.inc"

    #declare Wire_StartSeg=<304-51,0,101.5>
    object{SoderPoint translate Wire_StartSeg}
    #declare Wire_EndSeg=<304-51,0,95.5>
    #include "wire.inc"
    #declare Wire_EndSeg=<304-58,0,95.5>
    #include "wire.inc"
    #declare Wire_EndSeg=<304-59,0,94.5>
    #include "wire.inc"

    #declare Wire_StartSeg=<304-53.5,0,101.5>
    object{SoderPoint translate Wire_StartSeg}
    #declare Wire_EndSeg=<304-53.5,0,96>
    #include "wire.inc"
    #declare Wire_EndSeg=<304-71,0,96>
    #include "wire.inc"
    #declare Wire_EndSeg=<304-71,0,89>
    #include "wire.inc"

    #declare Wire_StartSeg=<304-55,0,107>
    #declare Wire_EndSeg=<304-55,0,99.5>
    #include "wire.inc"
    object{SoderPoint translate Wire_EndSeg}
    #declare Wire_EndSeg=<304-55,0,97>
    #include "wire.inc"
    #declare Wire_EndSeg=<304-74,0,97>
    #include "wire.inc"
    #declare Wire_EndSeg=<304-74,0,68>
    #include "wire.inc"
    #declare Wire_EndSeg=<304-72,0,66>
    #include "wire.inc"

    #declare Wire_StartSeg=<304-57.5,0,107>
    #declare Wire_EndSeg=<304-57.5,0,99>
    #include "wire.inc"
    object{SoderPoint translate Wire_EndSeg}
    #declare Wire_EndSeg=<304-57.5,0,98>
    #include "wire.inc"
    #declare Wire_EndSeg=<304-75,0,98>
    #include "wire.inc"
    #declare Wire_EndSeg=<304-75,0,67.5>
    #include "wire.inc"
    #declare Wire_EndSeg=<304-75+4,0,67.5+4>
    #include "wire.inc"

    #declare Wire_StartSeg=<304-60,0,107>
    #declare Wire_EndSeg=<304-60,0,99>
    #include "wire.inc"
    object{SoderPoint translate Wire_EndSeg}
    #declare Wire_EndSeg=<304-60,0,98.75>
    #include "wire.inc"
    #declare Wire_EndSeg=<304-76,0,98.75>
    #include "wire.inc"
    #declare Wire_EndSeg=<304-76,0,66.5>
    #include "wire.inc"
    #declare Wire_EndSeg=<304-76+5,0,66.5+5>
    #include "wire.inc"

    #declare Wire_StartSeg=<304-63,0,107>
    #declare Wire_EndSeg=<304-63,0,99>
    #include "wire.inc"
    #declare Wire_EndSeg=<304-76.5,0,99>
    #include "wire.inc"
    #declare Wire_EndSeg=<304-76.5,0,65>
    #include "wire.inc"
    #declare Wire_EndSeg=<304-76.5+6,0,65+6>
    #include "wire.inc"

    scale <1,.5,1>
    texture{T_Circuits}
}

#declare Motherboard_SoderPoints=union{
    object{SoderPoint translate<8.5,0,24>}
    object{SoderPoint translate<7.5,0,29>}
    object{SoderPoint translate<11,0,34.5>}
    object{SoderPoint translate<10,0,14>}
    object{SoderPoint translate<10,0,51>}
    object{SoderPoint translate<16,0,53>}
    object{SoderPoint translate<8.5,0,57.5>}
    object{SoderPoint translate<7.5,0,190>}
}

#declare Motherboard_TextHeight=0.25
#declare Motherboard_StandardTextSize=2/0.75

#declare Motherboard_Text=union{
     text {
           ttf "arial.TTF",
           "099",
           Motherboard_TextHeight, 0
           scale <Motherboard_StandardTextSize,Motherboard_StandardTextSize,Motherboard_TextHeight>
           rotate<90,0,0>
           rotate<0,90,0>
           translate<32,0,13>
     }
     text {
           ttf "arial.TTF",
           "098",
           Motherboard_TextHeight, 0
           scale <Motherboard_StandardTextSize,Motherboard_StandardTextSize,Motherboard_TextHeight>
           rotate<90,0,0>
           rotate<0,90,0>
           translate<32,0,24>
     }
     text {
           ttf "arial.TTF",
           "097",
           Motherboard_TextHeight, 0
           scale <Motherboard_StandardTextSize,Motherboard_StandardTextSize,Motherboard_TextHeight>
           rotate<90,0,0>
           rotate<0,90,0>
           translate<32,0,37>
     }
     text {
           ttf "arial.TTF",
           "096",
           Motherboard_TextHeight, 0
           scale <Motherboard_StandardTextSize,Motherboard_StandardTextSize,Motherboard_TextHeight>
           rotate<90,0,0>
           rotate<0,90,0>
           translate<32,0,50>
     }
     text {
           ttf "arial.TTF",
           "C7",
           Motherboard_TextHeight, 0
           scale <Motherboard_StandardTextSize,Motherboard_StandardTextSize,Motherboard_TextHeight>
           rotate<90,0,0>
           rotate<0,90,0>
           translate<9,0,18>
     }
     text {
           ttf "arial.TTF",
           "C9",
           Motherboard_TextHeight, 0
           scale <Motherboard_StandardTextSize,Motherboard_StandardTextSize,Motherboard_TextHeight>
           rotate<90,0,0>
           rotate<0,90,0>
           translate<8,0,49>
     }
     text {
           ttf "arial.TTF",
           "R6",
           Motherboard_TextHeight, 0
           scale <Motherboard_StandardTextSize,Motherboard_StandardTextSize,Motherboard_TextHeight>
           rotate<90,0,0>
           translate<1,0,65>
     }
     text {
           ttf "arial.TTF",
           "R9",
           Motherboard_TextHeight, 0
           scale <Motherboard_StandardTextSize,Motherboard_StandardTextSize,Motherboard_TextHeight>
           rotate<90,0,0>
           translate<1,0,68>
     }
     text {
           ttf "arial.TTF",
           "R8",
           Motherboard_TextHeight, 0
           scale <Motherboard_StandardTextSize,Motherboard_StandardTextSize,Motherboard_TextHeight>
           rotate<90,0,0>
           translate<1,0,70.5>
     }
     text {
           ttf "arial.TTF",
           "R7",
           Motherboard_TextHeight, 0
           scale <Motherboard_StandardTextSize,Motherboard_StandardTextSize,Motherboard_TextHeight>
           rotate<90,0,0>
           translate<1,0,73.2>
     }
     text {
           ttf "arial.TTF",
           "R10",
           Motherboard_TextHeight, 0
           scale <Motherboard_StandardTextSize,Motherboard_StandardTextSize,Motherboard_TextHeight>
           rotate<90,0,0>
           translate<1.25,0,78>
     }
    translate<0,.01,0>
    texture{T_ChipText}
}

// origin is lower left coner, top of board
#declare Motherboard=object{
    union{
        object{Motherboard_Circuits}
        object{Motherboard_SoderPoints}
        object{Motherboard_Text}

        difference{
            box{<0,0,0>,<304,-1.5,217>}
            cylinder{<5,0.01,5>,<5,-1.51,5>,2}
        }

        // chips (lower left corner
        // column 1 (left)
        object{Chip_74LS04 translate<13+19/2,0,9>}
        object{Chip_DM74LS175N translate<21.5,0,22>}
        object{Chip_74LS08 translate<13+19/2,0,35>}
        object{Chip_74LS74 translate<13+19/2,0,47>}
        object{Chip_PE_21712 translate<22,0,60>}
        object{Chip_SN74S280N translate<32.5-19/2,0,85>}

        // column 2 ( from left)
        object{Chip_74LS10 translate<39+19/2,0,9>}
        object{Chip_SN74LS04N translate<39+19/2,0,22>}
        object{Chip_DM74S74N translate<48,0,34.5>}
        object{Chip_74S00 translate<39+19/2,0,47>}
        object{Chip_SN74LS125AN translate<39+19/2,0,60>}
        object{Chip_SN74LS158N translate<37+19/2,0,73>}
        object{Chip_Q5500303 translate<37+21/2,0,85>}

        // column 3 (from left)
        object{Chip_Q5547008 translate<61.5+21/2,0,9>}
        object{Chip_74LS74 translate<74,0,22>}
        object{Chip_DM74LS138N translate<63+19/2,0,35>}

        // column 4 (from left)
        object{Chip_DM74LS02N translate<87+19/2,0,33>}
        object{Chip_DM74S08N translate<87+19/2,0,47.5>}
        object{Chip_DM74LS138N translate<88.5+19/2,0,60>}
        object{Chip_DM74LS138N translate<88.5+19/2,0,73>}
        object{Chip_DM74LS138N translate<88.5+19/2,0,86>}

        // left of ISA slots column
        object{Chip_SN74LS322AN translate<176,0,35>}
        object{Chip_DM74LS244N translate<176,0,47>}
        object{Chip_Q5582008 translate<168+21/2,0,60>}
        // (dip switches here)
        object{Chip_Q5547008 translate<168+21/2,0,85>}
        object{Chip_74LS670 translate<169+18/2,0,98>}
        object{Chip_DM74LS373N translate<176,0,110.5>}
        object{Chip_DM74LS244N translate<176,0,123>}
        object{Chip_DM74LS244N translate<176,0,136>}
        object{Chip_DM74LS244N translate<176,0,148>}

        // CPU Area chips - right
        object{Chip_Intel8088 translate<255,0,55+9>}
        object{Chip_Intel8087 translate<255,0,34+9>}
        object{Chip_D8259AC translate<248,0,86>}

        // CPU Area chips - left
        object{Chip_DM74LS373N translate<197+13,0,39>}
        object{Chip_DM74LS373N translate<197+13,0,55>}
        object{Chip_SN74LS245N translate<210,0,72>}
        object{Chip_DM74LS373N translate<197+13,0,90>}
        object{Chip_8243ED translate<199+23/2,0,23.5>}

        // memory
        #declare Counter=0
        #while (Counter<8)
            object{Chip_RAM1 translate<20,0,103+10*Counter>}
            object{Chip_RAM1 translate<46,0,103+10*Counter>}
            object{Chip_RAM2 translate<72.5,0,103+10*Counter>}
            #declare Counter=Counter+1
        #end

        object{Chip_RAM1 translate<20,0,200>}
        object{Chip_RAM1 translate<46,0,200>}
        object{Chip_RAM2 translate<72.5,0,200>}

        // left middle column 8255A chip etc...
        object{Chip_IP8255A_5 translate<159-52/2,0,29-13/2>}
        object{Chip_D8237AC_5 translate<108+25,0,43>}
        object{Chip_AM8253 translate<112+16,0,68>}
        object{Chip_1501476 translate<113+15,0,94>}
        object{Chip_MK36A71N_4 translate<113+15,0,114>}
        object{Chip_MK36A70N_4 translate<113+15,0,114+17>}
        object{Chip_XE5000021 translate<113+15,0,152>}
        object{Chip_MK36A68N_4 translate<113+15,0,152+17>}
        object{ChipHolder_Wide12Pin translate<113+15,0,198>}

        // middle column with dip switch boxes
        object{DipSwitchBlock1 translate<168+21/2,0,16.5+4.5>}
        object{DipSwitchBlock2 translate<169+21/2,0,68+4.5>}

        // capacitors
        object{Capacitor_BlobYellow translate<7.5,0,15>}
        object{Capacitor_BlobYellow translate<131.5,0,5>}
        object{Capacitor_BlobYellow translate<13,0,95>}

        object{Capacitor_K5M_473 rotate<0,90,0> translate<8,0,110>}

        object{Capacitor_K5M_473 rotate<0,90,0> translate<33,0,101>}
        object{Capacitor_K5M_473 rotate<0,90,0> translate<33,0,121>}
        object{Capacitor_K5M_473 rotate<0,90,0> translate<33,0,141>}
        object{Capacitor_K5M_473 rotate<0,90,0> translate<33,0,161>}

        object{Capacitor_K5M_473 rotate<0,90,0> translate<57,0,111>}
        object{Capacitor_K5M_473 rotate<0,90,0> translate<57,0,131>}
        object{Capacitor_K5M_473 rotate<0,90,0> translate<57,0,151>}
        object{Capacitor_K5M_473 rotate<0,90,0> translate<57,0,171>}

        object{Capacitor_K5M_473 rotate<0,90,0> translate<84,0,101>}
        object{Capacitor_K5M_473 rotate<0,90,0> translate<84,0,121>}
        object{Capacitor_K5M_473 rotate<0,90,0> translate<84,0,141>}
        object{Capacitor_K5M_473 rotate<0,90,0> translate<84,0,161>}

        object{Capacitor_K5M_473 rotate<0,90,0> translate<279,0,80>}
        object{Capacitor_K5M_473 rotate<0,90,0> translate<285,0,36>}
        object{Capacitor_K5M_473 rotate<0,90,0> translate<285,0,44>}
        object{Capacitor_K5M_473 translate<267,0,29>}
        object{Capacitor_K5M_473 translate<277,0,29>}

        object{RedBarrel translate<5,0,52>}
        object{RedBarrel translate<5,0,148>}

        object{Chip_YellowBlock translate<304-(7+9.2/2),0,83+19/2>}
        object{TallGreenBlock translate<111+6,0,9.5>}

        // jumper blocks
        object{JumperBlock_P4 translate<3.5+4.5/2,0,98.5+4.5/2>}

        // resistors
        object{Resistor_R6 translate<5+6,0,66>}
        object{Resistor_R9 translate<5+6,0,69>}
        object{Resistor_R8 translate<5+6,0,71>}
        object{Resistor_R7 translate<5+6,0,74>}
        object{Resistor_R10 translate<5+6,0,76>}

        // keyboard connectors
        object{KeyboardConnector translate<304,0,30+11>}
        object{KeyboardConnector translate<304,0,53+11>}

        object{Oscillator translate<208,0,9>}
        object{PowerConnector translate<(233+281)/2,0,1>}

        union{
            object{ISA_Slot}
            #ifdef (ShowFDDISABoard)
                union{
                    #include "Board_ISA_FDD.inc"
                    translate<-128,15,0>
                 }
            #end
            translate<197,0,109>
        }
        object{ISA_Slot translate<197,0,135>}
        object{ISA_Slot translate<197,0,160>}
        object{ISA_Slot translate<197,0,185>}
        object{ISA_Slot translate<197,0,210>}

        //object{Capacitor_K5M_473 scale 10 translate<130,0,30>}
    }

    texture{T_GreenBoard}
}

#ifdef (UseComputerCase)
// surrounding box
difference{
    box{<-500,-500,-500>,<3500,1900,2900> hollow}
    box{<-20,-10,-20>,<320,160,240> hollow}
    texture{
        pigment{
            color rgb<.15,.35,.55>
        }
        #if (QuickColour!=1)
            finish{
                //reflection rgb<.5,.5,.75> // 0.85
                reflection rgb<.7,.7,.95> // 0.85
            }
        #end
    }
}
#end

object{Motherboard}

