//////////////////////////////////////////////////////
// Capture the Night II
// Dan Connelly
// djconnel@flash.net
// http://www.flash.net/~djconnel/
//

#version 3.0
#include "shapes.inc"
#include "metals.inc"

// the following definitions are required by the spark files
#declare Luminous= finish{diffuse 0 ambient 1}
#declare White=    pigment{color rgb 1}

///////////////////////////////////////////////////////
// Render flags

#declare Full_Render_Flag  = 1    // overrides all of the following

// the following are true in full render
#declare Reflection_Flag   = (Full_Render_Flag | 0)    // allow reflection?
#declare Refraction_Flag   = (Full_Render_Flag | 0)    // allow refraction?
#declare Translucent_Flag  = (Full_Render_Flag | 0)    // allow translucency?
#declare ImageMap_Flag     = (Full_Render_Flag | 1)    // show imagemaps?
#declare Moon_Flag         = (Full_Render_Flag | 1)    // show moon?
#declare Earth_Flag        = (Full_Render_Flag | 1)    // show Earth?
#declare Cylinder_Flag     = (Full_Render_Flag | 1)    // show stadium cylinders?
#declare Stadium_Flag      = (Full_Render_Flag | 1)    // show primary stadium structure?
#declare Cage_Flag         = (Full_Render_Flag | 1)    // put a cage over the stadium?
#declare Monolith_Flag     = (Full_Render_Flag | 1)    // show monoliths?
#declare Water3D_Flag      = (Full_Render_Flag | 0)    // use heightfield for water?
#declare Lights_Flag       = (Full_Render_Flag | 1)    // use lights in sparks
#declare Sparks_Flag       = (Full_Render_Flag | 1)    // include sparks
#declare MoonGlow_Flag     = (Full_Render_Flag | 1)    // Does the Moon have a halo glow?

#declare TopView_Flag      = 0    // use top view instead of primary camera position
#declare Isometric_Flag    = 0    // isometric view (versus perspective)
#declare LightSphere_Flag  = 0    // debugging spheres used to place light locations
#declare Earthview_Flag    = 0    // position the camera to get a close-up of the Earth

///////////////////////////////////////////////////////
// Parameter declarations

// Camera Position (for standard view)
#declare Camera_Position=<0, 4, -79>

// Image parameters
#declare Isometric_Ratio    = 4/3          // W/H of image as rendered

// Moon
#declare Moon_Radius        = 18
#declare Moon_Altitude      = 5
#declare Moon_Position      = <0, Moon_Altitude + Moon_Radius, 0>

// Moon glow
#declare MoonGlow_Radius    = 1.5 * Moon_Radius

// Earth
#declare Earth_Radius       = 3.66 * Moon_Radius                       // this is correct
#declare Earth_Orbit        = 60.38 * Earth_Radius * 0.65              // 60.38 is natural ratio
#declare Earth_Rotate       = <-35, 25, 0>                             // rotation vector to place Earth
#declare Earth_Position0    = <0, 0, Earth_Orbit>                      // before rotation, to position texture
#declare Earth_Position     = vrotate(Earth_Position0, Earth_Rotate)   // after rotation
#declare Earth_Light_Radius = 5                                        // degrees : spotlight illuminating Earth
#declare Earth_Light_dy     = -20 * Earth_Radius / tan(radians(Earth_Light_Radius))  // kludge
#declare Earth_Light_Position = Earth_Position + <0, Earth_Light_dy, 0>

// Water
#declare Wave_Height        = -44         // peak-to-valley wave height, if 3D water is used

// Monolith
#declare Monolith_Scale     = 1
#declare Monolith_Size      = Monolith_Scale * <1, 4, 9>
#declare Monolith_Phi0      = 0
#declare Monolith_dPhi      = 30
#declare Monolith_NPhi      = 3
#declare Monolith_Theta0    = -12
#declare Monolith_N0        = 12
#declare Monolith_Orbit     = 45
#declare Monolith_y0        = Moon_Altitude + Moon_Radius
#declare Monolith_Texture   =
         texture {
           pigment {
             #if (Translucent_Flag)
               color rgbt<0, 0, 0, 0.5>
             #else
               color rgb<0,0,0>
             #end
           }
           finish {
             diffuse 1
             ambient 0.1
             specular 1        // specular highlighting
             roughness 0.01    // applies to specular highlighting
             #if (Reflection_Flag)
               reflection 0.13
             #end
             fade_distance Monolith_Scale
             fade_power    1
             refraction    Refraction_Flag
             ior           2.4   // diamond index of refraction
            }
         }

// Sparks
#declare Spark_Scale = Monolith_Orbit - Moon_Radius - Monolith_Size.x/2
#declare Spark_Color = color rgb <0.95, 0.9, 1>
#declare N_Spark_Files = 12

// Stadium
#declare Stadium_Radius     = 100
#declare Stadium_R_Outer    = 120
#declare Stadium_Height     = 60
#declare Stadium_Depth      = 30
#declare Stadium_Texture    =
    texture {
      pigment {
        color rgb<1, 0.8, 0.1>
      }
      finish {
        ambient 0.05
        diffuse 1
        specular 1
        roughness 0.01
        #if (Reflection_Flag)
          reflection 0.27
        #end
        // metallic          // uncomment to use metallic (less white) highlighting
        brilliance 10
      }
    }

// Cylinders (stadium pillars)
#declare Cylinder_Radius    = 20
#declare Cylinder_Height    = Stadium_Height
#declare Cylinder_Depth     = Stadium_Depth
#declare N_Cylinders        = 12
#declare Cylinder_dTheta    = (360/N_Cylinders)
#declare Cylinder_Theta0    = -0.8*Cylinder_dTheta
#declare Cylinder_Texture   = texture { Stadium_Texture }
#declare Cylinder_Position  = <Stadium_Radius, 0, 0>

// Cage (over stadium)
#declare Bars_N  = <11, 11, 11>
#declare Bar_r   = 0.01   // pre-scale
#declare Cage_R  = Stadium_Radius * (1 + 2*Bar_r)   // post-scale
#declare Cage_Position = <0, Stadium_Height, 0>
#declare Cage_Texture = texture { Stadium_Texture finish { reflection 0 metallic } }


// initialize random number generator
#declare Seed = seed(21327)

////////////////////////////////////////////////////////
// Render settings and cameras

global_settings {
  assumed_gamma 1.8                 // reciprical gamma correction
  ambient_light rgb<1, 0.85, 0.85>  // global ambient light
}
background {
  color rgb<0,0,0>
}

// define the camera.. this includes several debugging placements
camera {
  #if (TopView_Flag)
    #if (Isometric_Flag)
      orthographic
      up     <0,0,2 * Stadium_R_Outer>
      right  <2 * Isometric_Ratio * Stadium_R_Outer,0,0>
    #else
      perspective
    #end
    location <0, 200, 0>   // top view
    look_at  <0, 0, 0>
  #else
    #if (Isometric_Flag)
      orthographic
      #declare Camera_y = Stadium_Radius / Isometric_Ratio
      up     <0, 2*Camera_y, 0>
      right  <2 * Stadium_Radius, 0, 0>
      location <0, Camera_y, -79>
      look_at  <0, Camera_y, 0>
    #else
      perspective
      #if (Earthview_Flag)
        location Earth_Position0 - <0, 0, 3*Earth_Radius>
        rotate Earth_Rotate
        look_at Earth_Position
      #else
        location Camera_Position
        look_at  Moon_Position
      #end
    #end
  #end
}



///////////////////////////////////////////////////////
// objects

#if (Moon_Flag)
  // moon
  sphere {
    Moon_Position,
    Moon_Radius
    texture {
      pigment {
        #if (ImageMap_Flag)
          image_map {
            png "moon_texture.png"
          }
          scale 2*Moon_Radius
          translate <-Moon_Radius, Moon_Altitude, 0>
        #else
          color rgb<0.8, 0.8, 0.8>
        #end
      }
      finish {
        diffuse 1
        ambient 0.05
        specular 0
        roughness 0.05
      }
    }
  }
#end

#if (MoonGlow_Flag)
  sphere { 0, 1
    pigment { color rgbt <1, 1, 1, 1> }
    halo {
      emitting
      spherical_mapping
      cubic
       color_map {
        [ 0 color rgbt <0.8, 0.5, 1, 1> ]
        [ 1 color rgbt <0.8, 0.5, 1, -1> ]
      }
      samples 10
      aa_threshold 0.1
    }
    hollow
    scale MoonGlow_Radius
    translate Moon_Position
    no_shadow
  }
#end

#if (Earth_Flag)
  #render concat(
    "Placing Earth at <",
    str(Earth_Position.x,6,2),", ",
    str(Earth_Position.y,6,2),", ",
    str(Earth_Position.z,6,2),">\n"
  )
  #render concat(
    "Placing Earth light at <",
    str(Earth_Light_Position.x,6,2),", ",
    str(Earth_Light_Position.y,6,2),", ",
    str(Earth_Light_Position.z,6,2),">\n"
  )
  // define the planet itself
  sphere {
    Earth_Position0,
    Earth_Radius
    texture {
      pigment {
        #if (ImageMap_Flag)
          image_map {
            png "earth_texture.png"
          }
          scale 2*Earth_Radius
          translate Earth_Position0 - <Earth_Radius, Earth_Radius, 0>
        #else
          color rgb<0.8, 0.8, 0.8>
        #end
      }
      finish {
        diffuse 1
        ambient 0.05
        specular 0
        roughness 0.05
      }
    }
    rotate Earth_Rotate
    no_shadow
  }

  // define a light to represent the reflections off the Earth onto the scene
  light_source {
    Earth_Position
    color rgb 0.1
  }

  // define a light to illuminate the Earth
  light_source {
    Earth_Light_Position
    color rgb 1
    spotlight
    point_at Earth_Position
    radius Earth_Light_Radius
  }
#end

// Sky
#if (ImageMap_Flag)
  sky_sphere {
    pigment {
      image_map {
        gif "c:\djconnel\images\stars3.gif"
      }
      scale 0.5
    }
   }
#end

#if (Monolith_Flag | Lights_Flag | LightSphere_Flag | Sparks_Flag)

  // Monoliths, Sparks, plus associated lights and/or debugging spheres
  #declare Count_Phi = 0
  #declare Spark_Count = 0
  #while (Count_Phi < Monolith_NPhi)
    #declare Phi = Monolith_Phi0 + (Count_Phi * Monolith_dPhi)
    #declare Count = abs(floor( 0.5 + (cos(radians(Phi)) * Monolith_N0) ))
    #if (Count)
      #declare dTheta = 360 / Count

      #while (Count > 0)
        #declare Count = Count - 1
        #declare Theta = Monolith_Theta0 + (Count * dTheta)

        // load in the appropriate spark file to redefine the spark object...
        // these get cycled to avoid repetition
        #if (Sparks_Flag)
          #declare Spark_File = concat("Spark", str(mod(Spark_Count,N_Spark_Files)+1, 1, 0), ".inc")
          #declare Spark_Count = Spark_Count + 1
          #render concat("Include file = ", Spark_File, "\n")
          #include Spark_File
        #end

        union {
          #if (Monolith_Flag)
           box {
            -Monolith_Size/2,
            Monolith_Size/2
            texture {Monolith_Texture}
            translate <Monolith_Orbit,0,0>
           }
          #end
          #if (Sparks_Flag | LightSphere_Flag | Lights_Flag)
           union {
            #if (Sparks_Flag)
              object{
                spark                          // spark goes from <000> to <001>
                texture {
                  pigment {
                    color Spark_Color          // override default spark color
                  }
                  finish {
                    ambient 1
                    diffuse 0
                  }
                }
                rotate <0, 0, 360*rand(Seed)>  // randomize its orientation
                no_shadow
              }
            #end
            #if (Lights_Flag)
              light_source {                   // point source halfway along spark
                <0, 0, 0.5>
                color 0.1 * Spark_Color
                fade_distance Monolith_Orbit/2
                fade_power 1
              }
            #end
            #if (LightSphere_Flag)
              sphere {                        // debugging object halfway along spark
                <0, 0, 0.5>,
                0.05
                pigment {
                  color rgb <0, 1, 0>
                }
                finish {
                  diffuse 0.5
                  ambient 0.5
                }
              }
            #end
            rotate <0, 90, 0>              // transform to <100>
            scale  Spark_Scale
            translate <Moon_Radius, 0, 0>
           }
          #end
          rotate <0, 0, Phi>
          rotate <0, Theta, 0>
          translate <0, Monolith_y0, 0>
        }
      #end
    #end

    #declare Count_Phi = Count_Phi + 1
  #end

#end


///////////////////////////////////////////////////////
// Stadium and support cylinders and cage

// stadium : a torus-like ring (difference of 2 cylinders)
#if (Stadium_Flag)
  difference {
    cylinder {
      <0, -Stadium_Depth,  0>,
      <0, Stadium_Height, 0>,
      Stadium_R_Outer
    }
    cylinder {
      <0, -Stadium_Depth-1,  0>,
      <0, Stadium_Height+1, 0>,
      Stadium_Radius
    }
    texture { Stadium_Texture }
  }
#end

#if (Cylinder_Flag)
  // cylinders
  #declare Count = N_Cylinders
  #declare Theta = Cylinder_Theta0
  #while (Count > 0)
    #declare Count = Count - 1

    cylinder {
      Cylinder_Position - <0, Cylinder_Depth,  0>,
      Cylinder_Position + <0, Cylinder_Height, 0>,
      Cylinder_Radius
      rotate <0, Theta, 0>
      texture { Cylinder_Texture }
    }

    #declare Theta = Theta + Cylinder_dTheta
  #end
#end

#if (Cage_Flag)
  #declare Axis      = <1, 0, 0>
  #declare Rotation  = <0, 0, 90>
  #declare N_axis = 3
  #while (N_axis > 0)
    #declare N = Bars_N.x
    #if (N > 0)
      #declare X  = (1 - N) / N
      #declare dX = ((N = 1) ? 1 : (-2 * X / (N - 1)))
      #render concat("N = ", str(N,4,1), "\n")
      #render concat("dX = ", str(dX,6,4), "\n")

      #if (((1 - X) / dX) > 10000)
        #error "ERROR : limit of 10000 bars"
        #exit
      #else
        #while (X < 1)
          #declare Xnew = sin(pi * X / 2)
          #declare R  = sqrt(1 - Xnew * Xnew)
          difference {
            torus {
              R,
              Bar_r
              rotate Rotation
              translate Xnew * Axis
              scale Cage_R
            }
            plane {
              <0, 1, 0>
              0
            }
            texture {Cage_Texture}
            rotate <0, Cylinder_Theta0, 0>
            translate Cage_Position
          }
          #declare X = X + dX
        #end
      #end
    #end

    // shift to next axis
    #declare Axis     = <Axis.y, Axis.z, Axis.x>
    #declare Rotation = <Rotation.z, Rotation.x, Rotation.y>
    #declare Bars_N   = <Bars_N.z, Bars_N.x, Bars_N.y>
    #declare N_axis   = N_axis - 1
  #end
#end

///////////////////////////////////////////////////
// Water


#if (Water3D_Flag)
 height_field {
  tga "watermap.tga"
  smooth
  translate <-0.5, -0.2, -0.5>
  scale <2*Stadium_Radius, Wave_Height, 2*Stadium_Radius>
  texture {

#else
 disc {
  <0, 1, 0>     // surface normal
  0             // distance from origin
  1.1 * Stadium_Radius

  texture {
#end
    pigment {
      #if (Translucent_Flag)
        color rgbt<0, 0.2, 0.4, 0.5>
      #else
        color rgb<0, 0.2, 0.4>
      #end
    }
    finish {
      ambient 0.1
      specular 1         // specular highlighting
      roughness 0.0003   // applies to specular highlighting
      diffuse 1
      #if (Reflection_Flag)
        reflection 0.65
      #end
      refraction Refraction_Flag
      ior 1.33
      fade_distance Stadium_Depth/3
      fade_power    2
    }
  }
}
