#include "header.inc"
#declare Main = 1;

global_settings {
  ambient_light <0.1, 0.1, 0>
  max_trace_level 12
}

// special random number seeds to isolate phases
#declare Seed1 = 17;               // main crystal
#declare Rand1 = seed(Seed1);
#declare Seed2 = 171;              // gas
#declare Rand2 = seed(Seed2);
#declare Seed3 = 1713;             // y < 0 (needed to assign electron density to step atoms)
#declare Rand3 = seed(Seed3);
#declare Seed4 = 17131;            // abs(x) > z => a kludge to retain image with a code change
#declare Rand4 = seed(Seed4);

////////////////////////////////////////////////
// declare phase :
// -1 : small crystal, gas like phase 0-1
// 0 : the whole scene
// 1 : top part of crystal + full gas
// 2 : omit the distant gas particles & distant atoms
// 3 : small crystal, gas like phase 2
//

#declare Phase = 2;

////////////////////////////////////////////////
// rendering flags
//
// These switches define what is and is not rendered
//

#declare Full_Flag       = 1;               // use full-scene parameters
#declare Wafer_Flag      = (0 | Full_Flag); // whether to generate the solid lattice
#declare Blob_Flag       = (1 | Full_Flag); // whether to use blobs instead of spheres for orbitals (high Q renderings)
#declare Fog_Flag        = (1 | Full_Flag); // whether to render gas clouds as fog
#declare Atmosphere_Flag = 0;               // whether to render gas clouds as atmosphere
                                            // (incompatible with halo)
#declare Halo_Flag       = (0 | Full_Flag); // whether to render electron halo
#declare Gas_Flag        = (1 | Full_Flag); // whether to generate gas molecules
#declare Chamber_Flag    = (1 | Full_Flag); // whether to model quartz chamber
#declare Transparent_Flag= (0 | Full_Flag); // whether to use a transparent chamber
#declare Refraction_Flag = (0 | Full_Flag); // whether to use refraction on chamber
#declare Reflection_Flag = (0 | Full_Flag); // whether to use refraction on chamber
#declare Lamp_Array_Flag = (0 | Full_Flag); // whether to render the lamps
#declare MultiLight_Flag = (0 | Full_Flag); // use more than one light souce

//////////////////////////////////////////////////
// modeling parameters
//

// crystal : dimension depends on phase
#declare Nmax  = <25, 10, 25>;       // value for phase 1 : this is needed
                                     // by phase 2 to maintain rand() phase into gas
#if (Phase < 0)
    #declare Nmax         = <10, 2, 10>;   // rough sketch
#end

#declare Zphase2 = int(Nmax.z * 0.67);      // edge of phase2 region
#declare Zphase1 = Zphase2;                 // where phase1 has a step

#declare Wafer_Zmax   = 26;              // used for gas and halo

#declare Xstep        = 5;               // location of a step on the y surface
#declare Zstep        = 20;              // location of a step on the y surface
#declare H_coverage   = 0.85;            // surface bonds tied up by H atoms
#declare B_doping     = 0.01;            // fractional substitutional boron

// chamber
#declare Chamber_R    = 1e6 * <2, 1, 1>; // distance to chamber (reduced scale)
#declare Chamber_dR   = 2e5;             // thickness of chamber (reduced scale)
#declare Tube_R       = Chamber_R * <0.5, 0.25, 2>;
                                          // extension of chamber for slot

// enclosure
#declare Enclosure_R  = <Chamber_R.x + 2 * Chamber_dR,
                         Chamber_R.y + 2 * Chamber_dR,
                         Chamber_R.z + Tube_R.z + 2 * Chamber_dR>;

// lamp array
#declare Nlamps       = 8;
#declare Lamp_dR      = Chamber_R.x / (Nlamps + 1);
#declare Lamp_Rminor  = Lamp_dR / 3;
#declare Lamp_V       = <0, Chamber_R.y + 2 * Chamber_dR, Chamber_R.z - Chamber_R.x>;
#declare Chamber_Theta= -12;        // rotation of chamber relative to camera

// molecular gas
#declare Gas_boundary = <50, 50, 100>;   // gas from <-x, 0, -z/10> to <x, y, z>
#declare Gas_volume   = Gas_boundary.x * Gas_boundary.y * Gas_boundary.z * 2.42;
#declare H2_count     = 10;              // hydrogen molecules

#declare gas_density  = array[4];
#declare gas_density[0] = 0.0001;  // silane
#declare gas_density[1] = 0.00005; // germane
#declare gas_density[2] = 0.0005;  // hydrogen
#declare gas_density[3] = 0.00001; // diborane (not yet implemented)

#declare atomsize     = array[4];
#declare atomsize[0]  = 1;           // silicon (sp3)
#declare atomsize[1]  = 1.04;        // germanium (sp3)
#declare atomsize[2]  = 0.15;        // surface hydrogen (s)
#declare atomsize[3]  = 0.9;         // substitutional boron (sp3)

#declare atom_c    = array[4];
#declare atom_c[0] = <0.6, 0.6, 1.2>;
#declare atom_c[1] = <1.2, 0.6, 0.6>;
#declare atom_c[2] = <0.9, 0.9, 1.1>;
#declare atom_c[3] = <1.2, 1.2, 0.9>;

#declare atomcolor    = array[4];
#declare atomcolor[0] = pigment { color rgb atom_c[0] }       // silane
#declare atomcolor[1] = pigment { color rgb atom_c[1] }       // germanium
#declare atomcolor[2] = pigment { color rgb atom_c[2] }       // hydrogen
#declare atomcolor[3] = pigment { color rgb atom_c[3] transmit 0.25 } // boron

// continuous portion of gas (distant portion)
#declare gas_weight = array[4]
#declare N = 0
#while (N < 4)
  #declare gas_weight[N] = pow(atomsize[N],3) * gas_density[N];
  #debug concat( "gas_weight[", str(N, 1, 0), "] = ", str(gas_weight[N], 10, 8), "\n")
  #declare N = N + 1;
#end

#declare Gas_C =
 (
    ((atom_c[0]) * (gas_weight[0])) +
    ((atom_c[1]) * (gas_weight[1])) +
    ((atom_c[2]) * (gas_weight[2])) +
    ((atom_c[3]) * (gas_weight[3]))
  ) /
  (
    (gas_weight[0]) +
    (gas_weight[1]) +
    (gas_weight[2]) +
    (gas_weight[3])
   );

#declare Gas_Color = pigment { color rgb Gas_C }

#debug
  concat(
    "Gas_C = < ",
    str( Gas_C.x, 10, 8 ), " ",
    str( Gas_C.y, 10, 8 ), " ",
    str( Gas_C.z, 10, 8 ),
    ">\n"
  )

// electron gas
#declare halo_mag    = array[4]
#declare halo_mag[0] = 0.3;     // Si
#declare halo_mag[1] = 0.6;     // Ge
#declare halo_mag[2] = 0;       // H (shouldn't be used)
#declare halo_mag[3] = 1;       // B



////////////////////////////////////////////////
// define rotation angles of sp3 orbitals
// where the principle axes are (001) :
//
//         2   0   2   0   2
//       3   1   3   1   3   1
//         0   2   0   2   0
//       1   3   1   3   1   3
//         2   0   2   0   2
//       3   1   3   1   3   1
//         0   2   0   2   0
//       1   3   1   3   1   3
//
#declare T1 = degrees(atan2(1, sqrt(2)));
#declare T2 = degrees(atan2(1, 1));

#declare rotation    = array[4];
#declare rotation[0] = <0, -T1, -T2>;
#declare rotation[1] = <0, 180 + T1, -T2>;
#declare rotation[2] = <0, 180 - T1, T2>;
#declare rotation[3] = <0,  T1, T2>;


////////////////////////////////////////////////
// define parameters associated with crystal structure
//

#declare dV1    = array[2];
#declare dV1[0] = <0, 0, 0>;           // first atomic plane  : element 0
#declare dV1[1] = <0, -0.5, 0.5>;      // third atomic plane  : element 0

#declare dV2    = array[2];
#declare dV2[0] = <0, 0, 0>;           // first zincblend site
#declare dV2[1] = <0.25, -0.25, 0.25>; // second zincblend site

#declare dV3    = array[2];
#declare dV3[0] = <0, 0, 0>;           // first FCC site
#declare dV3[1] = <-0.5, 0, -0.5>;     // second FCC site


////////////////////////////////////////////////
// define wafer
//

#if (Wafer_Flag)
#if (Blob_Flag)
 blob {
  threshold 0.5
#else
 union {
#end
  #declare Z = 0;
  #while (Z < Nmax.z + 0.1)
    #declare Yphase1 = ((Z <= Zphase1) ? -1 : 0);       // edge of phase1 region
    #declare Xmax  = min(Z + ((Phase = 1) ? 0 : 2), Nmax.x);
    #declare X = -Xmax;
    #while (X < Xmax + 0.1)

      #declare Ymax = min(Z + 1, int(Nmax.y * exp((Nmax.y-Z)/4)));
      #declare Y = -Ymax;
      #declare Ytop = (((Z < Zstep + 0.1) & (X > Xstep - 0.1)) ? 1 : 0);     // note: surface has a step
      #declare YCount = 0;
      #while (Y <= Ytop)
        #declare AtomFlag = ((Phase = 1) ? (Y >= Yphase1) : ((Phase = 2) ? (Z <= Zphase2) : 1));

        #declare A = 0;
        #while (A < 2)  // atom type
          #declare Sz = ((A=0) ? 1 : -1);
          #declare NA = 0;
          #while (NA < 2)
            #declare NF = 0;
            #while (NF < 2)
              #declare V = <X, Y, Z> + dV1[NA] + dV2[A] + dV3[NF];

              // pick the right seed
              #declare Rand = ((Y > 0.1) ? Rand3 : ((abs(X) > Z) ? Rand4 : Rand1));

              // note : use special seed for top layer to allow reconstruction
              #declare Species = ((rand(Rand) < B_doping) ? 3 : A);   // 3: boron substitutionals

              #declare N = 0;   // orbital number
              #while (N < 4)
                #if (AtomFlag)
                  #declare YCount = YCount + 1;
                  sphere {
                    <sqrt(3)/12, 0, 0>,
                    0.3
                    #if (Blob_Flag)
                    , strength 1
                    #end
                    scale <1, 0.35, 0.35>
                    rotate rotation[N]
                    scale atomsize[Species] * <1, Sz, 1>
                    translate V
                    pigment { atomcolor[Species] }
                  }
                #end

                // if this is the top layer, it may need a terminating H atom
                #if ((A = 0) & (NA = 0) & (Y = Ytop))
                  #if (mod(N,2) = 1)
                    #if (rand(Rand) < H_coverage)
                      #if (AtomFlag)
                        sphere {
                          <0.25 * atomsize[Species], 0, 0>,
                          atomsize[2]
                          #if (Blob_Flag)
                          , strength 1
                          #end
                          rotate rotation[N]
                          scale atomsize[Species] * <1, Sz, 1>
                          translate V
                          pigment { atomcolor[2] }
                        }
                      #end
                    #end
                  #end
                #end
                #declare N = N + 1;
              #end
              #declare NF = NF + 1;
            #end
            #declare NA = NA + 1;
          #end
          #declare A = A + 1;
        #end
        #declare Y = Y + 1;
      #end

      #if (AtomFlag)
        #render concat(str(YCount,0,0),",")
      #else
        #render "-,"
      #end

      #declare X = X + 1;
    #end
    #declare Z = Z + 1;
    #render "\n"
  #end
  texture {
    finish { diffuse 1.5 specular 0.75 roughness 0.01 metallic }
  }
 }
#end  // end of wafer

////////////////////////////////////////////////
// dilute gas : this is the first version of the
// the distant gas
//

#if (Fog_Flag)
  fog{
      fog_type 1
      distance 25 * Chamber_R.z
      color rgb ( Gas_C * 5 )  // note: factor for added brightness
      turbulence 0
      omega 0.5
      lambda 1.01
      octaves 1
      scale Chamber_R.z * 5
  }
#end

////////////////////////////////////////////////
// atmosphere : this is a revised version of the
// distant gas.
//

#if (Atmosphere_Flag)
  atmosphere {
    type 2
    distance 10 * Chamber_R.z
    scattering 0.4 / Chamber_R.z
    samples 25
    jitter 0.2
    aa_threshold 0.25
    aa_level 2
    color rgb Gas_C filter 0.5
  }
#end

////////////////////////////////////////////////
// electron gas
//

// define the dimensions of the gas : it extends from the
// bottom of the crystal to the top, where it rapidly falls
// off...
//        y(x) = 1 / (1 + exp(x))
//
//  This is basically a hyperbolic tangent relation .
//

#declare Y0 =  0.1;          // scale length of falloff function
#declare Y1 = -Nmax.y;       // start of halo
#declare Y2 = 0.3;             // point of inflection of halo
#declare Y3 = 2 * (Y2 - Y1); // height of halo
#declare YA = Y3 / Y0;       // scale of normalized function
#debug concat( "Y0 = ", str(Y0, 6, 4), " )\n");
#debug concat( "Y1 = ", str(Y1, 6, 4), " )\n");
#debug concat( "Y2 = ", str(Y2, 6, 4), " )\n");
#debug concat( "Y3 = ", str(Y3, 6, 4), " )\n");
#debug concat( "YA = ", str(YA, 6, 4), " )\n");

#if (Halo_Flag)
  box {
    <-1, 0, -1>,
    <1,  1, 1>
    hollow
    pigment { rgbf 1 }
    halo {
      emitting
      planar_mapping
      linear
      max_value 1
      color_map {
        #declare Y = 0;
        #declare dY =0.01;
        #while (Y < 1 + dY/2)
          #declare F = 1 / (1 + exp(YA * (Y - 0.5)));
          [ Y rgbt <0, 1, 0, F> ]
          #render concat("(Y,F) = ( ", str(Y, 6, 4), ", ", str(F, 8, 6), " )\n")
          #declare Y = Y + dY;
        #end
      }
    }
    scale <Wafer_Zmax.x, Y3, Wafer_Zmax>
    translate <0, Y1, 0>
  }

  // add an electron gas to extra surface atoms
  // this can be done either with a superposition of halos or
  // one aggregate halo -- the former is rendering at approx 3 lines/hour,
  // but has the proper transitions into vacuum

  box {
    <Xstep - 0.5, -1, 0>,
    <Nmax.x, 2, Zstep + 0.5>
    hollow
    pigment {rgbf 1}

    #declare Rand3 = seed(Seed3);    // reset the alternate pseudorandom stream
    #declare Z = 0;
    #while (Z <= min(Nmax.z, Zstep))
      #declare Xmax = min(Z, Nmax.x);
      #declare X = max(-Xmax, Xstep);
      #while (X <= Xmax)
        #declare Ymax = min(Z + 1, int(Nmax.y * exp(-Z/Nmax.y)) );
        #declare Y = 1;
        #declare A = 0;
        #while (A < 2)  // atom type
          #render "[H]"
          #declare Sz = ((A=0) ? 1 : -1);
          #declare NA = 0;
          #while (NA < 2)
            #declare NF = 0;
            #while (NF < 2)
              #declare V = <X, Y, Z> + dV1[NA] + dV2[A] + dV3[NF];
              #declare Species = ((rand(Rand3) < B_doping) ? 3 : A);   // 3: boron substitutionals

              halo {
                emitting
                spherical_mapping
                cubic
                samples 10
                aa_threshold 0.3
                max_value halo_mag[A]
                color_map {
                  [ 0 rgbt <0, 1, 0, 1> ]
                  [ 1 rgbt <0, 1, 0, 0.98> ]
                }
                scale atomsize[Species] * <1, 1, 1>
                translate V
              }
              #declare NF = NF + 1;
            #end
            #declare NA = NA + 1;
          #end
          #declare A = A + 1;
        #end
        #declare X = X + 1;
        #render " "
      #end
      #declare Z = Z + 1;
      #render "\n"
    #end

  }
#end

////////////////////////////////////////////////
// define gas molecules
//

#if (Gas_Flag)
  #declare A = 0;
  #while (A < 4)
    #declare NG = gas_density[A];
    #declare NG = NG * Gas_volume;     // number of molecules of this species
    #render concat("NG[", str(A,1,0), "] = ", str(NG,1,0), "\n")

    #if (NG > 600)
      #error "Excessive molecule count encountered -- please adjust density or gas volume\n"
    #end

    // populate the gas volume with gas molecules of the particular type
    #declare I = 0;
    #while (I < NG)
      // assign vector of gas molecule
      #if ((A = 0) & (I = 0))
        #declare V = <-1.23, 1.3, 3.562>;
      #else
        #declare V = <0, -1, -1>;
        #while ( (V.y < 0) & (V.z < Wafer_Zmax) )
          #declare V = <rand(Rand2) * 2 - 1, 1.1 * rand(Rand2 - 0.1), 1.1 * rand(Rand2 - 0.1)> * Gas_boundary;
        #end
      #end
      #declare T = <rand(Rand2), rand(Rand2), rand(Rand2)> * 360;
                                                     // rotation angle of gas molecule

      // lay out the appropriate type of molecule
      #if ((Phase < 2) | (V.z < Wafer_Zmax))        // for phase 2, only consider potential shadow sources
        #if (Blob_Flag)
        blob {
          threshold 0.5
        #else
        union {
        #end
          #switch (A)
            #case(0)          // SiH4
            #case(1)          // GeH4
              #declare N = 0;
              #while (N < 4)
                sphere {
                  <sqrt(3)/12, 0, 0>,
                  0.3
                  #if (Blob_Flag)
                  , strength 1
                  #end
                  scale <1, 0.35, 0.35>
                  rotate rotation[N]
                  pigment { atomcolor[A] }
                }

                // terminating H atoms
                sphere {
                  <0.25 * atomsize[A], 0, 0>,
                  atomsize[2]
                  #if (Blob_Flag)
                  , strength 1
                  #end
                  rotate rotation[N]
                  pigment { atomcolor[2] }
                }
                #declare N = N + 1;
              #end
            #break
            #case (3)         // X2H4 : just fall through to H2 until this is implemented
            #case (2)         // H2
                sphere {
                  <-0.5, 0, 0>,
                  1
                  #if (Blob_Flag)
                  , strength 1
                  #end
                }
                sphere {
                  <0.5, 0, 0>,
                  1
                  #if (Blob_Flag)
                  , strength 1
                  #end
                }
                pigment { atomcolor[2] }
                #declare S = atomsize[2]
                scale S
            #break
          #end
          rotate T
          translate V
        }
      #end
      #declare I = I + 1;
    #end

    // now form some long-range molecules, rendered as spheres due to reduced resolution...
    // skip this in phase 2
    #declare I = 0;
    #while (I < 4.1 * NG)
      #declare V = <4 * rand(Rand2) - 2, 2.5 * rand(Rand2) - 0.5, 1 + rand(Rand2)> * Gas_boundary;
      #if (Phase < 2)
        sphere {
          V,
          1.5 * atomsize[2]
          pigment { Gas_Color }
        }
      #end
      #declare I = I + 1;
    #end
    #declare A = A + 1;
  #end
#end

/////////////////////////////////////////////////////////////////////////////
// chamber
//
union {
  #if (Chamber_Flag)
    difference {
      merge {
         superellipsoid {
           <0.2, 0.2>
           scale Chamber_R + Chamber_dR * <1, 1, 1>
         }
         superellipsoid {
           <0.2, 0.2>
           scale Tube_R + Chamber_dR * <1, 1, 1>
         }
       }
       union {
         superellipsoid {
           <0.2, 0.2>
           scale Chamber_R
         }
         superellipsoid {
           <0.2, 0.2>
           scale Tube_R + 1.1 * Chamber_dR * z;
         }
       }
       texture {
         pigment {
           color rgbf <0.25, 0.25, 0.25, 0.75 * Transparent_Flag>
         }
         normal {
           bozo 0.1
         }
         finish {
           ambient (Transparent_Flag ? 0.1 : 0.3)
           diffuse (Transparent_Flag ? 0.2 : 1)
           reflection .125 * Reflection_Flag
           refraction Refraction_Flag
           ior 2.2
           specular 0.5
           roughness 0.001
         }
       }
     }
  #end

  // enclosure
  box {
    -Enclosure_R,
    Enclosure_R
    inverse
    hollow
    texture {
      pigment { color rgb 0.5  }
      finish { metallic }
    }
  }

  // draw in lamp array
  #if (Lamp_Array_Flag)
    #declare N = 0;
    #while (N < Nlamps)
      #declare N = N + 1
      torus {
        Lamp_dR * N,
        Lamp_Rminor
        texture {
          pigment {
            color rgb 1
          }
          finish {
            ambient 5
            diffuse 0
          }
        }
        no_shadow
        translate Lamp_V
      }
    #end
  #end

  #if (Chamber_Theta)
    translate -<1, 0, 1> * Lamp_V
    rotate Chamber_Theta * y
    translate <1, 0, 1> * Lamp_V
  #end
}


///////////////////////////////////////////////
// lights  This should be above the
// glass, but no matter....
//

#if (MultiLight_Flag)
  union {
    light_source {
      <0.9, 0.9, 0.9> * Chamber_R
      color rgb 0.5
    }
    light_source {
      <-0.9, 0.9, 0.9> * Chamber_R
      color rgb 0.5
    }
    light_source {
      <0.9, 0.9, 0> * Chamber_R
      color rgb 0.5
    }
    light_source {
      <-0.9, 0.9, 0> * Chamber_R
      color rgb 0.5
    }
    light_source {
      <0.9, 0.9, -0.9> * Chamber_R
      color rgb 0.5
    }
    light_source {
      <-0.9, 0.9, -0.9> * Chamber_R
      color rgb 0.5
    }
    #if (Chamber_Theta)
      translate -<1, 0, 1> * Lamp_V
      rotate Chamber_Theta * y
      translate <1, 0, 1> * Lamp_V
    #end
  }
  #declare Light_At_Camera = 0;
#else
  #declare Light_At_Camera_Flag = 1;
#end

///////////////////////////////////////////////
// camera
//

#declare Camera_Type = Camera_Main;
#declare Camera_Target   = <0, 0, 3>;
#declare Camera_Location = <0, 1, 0>;
#declare Camera_Angle = 90;
#declare Camera_Scale = 2;
#include "camera.pov"



