#ifdef RCSIDS
static char rcsid[] = "$Id: glass.sl,v 1.4 1996-03-11 19:04:51-08 lg Exp $";
#endif

/*
 * glass.sl -- Shiny reflective & refractive glass, using ray tracing.
 *
 * DESCRIPTION:
 *   Makes semi-transparent glass, using ray tracing to calculate
 *   reflections and refractions of the environment.
 * 
 * PARAMETERS:
 *    Ka, Kd, Ks, roughness, specularcolor - The usual meaning
 *    Kr - coefficient for mirror-like reflections of environment
 *    Kt - coefficient for refracted transmission
 *    transmitcolor - color of the glass
 *    eta - the coefficient of refraction of the glass
 *    blur - how blurry are the reflections/refractions? (0 = perfectly sharp)
 *    samples - set to higher than 1 for oversampling of blur
 *
 * AUTHOR: written by Larry Gritz, 1991
 *
 * HISTORY:
 *
 * $Revision: 1.4 $      $Date: 1996-03-11 19:04:51-08 $
 *
 * $Log: glass.sl,v $
 * Revision 1.4  1996-03-11 19:04:51-08  lg
 * Don't use Os -- we just want to set the color from the rays
 *
 * Revision 1.3  1996-02-29 10:11:23-08  lg
 * Minor correction and dead code removal
 *
 * Revision 1.2  1996-02-12 17:34:53-08  lg
 * Fixed use of fresnel, chose better default parameters
 *
 * Revision 1.1  1995-12-05 15:05:47-08  lg
 * Initial RCS-protected revision
 *
 * 25 Jan 1994 -- recoded by lg in correct shading language.
 * Aug 1991 -- written by lg in C
 *
 */




surface
BMGlass ( float Ka = 0.2, Kd = 0, Ks = 0.5;
	float Kr = 1, Kt = 1, roughness = 0.05, blur = 0, eta = 1.5;
	color specularcolor = 1, transmitcolor = 1;
	float samples = 1; )
{
  point IN;               /* Normalized incident vector */
  point Nf;               /* Forward facing normal vector */
  point Rfldir, Rfrdir;   /* Smooth reflection/refraction directions */
  point uoffset, voffset; /* Offsets for blur */
  color ev = 0;           /* Color of the environment reflections */
  color cr = 0;           /* Color of the refractions */
  point R;                /* Direction to cast the ray */
  float i;
  float kr, kt;
  float eeta;
float tmp;

  /* Construct a normalized incident vector */
  IN = normalize (I);

  /* Construct a forward facing surface normal */
  Nf = faceforward (normalize(N), I);

  fresnel (IN, Nf, (I.N < 0) ? 1.0/eta : eta, kr, kt, Rfldir, Rfrdir);
  kr *= Kr;
  kt *= Kt;

  /* Calculate the reflection color */
  if (kr > 0.001) {
      /* Rfldir gets the perfect reflection direction */
      Rfldir = normalize (Rfldir);
      if (blur > 0) {
	  /* Construct orthogonal components to Rdir */
	  uoffset = normalize (point (zcomp(Rfldir) - ycomp(Rfldir),
				      xcomp(Rfldir) - zcomp(Rfldir),
				      ycomp(Rfldir) - xcomp(Rfldir)));
	  voffset = Rfldir ^ uoffset;
	}
      for (i = 0;  i < samples;  i += 1) {
	  R = Rfldir;
	  if (blur > 0) {
	      /* Add a random offset to the smooth reflection vector */
	      R += ((float random()-0.5) * blur) * uoffset;
	      R += ((float random()-0.5) * blur) * voffset;
	      R = normalize (R);
	    }
	  ev += trace (P, R);
	}
      ev *= kr / samples;
    }

  /* Calculate the refraction color */
  if (kt > 0.001) {
      /* Rdir gets the perfect refraction direction */
      Rfrdir = normalize (Rfrdir);
      if (blur > 0) {
	  /* Construct orthogonal components to Rdir */
	  uoffset = normalize (point (zcomp(Rfrdir) - ycomp(Rfrdir),
				      xcomp(Rfrdir) - zcomp(Rfrdir),
				      ycomp(Rfrdir) - xcomp(Rfrdir)));
	  voffset = Rfrdir ^ uoffset;
	}
      for (i = 0;  i < samples;  i += 1) {
	  R = Rfrdir;
	  if (blur > 0) {
	      /* Add a random offset to the smooth reflection vector */
	      R += (float random()-0.5) * blur * uoffset;
	      R += (float random()-0.5) * blur * voffset;
	      R = normalize (R);
	    }
	  cr += trace (P, R);
	}
      cr *= (kt / samples);
    }

  Oi = 1;
  Ci = ( Cs * (Ka*ambient() + Kd*diffuse(Nf)) +
	 specularcolor * (ev + Ks*specular(Nf,-IN,roughness)) +
	 transmitcolor * cr );
}
