/*
 * BSThread.sl -- Make threads in shader space
 *
 * DESCRIPTION:
 *    Makes a threaded bump map in shader space, with selectable
 *    direction and size in space.
 *
 * PARAMETERS:
 *    Km - displacement magnitude
 *    frequency: number of threads per spatial unit
 *    phase - phase of the "screw"
 *    offset - an offset to the threads
 *    dampzone - adjustable damp at end and beginning of threads
 *    thr_start, thr_end - start and end of threads in shader space
 *    displ_direction - 0 means x, 1 meanz y, 2 means z
 *
 * AUTHOR: written by Bernd Sieker, 1998,
 *
 * HISTORY:
 *      Oct 1998 -- written
 *
 */

displacement 
BSThreads ( 
	float   Km		=  .1,
		frequency	= 5.0,
		phase		=  .0,
		offset		=  .0,
		dampzone	=  .05,
	        thr_start       = .5,
	        thr_end         = 1.0,
                displ_direct    = 1.0)
{
	float magnitude, thr;

	point Psh;

	/* Calculate the undamped displacement */

	Psh = transform("shader", P);

	if (displ_direct < .5)
	  thr = xcomp(Psh);
	if ((displ_direct >= .5) && (displ_direct <1.5))
	  thr = ycomp(Psh);
	if (displ_direct >=1.5)
	  thr = zcomp(Psh);

	if ((thr > thr_start) && (thr < thr_end)) {

	  magnitude = (sin( PI*2*(thr*frequency + thr + phase))+offset) * Km;

	  /* Damp the displacement to 0 at each end */

	  if( thr > (thr_end-dampzone))
	    magnitude *= (thr_end-thr) / dampzone;
	  else if( thr < thr_start+dampzone )
	    magnitude *= (thr-thr_start) / dampzone;
	}

	/* Do the displacement */
	P += normalize(N) * magnitude;
	N = calculatenormal(P);
}
