/*
 * BSToothPasteStripes.sl -- Make a striped toothpaste.
 *
 * DESCRIPTION:
 *    Makes stripes on a shiny material, parts based on Larry Gritz' Shiny
 *    and using definitions from the RManNotes.
 *
 * PARAMETERS:
 *    Ka, Kd, Ks, roughness, specularcolor - The usual meaning
 *    Kr - coefficient for mirror-like reflections of environment
 *    blur - how blurry are the reflections? (0 = perfectly sharp)
 *    samples - set to higher than 1 for oversampling of blur
 *    pastecolor, stripecolor - the color of the base toothpaste and
 *                              of the stripes
 *    stripes - number of colored stripes
 *    ratio - ratio between base color and stripe
 *    fuzz - fuzziness of stripes
 *
 * AUTHOR: written by Bernd Sieker, 1998,
 *
 * HISTORY:
 *      Oct 1998 -- written
 *
 */

#define pulse(a,b,fuzz,x) (smoothstep((a)-(fuzz),(a),(x)) - \
			   smoothstep((b)-(fuzz),(b),(x)))

#define repeat(x,freq)    (mod((x) * (freq), 1.0))

surface
BSToothPasteStripes (float Ka = 1, Kd = 0.5, Ks = 1;
	 float Kr = 0.5, roughness = 0.005, blur = 0;
	 color specularcolor = color(1.0, 1.0, 1.0);
	 color pastecolor = color(1.0, 1.0, 1.0);
         color stripecolor = color(0.5, 0.85, 0.5);
	 float samples = 1;
	 float stripes = 4,
	       ratio = .4,
	       fuzz = 0.1;)
{
  uniform float ratio2 = ratio / 2;

  float tt;
  color Ct;               /* calculated stripe color */
  point Nf;               /* Forward facing normal vector */
  point IN;               /* normalized incident vector */
  point Rdir;             /* Smooth reflection direction */
  point uoffset, voffset; /* Offsets for blur */
  color ev;               /* Color of the reflections */
  point R;                /* Direction to cast the ray */
  float i;

  /* Construct a forward facing surface normal */
  Nf = faceforward (normalize(N), I);
  IN = normalize (I);
  ev = 0;

  /* Calculate the reflection color */
  if (Kr > 0) {
      /* Rdir gets the perfect reflection direction */
      Rdir = normalize (reflect (IN, Nf));

      if (blur > 0) {
	  /* Construct orthogonal components to Rdir */
	  uoffset = normalize (point (zcomp(Rdir) - ycomp(Rdir),
				      xcomp(Rdir) - zcomp(Rdir),
				      ycomp(Rdir) - xcomp(Rdir)));
	  voffset = Rdir ^ uoffset;
	}
      R = Rdir;
      ev = trace (P, R);
      ev *= Kr / samples;
    }

  /* calculate the texture color */

  tt = repeat(t, stripes);

  Ct = mix(pastecolor, stripecolor, \
	   pulse(0.5 - ratio2, 0.5 + ratio2, \
		 fuzz, tt));

  /* The rest is like the plastic shader */
  Oi = Os;
  Ci = Os * ( Ct * (Ka*ambient() + Kd*diffuse(Nf)) +
	      specularcolor * (ev + Ks*specular(Nf,-IN,roughness)));
}
