// Persistence of Vision Ray Tracer Scene Description File
// File: voyager.inc
// Vers: POV-Ray 3.1
// Desc: This is written for the Nov-Dec 1998 IRTC whose theme is "First Encounters". 
//       The probe Voyager has passed beyond Pluto and entered deep space.  After who 
//       knows how long it meets up with an alien exploration vessel.  This is mankinds
//       first encounter with aliens.
// Date: Started Nov 8, 1998 
// Auth: David Ray
// 
// ==== Standard POV-Ray Includes ====
#include "colors.inc"	// Standard Color definitions
#include "textures.inc"	// Standard Texture definitions
#include "finish.inc"   // Standard finished include file

// Voyager shape deinition.  The scale of voyager is that the diameter of the dish is 3.7m

#declare Strut_Width = 0.03;

#declare High_Gain_Antenna_Dish = object {  
    lathe {
        quadratic_spline
        16, // number of points
        <0, 0>, <0.5, 0.07>, <0.75, 0.156>, <1.0, 0.278>, 
        <1.25, 0.43>, <1.5, 0.625>, <1.75, 0.85>, <1.85, 0.95>,
        <1.85, 0.97>, <1.75, 0.87>, <1.5, 0.645>, <1.25, 0.45>,
        <1.0, 0.298>, <0.75, 0.176>, <0.5, 0.09>, <0, 0.02>
        pigment { color White }
    }
}

#declare High_Gain_Antenna = object {
    union {
        union {
            cylinder { <0, 0.01, 0>, <0, 0.16, 0>, 0.2 }
            cone { <0, 0.16, 0>, 0.2, <0, 0.34, 0>, 0.12 }
            cylinder { <0, 0.34, 0>, <0, 0.44, 0>, 0.12 }
            cylinder { <0, 0.44, 0>, <0, 0.5, 0>, 0.10 }
            cylinder { <0, 0.5, 0>, <0, 0.54, 0>, 0.12 }
            pigment { color Silver }
        }
        cylinder { 
            <0, 0.54, 0>, <0, 0.541, 0>, 0.1
            pigment { color Black }
        }
    }
}                                                                            
                                                                            
#declare High_Gain_Antenna_Collar = object {  
    lathe {
        linear_spline
        4, // number of points
        <0.84, 0.196>, <0.84, -0.02>, <0.92, -0.02>, <0.92, 0.236>
        pigment { color White }
    }    
}   

#declare Subreflector_Dish = object {  
    lathe {
        quadratic_spline
        6, // number of points
        <0, 1.2>, <0.1, 1.2155>, <0.2, 1.262>, <0.3, 1.339>, <0.35, 1.39>, <0, 1.39>
        // <0.35, 1.4>, <0.3, 1.349>, <0.2, 1.272>, <0.1, 1.2255>, <0, 1.21>
        pigment { color White }
    }
}

#declare Subreflector_Strut = object {
    union {
        cylinder { <0.35*.866,  1.38, 0.175 >, <0, 0.23, 0.9>, 0.02 }
        cylinder { <-0.35*.866, 1.38, 0.175 >, <0, 0.23, 0.9>, 0.02 }
        pigment { color White }
    }
}

#declare Subreflector_Tripod = object {
    union {
        object { 
            Subreflector_Strut 
        }
        object { 
            Subreflector_Strut 
            rotate <0, 120, 0>
        }
        object { 
            Subreflector_Strut 
            rotate <0, 240, 0>
        }
    }
}

#declare Low_Gain_Antenna = object {
    union {
        cone { 
            <0, 1.225, 0>, 0.12, 
            <0, 1.63, 0>, 0.03 
        }
        cylinder {
            <0, 1.63, 0>, 
            <0, 1.64, 0>,
            0.02
        }
        torus {
            0.05, 
            0.01
            translate <0, 1.65, 0>
        }
        cylinder {
            <0, 1.64, 0>,
            <0, 1.66, 0>,
            0.05
        }
        sphere {
            <0, 1.66, 0>, 0.03
        }
        pigment { color Gold }
        finish { Phong_Glossy }
    }
}

#declare Bus_Face = object {
    box {
        <-0.311, 0, 0> <0.311, -0.35, -0.2>
        translate <0, -.35, 0.9673>
        pigment { color White }
    }
}


#declare Bus_Strut = object {
    union {
        cylinder { <-0.3, -0.05, -0.1 + Strut_Width>, <0, 0.35, -0.1 + Strut_Width>, Strut_Width }
        cylinder { <0.3, -0.05, -0.1 + Strut_Width>, <0, 0.35, -0.1 + Strut_Width>, Strut_Width }
    }
    translate <0, -.35, 0.95>
    pigment { color White }
} 

// 3 copies of Bus_Strut make up the bus struts 

#declare Bus_Struts = object {
    union {
        object { Bus_Strut rotate <0, 36, 0> }
        object { Bus_Strut rotate <0, 144, 0> }
        object { Bus_Strut rotate <0, -72, 0> }
    }

}

// 10 copies of the bus face make up the bus - each rotated at 36 degree intervals
#declare Electronics_Bus = object {
    union { 
        #declare Count = 0;
        #while (Count < 10)
            object{ Bus_Face rotate y*Count*36 }
            #declare Count = Count + 1;
        #end
    }
}

#declare Propellant_Tank = object {
    sphere {
        <0, -0.5, 0>, 0.4
        pigment { color White }
    }
}


#declare Leg_Strut = object {
    union {
        // The longer support strut just inside bus
        cylinder { <0.6527, Strut_Width/2, 0.4742>, <-0.6527, Strut_Width/2, 0.4742>, Strut_Width }
        // The 2 shorter support struts inside bus
        cylinder { <0.6527, Strut_Width/2, 0.4742>, <0.6527, Strut_Width/2, -0.4742>, Strut_Width }
        // The propellant tank struts
        cylinder { <0.6527, Strut_Width/2, 0.4742>, <0, 0.4, 0>, Strut_Width } 
        cylinder { <0.6527, Strut_Width/2, -0.4742>, <0, 0.4, 0>, Strut_Width } 
        // The 2 longer legs
        cylinder { <0.6527, 0, 0.4742>, <0, -1.1, 0.4742>, Strut_Width }
        cylinder { <-0.6527, 0, 0.4742>, <0, -1.1, 0.4742>, Strut_Width }
        // The 2 shorter legs
        cylinder { <0.6527, 0, 0.4742>, <0.6527, -1.1, 0>, Strut_Width }
        cylinder { <-0.6527, 0, 0.4742>, <-0.6527, -1.1, 0>, Strut_Width }
        // The 2 "feet"
        box {
            <-0.07, -1.1+Strut_Width, 0.4742-Strut_Width>, 
            <0.07, -1.1-Strut_Width, 0.4742+Strut_Width> 
        }
        box {
            <0.6527-Strut_Width, -1.1+Strut_Width, -0.07>, 
            <0.6527+Strut_Width, -1.1-Strut_Width, 0.07> 
        }
        pigment { color White }
    }                          
    translate <0, -0.7, 0>
}

#declare Leg_Struts = object {
    union {
        object { Leg_Strut }
        object { 
            Leg_Strut 
            rotate <0, 180, 0>
        }
    }
}

#declare Optical_Cal_Target = object {
    box {
        <0.25, 0, 0>, 
        <-0.25, 0.5, 0.05>
        rotate <-25, 0, 0>
        translate <0, -1.7, 1>
        pigment { color White }
        rotate <0, 90, 0>
    }
}

#declare Optical_Cal_Target_Struts = object {
    union {
        // Top 2 struts
        cylinder { <0.6527, 0, 0.4742>, <0.83, -0.57, 0.25-Strut_Width/2>, Strut_Width*3/4 }
        cylinder { <0.6527, 0, -0.4742>, <0.83, -0.57, -0.25+Strut_Width/2>, Strut_Width*3/4 }
        // Bottom 2 struts
        cylinder { <0.6527, -1.1+(2/3*Strut_Width), 0>, <1.0, -0.96, 0.25-Strut_Width/2>, Strut_Width*3/4 }
        cylinder { <0.6527, -1.1+(2/3*Strut_Width), 0>, <1.0, -0.96, -0.25+Strut_Width/2>, Strut_Width*3/4 }
    }
    translate <0, -0.7, 0>
    pigment { color White }
}

#declare LF_Magnetometer_Spar = object {
    union {
        cylinder { <-0.1, 0, 0.0577>, <-0.1, 0.15, 0.0577>, 0.01 }    
        cylinder { <-0.1, 0, 0.0577>, <0.1, 0, 0.0577>, 0.01 }
        cylinder { <-0.1, 0, 0.0577>, <0.1, 0.15, 0.0577>, 0.01 }
        cylinder { <0.1, 0, 0.0577>, <-0.1, 0.15, 0.0577>, 0.01 }
    }
}

#declare LF_Magnetometer_Face = object {
    union { 
        #declare Count = 0;
        #while (Count < 40)
            object{ LF_Magnetometer_Spar translate <0, Count*0.15, 0> }
            #declare Count = Count + 1;
        #end                       
        #declare Count = Count + 1;
        #while (Count < 45)
            object{ LF_Magnetometer_Spar translate <0, Count*0.15, 0> }
            #declare Count = Count + 1;
        #end
        // The pointy end on the magentometer boom
        cylinder { <-0.1, 6.75, 0.0577>, <0.1, 6.75, 0.0577>, 0.01 }
        cylinder { <-0.1, 6.75, 0.0577>, <0, 6.90, 0>, 0.01 }
        sphere { <0, 6.90, 0>, 0.01 } 
    }
    translate <0, 0.45, 0>
    pigment { color White }        
}                 

#declare LF_Mag_Base = object {
    union {
        cylinder { <0, 0, 0>, <0, 0.45, 0>, 0.12 }
        torus { 
            0.12, 0.02
            translate <0, 0.43, 0>
        }
        torus { 
            0.12, 0.02
            translate <0, 0.225, 0>
        }
        torus { 
            0.12, 0.02
            translate <0, 0.02, 0>
        }
    }
    pigment { color White }
}
                 
#declare Low_Field_Magnetometer = object {
    union {
        object { LF_Mag_Base }
        object { LF_Magnetometer_Face } 
        object { 
            LF_Magnetometer_Face        
            rotate <0, 120, 0>
        }
        object { 
            LF_Magnetometer_Face        
            rotate <0, 240, 0>
        }
        // The magnetometer unit itself
        box { 
            <-0.11, 6.45, -0.11>, 
            <0.11, 6.60, 0.07>
            pigment { color White }
        }
        // Electronic's Bus Mounting struts
        object {
            union {
                cylinder { <0, 0, -0.12>, <0, -0.1, -0.22>, Strut_Width*2/3 }
                cylinder { <0, 0, -0.12>, <0, 0.1, -0.22>, Strut_Width*2/3 }
                cylinder { <0, 0.45, -0.12>, <0, 0.55, -0.22>, Strut_Width*2/3 }
                // Long mounting strut
                cylinder { <0, 0.55, -0.22>, <0, -0.55, -0.22>, Strut_Width*2/3 }
            }
            pigment { color White }
        } 
    }
    // Move it into position 
    translate <0, 1.0, 0>
    rotate <45, 0, 0>
    rotate <0, 239, 0>
    translate <-0.63, -0.83, 0>
}

#declare PRA_PWS_Module = object {
    union {
        difference {
            cylinder { 
                <0, -0.01, 0>, <0,-0.09,0>, 0.075 
            }
            box {
                <1, 0, 0.05>, <-1, -1, 0.1>
            }
            scale <1, 1, 0.8>
        }
        cylinder { <0, -0.05, 0>, <0, 0.05, 3>, 0.01 }
    }
}
 
#declare PRA_PWS = object {
    union {
        box { <-0.17, 0, 0> <0.17, -0.1, 0.1> }
        box { <-0.15, -0.01, 0.1> <0.15, -0.09, 0.13> }
        object { 
            PRA_PWS_Module
            rotate <20, 0, 0>
            rotate <0, 45, 0>
            translate <0.07, 0, 0.15>
        }
        object { 
            PRA_PWS_Module
            rotate <20, 0, 0>
            rotate <0, -45, 0>
            translate <-0.07, 0, 0.15>
        }
    }
    translate <0, -0.5, 0.967>
    rotate <0, 36*7, 0>
    pigment { color White }
}

#declare Strut_Width = Strut_Width/2;

#declare RTG_Leg_Struts = object {
    union {
        // The small struts coming off the RTG itself
        //  The 2 joined ones with a sphere at the join
        cylinder { <0, 0.50, 0.16>, <0, 0.75, 0.41>, Strut_Width/2 }
        cylinder { <0, 0.75, 0.41>, <0, 1.0, 0.16>, Strut_Width/2 }
        sphere { <0, 0.75, 0.41>, Strut_Width/2 }      
        //  The single short one
        cylinder { <0, 1.0, 0.16>, <0, 1.15, 0.31>, Strut_Width/2 }
    }
    pigment { color White }
}

#declare RTG_Fins = object {
    difference {
        union {
            #declare Count=1;
            #while (Count <= 6)
                 box { 
                     <0.23, 0, -0.02>, <-0.23, 1.48, 0.02> 
                     rotate <0, Count*60, 0> 
                 }
                 #declare Count=Count+1;
            #end                    
        }
        union {
            cylinder { <0, 0.48, 0>, <0, 0.52, 0>, 0.25 }
            cylinder { <0, 0.98, 0>, <0, 1.02, 0>, 0.25 }
        }
    }
}
    
#declare RTG = object {
    union {
        // Main body
        union {
            cylinder { <0, 0, 0>, <0, 1.48, 0>, 0.16 }  // Central shaft
            cylinder { <0, 0, 0>, <0, 0.02, 0>, 0.2 }   // end plate
            cylinder { <0, 0.2, 0>, <0, 0.46, 0>, 0.17 }
            cylinder { <0, 0.46, 0>, <0, 0.48, 0>, 0.2 } // end plate
            cylinder { <0, 0.52, 0>, <0, 0.54, 0>, 0.2 } // end plate
            cylinder { <0, 0.54, 0>, <0, 0.96, 0>, 0.17 }
            cylinder { <0, 0.96, 0>, <0, 0.98, 0>, 0.2 } // end plate
            cylinder { <0, 1.02, 0>, <0, 1.04, 0>, 0.2 } // end plate
            cylinder { <0, 1.04, 0>, <0, 1.44, 0>, 0.17 }
            cylinder { <0, 1.44, 0>, <0, 1.46, 0>, 0.2 }  // end plate
            // fins
            intersection {        
                object { RTG_Fins }
                cylinder { <0, 0, 0>, <0, 1.48, 0>, 0.2 }
            }
        }
        // RTG Top struts connecting each module together
        //   Long connecting cylinder
        cylinder { <0, 1.07, -0.24 >, <0, 0.05, -0.24>, Strut_Width/2 }
        //   The short cylinders going from RTG to long connector
        cylinder { <0, 1.07, -0.24>, <0, 1.07, 0>, Strut_Width/2 }
        cylinder { <0, 0.93, -0.24>, <0, 0.93, 0>, Strut_Width/2 }
        cylinder { <0, 0.57, -0.24>, <0, 0.57, 0>, Strut_Width/2 }
        cylinder { <0, 0.43, -0.24>, <0, 0.43, 0>, Strut_Width/2 }
        cylinder { <0, 0.05, -0.24>, <0, 0.05, 0>, Strut_Width/2 }
        // Far end of RTG
        cylinder { <0, 1.48, 0>, <0, 1.52, 0>, 0.22 }
        intersection {
            sphere { <0, 1.28, 0>, 0.3 }
            cylinder { <0, 1.52, 0>, <0, 1.70, 0>, 0.22 }
        }
        object {
            RTG_Leg_Struts
            rotate <0, 30, 0>
        }
        object {
            RTG_Leg_Struts
            rotate <0, -30, 0>
        }
    }
    pigment { color White }
    translate <0, 2.0, 0>
    rotate <100, 0, 0>
    rotate <0, 239, 0>
    translate <-0.6, -0.3, 0>
}

#declare RTG_Struts = object {
    union {
        // Upper 2 struts coming from bus 
        cylinder { <-0.3, 0, 0>, <0, 0.1, 0.4>, Strut_Width*1.5 }
        cylinder { <0.3, 0, 0>, <0, 0.1, 0.4>, Strut_Width*1.5 }
        // Lower 2 struts coming off bus
        cylinder { <-0.3, -0.3, 0>, <0, 0.1, 0.4>, Strut_Width*1.5 }
        cylinder { <0.3, -0.3, 0>, <0, 0.1, 0.4>, Strut_Width*1.5 }
        // The struts going from the point (struts from bus) to the RTG base
        cylinder { <0, 0.1, 0.4>, <-0.23, -0.15, 1.6>, Strut_Width*1.5 }  // top 1
        cylinder { <0, 0.1, 0.4>, <-0.35, -0.40, 1.55>, Strut_Width*1.5 }  // bottom 2
        cylinder { <0, 0.1, 0.4>, <-0.13, -0.38, 1.65>, Strut_Width*1.5 }
    }
    translate <0, -0.37, 0.96>
    rotate <0, 36*7, 0>
    pigment { color White }
}

#declare Science_Boom = object {
    union {
        // End closest to the body
        cylinder { <0, -0.125, -0.15>, <0, -0.125, 0.15>, Strut_Width }
        cylinder { <0, -0.125, -0.15>, <0.22, -0.25, -0.15>, Strut_Width }
        cylinder { <0, -0.125, 0.15>, <0.22, -0.25, 0.15>, Strut_Width }
        cylinder { <0, -0.125, -0.15>, <0.22, 0, -0.15>, Strut_Width }
        cylinder { <0, -0.125, 0.15>, <0.22, 0, 0.15>, Strut_Width }
        cylinder { <0.22, 0, -0.15>, <0.22, -0.25, -0.15>, Strut_Width }
        cylinder { <0.22, 0, 0.15>, <0.22, -0.25, 0.15>, Strut_Width }
        cylinder { <0.22, 0, -0.15>, <0.22, 0, 0.15>, Strut_Width }
        cylinder { <0.22, -0.125, -0.15>, <0.22, -0.125, 0.15>, Strut_Width }
        #declare Count=1;
        #while (Count <= 8)
            // Horizontal pieces
            cylinder { <0.22*Count, 0, -0.15>, <0.22*(Count+1), 0, -0.15>, Strut_Width }
            cylinder { <0.22*Count, 0, 0.15>, <0.22*(Count+1), 0, 0.15>, Strut_Width }
            cylinder { <0.22*Count, -0.25, -0.15>, <0.22*(Count+1), -0.25, -0.15>, Strut_Width }
            cylinder { <0.22*Count, -0.25, 0.15>, <0.22*(Count+1), -0.25, 0.15>, Strut_Width }
            // Vertical pieces
            cylinder { <0.22*(Count+1), 0, -0.15>, <0.22*(Count+1), 0, 0.15>, Strut_Width }
            cylinder { <0.22*(Count+1), -0.25, -0.15>, <0.22*(Count+1), -0.25, 0.15>, Strut_Width }
            cylinder { <0.22*(Count+1), 0, -0.15>, <0.22*(Count+1), -0.25, -0.15>, Strut_Width }
            cylinder { <0.22*(Count+1), 0, 0.15>, <0.22*(Count+1), -0.25, 0.15>, Strut_Width }
            // Diagonal pieces
            cylinder { <0.22*(Count), 0, -0.15>, <0.22*(Count+1), -0.25, -0.15>, Strut_Width }
            cylinder { <0.22*(Count), 0, 0.15>, <0.22*(Count+1), -0.25, 0.15>, Strut_Width }
            #declare Count=Count+1;
        #end
    }
    pigment { color White }
    translate <1.2, 0.25, 0>
}

#declare Science_Boom_Struts = object {
    union {
        // 2 struts coming off the same point of bus
        cylinder { <1.0170, -0.35, 0>, <1.2, 0.125, -0.15>, Strut_Width }
        cylinder { <1.0170, -0.35, 0>, <1.2, 0.125, 0.15>, Strut_Width }
        // 2 struts coming off opposite points of bus
        cylinder { <0.822, -0.34, -0.597>, <1.2, 0.125, -0.15>, Strut_Width }
        cylinder { <0.822, -0.34,  0.597>, <1.2, 0.125, 0.15>, Strut_Width }
        // Long strut going to the 3rd frame
        cylinder { <1.0170, -0.35, 0>, <1.8, 0, 0>, Strut_Width }
    }
    pigment { color White }
}

#declare Science_Boom_Scan_Platform = object {
    union {
        // The mounting pipe that mounts the scan platform to the science boom
        box { <0, 0.25, -0.15>, <0.015, 0, 0.15> }
        cylinder { <0.06, 0.21, 0>, <0.06, 0.04, 0>, 0.06 }
        cylinder { <0.06, 0.36, 0>, <0.06, -0.50, 0>, 0.05 }
        cylinder { <-0.04, -0.50, 0>, <0.16, -0.50, 0>, 0.05 }
        // The inboard part of the scan platform
        box { <-0.04, -0.25, -0.2>, <-0.08, -0.75, 0.2> }
        box { <-0.08, -0.25, -0.2>, <-0.1, -0.65, 0.2> }
        // The outboard part of the scan platform
        box { <0.16, -0.35, -0.13>, <0.3, -0.65, 0.13> }
    }
    pigment { color White }
    translate <3.18, 0, 0>
}

#declare IRIS = object {
    union {
        lathe {
            linear_spline
            11, // number of points
            <0, -0.55>, <0.06, -0.55>, <0.1, -0.51>, <0.12, -0.40>, 
            <0.05, -0.40>, <0.05, -0.09>, <0.03, -0.09>, <0.03, 0.06>, 
            <0.08, 0.06>, <0.08, 0.35>, <0, 0.35>
            pigment { color White }
        }        
        lathe {
            linear_spline
            10, // number of points
            <0.29, -0.3>, <0.30, -0.3>, <0.30, -0.14>, <0.11, -0.10>, 
            <0.11, -0.09>, <0, -0.09>, <0, -0.10>, <0.09, -0.10>, 
            <0.29, -0.15>, <0.29, -0.3>
            pigment { color White }
        }        
        lathe {
            quadratic_spline
            4, // number of points
            <0.28, -0.17>, <0.25, -0.15>, <0.15, -0.12>, <0.05, -0.10> 
            pigment { color Gold }
            finish { Phong_Glossy }
        }
        box { 
            <-0.06, 0.06, 0.06>, <0, -0.09, -0.06> 
            pigment { color White }
        }
        cylinder {
            <-0.18, -0.015, 0>, <-0.06, -0.015, 0>, 0.06
            pigment { color White }
        }        
    }
    scale 0.75
    rotate <0, -90, 0>
    rotate <90, 0, 0>
    translate <3.4, -0.75, -0.07>
}

#declare Narrow_Angle_Camera = object {
    union {
        // Camera body shape
        lathe {
            linear_spline
            11, // number of points
            <0, 0.25>, <0.03, 0.25>, <0.1, 0.23>, <0.1, -0.2>, 
            <0.12, -0.22>, <0.12, -0.4>, <0.11, -0.4>, <0.11, -0.22>, 
            <0.09, -0.2>, <0.09, 0.22>, <0, 0.25>
            pigment { color White }
        }
        // Camera Lens        
        cylinder { 
            <0, -0.2, 0>, <0, -0.19, 0>, 0.09
            pigment { color White }  // Should be a glass texture here
        }
        // The two mounting brackets
        #declare Count=1;
        #while (Count <= 2)
            union {
                cylinder { <0, 0.1, 0>, <0, 0.15, 0>, 0.12 }
                box { <0, 0.1, 0.12>, <-0.12, 0.15, -0.12> }
                pigment { color White }
                #if (Count > 1)
                    translate <0, -0.25, 0>
                #end            
            }
            #declare Count = Count + 1;
        #end
    }
    scale 0.7
    rotate <90, 0, 0>
    rotate <0, 0, 180>
    translate <3.0, -0.5, 0.08>
}

#declare Wide_Angle_Camera = object {
    union {
        cone { <-0.21, 0.035, 0>, 0.025, <0.01, 0.035, 0>, 0 open }
        cylinder { <0.09, 0.035, 0>, <0.05, 0.035, 0>, 0.02 }
        cone { <-0.09, 0.035, 0>, 0.015, <-0.04, 0.035, 0>, 0.02 }
        box { <-0.04, 0.11, -0.06>, <-0.01, 0.01, 0.06> }                       
        box { <-0.01, 0.095, -0.04>, <0.01, 0.01, 0.04> }
        box { <0.01, 0.01, 0.035>, <0.03, 0.085, -0.035> }
        box { <0.03, 0.01, 0.04>, <0.08, 0.09, -0.04> }
        // mount to the scan platform
        box { <0.01, 0.01, 0.02>, <-0.01, -0.02, -0.02> }
    }
    pigment { color White }
    scale 1.5
    rotate <90, 0, 0>
    rotate <0, -90, 0>
    translate <3.08, -0.305, 0.08>
}

#declare Ultraviolet_Spectrometer = object {
    union {
        box { <-0.1, 0, -0.035>, <0.1, 0.015, 0.035> } // base 
        box { <0.1, 0.015, -0.035>, <0.095, 0.065, 0.035> } // end support s
        box { <-0.1, 0.015, -0.035>, <-0.095, 0.065, 0.035> }
        // Scoop type thing at front
        difference {
            difference {
                box { <-0.1, 0.02, -0.03>, <-0.135, 0.04, 0.03> }
                box { <-0.1, 0.025, -0.025>, <-0.14, 0.05, 0.025> }
            }
            box { 
                <0, 0, -0.04>, <-0.04, 0.07, 0.04>
                rotate <0, 0, -60>
                translate <-0.135, 0.02, 0>
            }
        }
        // General shape of the unit
        box { <-0.095, 0.015, -0.03>, <-0.02, 0.05, 0.03> }
        cylinder { <-0.02, 0.065, 0>, <0.06, 0.03, 0>, 0.013 }
        box {
            <0, 0, -0.035>, <-0.005, 0.07, 0.035>
            rotate <0, 0, -25>
            translate <-0.03, 0.015, 0>
        }
        // Struts
        //  Both sides
        cylinder { <-0.005, 0.07, -0.03>, <0.095, 0.045, -0.03>, 0.005 }
        cylinder { <-0.005, 0.07, 0.03>, <0.095, 0.045, 0.03>, 0.005 }
        cylinder { <-0.005, 0.07, -0.03>, <0.02, 0.01, -0.03>, 0.005 }
        cylinder { <-0.005, 0.07, 0.03>, <0.02, 0.01, 0.03>, 0.005 }
        cylinder { <0.02, 0.01, -0.03>, <0.055, 0.055, -0.03>, 0.005 }
        cylinder { <0.02, 0.01, 0.03>, <0.055, 0.055, 0.03>, 0.005 }
        cylinder { <0.055, 0.055, -0.03>, <0.095, 0.01, -0.03>, 0.005 }
        cylinder { <0.055, 0.055, 0.03>, <0.095, 0.01, 0.03>, 0.005 }
        //  Top criss-cross
        cylinder { <-0.005, 0.07, -0.03>, <0.095, 0.045, 0.03>, 0.005 }
        cylinder { <-0.005, 0.07, 0.03>, <0.095, 0.045, -0.03>, 0.005 }
    }
    pigment { color White }
    scale 2
    rotate <0, -90, 180>
    translate <3.55, -0.38, -0.07>
}

#declare Photopolarimeter = object {
    lathe {
        linear_spline
        16, // number of points
        <0, 0.14>, <0.015, 0.0135>, <0.015, 0.013>, <0.04, 0.013>, 
        <0.04, 0.012>, <0.07, 0.1>, <0.07, 0.025>, <0.105, 0>, 
        <0.105, -0.26>, <0.1, -0.26>, <0.1, -0.01>, <0.035, 0>,
        <0.035, -0.05>, <0.05, -0.12>, <0, -0.12>, <0, 0.14>
    }
    pigment { color White }
    rotate <90, 0, 0>
    translate <3, -0.7, 0.05>
}

#declare Plasma_Investigation = object {
    union {
        union {  
            intersection {
                prism {
                    linear_sweep
                    linear_spline
                    0, // sweep the following shape from here ...
                    1, // ... up through here
                    31, // number of points
                    <-1.5, -1.2>, <-1.5, -0.9>, <-1.7, -0.6>, <-0.3, 1.9>,  // The outline
                    <-0.15, 1.8>, <-0.15, 2.1>, <0.15, 2.1>, <0.15, 1.8>, 
                    <0.3, 1.9>, <1.7, -0.6>, <1.5, -0.9>, <1.5, -1.2>, <-1.5, -1.2>,   
                    <-1.2, -1.0>, <-1.2, -0.85>, <0, -0.15>, // The first hole
                    <1.2, -0.85>, <1.2, -1.0>, <-1.2, -1.0>, 
                    <-1.45, -0.6>, <-0.2, 1.6>, <-0.1, 1.5>, // The next hole
                    <-0.1, 0.1>, <-1.3, -0.7>, <-1.45, -0.6>,
                    <1.45, -0.6>, <0.2, 1.6>, <0.1, 1.5>, // The last hole
                    <0.1, 0.1>, <1.3, -0.7>, <1.45, -0.6>
                }
                cone { 
                    <0, 0.0, 0>, 3.0,
                    <0, 0.4, 0>, 0.5
                }
            }
            cylinder { <0, -0.2, 0>, <0, 0, 0>, 1.4 }
            box { <-0.3, 0, 1>, <0.3, -0.2, 2.1> }
            box { 
                <-0.4, 0, 1>, <0.4, -0.2, 1.8> 
                rotate <0, 120, 0>
            }
            box { 
                <-0.4, 0, 1>, <0.4, -0.2, 1.8> 
                rotate <0, -120, 0>
            }
            cylinder {
                <0, 0, 2.1>,
                <0, 0, 2.5>,
                0.3
            }
            scale 0.11
            pigment { color White }
            rotate <0, 90, 0>
            translate <2.8, 0.4, 0>
        }
        // Struts holding it to the Science Boom
        union {
            cylinder { <2.74, 0.25, -0.15>, <2.74, 0.38, -0.15>, Strut_Width }
            cylinder { <2.74, 0.25, 0.15>, <2.74, 0.38, 0.15>, Strut_Width }
            cylinder { <2.96, 0.25, -0.15>, <2.88, 0.38, -0.12>, Strut_Width }
            cylinder { <2.96, 0.25, 0.15>, <2.88, 0.38, 0.12>, Strut_Width }
            pigment { color White }
        }
    }
}

#declare Cosmic_Ray_Subsystem_Detector = object {
    union {              
        // Main body 
        cylinder { <-0.025, 0, 0>, <0.025, 0, 0>, 0.02 }
        // Central shaft
        cylinder { <-0.04, 0, 0>, <0.04, 0, 0>, 0.007 }
        // The two end pieces
        difference {
            cylinder { <-0.035, 0, 0>, <-0.05, 0, 0>, 0.015 }
            cylinder { <-0.036, 0, 0>, <-0.055, 0, 0>, 0.01 }
        }
        difference {
            cylinder { <0.035, 0, 0>, <0.05, 0, 0>, 0.015 }
            cylinder { <0.036, 0, 0>, <0.055, 0, 0>, 0.01 }
        }
    }
}

#declare Cosmic_Ray_Subsystem = object {
    union {
        union {
            // main body
            difference {
                box { <-0.1, 0, -0.1>, <0.1, 0.15, 0.1> }
                box { <-0.08, 0.02, -0.11>, <0.08, 0.06, 0.11> }
                box { <-0.08, 0.09, -0.11>, <0.08, 0.13, 0.11> }
            }
            // interior of main body
            box { <0.09, 0.01, 0.09>, <-0.09, 0.14, -0.09> }
            // The top with pieces cut out of it
            difference {
                box { <-0.095, 0.15, -0.095>, <0.095, 0.2, 0.095> }
                box { <-0.11, 0.14, -0.11>, <-0.05, 0.21, -0.05> }
                box { <-0.11, 0.14, 0.11>, <-0.05, 0.21, 0.05> }
                box { <0.11, 0.14, 0.11>, <0.05, 0.21, 0.00> }
            }
            // The various detectors placed on the top of the unit
            object { 
                Cosmic_Ray_Subsystem_Detector 
                rotate <0, 45, 0>
                translate <-0.083, 0.18, -0.083>
            }
            object { 
                Cosmic_Ray_Subsystem_Detector 
                rotate <0, -45, 0>
                translate <-0.083, 0.18, 0.083>
            }
            object { 
                Cosmic_Ray_Subsystem_Detector 
                translate <0.08, 0.18, 0.02>
            }
            object { 
                Cosmic_Ray_Subsystem_Detector 
                rotate <0, 0, 90>
                translate <0.075, 0.18, 0.02>
            }
            object { 
                Cosmic_Ray_Subsystem_Detector 
                rotate <0, 30, 0>
                translate <0.083, 0.18, 0.083>
            }
            object { 
                Cosmic_Ray_Subsystem_Detector 
                rotate <0, 120, 0>
                translate <0.075, 0.18, 0.086>
            }
            //scale 3
            pigment { color White }
            //rotate <-90, 15, 0>
            translate <2.42, 0.38, 0>
        }
        union {
            cylinder { <2.30, 0.25, -0.15>, <2.34, 0.38, -0.08>, Strut_Width }
            cylinder { <2.30, 0.25, 0.15>, <2.34, 0.38, 0.08>, Strut_Width }
            cylinder { <2.52, 0.25, -0.15>, <2.48, 0.38, -0.08>, Strut_Width }
            cylinder { <2.52, 0.25, 0.15>, <2.48, 0.38, 0.08>, Strut_Width }
            pigment { color White }
        }
    }
}

#declare Low_Energy_Charged_Particles_Detector = object {
    union {
        union {
            box { <0, 0, -0.1>, <0.1, 0.2, 0.1> }
            // Hexagonal "Nut" shape - Due to small size this may not be implemented
            box { <0.1, 0.01, -0.09>, <0.12, 0.19, 0.09> }
            // Rest of the subsystem
            cylinder { <0.12, 0.09, 0>, <0.13, 0.09, 0>, 0.06 }
            box { <0.13, 0.03, -0.06>, <0.3, 0.15, 0.06> }
            sphere { <0.3, 0.09, 0>, 0.04 }
            sphere { <0.34, 0.09, 0>, 0.01 }
            cylinder { <0.18, 0.15, 0>, <0.18, 0.17, 0>, 0.02 }
            cylinder { <0.25, 0.15, 0>, <0.25, 0.17, 0>, 0.02 }
            pigment { color White }
            scale 0.8
            rotate < 180, 0, 0 >
            translate <2.3, -0.15, 0>
        }
        // Struts 
        union {                    
            cylinder { <2.30, 0, -0.15>, <2.36, -0.18, -0.05>, Strut_Width }
            cylinder { <2.30, 0, 0.15>, <2.36, -0.18, 0.05>, Strut_Width }
            cylinder { <2.52, 0, -0.15>, <2.48, -0.18, -0.01>, Strut_Width }
            cylinder { <2.52, 0, 0.15>, <2.48, -0.18, 0.01>, Strut_Width }
            pigment { color White }
        }
    }
}

#declare Canopus_Tracker = object {
    union {
        difference {
            union {
                cone { 
                    <1.25, 0.43, 0>, 0.08,
                    <1.32, 0.34, 0>, 0.06
                }
                cylinder { <1.32, 0.34, 0>, <1.33, 0.33, 0>, 0.08 }
            }
            cylinder { <1.32, 0.34, 0>, <1.33, 0.33, 0>, 0.05 }
        }
        // The strut
        cylinder { <1.25, 0.43, 0>, <0.822, -0.34, 0>, Strut_Width/2 }
    }
    pigment { color White }
    rotate <0, 90, 0>
}

#declare Gold_Record = object {
    cylinder {
        <0.967, -0.52, 0>,
        <0.968, -0.52, 0>,
        0.15
    }
    pigment { color Gold }
    finish { Phong_Glossy }
    rotate <0, 90, 0>
}
     
#declare Bus_Detail_Vertical_Grates = object {
    union {
        box { <0.967, -0.355, -0.305>, <0.98, -0.37, 0.305> } // Top piece
        box { <0.967, -0.51, -0.305>, <0.98, -0.53, 0.305> } // Central piece
        box { <0.967, -0.685, -0.305>, <0.98, -0.67, 0.305> } // Bottom piece
        box { <0.967, -0.355, -0.29>, <0.98, -0.67, -0.305> } // Left piece
        box { <0.967, -0.355, 0.29>, <0.98, -0.67, 0.305> } // Right piece
        // The grates
        #declare Count=1;
        #while (Count <= 12) 
            box { 
                <0, -0.685, 0>, 
                <0.01, -0.355, -0.01>
                rotate <0, 45, 0>
                translate <0.97, 0, -0.32 + Count * 0.05>
            }
            #declare Count = Count + 1;
        #end
    }
    pigment { color Silver }
    finish { Phong_Glossy }
}

#declare Bus_Detail_Half_Vertical_Grates = object {
    union {
        union {
            box { <0.967, -0.355, -0.305>, <0.98, -0.37, 0.305> } // Top piece
            box { <0.967, -0.51, -0.305>, <0.98, -0.53, 0.305> } // Bottom piece
            box { <0.967, -0.355, -0.29>, <0.98, -0.53, -0.305> } // Left piece
            box { <0.967, -0.355, 0.29>, <0.98, -0.53, 0.305> } // Right piece
            // The grates
            #declare Count=1;
            #while (Count <= 12) 
                box { 
                    <0, -0.51, 0>, 
                    <0.01, -0.355, -0.01>
                    rotate <0, 45, 0>
                    translate <0.97, 0, -0.32 + Count * 0.05>
                }
                #declare Count = Count + 1;
            #end
            pigment { color Silver }
            finish { Phong_Glossy }
        }
        // The bottom large rectangular shape
        box { 
            <0.967, -0.67, -0.29>, <0.97, -0.55, 0.29> 
            pigment { color Gray50 }
        }
    }
}     
     
#declare Bus_Detailing = object {
    union {
        object { 
            Bus_Detail_Half_Vertical_Grates 
            rotate <0, 54, 0>
        }
        object { 
            Bus_Detail_Vertical_Grates
            rotate <0, 126, 0>
        }
        object { 
            Bus_Detail_Vertical_Grates
            rotate <0, 234, 0>
        }
        object { 
            Bus_Detail_Vertical_Grates
            rotate <0, 306, 0>
        }
    }
}
 
#declare NASA_Voyager = object {
    union {
        object { High_Gain_Antenna_Dish }
        object { High_Gain_Antenna }
        object { High_Gain_Antenna_Collar }
        object { Subreflector_Dish }
        object { Subreflector_Tripod }
        object { Low_Gain_Antenna }
        object { Electronics_Bus }
        object { Bus_Struts }
        object { Propellant_Tank }
        object { Leg_Struts }
        object { Optical_Cal_Target }
        object { Optical_Cal_Target_Struts }
        object { Low_Field_Magnetometer }
        object { PRA_PWS }
        object { RTG_Struts }
        object { RTG } // Power unit
        object { Science_Boom }
        object { Science_Boom_Struts } 
        object { Science_Boom_Scan_Platform }
        object { IRIS } // Infrared Interferometer Spectrometer and Radiometer
        object { Narrow_Angle_Camera }
        object { Wide_Angle_Camera }
        object { Ultraviolet_Spectrometer }
        object { Photopolarimeter }
        object { Plasma_Investigation }
        object { Cosmic_Ray_Subsystem }
        object { Low_Energy_Charged_Particles_Detector }
        object { Canopus_Tracker }
        object { Gold_Record }
        object { Bus_Detailing }
        pigment { color White }
    } 
}
