/* spring.inc.  Makes smooth, continous helices with circular or elliptical
cross sections (aka. "springs"). Last modified Dec 17 1998  */

#declare Pi = 3.141592653589
#declare Count = 0


/*

   Basic constants:  Pitch = height change per complete turn
                     TurnTotal = total # of turns                  
                     SpringRadius = distance from spring axis to spring
                     Pitch, TurnTotal, Radius as before
 		     RVert, RHor:  radii in vertical (+y) and horizontal (+x,z)
                       directions


       
   Construction:     A spring is made of tilted cylinders with squared-off
                     ends (cross section is a parallelogram) 

*/



/*  Define textures... */

#declare SpringTexture = 
 texture {pigment {color Silver} 
          finish {ambient 0 diffuse 1 reflection 0.75 roughness 0.001 metallic}} 

/* These numbers should be set by the user. */
   
#declare Pitch = 0.08 / 12
#declare TurnTotal = 12
#declare SpringRadius = 0.0075
#declare RVert = 0.0006
#declare RHor = 0.0006

/* The following ones generally be left alone, except for HeightInc.
   Make HeightInc as large as is reasonably possible without introducing
   artifacts.  It'll need to be smaller if Pitch is small.  0.01 is a decent        starting place */

/* HeightInc = increment in height between successive helix / spring pieces */

#declare HeightInc = 0.0001

/* PerTurn is the number of such pieces per full turn   */

#declare PerTurn = Pitch/HeightInc

/* Alpha is the angle between pieces */

#declare Alpha = 360 / PerTurn

/* Total is the total number of pieces */

#declare Total = TurnTotal*PerTurn

/* PhiS is the angle the spring's "trace" makes with the horizontal */
 
#declare PhiS = degrees(atan2(Pitch, 2*Pi*SpringRadius))

/* calculate how wide each segment needs to be for a
   given radius and height increment   */

#declare CylWidth= 
 2*(SpringRadius+RHor)*tan(radians(Alpha/2)) 


/* Define the pieces of a spring ... */


/* Size of box needed to "square off" -- make parallel to the helix axis -- the ends of the cylinder, once the cylinder is rotated */

#declare BoxWidth = CylWidth
#declare BoxHeight = 2 * RVert / cos (PhiS)

/*Size of cylinder *just* large enough to have its ends clipped by the box */

#declare CylDiag = sqrt (BoxWidth*BoxWidth + BoxHeight*BoxHeight)
#declare CylLength = (sqrt (CylDiag*CylDiag - RVert*RVert))

#declare SpringSegment = intersection {
 box {<-BoxWidth/2, -BoxHeight/2, -RVert>, <BoxWidth/2, BoxHeight/2, RVert>}
     
 cylinder {<-CylLength/2, 0, 0>, <CylLength/2, 0, 0>, RVert
            rotate z*PhiS}

 bounded_by {
  box {<-BoxWidth/2, -BoxHeight/2, -RVert>, <BoxWidth/2, BoxHeight/2, RVert>}
            }
 scale <1,1,RHor/RVert> translate <0,0,SpringRadius>
 }

#declare Spring = union {
#declare Count=0
 #while (Count < Total)
  object {SpringSegment 
   rotate y*(Count*Alpha)            // to move along helix locus
   translate y*(Count*HeightInc) }   // to move to proper height
  #declare Count=Count+1
 #end
}
