#ifndef (beddetail)
  #declare beddetail=1;
#end

#local sponge=box {
  <-1.5,0,-2.5>,<1.5,.75,2.5>
  texture {
    pigment { color rgb <1,.5,.7> }
    normal { 
      leopard 2
      scallop_wave
      scale .02
      turbulence 3
    } 
    finish {
      diffuse .5
      ambient .2
      phong 0
      specular 0
    }
  }
}

// here begins the bedsheet

// xPatch is the number of patches to use in the x direction.  The 
// more patches, the more wrinkles will appear along the end of the
// object.                                           
#local xPatch=10;

// zPatch is the number of patches to use in the z direction.  The 
// more patches, the more wrinkles will appear along the sides of the
// object.                                           
#local zPatch=25;  
                              
// Corner1 and Corner2 define the corners of the surface being covered.
// If the surface has rounded corners, use Curve to indicate that and 
// set the corners to the edges of the flat part.
#local Corner1=<-1.5, 0, -1.0>;
#local Corner2=<1.5, .75, 2.5>;

// Curve is the radius of curvature of the "corners" along the top edges.
// It must be nonzero, and it should be a little bigger than the actual 
// curvature of the corners being covered.
#local Curve= .02;
#local TopWrink=.05;

// Puff governs how far the corners puff out as well as how wavy the sides are. 
// It is roughly a measure of the stiffness of the fabric.  If you increase it
// too far, though, you WILL force the cloth to intersect the covered box due
// to the differentiability constraints at patch boundaries.
#local Puff = .2;

// Clear is the clearance from the floor to the edge of the cloth.
#local Clear= .1; 

// Up is the space between the top surface of the cloth and the top surface
// of the supporting object.  This is mainly to avoid coincident surfaces.
#local Up   = .01;

// random seed for the waviness on the hanging parts of the cloth
#local s=seed(31252);

// Swing governs the side-to-side variation of the ridges along the sides, but
// it should not be modified independently of xPatch and zPatch.  You may modify
// the constant at the end, but don't go below about 6.
#local Swingx=(Corner2.x-Corner1.x)/xPatch/9;
#local Swingz=(Corner2.z-Corner1.z)/zPatch/9;

#local Points=array[xPatch*3+1][zPatch*3+1]

// Initialize the points on top of the box (surface points)

// A nice mod would be to add some wrinkles on top, but they would need
// to interact with the drape wrinkles and the differentiability 
// constraints at patch boundaries would have to be maintained by
// moving the boundary points around as well.  Wrinkles in the 
// corners would be particularly messy.

#if (beddetail)

  #local i=3;
  #while ( i <= xPatch*3-3 )
    #local j=0;
    #while ( j <= zPatch*3-3 )
      #declare Points[i][j] = <Corner1.x+(Corner2-Corner1).x*(i-3)/(xPatch*3-6),
                     Corner2.y+Up,
                     Corner1.z+(Corner2-Corner1).z*j/(zPatch*3-3)>;
      #local j=j+1;                   
    #end
    #local i=i+1;
  #end
  
  // Initialize the end boundary points - These help set the top edge curvature and they
  // enforce the differentiability criterion at the surface-drape boundary.
  #local i=3;
  #while ( i <= xPatch*3-3 ) 
    #declare Points[i][zPatch*3-2] = <Corner1.x+(Corner2-Corner1).x*(i-3)/(xPatch*3-6), 
                     Corner2.y+Up,
                     Corner2.z+Curve>;
    #local i=i+1;
  #end
  
  // Initialize the side boundary points
  #local i=0;
  #while ( i <= zPatch*3-3 ) 
    #declare Points[2][i] = <Corner1.x-Curve,
                   Corner2.y+Up,
                   Corner1.z+(Corner2-Corner1).z*i/(zPatch*3-3)>;
    #declare Points[xPatch*3-2][i] = <Corner2.x+Curve,
                            Corner2.y+Up,
                            Corner1.z+(Corner2-Corner1).z*i/(zPatch*3-3)>;
    #local i=i+1;
  #end     
  
  // Initialize the corner boundary points
  #declare Points[2][zPatch*3-2]=<Corner1.x-Curve,Corner2.y+Up,Corner2.z+Curve>;
  #declare Points[xPatch*3-2][zPatch*3-2]=<Corner2.x+Curve, Corner2.y+Up, Corner2.z+Curve>;
  
  // Initialize the end drape points - the upper drape points help set the top edge 
  // curvature, and the lower ones define the edge of the cloth.
  #local i=2;
  #while ( i <= xPatch*3-2 )
    // upper 
    #declare Points[i][zPatch*3-1] = 
                    <Points[i][zPatch*3-2].x, 
                     Corner2.y+Up-Curve,
                     Corner2.z+Curve>;
    // lower
    #declare Points[i][zPatch*3] = 
                    <Points[i][zPatch*3-2].x, 
                     Corner1.y+Clear,
                     Corner2.z+Curve>;
    #local i=i+1;
  #end
  
  // Initialize the side drape points
  #local i=0;
  #while ( i <= zPatch*3-2 ) 
    // lower -x
    #declare Points[0][i] = <Corner1.x-Curve,
                   Corner1.y+Clear,
                   Points[2][i].z>;
    // upper -x                 
    #declare Points[1][i] = <Corner1.x-Curve,
                             Corner2.y+Up-Curve,
                             Points[2][i].z>;
    // lower +x                           
    #declare Points[xPatch*3][i] = <Corner2.x+Curve,
                            Corner1.y+Clear,
                            Points[2][i].z>;
    // upper +x                          
    #declare Points[xPatch*3-1][i] = <Corner2.x+Curve,
                            Corner2.y+Up-Curve,
                            Points[2][i].z>;
    #local i=i+1;
  #end     
                   
  // Initialize the corner drape points
  
  /*
  
          E---E---B---S
          |   |   |   |
          |   |   |   |
          U---V---B---B
          |   |   |   |
          |   |   |   |
          X---X---V---E
          |   |   |   |
          |   |   |   |
          X---X---U---E
          
          E    - normal drape point.
          B    - normal boundary point.
          S    - normal surface point.
          U, V - normal "drape" point, already initialized but moved slightly away from the corner
                 to avoid coinciding with the drape point with the same letter on the same patch.
          X    - undetermined point; these are "folded" to provide a typical loose corner.  This 
                 forms a low-curvature fold that follows the B-V-U lines, with a "billow" away
                 from the corner of the covered box.  One nice mod would be some perturbation of
                 the X points to avoid having all the corners be identical and perfect.
  */
  
  // -x, +z corner
  // O points
  #declare Points[0][zPatch*3-2]=Points[0][zPatch*3-2]-Curve/2*z;
  #declare Points[1][zPatch*3-2]=Points[1][zPatch*3-2]-Curve/2*z;
  #declare Points[2][zPatch*3-1]=Points[2][zPatch*3-1]+Curve/2*x;
  #declare Points[2][zPatch*3]=Points[2][zPatch*3]+Curve/2*x;
  // X points
  #declare Points[0][zPatch*3-1]= <Corner1.x-Puff, Corner1.y+Clear, Corner2.z+Curve>;
  #declare Points[1][zPatch*3-1]= <Corner1.x-Puff/2, Corner2.y+Up-2*Curve, Corner2.z+Puff/2>;
  #declare Points[0][zPatch*3]= <Corner1.x-Puff, Corner1.y+Clear, Corner2.z+Puff>;
  #declare Points[1][zPatch*3]= <Corner1.x-Curve, Corner1.y+Clear, Corner2.z+Puff>;
  
  // +x, +z corner
  // O points
  #declare Points[xPatch*3][zPatch*3-2]=Points[xPatch*3][zPatch*3-2]-Curve/2*z;
  #declare Points[xPatch*3-1][zPatch*3-2]=Points[xPatch*3-1][zPatch*3-2]-Curve/2*z;
  #declare Points[xPatch*3-2][zPatch*3-1]=Points[xPatch*3-2][zPatch*3-1]-Curve/2*x;
  #declare Points[xPatch*3-2][zPatch*3]=Points[xPatch*3-2][zPatch*3]-Curve/2*x;
  // X points
  #declare Points[xPatch*3][zPatch*3-1]= <Corner2.x+Puff, Corner1.y+Clear, Corner2.z+Curve>;
  #declare Points[xPatch*3-1][zPatch*3-1]= <Corner2.x+Puff/2, Corner2.y+Up-2*Curve, Corner2.z+Puff/2>;
  #declare Points[xPatch*3][zPatch*3]= <Corner2.x+Puff, Corner1.y+Clear, Corner2.z+Puff>;
  #declare Points[xPatch*3-1][zPatch*3]= <Corner2.x+Curve, Corner1.y+Clear, Corner2.z+Puff>;
  
  // Perturb the end drape points
  // We do not perturb points within 5 of the draped corners to prevent creases at 
  // the corners due to the unperturbed and generally immovable points there.
  #local i=5;
  #while ( i <= xPatch*3-5 ) 
    #local randVect=<rand(s)*Swingx*2-Swingx, 0, rand(s)*(Puff/2+Curve)-Curve+Clear>;
    #if ( mod(i,3) = 1 )
      // This enforces differentiability at patch boundaries.  Without it, you get
      // sharp creases.
      #local randVect=<randVect.x, 0,
                       (Points[i][zPatch*3].x+randVect.x-Points[i-1][zPatch*3].x)/
                       (Points[i-1][zPatch*3].x-Points[i-2][zPatch*3].x)*
                       (Points[i-1][zPatch*3].z-Points[i-2][zPatch*3].z) - 
                       Points[i][zPatch*3].z + Points[i-1][zPatch*3].z
                       >;
    #end
    #declare Points[i][zPatch*3-1] = Points[i][zPatch*3-1]+randVect/3;
    #declare Points[i][zPatch*3] = Points[i][zPatch*3]+randVect;
    #local i=i+1;
  #end
  
  // Perturb the side drape points
  #local i=0;
  #while ( i <= zPatch*3-5 ) 
    #local randVect=<-(rand(s)*(Puff/2+Curve)-Curve+Clear), 0, -(rand(s)*Swingz*2-Swingz)>;
    #if ( i > 3 & mod(i,3) = 1 )
      // This is the same as above, but the i>3 is recognition of the fact that 
      // there is no differentiability constraint at exterior boundaries.
      #local randVect=<(Points[0][i].z+randVect.z-Points[0][i-1].z)/
                       (Points[0][i-1].z-Points[0][i-2].z)*
                       (Points[0][i-1].x-Points[0][i-2].x) - 
                       Points[0][i].x + Points[0][i-1].x,
                       0, randVect.z>;
    #end
    #declare Points[0][i] = Points[0][i]+randVect;
    #declare Points[1][i] = Points[1][i]+randVect/3;
    #local randVect=<(rand(s)*(Puff/2+Curve)-Curve+Clear), 0, (rand(s)*Swingz*2-Swingz)>;
    #if ( i > 3 & mod(i,3) = 1 )
      #local randVect=<(Points[xPatch*3][i].z+randVect.z-Points[xPatch*3][i-1].z)/
                       (Points[xPatch*3][i-1].z-Points[xPatch*3][i-2].z)*
                       (Points[xPatch*3][i-1].x-Points[xPatch*3][i-2].x) - 
                       Points[xPatch*3][i].x + Points[xPatch*3][i-1].x,
                       0, randVect.z>;
    #end
    #declare Points[xPatch*3][i] = Points[xPatch*3][i]+randVect;
    #declare Points[xPatch*3-1][i] = Points[xPatch*3-1][i]+randVect/3;
    #local i=i+1;
  #end     

  // Add a few wrinkles to the top
  
  #local c=0;
  #local d=(Corner2.z-Corner1.z)/(zPatch-1)/2;
  #while ( c < 50 )
    #local l=rand(s)*20+5;
    #local i=floor(rand(s)*(xPatch*3-6-l)+3);
    #local j=floor(rand(s)*(zPatch-3)+1)*3; 
    #local h=(rand(s)*TopWrink);
    #local k=0;
    #while (k<l & i+k<xPatch*3-3)
      #declare hh=h*pow((1-2*abs(k/l-.5)),2);
      #declare Points[i+k][j] = Points[i+k][j]+hh*y;
      #declare Points[i+k][j-1] = Points[i+k][j-1]+hh*y;
      #declare Points[i+k][j+1] = Points[i+k][j+1]+hh*y;
      #local k=k+1;
    #end
    #declare c=c+1;
  #end
  
  
  
  // Create the patches
  #declare bedspread=union {
  
    #local pI=0;
    #while ( pI < xPatch )
      #local pJ=0;
      #while ( pJ < zPatch )
        bicubic_patch {
          type 1
          u_steps 3 
          v_steps 3
          #local i=0;
          #while ( i < 4 )
            #local j=0;
            #while ( j < 4 )
              Points[pI*3+i][pJ*3+j]
              #local j=j+1;
            #end
            #local i=i+1;
          #end
        }
        #local pJ=pJ+1;
      #end
      #local pI=pI+1;
    #end
  
    /*
    // The following code draws the control mesh over a box representing the
    // covered surface.  It is mainly useful for debugging changes to the
    // mesh calculation code.  It might help to comment out the patch creation
    // loops above (everything from the #local pI=0 to here) if you plan to use 
    // the debug code.  This code is not optimal: it creates two of everything
    // on interior patch boundaries.
  
    #local pI=0;
    #while ( pI < xPatch )
      #local pJ=0;
      #while ( pJ < zPatch )
          #local i=0;
          #while ( i < 4 )
            #local j=0;
            #while ( j < 4 )
              sphere {Points[pI*3+i][pJ*3+j], .02 pigment {green 1 }}
              
              #if ( pI<xPatch-1 | i < 3 ) 
                cylinder {Points[pI*3+i][pJ*3+j] Points[pI*3+i+1][pJ*3+j] .01 
                  pigment {
                    #if ( mod(j,3)=0 ) 
                      rgb 1
                    #else
                      blue 1
                    #end
                  }
                }
              #end
              #if ( pJ<xPatch-1 | j < 3 )
                cylinder {Points[pI*3+i][pJ*3+j] Points[pI*3+i][pJ*3+j+1] .01
                  pigment {
                    #if ( mod(i,3)=0 ) 
                      rgb 1
                    #else
                      blue 1
                    #end
                  }
                }
              #end
              
              #local j=j+1;
            #end
            #local i=i+1;
          #end
        #local pJ=pJ+1;
      #end
      #local pI=pI+1;
    #end
  
    // These cylinders highlight the edges of the covered surface in red.
    cylinder { <Corner1.x,Corner1.y,Corner1.z>, <Corner2.x, Corner1.y, Corner1.z> .01 pigment {red 1}}
    cylinder { <Corner1.x,Corner1.y,Corner1.z>, <Corner1.x, Corner2.y, Corner1.z> .01 pigment {red 1}}
    cylinder { <Corner1.x,Corner1.y,Corner1.z>, <Corner1.x, Corner1.y, Corner2.z> .01 pigment {red 1}}
    cylinder { <Corner2.x,Corner2.y,Corner2.z>, <Corner2.x, Corner2.y, Corner1.z> .01 pigment {red 1}}
    cylinder { <Corner2.x,Corner2.y,Corner2.z>, <Corner2.x, Corner1.y, Corner2.z> .01 pigment {red 1}}
    cylinder { <Corner2.x,Corner2.y,Corner2.z>, <Corner1.x, Corner2.y, Corner2.z> .01 pigment {red 1}}
    cylinder { <Corner2.x,Corner2.y,Corner1.z>, <Corner1.x,Corner2.y,Corner1.z> .01 pigment {red 1}}
    cylinder { <Corner1.x,Corner2.y,Corner1.z>, <Corner1.x,Corner2.y,Corner2.z> .01 pigment {red 1}}
    cylinder { <Corner1.x,Corner2.y,Corner2.z>, <Corner1.x,Corner1.y,Corner2.z> .01 pigment {red 1}}
    cylinder { <Corner1.x,Corner1.y,Corner2.z>, <Corner2.x,Corner1.y,Corner2.z> .01 pigment {red 1}}
    cylinder { <Corner2.x,Corner1.y,Corner2.z>, <Corner2.x,Corner1.y,Corner1.z> .01 pigment {red 1}}
    cylinder { <Corner2.x,Corner1.y,Corner1.z>, <Corner2.x,Corner2.y,Corner1.z> .01 pigment {red 1}}
    box {Corner1,Corner2 pigment {rgb <1,1,0>}}
    no_shadow
    */
  
    scale <1,1,-1>
    texture {
      pigment {color rgb 1}
      finish {diffuse .5 phong .2 ambient .1 specular 0}
    }
  }
#else
  #declare bedspread=box{
    Corner1+<-Curve,Clear,0>, Corner2+<Curve,Up,Curve>
    scale <1,1,-1>
    texture {
      pigment {color rgb 1}
      finish {diffuse .5 phong .2 ambient .1 specular 0}
    }
  }
#end
#declare bed=union {               
  object {sponge}           
  object {bedspread}
} 

/*           
object {bed rotate 45*y}
plane {y,0 pigment {checker rgb 0, rgb 1 scale .5}}
light_source {<-20,20,-20> rgb 1}
light_source {2*y-5*z rgb 1 spotlight radius 15 falloff 30 point_at 0}
camera { location 2*y-5*z look_at 0}
*/  