/*************************************************************************
    FILE OUTPUT CITY GENERATOR MACRO FILE FOR PERSISTENCE OF VISION 3.x
**************************************************************************

Created by Chris Colefax, 1 February 1999
NOTE: This file is designed for use with City.inc, eg:

      #declare city_output_file = "MYCITY.INC"  // Default is City.fil
      #declare city_macro_file = "FILECITY.MCR"
      #include "CITY.INC"

 The file produces the same cities as the default City.mcr file (when the
 same options are used).  As the city is parsed, however, all the objects
 are written to a separate POV file, which can then be included into your
 scene *instead* of City.inc.  This reduces parsing time substantially, and
 is very useful when creating animations or designing building objects (as
 the city doesn't need to be regenerated for every render).

 CAUTION: the output file you declare is overwritten every time you use
 CITY.INC with this macro file, so make sure you are using the filename
 you want.

*************************************************************************/

// CHECK OPTIONS AND ASSIGN DEFAULTS
   #ifndef (city_seed) #local city_seed = 0; #end
   #declare _CT_rand1 = seed(city_seed); // For traffic
   #declare _CT_rand2 = seed(city_seed); // For pavement objects
   #declare _CT_rand3 = seed(city_seed); // For buildings
   #declare _CT_rand4 = seed(city_seed); // For building levels

   #ifndef (building_width) #declare building_width = 25; #end
   #ifndef (min_building_height) #declare min_building_height = building_width * .5; #end
   #ifndef (max_building_height) #declare max_building_height = building_width * 2; #end
   #ifndef (building_height_falloff) #declare building_height_falloff = 2; #end
   #ifndef (building_height_turb) #declare building_height_turb = .5; #end

   #ifndef (pavement_height) #declare pavement_height = .15; #end
   #ifndef (traffic_spacing) #declare traffic_spacing = 10; #end
   #ifndef (traffic_lanes) #declare traffic_lanes = 1; #end
   #ifndef (traffic_width) #declare traffic_width = 3; #end
   #declare street_width = traffic_width*2*(traffic_lanes + .5);

   #ifndef (city_output_file) #declare city_output_file = "CITY.FIL" #end

   #ifndef (city_default_objects) #declare city_default_objects = true; #end
   #ifndef (building_types) #declare building_types = 0; #end

   #if (city_default_objects)
      #include "DEFAULT.OBJ"
      #if (traffic_spacing > 0) #include "VEHICLES.OBJ" #end
      #include "HOTELS.OBJ" #include "FLATS.OBJ" #include "OFFICES.OBJ"
      #declare _CT_writeincludes = true;
   #else
      #declare _CT_writeincludes = false;
   #end

// CITY CREATION MACROS
   #macro city_base (Corner1, Corner2)
      #declare _CT_extents = (abs(city_corner1.x) + abs(city_corner1.z)) / 2;
      #if (debug_progress) #debug concat ("  Opening \"", city_output_file, "\" for city output...\n") #end
      #fopen _CT_file city_output_file write
      #write (_CT_file, "// CITY GENERATOR INCLUDE FILE: File output using FileCity.mcr\n\n")

      #if (_CT_writeincludes) #write (_CT_file, "   #ifndef (city_default_objects) #include \"DEFAULT.OBJ\"  ")
         #if (traffic_spacing > 0) #write (_CT_file, "#include \"VEHICLES.OBJ\"  ") #end
         #write (_CT_file, "#include \"HOTELS.OBJ\"  #include \"FLATS.OBJ\"  #include \"OFFICES.OBJ\" #end\n\n")
      #end

      #write (_CT_file, "   plane {y, 0 hollow ",
         "clipped_by {box {",<Corner1.x, -.1, Corner1.z>,",",<Corner2.x, .1, Corner2.z>,"}} ",
         "texture {street_texture}}\n")
   #end

   #macro city_pavement (Corner1, Corner2)
      #write (_CT_file, "   box {",Corner1,",",Corner2 + y*pavement_height," texture {pavement_texture}}\n")

      #ifdef (pavement_objects) #local PSize = Corner2 - Corner1;
      #local V = 0; #while (V < dimension_size(pavement_objects, 1))
         #local P = pavement_object_offsets[V] + rand(_CT_rand2)*pavement_object_turb[V]; #while (P < PSize.x - pavement_object_offsets[V])
            #write (_CT_file, "   object {pavement_objects[",V,"] rotate y*90 translate ",Corner1 + <P, pavement_height, 0>,"} ",
            "object {pavement_objects[",V,"] rotate y*270 translate ",Corner2 + <-P, pavement_height, 0>,"}\n")
         #if (pavement_object_spacings[V] <= 0) #local P = PSize.x; #else
         #local P = P + pavement_object_spacings[V] + rand(_CT_rand2)*pavement_object_turb[V]; #end #end

         #local P = pavement_object_offsets[V] + rand(_CT_rand2)*pavement_object_turb[V]; #while (P < PSize.z - pavement_object_offsets[V])
            #write (_CT_file, "   object {pavement_objects[",V,"] translate ",<Corner2.x, pavement_height, Corner1.z + P>,"} ",
            "object {pavement_objects[",V,"] rotate y*180 translate ",<Corner1.x, pavement_height, Corner2.z - P>,"}\n")
         #if (pavement_object_spacings[V] <= 0) #local P = PSize.z; #else #local P = P + pavement_object_spacings[V]; #end #end
      #local V = V + 1; #end #end
   #end

   #macro city_traffic ()
      #local V = dimension_size(city_vehicles, 1);
      #if (Direction = 0) #local I = BlockSize.x; #else #local I = BlockSize.z; #end #local I = I + street_width;
      #local L = -traffic_lanes; #while (L <= traffic_lanes)
         #local VD = (L < 0 ? -1 : 1); #local VZ = From.z + traffic_width*VD*(abs(L)-.5);
         #local VX = From.x + street_width + rand(_CT_rand1)*traffic_spacing; #while (VX < To.x - (city_tileable ? 0 : street_width))
            #local VT = int(V*rand(_CT_rand1));
            #write (_CT_file, "   object {city_vehicles[",VT,"] rotate ",y*90*VD," translate ",<VX, 0, VZ>," rotate ",-y*90*Direction,"}\n")
         #local VX = VX + city_vehicle_spacings[VT] + (1 + 2*rand(_CT_rand1))*traffic_spacing;
         #if (mod(VX - From.x, I) < street_width) #local VX = VX + street_width; #end #end
      #local L = L + (L = -1 ? 2 : 1); #end
   #end

   #macro city_street (From, To, Direction)
      #if (Direction != 0) #local From = vrotate(From, y*90*Direction); #local To = vrotate(To, y*90*Direction); #end

      #ifdef (street_overlay) #write (_CT_file, "   box {",From - <0, 0, street_width*.5>,",",To + <0, (Direction+1)*.001, street_width*.5>,
         " texture {street_overlay scale ",traffic_width," translate ",From,"}",
         " rotate ",-y*90*Direction,"}\n") #end

      #if (traffic_spacing > 0 & defined(city_vehicles)) city_traffic () #end
   #end

   #macro city_building (Corner1, Corner2, Direction)
      #if (building_types > 0 & defined(building_size))
         #local BCentre = (Corner1 + Corner2)/2;
         #local BHeight = pow(1.2 - (vlength(BCentre) / _CT_extents), building_height_falloff);
         #local BHeight = BHeight + (rand(_CT_rand3) - .5)*building_height_turb;
         #local BHeight = min_building_height + BHeight*(max_building_height - min_building_height);
         #if (BHeight > 0) city_assemble_building (int(building_types*rand(_CT_rand3)), (Corner2-Corner1) + y*BHeight, -y*90*Direction, BCentre + y*pavement_height) #end
      #end
   #end

   #macro city_option (Option) (mod(building_options[Type], Option*2) >= Option ? true : false) #end
   #macro city_assemble_building (Type, Size, Rotate, Translate)
      #local Levels = ceil((Size.y / Size.x) / (building_size[Type].y / building_size[Type].x));
      #local LHeight = y*building_size[Type];
      #local Scale = Size/building_size[Type]; #local Scale = <Scale.x, (Scale.x+Scale.z)/2, Scale.z>;
      #local CLevel = 0;

      #write (_CT_file, "   union {\n")
         #if (city_option(build_profile)) #write (_CT_file, "   object {building_profile[",Type,"] scale ",<1, Levels, 1>,"}\n") #end
         #if (city_option(build_base)) #write (_CT_file, "   object {building_base[",Type,"]}\n") #local CLevel = 1; #end
         #if (city_option(build_details)) #while (CLevel < Levels)
            #write (_CT_file, "   object {building_details[",Type,"] translate ",LHeight*CLevel,"}\n")
            #local CLevel = CLevel + 1; #end #end

         #if (city_option(build_window_profile)) #write (_CT_file, "   object {building_windows[",Type,"] scale ",<1, Levels, 1>," texture {building_window_texture[",Type,"] ",
            "translate ",<int(rand(_CT_rand4)*100)-50, int(rand(_CT_rand4)*100), int(rand(_CT_rand4)*100)-50>*building_window_size[Type],"}}\n") #end
         #if (city_option(build_window_levels)) #local CLevel = 0; #while (CLevel < Levels)
            #write (_CT_file, "   object {building_windows[",Type,"] texture {building_window_texture[",Type,"] ",
            "translate ",<int(rand(_CT_rand4)*100)-50, int(rand(_CT_rand4)*100), int(rand(_CT_rand4)*100)-50>*building_window_size[Type],"} ",
            "translate ",LHeight*CLevel,"}\n")
            #local CLevel = CLevel + 1; #end #end

         #if (city_option(build_roof)) #write (_CT_file, "   object {building_roof[",Type,"] translate ",LHeight*Levels,"}\n")
            #else #write (_CT_file, "   sphere {0, 0 pigment {rgb 0}}\n") #end

         #if (city_option(build_fit_texture)) #write (_CT_file, "   texture {building_texture[",Type,"]}\n") #end
         #write (_CT_file, "   rotate ",Rotate," scale ",Scale," translate ",Translate,"\n")
         #if (city_option(build_texture)) #write (_CT_file, "   texture {building_texture[",Type,"] scale ",Scale,"}\n") #end
         #write (_CT_file, "   }\n\n")
   #end

   #macro city_finish ()
      #if (debug_progress) #debug concat("  Closing \"", city_output_file, "\"...") #end
      #fclose _CT_file  #include city_output_file
      #if (debug_progress) #debug concat("\r  City has been saved to \"", city_output_file, "\"!\n") #end
   #end
