// halls.inc
// Opera House Halls definitions

// development switches
#declare ShowCopper = 0;
#declare ShowGlass = 1;
#declare TransparentGlass = 1;
#declare ShowShells = 1;

// Building dimension constants
// Hall shells:
// Shell A - Largest north-facing shell
// Shell B - Next north-facing shell
// Shell C - Smallest north-facing shell
// Shell D - South-facing shell
// Shell E - Minor shell between A and B
// Shell F - Minor shell between B and C
// Restaurant shells:
// Shell G - North-facing restaurant shell
// Shell H - South-facing restaurant shell

#declare Radius = 75;
#declare GlassSize = 200;
#declare Fudge = 0.85;
// Hall shell centres
#declare NumShells = 6;
#declare XCs = array[NumShells] { -45.01, -38.92, -38.69, -58.16, -32, -31 }
#declare YCs = array[NumShells] {      0, -18.85, -29.89, -14.32,  -5, -20 }
#declare ZCs = array[NumShells] {      0, -13.41, -23.04,  81.63,  73,  44 }
// Hall shell angles
#declare Phi1s   = array[NumShells] { 22.88, 20.66, 20.72,  6.95, -38, -17 }
#declare Theta1s = array[NumShells] {  8.20,  7.00, 18.00, 41.80,  30,  35 }
#declare Phi2s   = array[NumShells] { 19.58, 10.24, -3.04, -8.89, -40, -23.8 }
#declare Theta2s = array[NumShells] { -56.5, -24.5, -21.0, -22.4,  13,  20 }

// Copper ribbing z-axis locations for shells A and B
#declare Ribs = array[2] { 28, 7.5 }

// Glass wall constants
#declare GlassThickness = 0.019;
#declare NumGlassPanels = 16;
#declare GlassConeAngle1 = 36;
#declare GlassConeAngle1a = 35;
#declare GlassConeAngle2 = 72;
#declare GlassConeAngle2a = 71;
#declare GlassConeAngle3 = -45;
#declare GlassConeZrot1 = 0.8;
#declare GlassConeYtrans1 = 5;
#declare GlassConeYtrans2 = 9;
#declare GlassConeYtrans3 = 9;
#declare GlassRayAngles = array[NumGlassPanels] { 2.5, 7.8, 13.0, 18.0, 22.9, 27.6, 32.0, 36.1,
                                                40.2, 44.0, 47.7, 51.0, 54.6, 58.5, 62.5, 66.5 }
#declare GlassWallAngles = array[NumGlassPanels]
#declare GlassWallDistances = array[NumGlassPanels]
#declare GlassWallAngles[0] = 0.0;
#declare GlassWallDistances[0] = 9.6;
#local i = 1;
#while (i < NumGlassPanels)
    #declare GlassWallAngles[i] = 20.0 + (i-1)*4/(NumGlassPanels-2);
    #declare GlassWallDistances[i] = 9.8 + (i-1)*0.6/(NumGlassPanels-2);
    #local i = i + 1;
#end

// Textures
#declare CopperTexture = texture {
    pigment { bozo
        colour_map {
            [0.0 rgb <0,0.5,0.4>]
            [0.7 rgb <0.1,0.5,0.4>]
            [0.8 rgb <0.5,0.5,0.4>]
            [1.0 rgb <0.5,0.5,0.4>]
        }
        turbulence 0.7
        translate <0,10,0>
        scale 6
    }
    normal { gradient x, 1
        triangle_wave
        slope_map {
            [0.0 <0, 0>]
            [0.7 <0, 0>]
            [0.7 <0, 3>]
            [0.8 <1, 3>]
            [0.8 <1, 0>]
            [0.9 <1, 0>]
            [0.9 <1, -3>]
            [1.0 <0, -3>]
            [1.0 <0, 0>]
        }
        scale 1
    }
    finish {
        metallic
    }
}
#declare GlassMaterial = material {
    texture {
        pigment { rgbt <1,0.9,0.85,0.95> }
        finish {
            reflection 0.2
        }
    }
    interior {
        ior 1.4
    }
}
#declare GlassDevMaterial = material {
    texture {
        pigment { rgb <0.8,0.7,0.6> }
        finish { ambient 0.1 }
    }
}

#macro Hall(Yrot, Size, Xtrans, Ytrans, Ztrans, HollowShells, Bubbles)
union {
    union {
        #if (ShowShells)
        // shells
        #local i = 0;
        #while (i < NumShells)
            Shell(XCs[i], YCs[i], ZCs[i], Radius, Phi1s[i], Theta1s[i], Phi2s[i], Theta2s[i], HollowShells, Bubbles)
            #local i = i + 1;
        #end
        #end
    
        #if (ShowCopper)
        // copper ribbing inside shells A and B
        #local i = 0;
        #while (i < 2)
            intersection {
                box { <Radius, Radius, 0> <-Radius, 0, 0.1>
                    rotate <-7, 0, 0>
                    translate <0, 0, Ribs[i]>
                }
                sphere { <XCs[i], YCs[i], ZCs[i]>, Radius-SkinThickness }
                sphere { <-XCs[i], YCs[i], ZCs[i]>, Radius-SkinThickness }
                texture { CopperTexture }
            }
            #local i = i + 1;
        #end
        #end
        scale Fudge
    }

    #if (ShowGlass)
    // glass wall inside shell C
    intersection {
        // all the glass
        union {
            // central panes
            intersection {
                union {
                    // vertical pane
                    intersection {
                        box { <GlassSize, GlassSize, 0> <-GlassSize, 0, GlassThickness>
                            translate <0, 0, -GlassWallDistances[0]>
                        }
                        plane { z, 0
                            rotate <GlassConeAngle1, 0, 0>
                            translate <0, GlassConeYtrans1, -2*GlassWallDistances[0]>
                        }
                    }
                    // first cone pane
                    intersection {
                        box { <GlassSize, GlassSize, 0> <-GlassSize, 0, GlassThickness>
                            rotate <GlassConeAngle1, 0, 0>
                            translate <0, GlassConeYtrans1, -2*GlassWallDistances[0]>
                        }
                        plane { z, 0
                            translate <0, 0, -GlassWallDistances[0]>
                        }
                        plane { z, 0
                            rotate <GlassConeAngle2, 0, 0>
                            translate <0, GlassConeYtrans2, -3*GlassWallDistances[0]>
                        }
                    }
                    // second cone pane
                    intersection {
                        box { <GlassSize, GlassSize, 0> <-GlassSize, 0, GlassThickness>
                            rotate <GlassConeAngle2, 0, 0>
                            translate <0, GlassConeYtrans2, -3*GlassWallDistances[0]>
                        }
                        plane { z, 0
                            rotate <GlassConeAngle1, 0, 0>
                            translate <0, GlassConeYtrans1, -2*GlassWallDistances[0]>
                        }
                        plane { -z, 0
                            rotate <GlassConeAngle3, 0, 0>
                            translate <0, GlassConeYtrans3, -1.3*GlassWallDistances[0]>
                        }
                    }
                    // third cone pane
                    intersection {
                        box { <GlassSize, GlassSize, 0> <-GlassSize, 0, GlassThickness>
                            rotate <GlassConeAngle3, 0, 0>
                            translate <0, GlassConeYtrans3, -1.3*GlassWallDistances[0]>
                        }
                        plane { -z, 0
                            rotate <GlassConeAngle2, 0, 0>
                            translate <0, GlassConeYtrans2, -3*GlassWallDistances[0]>
                        }
                    }
                }
                // intersect off adjoining panes to east and west
                plane { x, 0
                    rotate <0, -GlassRayAngles[0], 0>
                }
                plane { -x, 0
                    rotate <0, GlassRayAngles[0], 0>
                }
            }
            // panes around edges on each side
            #local i = 1;
            #while (i < NumGlassPanels)
                // eastern side
                intersection {
                    union {
                        // vertical pane
                        intersection {
                            box { <Radius, Radius, 0> <-Radius, 0, GlassThickness>
                                rotate <0, GlassWallAngles[i], 0>
                                translate <0, 0, -GlassWallDistances[i]>
                            }
                            plane { z, 0
                                rotate <GlassConeAngle1a, 0, 0>
                                rotate <0, 0, GlassConeZrot1*i>
                                rotate <0, GlassWallAngles[i], 0>
                                translate <0, GlassConeYtrans1, -2*GlassWallDistances[i]>
                            }
                        }
                        // first cone pane
                        intersection {
                            box { <Radius, Radius, 0> <-Radius, 0, GlassThickness>
                                rotate <GlassConeAngle1a, 0, 0>
                                rotate <0, 0, GlassConeZrot1*i>
                                rotate <0, GlassWallAngles[i], 0>
                                translate <0, GlassConeYtrans1, -2*GlassWallDistances[i]>
                            }
                            plane { z, 0
                                rotate <0, GlassWallAngles[i], 0>
                                translate <0, 0, -GlassWallDistances[i]>
                            }
                            plane { z, 0
                                rotate <GlassConeAngle2a, 0, 0>
                                //rotate <0, 0, GlassConeZrot1*i>
                                rotate <0, GlassWallAngles[i], 0>
                                translate <0, GlassConeYtrans2, -3*GlassWallDistances[i]>
                            }
                        }
                        // second cone pane
                        intersection {
                            box { <Radius, Radius, 0> <-Radius, 0, GlassThickness>
                                rotate <GlassConeAngle2a, 0, 0>
                                //rotate <0, 0, GlassConeZrot1*i>
                                rotate <0, GlassWallAngles[i], 0>
                                translate <0, GlassConeYtrans2, -3*GlassWallDistances[i]>
                            }
                            plane { z, 0
                                rotate <GlassConeAngle1a, 0, 0>
                                rotate <0, 0, GlassConeZrot1*i>
                                rotate <0, GlassWallAngles[i], 0>
                                translate <0, GlassConeYtrans1, -2*GlassWallDistances[i]>
                            }
                            plane { -z, 0
                                rotate <GlassConeAngle3, 0, 0>
                                //rotate <0, 0, GlassConeZrot1*i>
                                rotate <0, GlassWallAngles[i], 0>
                                translate <0, GlassConeYtrans3, -1.3*GlassWallDistances[i]>
                            }
                        }
                        // third cone pane
                        intersection {
                            box { <Radius, Radius, 0> <-Radius, 0, GlassThickness>
                                rotate <GlassConeAngle3, 0, 0>
                                //rotate <0, 0, GlassConeZrot1*i>
                                rotate <0, GlassWallAngles[i], 0>
                                translate <0, GlassConeYtrans3, -1.3*GlassWallDistances[i]>
                            }
                            plane { -z, 0
                                rotate <GlassConeAngle2a, 0, 0>
                                //rotate <0, 0, GlassConeZrot1*i>
                                rotate <0, GlassWallAngles[i], 0>
                                translate <0, GlassConeYtrans2, -3*GlassWallDistances[i]>
                            }
                        }
                    }
                    // intersect off adjoining panes to east and west
                    plane { x, 0
                        rotate <0, GlassRayAngles[i-1], 0>
                    }
                    plane { -x, 0
                        rotate <0, GlassRayAngles[i], 0>
                    }
                }
                // western side
                intersection {
                    union {
                        // vertical pane
                        intersection {
                            box { <Radius, Radius, 0> <-Radius, 0, GlassThickness>
                                rotate <0, -GlassWallAngles[i], 0>
                                translate <0, 0, -GlassWallDistances[i]>
                            }
                            plane { z, 0
                                rotate <GlassConeAngle1a, 0, 0>
                                rotate <0, 0, -GlassConeZrot1*i>
                                rotate <0, -GlassWallAngles[i], 0>
                                translate <0, GlassConeYtrans1, -2*GlassWallDistances[i]>
                            }
                        }
                        // first cone pane
                        intersection {
                            box { <Radius, Radius, 0> <-Radius, 0, GlassThickness>
                                rotate <GlassConeAngle1a, 0, 0>
                                rotate <0, 0, -GlassConeZrot1*i>
                                rotate <0, -GlassWallAngles[i], 0>
                                translate <0, GlassConeYtrans1, -2*GlassWallDistances[i]>
                            }
                            plane { z, 0
                                rotate <0, -GlassWallAngles[i], 0>
                                translate <0, 0, -GlassWallDistances[i]>
                            }
                            plane { z, 0
                                rotate <GlassConeAngle2a, 0, 0>
                                //rotate <0, 0, -GlassConeZrot1*i>
                                rotate <0, -GlassWallAngles[i], 0>
                                translate <0, GlassConeYtrans2, -3*GlassWallDistances[i]>
                            }
                        }
                        // second cone pane
                        intersection {
                            box { <Radius, Radius, 0> <-Radius, 0, GlassThickness>
                                rotate <GlassConeAngle2a, 0, 0>
                                //rotate <0, 0, -GlassConeZrot1*i>
                                rotate <0, -GlassWallAngles[i], 0>
                                translate <0, GlassConeYtrans2, -3*GlassWallDistances[i]>
                            }
                            plane { z, 0
                                rotate <GlassConeAngle1a, 0, 0>
                                rotate <0, 0, -GlassConeZrot1*i>
                                rotate <0, -GlassWallAngles[i], 0>
                                translate <0, GlassConeYtrans1, -2*GlassWallDistances[i]>
                            }
                            plane { -z, 0
                                rotate <GlassConeAngle3, 0, 0>
                                //rotate <0, 0, -GlassConeZrot1*i>
                                rotate <0, -GlassWallAngles[i], 0>
                                translate <0, GlassConeYtrans3, -1.3*GlassWallDistances[i]>
                            }
                        }
                        // third cone pane
                        intersection {
                            box { <Radius, Radius, 0> <-Radius, 0, GlassThickness>
                                rotate <GlassConeAngle3, 0, 0>
                                //rotate <0, 0, -GlassConeZrot1*i>
                                rotate <0, -GlassWallAngles[i], 0>
                                translate <0, GlassConeYtrans3, -1.3*GlassWallDistances[i]>
                            }
                            plane { -z, 0
                                rotate <GlassConeAngle2a, 0, 0>
                                //rotate <0, 0, -GlassConeZrot1*i>
                                rotate <0, -GlassWallAngles[i], 0>
                                translate <0, GlassConeYtrans2, -3*GlassWallDistances[i]>
                            }
                        }
                    }
                    // intersect off adjoining panes to east and west
                    plane { -x, 0
                        rotate <0, -GlassRayAngles[i-1], 0>
                    }
                    plane { x, 0
                        rotate <0, -GlassRayAngles[i], 0>
                    }
                }
                #local i = i + 1;
            #end
            scale 1.2
            translate <0,-5,-10>
        }
        // intersect off the shell roof so it doesn't stick out the top!
        union {
            intersection {
                sphere { <XCs[2], YCs[2], ZCs[2]>, Radius-SkinThickness }
                sphere { <-XCs[2], YCs[2], ZCs[2]>, Radius-SkinThickness }
                scale Fudge
            }
            plane { y, 17 }
        }

        #if (TransparentGlass)
            material { GlassMaterial }
        #else
            material { GlassDevMaterial }
        #end
    }
    #end // #if (ShowGlass)

    // rotate, scale and translate into position
    rotate <0,Yrot,0>
    scale Size
    translate <Xtrans, Ytrans, Ztrans>
}
#end
