// homestead.inc
// A run-down homestead
// Uses Chris Colefax's Bend.inc file

#declare Texture_Dirt = texture {
    pigment { bozo
        colour_map {
            [ 0.0 rgbt <1,1,1,1> ]
            [ 1.0 rgbt <0.4,0.3,0.1,0> ]
        }
        turbulence 0.7
        omega 0.8
        scale 0.7
    }
    normal { granite 0.3
        turbulence 0.2
    }
}

#declare Julia = julia_fractal { <-0.083,0.2,0.83,0.25>
    max_iteration 8
    precision 15
}

#declare Ellip = superellipsoid { <0.5,0.5>
}

// straw macro
#macro Straw(x1,y1,z1,Stalk,Height)
    #local x2 = (rand(Rseed1)-0.5)*0.3;
    #local y2 = rand(Rseed1)*Height;
    #local z2 = (rand(Rseed1)-0.5)*0.3;
    cylinder { <0,0,0>, <x2,y2,z2> Stalk
        texture {
            pigment { rgb <1,0.8,0.5> }
        }
        translate <x1,y1,z1>
    }
#end // macro Straw

// MAIN MACRO
#macro Homestead(xTrans,yTrans,zTrans)

// defining constants
#local BoxLength = 8;
#local BoxDepth = 6;
#local BoxHeight = 2.5;
#local SkirtHeight = 0.5;
#local SkirtWidth = 0.05;
#local DoorWidth = 1;
#local DoorHeight = 2;
#local WindowWidth = 0.8;
#local WindowHeight = 0.6;
#local BaseHeight = 0.8;
#local BaseDepth = 1.2;
#local EaveHeight = 0.1;
#local EaveWidth = 0.15;
#local InsRadius = 0.05;
#local GutterWidth = 0.1;
#local GutterHeight = 0.1;
#local RoofHeight = 1.8;
#local RoofLength = 4;
#local RoofPipe = 0.025;
#local ChimneyPos = 2.8;
#local ChimneySize = 0.4;
#local ChimneyHeight = 1.8;
#local ChimneyEave = 0.05;
#local ChimneyEaveHeight = 0.1;
#local TankHeight = 2.4;
#local TankRadius = 0.5;
#local TankPosX1 = 0.3;
#local TankPosX2 = 1.5;
#local TankPosZ = 3;
#local StepDepth = 0.25;
#local NumSteps = 5;
#local RailWidth = 0.2;
#local StrawStalk = 0.005; // radius of Straw stalk
#local Rseed1 = seed(2710);
#local Rseed2 = seed(3013);
#local NumStraws = 900; // 1000
#local StrawWidth = 1.5;
#local StrawDepth = 2;
#local NumRocks = 200;
#local MetalX = 2.2;
#local MetalZ = -5.5;
#local MetalRot = -30;

// derived constants
#local BaseY = BaseHeight;
#local BoxX = BoxLength / 2;
#local BoxZ = BoxDepth / 2;
#local BoxY = BoxHeight + BaseY;
#local SkirtY = BaseY + SkirtHeight;
#local SkirtX = BoxX + SkirtWidth;
#local SkirtZ = BoxZ + SkirtWidth;
#local BaseZ = -BoxZ - BaseDepth;
#local DoorX1 = DoorWidth / 2;
#local DoorX2 = (BoxLength + DoorWidth) / 4 - WindowWidth / 2;
#local DoorX3 = (BoxLength + DoorWidth) / 4 + WindowWidth / 2;
#local DoorY = BaseY + DoorHeight;
#local WindowY = BaseY + WindowHeight;
#local EaveY = BoxY + EaveHeight;
#local EaveX = BoxX + EaveWidth;
#local EaveZ = BoxZ + EaveWidth;
#local GutterY = EaveY + GutterHeight;
#local GutterX = EaveX + GutterWidth;
#local GutterZ = EaveZ + GutterWidth;
#local RoofX = RoofLength / 2;
#local RoofY = sqrt(RoofHeight*RoofHeight + EaveZ*EaveZ);
#local RoofY2 = RoofY * (2/3);
#local ChimneyX1 = ChimneyPos - ChimneySize/2;
#local ChimneyX2 = ChimneyPos + ChimneySize/2;
#local ChimneyY = GutterY + ChimneyHeight;
#local ChimneyZ = ChimneySize/2;
#local ChimneyEaveX1 = ChimneyX1 - ChimneyEave;
#local ChimneyEaveX2 = ChimneyX2 + ChimneyEave;
#local ChimneyEaveY = ChimneyY + ChimneyEaveHeight;
#local ChimneyEaveZ = ChimneyZ + ChimneyEave;
#local TankX1 = BoxX + TankPosX1;
#local TankX2 = BoxX + TankPosX2;
#local TankZ = BoxZ + TankPosZ;
#local StepHeight = BaseHeight / NumSteps;
#local RailX = DoorX1 + RailWidth;
#local RailY = BaseY + StepHeight;
#local RailZ = BaseZ - (NumSteps + 1) * StepDepth;
#local StrawX = BoxX + StrawWidth;
#local StrawZ = BaseZ - StrawDepth;
#local RockX = BoxX * 3;
#local RockZ = BaseZ * 3;

union {
    // base
    // box { <-BoxX, 0, BaseZ>, <BoxX, BaseY, BoxZ>
    superellipsoid { <0.05,0.03>
        rotate <0,90,0>
        scale <BoxX, BaseY, -BaseZ>
        texture {
            pigment {
                image_map { png "stonewall.png"
                }
                scale BoxX
                rotate <45,0,0>
            }
            normal { granite 0.3
                turbulence 0.2
            }
        }
        texture { Texture_Dirt }
    }
    // steps
    union {
        #local i = 0;
        #while (i < NumSteps)
            #local StepZ = BaseZ - (NumSteps - i) * StepDepth;
            #local StepY1 = i * StepHeight;
            #local StepY2 = (i + 1) * StepHeight;
            #if (i = 0)
                box { <-RailX-RailWidth, StepY1, StepZ>, <RailX+RailWidth, StepY2, BaseZ> }
            #else
                box { <-RailX, StepY1, StepZ>, <RailX, StepY2, BaseZ> }
            #end
            #local i = i + 1;
        #end
        // step rails
        #declare bend_object = intersection {
            box { <DoorX1, 0, RailZ>, <RailX, RailY, BaseZ> }
            plane { <0, NumSteps*StepDepth, -BaseY>,
                -(RailZ - StepDepth) * sin(atan2(BaseY, NumSteps*StepDepth))
            }
        }
        #declare object_axis1 = <(DoorX1+RailX)/2, 0, BaseZ>;
        #declare object_axis2 = <(DoorX1+RailX)/2, 0, RailZ>;
        #declare bend_angle = 90;
        #declare bend_direction = x;
        #declare bend_start = 0.5;
        #include "Bend.inc"
        #declare bend_object = intersection {
            box { <-DoorX1, 0, RailZ>, <-RailX, RailY, BaseZ> }
            plane { <0, NumSteps*StepDepth, -BaseY>,
                -(RailZ - StepDepth) * sin(atan2(BaseY, NumSteps*StepDepth))
            }
        }
        #declare object_axis1 = <-(DoorX1+RailX)/2, 0, BaseZ>;
        #declare object_axis2 = <-(DoorX1+RailX)/2, 0, RailZ>;
        #declare bend_angle = 90;
        #declare bend_direction = -x;
        #declare bend_start = 0.5;
        #include "Bend.inc"
        texture {
            pigment { rgb <1,0.6,0.1>
            }
            normal { granite 0.3
                turbulence 0.2
            }
        }
        texture { Texture_Dirt }
    }
    // main box
    difference {
        union {
            //box { <-BoxX, BaseY, -BoxZ>, <BoxX, BoxY, BoxZ>
            superellipsoid { <0.06,0.03>
                rotate <0,90,0>
                scale <BoxX, (BoxY-BaseY)/2, BoxZ>
                translate <0, (BoxY+BaseY)/2, 0>
                texture {
                    pigment {
                        image_map { png "stonewall.png"
                        }
                        scale BoxX
                    }
                    normal { granite 0.3
                        turbulence 0.2
                    }
                }
                texture { Texture_Dirt }
            }
            // sandstone skirting
            //box { <-SkirtX, BaseY, -SkirtZ>, <SkirtX, SkirtY, SkirtZ>
            superellipsoid { <0.02,0.02>
                scale <SkirtX, (SkirtY-BaseY)/2, SkirtZ>
                translate <0, (SkirtY+BaseY)/2, 0>
                texture {
                    pigment { rgb <1,0.6,0> }
                    normal { granite 0.3
                        turbulence 0.2
                    }
                }
                texture { Texture_Dirt }
            }
        }
        // difference away door and windows
        box { <-DoorX1, BaseY, -BoxZ-0.1>, <DoorX1, DoorY, BoxZ>
            texture {
                pigment { rgb <1,0.6,0> }
            }
        }
        box { <DoorX2, WindowY, -BoxZ-0.1>, <DoorX3, DoorY, BoxZ> }
        box { <-DoorX2, WindowY, -BoxZ-0.1>, <-DoorX3, DoorY, -BoxZ+0.2>
            texture {
                pigment { wood
                    colour_map {
                        [ 0.0 rgb <0.8,0.8,0.8> ]
                        [ 1.0 rgb <0.8,0.5,0.2> ]
                    }
                    rotate <30,5,10>
                    turbulence 0.3
                    scale 0.1
                }
                normal { gradient y 0.5
                    scale 0.1
                }
            }
            texture { Texture_Dirt }
        }
    }
    // stuff in doorway
    union {
        box { <-DoorX1, DoorY-0.3, -BoxZ+0.1>, <DoorX1, DoorY, -BoxZ+0.15> }
        box { <-DoorX1+0.05, BaseY, -BoxZ+0.1>, <-DoorX1+0.1, DoorY, -BoxZ+0.15> }
        box { <-DoorX1+0.1, BaseY, -BoxZ+0.15>, <-DoorX1+0.15, DoorY, -BoxZ+0.2> }
        box { <DoorX1-0.1, BaseY, -BoxZ+0.1>, <DoorX1-0.15, DoorY, -BoxZ+0.15> }
        box { <-1,0,-0.05>, <1,0.07,0>
            rotate <0,0,-10>
            translate <-1.5,DoorY,-BoxZ>
        }
        texture {
            pigment { wood
                colour_map {
                    [ 0.0 rgb <0.9,0.7,0.3> ]
                    [ 1.0 rgb <0.6,0.4,0.1> ]
                }
                turbulence 0.3
                scale 0.3
            }
        }
    }
    // wooden eaves
    box { <-EaveX, BoxY, -EaveZ>, <EaveX, EaveY, EaveZ>
        texture {
            pigment { wood
                colour_map {
                    [ 0.0 rgb <0.9,0.7,0.3> ]
                    [ 1.0 rgb <0.6,0.4,0.1> ]
                }
                turbulence 0.3
            }
        }
    }
    // power insulators
    union {
        cylinder { <DoorX1, BoxY, -EaveZ-2*InsRadius>, <DoorX1, EaveY, -EaveZ-2*InsRadius>, InsRadius }
        cylinder { <DoorX1+10*InsRadius, BoxY, -EaveZ-2*InsRadius>, <DoorX1+10*InsRadius, EaveY, -EaveZ-2*InsRadius>, InsRadius }
        texture {
            pigment { rgb <0.9,0.9,0.9> }
        }
    }
    // steel gutter
    intersection {
        union {
            // front
            intersection {
                union {
                    cylinder { <-GutterX, EaveY + GutterHeight/2, -EaveZ>,
                               <GutterX, EaveY + GutterHeight/2, -EaveZ>, GutterHeight / 2 }
                    cylinder { <-GutterX, GutterY, -EaveZ - GutterHeight/2>,
                               <GutterX, GutterY, -EaveZ - GutterHeight/2>, GutterHeight / 2 }
                }
                plane { <1,0,1>, (BoxLength - BoxDepth) / (2 * sqrt(2)) }
                plane { <-1,0,1>, (BoxLength - BoxDepth) / (2 * sqrt(2)) }
            }
            // right
            intersection {
                union {
                    cylinder { <EaveX, EaveY + GutterHeight/2, -GutterZ>,
                               <EaveX, EaveY + GutterHeight/2, GutterZ>, GutterHeight / 2 }
                    cylinder { <EaveX + GutterHeight/2, GutterY, -GutterZ>,
                               <EaveX + GutterHeight/2, GutterY, GutterZ>, GutterHeight / 2 }
                }
                plane { <-1,0,-1>, -(BoxLength - BoxDepth) / (2 * sqrt(2)) }
                plane { <-1,0,1>, -(BoxLength - BoxDepth) / (2 * sqrt(2)) }
            }
        }
        // take off material above gutter lip
        plane { y, GutterY }
        texture {
            pigment { rgb <0.8,0.8,0.8> }
        }
        // rust layer
        texture {
            pigment { Pigment_Rust
            }
        }
    }
    // steel roof
    union {
        polygon { 7, <-EaveX, 0>, <-RoofX, RoofY2>, <-RoofX, RoofY>,
                <RoofX, RoofY>, <RoofX, RoofY2>, <EaveX, 0>, <-EaveX, 0>
            rotate <degrees(atan2(EaveZ,RoofHeight)),0,0>
            translate <0,GutterY,-EaveZ>
        }
        triangle { <RoofX,GutterY+RoofHeight,0>, <RoofX,GutterY,-EaveZ>, <RoofX,GutterY,EaveZ> }
        texture {
            pigment { rgb <0.8,0.8,0.8> }
            normal { gradient x 0.5
                sine_wave
                scale 0.1
                translate <-GutterX,0,0>
            }
        }
        // rust layer
        texture {
            pigment { brick
                pigment {  rgbt <1,1,1,1>
                }
                pigment { Pigment_Rust
                }
                brick_size <0.8,10,0.8>
                mortar 0.1
                rotate <20,0,0>
                translate <0,0,1>
            }
            normal { gradient x 0.5
                sine_wave
                scale 0.1
                translate <-GutterX,0,0>
            }
        }
    }
    // roof piping
    union {
        cylinder { <-RoofX, GutterY+RoofHeight, 0>, <RoofX, GutterY+RoofHeight, 0>, RoofPipe }
        cylinder { <RoofX, GutterY+RoofHeight*(2/3), -EaveZ/3>, <EaveX, GutterY, -EaveZ>, RoofPipe }
        cylinder { <-RoofX, GutterY+RoofHeight*(2/3), -EaveZ/3>, <-EaveX, GutterY, -EaveZ>, RoofPipe }
        texture {
            pigment { rgb <0.8,0.8,0.8> }
        }
        // rust layer
        texture {
            pigment { Pigment_Rust
            }
        }
    }
    // chimneys
    union {
        box { <ChimneyX1, GutterY, -ChimneyZ>, <ChimneyX2, ChimneyY, ChimneyZ> }
        box { <ChimneyEaveX1, ChimneyY, -ChimneyEaveZ>, <ChimneyEaveX2-0.2, ChimneyEaveY, ChimneyEaveZ> }
        box { <-ChimneyX1, GutterY, -ChimneyZ>, <-ChimneyX2, ChimneyY, ChimneyZ> }
        box { <-ChimneyEaveX1, ChimneyY, -ChimneyEaveZ>, <-ChimneyEaveX2, ChimneyEaveY, ChimneyEaveZ> }
        texture {
            pigment {
                image_map { png "stonewall.png"
                }
                scale BoxX/2
            }
            normal { granite 0.3
                turbulence 0.2
            }
        }
    texture { Texture_Dirt }
    }
    // water tanks
    union {
        cylinder { <TankX1, 0, TankZ>, <TankX1, TankHeight, TankZ>, TankRadius }
        cylinder { <TankX2, 0, TankZ>, <TankX2, TankHeight, TankZ>, TankRadius }
        texture {
            pigment { rgb <0.8,0.8,0.8> }
            normal { gradient y 0.5
                sine_wave
                scale 0.1
            }
            //finish { ambient 0.5 }
        }
    }

    // straw
    #local BaseZ2 = BaseZ * 2;
    #local BoxX2 = BoxX * 2;
    #if (ShowStraw)
        #local NumStrawsSide = NumStraws * 3;
        #local NumStrawsFront = NumStraws * 5;
        #local NumStrawsBase = NumStraws * 0.5;
        // straw on side
        #local i = 0;
        #while (i < NumStrawsSide)
            Straw(rand(Rseed1)*StrawWidth + BoxX,
                 0,
                 (rand(Rseed1)-0.5)*BaseZ2,
                 StrawStalk, 0.3)
            #local i = i + 1;
        #end
        // straw at front
        difference {
            union {
                #local i = 0;
                #while (i < NumStrawsFront)
                    Straw((rand(Rseed1)-0.5)*BoxX2,
                         0,
                         BaseZ - rand(Rseed1)*StrawDepth,
                         StrawStalk, 0.3)
                    #local i = i + 1;
                #end
            }
            // subtract area for sheet of metal lying on ground
            box { <-0.6,0,-0.4>, <0.6,1,0.4>
                rotate <0,MetalRot,0>
                translate <MetalX,0,MetalZ>
            }
        }
        // straw in corner
        #local i = 0;
        intersection {
            union {
                #while (i < NumStraws)
                    Straw(rand(Rseed1)*StrawWidth + BoxX,
                         0,
                         BaseZ - rand(Rseed1)*StrawDepth,
                         StrawStalk, 0.3)
                    #local i = i + 1;
                #end
            }
            cylinder { <0,0,0>, <0,1,0>, 1
                scale <StrawWidth, 1, StrawDepth>
                translate <BoxX, 0, BaseZ>
            }
        }
        // straw everywhere
        #local i = 0;
        #while (i < NumStraws)
            Straw((rand(Rseed1)-0.3)*RockX,
                 0,
                 rand(Rseed1)*RockZ,
                 StrawStalk, 0.1)
            #local i = i + 1;
        #end
        // straw on base
        #local i = 0;
        #while (i < NumStrawsBase)
            Straw((rand(Rseed1)-0.5)*BoxX2,
                 BaseY,
                 BaseZ + rand(Rseed1)*BaseDepth,
                 StrawStalk, 0.1)
            #local i = i + 1;
        #end
    #end // if ShowStraw

    // sheet of metal lying on ground
    box { <-0.6,0.05,-0.4>, <0.6,0.06,0.4>
        texture {
            pigment { rgb <0.8,0.8,0.8> }
            normal { gradient z 0.5
                sine_wave
                scale 0.1
                translate <0,0,1>
            }
        }
        // rust layer
        texture {
            pigment { Pigment_Rust
            }
            normal { gradient z 0.5
                sine_wave
                scale 0.1
                translate <0,0,1>
            }
        }
        rotate <0,MetalRot,0>
        translate <MetalX,0,MetalZ>
    }

    // rocks scattered on ground
    #if (ShowRocks)
        #include "rock.inc"
        #local i = 0;
        #while (i < NumRocks)
            Rock((rand(Rseed1)-0.3)*RockX, 0, rand(Rseed1)*RockZ)
            #local i = i + 1;
        #end // while
        Rock2(-2,0,BaseZ-StrawDepth+0.3,0.25,0.2,0.25)
    #end

    // basic texture
    // should be overridden by each object
    texture {
        pigment { rgb <0,1,0> }
    }

    // rotate and move into position
    //rotate<0,rot,0>
    translate <xTrans,yTrans,zTrans>
}

#end // macro