// plates.inc - hull plating for spherical/cylindrical starship parts.
//              Also works well for brick columns, domed ceilings, etc.
//
// Author:   Jeff Lee <shipbrk@gate.net>
// Date:     Wednesday, 30 June 1999
// Version:  1.0
// Platform: POV-Ray 3.1
//
// If this file was distributed without documentation, you may obtain the
// documentation, a sample scene and the latest version of this file from:
//
//       http://www.gate.net/~shipbrk/raytrace/plates.html
//
// Known problems: Due to blend_map limitations, spherical plates will
// crash if there are more than 128 latitudinal divisions (256 if a zero
// width is used for the "mortar" between the plates).  For this reason,
// the height of individual plates should be at least 2.5% of the
// sphere's radius.

#ifndef (safeatan) #macro safeatan(a,b) #if (a|b) degrees(atan2(a,b)) #else 0.0 #end #end #end

// MakeSphericalPlates() macro
//
// MainRad:  Radius of target object
// PWidth:   Width of plates (at widest point)
// PHeight:  Height of plates
// JWidth:   Width (and height) of the joining seams ("mortar").
// PlateTex: Base texture of plates.  Should be aligned in XY plane, pointing towards
//           -Z; will be rotated to lie on the face of each individual plate.
// JoinTex:  Base texture of the plate joins (will not be rotated).
//
#macro MakeSphericalPlates ( MainRad, PWidth, PHeight, JWidth, PlateTex, JoinTex )
  texture {
    #local T_Sphere1 = texture {
      gradient y
      texture_map {
        #local TexOfs = 0;
        #local ShiftPlates = 0;
        #local foo = MainRad * 0.5 * pi;
        #local NumPlates = int(foo / (PHeight+JWidth));
        #local PlateHeight = (foo - (NumPlates*JWidth))  / NumPlates;
        #local HalfJoinHeight = JWidth / foo / 2;
        #local VertIter = 0;
        #while (VertIter < NumPlates)
          #local BottomPoint = cos( radians( 90-(90*VertIter/NumPlates)-(90*HalfJoinHeight) ));
          #local TopPoint    = cos( radians( 90-(90*(VertIter+1)/NumPlates)+(90*HalfJoinHeight) ));
          #local MedPointX   = sin( radians( 90-(90*(VertIter+0.5)/NumPlates)));
          #local MedPointY   = cos( radians( 90-(90*(VertIter+0.5)/NumPlates)));
          #local PlateRotX   = 90-(safeatan( MedPointX, MedPointY ));
          #if (VertIter = NumPlates-1) #local PlateRotX=90; #end
          #local NewYTex = texture { 
            radial
            texture_map {
              #local UpperLimit=63;
              #if (JWidth = 0) #local UpperLimit=126; #end
              #local Freq = 1;
              #local LevelCirc  = MainRad * pi * 2 * sin( radians( 90-(90*VertIter/NumPlates)-(90*HalfJoinHeight)));
              #local RowPlates  = int(LevelCirc / (PWidth+JWidth));
              #while (RowPlates/Freq > UpperLimit) #declare Freq = Freq + 1; #end
              #local RowPlates = RowPlates / Freq;
              #local PlateWidth = (LevelCirc - (RowPlates*JWidth))  / RowPlates;
              #local HalfJoinWidth = JWidth / LevelCirc / 2;
              #local HorIter = 0;
              #while (HorIter < RowPlates)
                #local RightPoint = (HorIter/RowPlates)+HalfJoinWidth;
                #local LeftPoint  = ((HorIter+1)/RowPlates)-HalfJoinWidth;
                #local MedPointX  = sin( radians( (360*(HorIter+0.5)/RowPlates)));
                #local MedPointY  = cos( radians( (360*(HorIter+0.5)/RowPlates)));
                #local PlateRotY  = (safeatan( MedPointX, MedPointY ));
                #if (PlateRotY<0) #local PlateRotY=360+PlateRotY; #end
                #local NewXTex = texture { PlateTex scale 1/MainRad translate z*TexOfs rotate <PlateRotX,PlateRotY-90,0> }
                #if (JWidth != 0) [RightPoint JoinTex] #end
                [RightPoint NewXTex]
                [LeftPoint  NewXTex]
                #if (JWidth != 0) [LeftPoint  JoinTex] #end
                #local HorIter=HorIter+1;
                #local TexOfs=TexOfs+1234;
              #end
            }
            #if (Freq > 1) frequency Freq #end
            #if (VertIter=0) #local BottomRot = 360*((0.5/RowPlates)+HalfJoinWidth); #end
            rotate ShiftPlates*y*45
            #local ShiftPlates=1-ShiftPlates;
          }
          #if (JWidth != 0) [BottomPoint JoinTex] #end
          [BottomPoint NewYTex]
          [TopPoint    NewYTex]
          #if (JWidth != 0) [TopPoint    JoinTex] #end
          #local VertIter=VertIter+1;
        #end
      }
      rotate y*-90 // Put the seams in +Z
      scale <1,0.5,1>
    }
    gradient y
    texture_map {
      [0.5 T_Sphere1 translate y*0.5 rotate y*BottomRot]
      [0.5 T_Sphere1 translate y*0.5]
    }
    translate y*-0.5
    scale <1,2,1>*MainRad
  }
#end

// MakeCylindricalPlates() macro
//
// Same parameters as MakeSphericalPlates().
//
#macro MakeCylindricalPlates ( MainRad, PWidth, PHeight, JWidth, PlateTex, JoinTex )
  texture {
    #local T_Cylinder1 = texture {
      radial
      texture_map {
        #local UpperLimit=63;
        #if (JWidth = 0) #local UpperLimit=126; #end
        #local Freq = 1;
        #local TexOfs = 0;
        #local LevelCirc  = MainRad * pi * 2;
        #local RowPlates  = int(LevelCirc / (PWidth+JWidth));
        #while (RowPlates/Freq > UpperLimit) #declare Freq = Freq + 1; #end
        #local RowPlates = RowPlates / Freq;
        #local PlateWidth = (LevelCirc - (RowPlates*JWidth))  / RowPlates;
        #local HalfJoinWidth = JWidth / LevelCirc / 2;
        #local HorIter = 0;
        #local OffsetRot = 360/RowPlates/(2*Freq);
        #if (mod(RowPlates,2)=1) #local OffsetRot=OffsetRot*2; #end
        #while (HorIter < RowPlates)
          #local RightPoint = (HorIter/RowPlates)+HalfJoinWidth;
          #local LeftPoint  = ((HorIter+1)/RowPlates)-HalfJoinWidth;
          #local MedPointX  = sin( radians( (360*(HorIter+0.5)/RowPlates)));
          #local MedPointY  = cos( radians( (360*(HorIter+0.5)/RowPlates)));
          #local PlateRotY  = (safeatan( MedPointX, MedPointY ));
          #if (PlateRotY<0) #local PlateRotY=360+PlateRotY; #end
          #local NewXTex = texture { PlateTex scale <1/MainRad,1/PHeight,1/MainRad> translate z*TexOfs rotate y*(PlateRotY-90) }
          #if (JWidth != 0) [RightPoint JoinTex] #end
          [RightPoint NewXTex]
          [LeftPoint  NewXTex]
          #if (JWidth != 0) [LeftPoint  JoinTex] #end
          #local HorIter=HorIter+1;
          #local TexOfs=TexOfs+1234;
        #end
      }
      #if (Freq > 1) frequency Freq #end
    }
    #local BottomPoint = (JWidth / PHeight) / 2;
    #local TopPoint = 1-BottomPoint;
    #local T_Cylinder2 = texture {
      gradient y
      texture_map {
        #if (JWidth != 0) [BottomPoint JoinTex] #end
        [BottomPoint T_Cylinder1]
        [TopPoint    T_Cylinder1]
        #if (JWidth != 0) [TopPoint    JoinTex] #end
      }
      scale <1,0.5,1>
    }
    gradient y
    texture_map {
      [0.5 T_Cylinder2 ]
      [0.5 T_Cylinder2 translate y*30 rotate y*(180+OffsetRot)]
    }
    warp { repeat y offset y*50 }
    scale <MainRad,2*PHeight,MainRad>
  }
#end
