//------------------------------------------------------------------------------
// Persistence of Vision Ray Tracer Scene Include File
// File: CityObj.inc
// Vers: 3.1g
// Desc: Textures for IRTC The City
// Date: 27/04/2000
// Auth: Tony Vrnjas
// Mail: tonyv@iafrica.com
//

//------------------------------------------------------------------------------
// macro: MakeSheet (TheWidth,Thelength,Thickness,TheDegrees,NumberPairs)
//
//        Creates corrugated sheets of a specified length and width which can
//        then be textured as corrugated iron, steel or cardboard.
//
// where: TheWidth    is the width of piece of corrugation in the X-axis
//        TheLength   is the length of piece of corrugation in the Y-axis
//        Thickness   is the thickness of the corrugated material. Typically
//                    use values in range of 0.001 to 0.5.                          
//        TheDegrees  is a value that determines the depth of the corrugations.
//                    Use values from 20 degrees to 70 degress for the best
//                    results. The smaller the angle the greater the corrugation
//                    depth.
//        NumberPairs is the number of in/out pairs of corrugations in the width
//                    of the object.
//
// usage: object {MakeSheet(2,3,0.003,30,20)}
//
//------------------------------------------------------------------------------
#macro MakeSheet (TheWidth,Thelength,Thickness,TheDegrees,NumberPairs)
  union {
    #local TheBottom = 0;
    #local TheLeft = 0;
    #local TheAngle = radians(TheDegrees);
    #local InnerRadius = (TheWidth/NumberPairs/4/cos(TheAngle))-(Thickness/2);
    #local OuterRadius = InnerRadius + Thickness;
    #local Hypoteneuse = InnerRadius + Thickness + InnerRadius;
    #local OppositeSide = Hypoteneuse * sin(TheAngle);
    #local AdjacentSide = Hypoteneuse * cos(TheAngle);
    #local XMoveToZero = OuterRadius * cos(TheAngle);
    #local ZMoveToZero = OuterRadius - (Hypoteneuse * sin(TheAngle));
    #local NextStep = TheLeft;
    #while (NextStep < (NumberPairs*2))
      #if (mod(NextStep,2) = 0)
        #local ZValue = ZMoveToZero;
        #local DegValue = TheDegrees;
      #else
        #local ZValue = ZMoveToZero + OppositeSide;
        #local DegValue = TheDegrees + 180;
      #end
      #local XValue = XMoveToZero + (NextStep * AdjacentSide);
      difference { // this is THE object
        cylinder {
          <XValue, TheBottom, ZValue>,
          <XValue, Thelength, ZValue>,
          OuterRadius
        }
        cylinder {
          <XValue, TheBottom - 1, ZValue>,
          <XValue, Thelength +1 , ZValue>,
          InnerRadius
        }
        #if (mod(NextStep,2) = 0)
          plane { z, 0
            rotate <0, TheDegrees, 0>
            translate <XValue,0,ZValue>
          }
          plane { z, 0
            rotate <0, -TheDegrees, 0>
            translate <XValue,0,ZValue>
          }
        #else
          plane { z, 0 inverse
            rotate <0, -TheDegrees, 0>
            translate <XValue,0,ZValue>
          }
          plane { z, 0 inverse
            rotate <0, TheDegrees, 0>
            translate <XValue,0,ZValue>
          }
        #end
      }
      #declare NextStep = NextStep + 1;
    #end
  }
  #declare TheDepth = OppositeSide + (2*(OuterRadius-OppositeSide));
//  #warning concat("TheDepth value is: ",str(TheDepth,5,15),"\n")
#end                                   

//------------------------------------------------------------------------------
// macro: MakeLampPost1 (LightColour, FadeDistance, FadePower)
//
// where: LightColour  is the colour of the light attached. Use a rgb/t vector
//                     or a previously #declared colour.
//        FadeDistance is the fade_distance value used in a light_source
//                     declaration
//        FadePower    is the fade_power value used in a light_source
//                     declaration
//
// usage: object {MakeLampPost1(rgb <0.5, 0.5, 0.5>,6,2)}
//        object {MakeLampPost1(Red,6,2)}
//
// to-do: - include a parameter that allows the passing of a texture. Possibly
//        extend this to include two or three textures so the lamp-holder could
//        be textured differently.
//        - switches for different-lamp holders
//        - extra parameter that would allow the light to be toggled on/off.
//
//------------------------------------------------------------------------------
#macro MakeLampPost1 (LightColour, FadeDistance, FadePower, InTexture)
  union {
    cylinder {<0,0,0>,<0,1,0>,0.10}
    torus {0.09,0.02 translate <0,1,0>}
    cone {<0,1,0>,0.08,<0,2.5,0>,0.03}
    difference {
      torus {1.0,0.03 rotate <0,0,90>}
      box {<-1,-2,0><1,2,2>}
      box {<-1,-2,-2><1,0,2>}
      translate <0,2.5,1>
    }
    cone {<0,3.5,1>,0.03,<0,3.5,1.5>,0.02}
    union {
      difference {
        union {
          cone {<0,0,-0.1>,0.1,<0,0,-0.5>,0.02}
          sphere {<0,0,0> 0.15}
          sphere {<0,0,0> 0.14}
        }
        box{<1,-1,1><-1,0,-1>}
      }
      light_source {
        <0, 0, 0>
        colour LightColour
        fade_distance FadeDistance
        fade_power FadePower
      }
      translate <0,3.5-0.02,1.65>
    }
    texture {InTexture}
  }
#end

//------------------------------------------------------------------------------
// Simple three-legged pine stool
#declare PineStool = union {
  union {
    torus { 0.15, 0.02 }
    cylinder {<0,0,0>,<0,0.02,0>,0.15}
    texture {Yellow_Pine
      rotate <0,20,0>
    }
  }
  cylinder {<0,0,0.11>,<0,-0.50,0.15>,0.025
    texture {Yellow_Pine}
  }
  cylinder {<sin(30)*0.11,0,-cos(30)*0.11>,<sin(30)*0.15,-0.50,-cos(30)*0.15>,0.025
    texture {Yellow_Pine}
  }
  cylinder {<-sin(30)*0.11,0,-cos(30)*0.11>,<-sin(30)*0.15,-0.50,-cos(30)*0.15>,0.025
    texture {Yellow_Pine}
  }
  translate <0,0.50,0>
}

//------------------------------------------------------------------------------
// Simple Pine Table
// size is 1.40 x 0.60
// future versions will allow sizing
#declare PineTable = union {
  union {
    cylinder {<-0.67,0.8, 0.27>,<-0.67,0.83, 0.27>,0.03 }
    cylinder {<-0.67,0.8,-0.27>,<-0.67,0.83,-0.27>,0.03 }
    cylinder {< 0.67,0.8, 0.27>,< 0.67,0.83, 0.27>,0.03 }
    cylinder {< 0.67,0.8,-0.27>,< 0.67,0.83,-0.27>,0.03 }
    box {<-0.67,0.8,0.30><0.67,0.83,-0.30> }
    box {<-0.70,0.8,0.27><0.70,0.83,-0.27> }
    texture { Yellow_Pine rotate <0,90,0> }
  }
  box {<-0.64,0.8001, 0.24><0.64,0.70, 0.23> texture { Yellow_Pine rotate <90,0,90> } }
  box {< 0.64,0.8001,0.24>< 0.63,0.70,-0.24> texture { Yellow_Pine rotate <0, 0,90> } }
  box {<-0.64,0.8001,-0.24><0.64,0.70,-0.23> texture { Yellow_Pine rotate <90,0,90> } }
  box {<-0.64,0.8001,0.24><-0.63,0.70,-0.24> texture { Yellow_Pine rotate <0, 0,90> } }

  box {< 0.63,0, 0.23>< 0.56,0.8001, 0.16> texture { Yellow_Pine rotate <90,0,0> } }
  box {< 0.63,0,-0.23>< 0.56,0.8001,-0.16> texture { Yellow_Pine rotate <90,0,0> } }
  box {<-0.63,0, 0.23><-0.56,0.8001, 0.16> texture { Yellow_Pine rotate <90,0,0> } }
  box {<-0.63,0,-0.23><-0.56,0.8001,-0.16> texture { Yellow_Pine rotate <90,0,0> } }
}                                                                                   

//------------------------------------------------------------------------------
// A proper octagonal traffic stop sign; already textured.
#declare StopSign = 
union {
  cylinder {
    <0,0.00,0>,<0,1.50,0>,0.05
    texture {
      pigment {
        gradient y
        colour_map {
          [0.0 colour White]
          [0.1 colour White]
          [0.1 colour Black]
          [0.2 colour Black]
          [0.2 colour White]
          [0.3 colour White]
          [0.3 colour Black]
          [0.4 colour Black]
          [0.4 colour White]
          [0.5 colour White]
          [0.5 colour Black]
          [0.6 colour Black]
          [0.6 colour White]
          [0.7 colour White]
          [0.7 colour Black]
          [0.8 colour Black]
          [0.8 colour White]
          [0.9 colour White]
          [0.9 colour Black]
          [1.0 colour Black]
        }
      }
      finish {
        metallic
      }
      scale <1,1.5,1>
    }
    texture {
      pigment {
        granite
        turbulence 1.0
        colour_map {
          [0.0 colour White filter 1]
          [0.3 colour White filter 1]
          [0.3 colour Black filter 0.4]
          [0.7 colour White filter 0.4]
          [0.7 colour White filter 1]
          [1.0 colour White filter 1]
        }
      }
    }
  }
  box {<-0.15,1.40,-0.05><0.15,1.45,-0.07>
    texture {
      pigment { colour Gray75 }
      finish { metallic }
    }
  }
  box {<-0.15,1.25,-0.05><0.15,1.30,-0.07>
    texture {
      pigment { colour Gray75 }
      finish { metallic }
    }
  }
  cylinder {<-0.14,1.425,-0.06>,<-0.14,1.425,-0.105>,0.01 pigment {colour Gray10}}
  cylinder {< 0.14,1.425,-0.06>,< 0.14,1.425,-0.105>,0.01 pigment {colour Gray10}}
  cylinder {<-0.14,1.275,-0.06>,<-0.14,1.275,-0.105>,0.01 pigment {colour Gray10}}
  cylinder {< 0.14,1.275,-0.06>,< 0.14,1.275,-0.105>,0.01 pigment {colour Gray10}}
  union {
    intersection {
      box {<-0.30,-0.30,-0.000001>< 0.30,0.30,-0.02>}
      box {<-0.30,-0.30, 0.00>< 0.30,0.30,-0.02> rotate <0,0,45>}
      texture { pigment { colour Gray75 } finish {metallic}}
    }
    intersection {
      box {<-0.30,-0.30,-0.020001>< 0.30,0.30,-0.03>}
      box {<-0.30,-0.30,-0.02>< 0.30,0.30,-0.03> rotate <0,0,45>}
      texture {
        pigment {
          image_map {
            png "sign222-600.png"
            map_type 0
            once
            interpolate 2
          }
          scale <0.60,0.60,1>
          translate <-0.30,-0.30,0>
        }
      }
      texture {
        pigment {
          granite
          turbulence 1.0
          colour_map {
            [0.0 colour White filter 1]
            [1.0 colour Black]
          }
        }
      }
    }
    translate <0,1.35,-0.07>
  }
}

