// Persistence of Vision Ray Tracer Scene Description File
// File: Road.inc
// Vers: 3.1
// Desc: Road building blocks
// Date: 3/29/00
// Auth: Katherine Smith
// URL: http://www.bigfoot.com/~kathsth/
// EMail: kathsth@bigfoot.com
/*
	Copyright (C) 2000  Katherine Smith

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.
	
	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.
	
	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/



#ifndef (Render)
	// We're not being included, but rendered on our own,
	// so we need to create our own camera and lights
	#local Render = 1;
#else
	#local Render = 0;
#end

#ifndef (Detail_Level)
	#local Detail_Level = 10; // 0..10
#end

#include "SimpleTree.inc"

#declare RoadGray_P = <0.5,0.5,0.5>;
#declare RoadGrayClear_P = <0.5,0.5,0.5,1>;
#declare RoadWhiteStripe_P = <1,1,1>;
#declare RoadGrime_P = <0.3,0.3,0.3>;

#declare RoadWhiteStripe_T = texture {
	pigment {
		gradient x
		color_map {
			[0.47 color RoadGray_P]
			[0.47 color RoadWhiteStripe_P]
			[0.53 color RoadWhiteStripe_P]
			[0.53 color RoadGray_P]
		}
	}
}
#declare RoadWhiteStripeCutter_T = texture {
	pigment {
		gradient z
		color_map {
			[0.3 color rgb RoadGray_P]
			[0.3 color rgbt RoadGrayClear_P]
			[0.7 color rgbt RoadGrayClear_P]
			[0.7 color rgb RoadGray_P]
		}
		scale <1,1,1>
	}
}
#declare RoadOilStripe_T = texture {
	pigment {
		gradient x
		color_map {
			[0.00 color rgb RoadGrime_P]
			[0.35 color rgbt RoadGrayClear_P]
			[0.65 color rgbt RoadGrayClear_P]
			[1.00 color rgb RoadGrime_P]
		}
	}
}

#declare RoadOneWay_T =
	texture {RoadWhiteStripe_T}
	texture {RoadWhiteStripeCutter_T}
	texture {RoadOilStripe_T}

#declare RoadStripeEdge = 0.47;
#declare RoadLaneWidth = 1;
#macro RoadOneWay(NLanes,NLength)
	box {
		<-RoadStripeEdge,-1,0>,
		<NLanes-1+RoadStripeEdge,0,NLength>
		texture {RoadOneWay_T}
		translate <-(NLanes-1)/2,0,0>
		scale <1,1,RoadLaneWidth>
		rotate 90*y
	}
#end

#declare SideWalkGray_P = <165,155,142>/256;//<0.8,0.8,0.7>;
#declare SideWalkCrack_P = <1,1,1>*0.4;

#declare SideWalk_T = texture {
	pigment {
		gradient z
		color_map {
			[0.00 color SideWalkCrack_P]
			[0.05 color SideWalkGray_P]
			[0.95 color SideWalkGray_P]
			[1.00 color SideWalkCrack_P]
		}
		scale <1,1,0.5>
	}
}

#declare SideWalkCurb_T = texture { pigment {color rgb SideWalkGray_P}}

#declare SideWalkGrass_T = texture {
	pigment {
		granite 
		color_map {
			[0.0 color rgb <69,82,47>/256]
			[0.5 color rgb <71,87,44>/256]
			[1.0 color rgb <108,105,65>/256]

/*			[0.0 color <0.0,0.6,0.0>]
			[0.3 color <0.2,0.5,0.7>]
			[0.6 color <0.2,0.8,0.4>]
			[0.9 color <0.8,1.0,0.6>]
*/		}
		triangle_wave
		scale 0.1
	}
}
	
#declare SideWalkHeight = 0.1;
#declare SideWalkWidth = 0.5;
#declare CurbWidth = SideWalkHeight;
#declare SideWalkGrassWidth = 0.3;

#macro SideWalkWGrass (Length,Height,CurbWidth,GrassWidth,SideWalkWidth)
	// curb
	union {
		box {
			<0,-1,0>
			<CurbWidth,0,Length>
			texture {SideWalkCurb_T}
		}
		box {
			<0,-1,0>
			<SideWalkGrassWidth,0,Length>
			texture {SideWalkGrass_T}
			translate <CurbWidth,0,0>
		}
		box {
			<0,-1,0>
			<SideWalkWidth,0,Length>
			texture {SideWalk_T}
			translate <CurbWidth+GrassWidth,0,0>
		}
		translate <0,Height,0>
	}
#end

// *******************************************
// Sidewalk corner definition
// *******************************************
#declare SideWalkWGrassWidth = CurbWidth+SideWalkGrassWidth+SideWalkWidth;
#declare SideWalkWGrassCorner = union {
	box {
		<0,-1,0>
		<SideWalkWGrassWidth,0,SideWalkWGrassWidth>
		texture {SideWalkCurb_T}
	}
	box {
		<0,-1,0>
		<SideWalkWGrassWidth-CurbWidth,0.001,SideWalkWGrassWidth-CurbWidth>
		pigment {color rgb SideWalkGray_P}
		translate <CurbWidth,0,CurbWidth>
	}
	translate <0,SideWalkHeight,0>
}
	

#declare RoadLame_T = texture {
		pigment {
			checker
			pigment {
				gradient x
				color_map {
					[0.0 color rgb <0,0,0>]
					[0.5 color rgb <0,0,0>]
					[0.5 color rgb <1,1,1>]
					[0.6 color rgb <1,1,1>]
					[0.6 color rgb <0,0,0>]
				}
			}
			pigment { color <0.1,0.1,0.1> }
		}
}

#declare MedianWidth = (1-RoadStripeEdge)*2;
#declare MedianEnd = union {
	cylinder {
		<0,0,0>, // base point
		<0,SideWalkHeight,0>, // cap point
		MedianWidth/2 // radius
		texture {SideWalkCurb_T}
	}
	cylinder {
		<0,0,0>, // base point
		<0,SideWalkHeight+0.001,0>, // cap point
		MedianWidth/2-CurbWidth // radius
		texture {SideWalkGrass_T}
	}
	clipped_by {
		plane {
			z,
			0
		}
	}
}

#declare MedianBody = union {
	box {
		<-MedianWidth/2,0,0>, 
		< MedianWidth/2,SideWalkHeight,1> 
		texture {SideWalkCurb_T}
	}
	box {
		<-(MedianWidth/2-CurbWidth),0,1>, 
		<MedianWidth/2-CurbWidth,SideWalkHeight+0.001,0>
		texture {SideWalkGrass_T}
	}
}

#include "SimplePalm.inc"
#declare RTreeHeight = seed(0);
#macro Median(Length,NumTrees,MedianType)
	union {
		object {
			MedianEnd
			translate <0,0,MedianWidth/2>
		}
		object {
			MedianBody
			scale <1,1,Length-MedianWidth >
			translate <0,0,MedianWidth/2>
		}
		object {
			MedianEnd
			rotate 180*y
			translate <0,0,Length-MedianWidth/2>
		}
		#if (MedianType = 1) // with trees
			#local Count = 0;
			#while (Count < NumTrees)
				object {
					SimplePalm(rand(RTreeHeight)+2.5)
					//box{ 0,1 texture{SideWalk_T}}
					scale 0.5
					translate <0,SideWalkHeight,MedianWidth/2 + Count*(Length-MedianWidth)/(NumTrees-1)>
				}
				#local Count = Count + 1;
			#end
		#end
	}
#end		


// *******************************************
// Street Descriptor creation
// *******************************************
// a road value is <Width in lanes,Gap to next road, Road type, Median Type, Sidewalk Type, Num cars per unit length>
#ifndef(NumRoadsH)
	#declare NumRoadsH = 16;
	#declare RoadDescriptionH = array[NumRoadsH][6] {{9,30,1,1,1,0.25},{7,30,1,0,1,0.25},{5,20,1,0,1,0.25},{9,20,1,0,1,0.25},{5,30,1,0,1,0.25},{7,30,1,0,1,0.25},{5,20,1,0,1,0.25},{9,20,1,0,1,0.25},{5,30,1,0,1,0.25},{7,30,1,0,1,0.25},{5,20,1,0,1,0.25},{9,20,1,0,1,0.25},{5,30,1,0,1,0.25},{7,30,1,0,1,0.25},{5,20,1,0,1,0.25},{9,20,1,0,1,0.25}}
#end
#ifndef(NumRoadsV)
	#declare NumRoadsV = 16;
	#declare RoadDescriptionV = array[NumRoadsH][6] {{5,30,1,1,1,0.25},{7,30,1,0,1,0.25},{5,20,1,0,1,0.25},{9,20,1,0,1,0.25},{5,30,1,0,1,0.25},{7,30,1,0,1,0.25},{5,20,1,0,1,0.25},{9,20,1,0,1,0.25},{5,30,1,0,1,0.25},{7,30,1,0,1,0.25},{5,20,1,0,1,0.25},{9,20,1,0,1,0.25},{5,30,1,0,1,0.25},{7,30,1,0,1,0.25},{5,20,1,0,1,0.25},{9,20,1,0,1,0.25}}
#end
#declare aRoadNLanes = 0;
#declare aRoadSpacing = 1;
#declare aRoadType = 2;
#declare aMedianType = 3;
#declare aSidewalkType = 4;
#declare aProbCars = 5;

#include "Cars.inc"
#declare RCarPos = seed(0);

// *******************************************
// Street Section creation
// *******************************************
#macro MakeStreetSegment(NLanes,RoadType,MedianType,SidewalkType,ProbCars,Length)
	//#local = InRoadDescription[aRoadNLanes];
	//#local RoadType = InRoadDescription[aRoadType];
	//#local MedianType = InRoadDescription[aMedianType];
	//#local SidewalkType = InRoadDescription[aSidewalkType];

	// ******************************************
	// Create Road base
	#switch (RoadType)
	 #case (0)
		#break
	 #case (1)
		object {
			RoadOneWay(NLanes,Length)
			rotate -90*y
		}
		#break
	#end

	// ******************************************
	// Create Medians if any
	Median(Length,5,MedianType)
	#declare MedianLane = ceil(NLanes/2)-1;
	
	// ******************************************
	// Create Cars if any.
	// Currently cars are quadrilateraly symetric, it doesn't matter which way they're pointing.
	#if (ProbCars != 0)
		#local Lane = 0;
		#while (Lane < NLanes)
			#local Z = 0;
			#while (Z < Length)
				#if (rand(RCarPos) < ProbCars)
					object {
						MakeCar()
						translate <(Lane-NLanes/2+1/2)* RoadLaneWidth, 0, Z>
					}
					#local Z = Z + CarMeanLength + rand(RCarPos);
				#else
					#local Z = Z + CarLength + rand(RCarPos); // set by MakeCar.
				#end
			#end
			#local Lane = Lane + 1;		
			#if ((Lane = MedianLane) & (MedianType != 0))
				#local Lane = Lane + 1;
			#end
		#end
	#end
	
	// ******************************************
	// Create Sidewalks if any
	#switch (SidewalkType)
	 #case (0)
	 	#local TheSidewalk = sphere { <0,-1000, 0>, 0.0001}
		#break
	 #case (1)
		#local TheSidewalk = object {
			SideWalkWGrass(Length,SideWalkHeight,CurbWidth,SideWalkGrassWidth,SideWalkWidth)
		}
		#break
	 #case (2)
		#local TheSidewalk = union {
			object {
				SideWalkWGrass(Length,SideWalkHeight,CurbWidth,SideWalkGrassWidth,SideWalkWidth)
			}
			#local Z = 0;
			#while (Z < Length)
				#if (rand(RTreeHeight) < 0.50)
					object {
						LollypopTree(3,1,0.125)
						scale 0.5+rand(RTreeHeight)/2
						translate < CurbWidth+ SideWalkGrassWidth/2, 0, Z>
					}
				#end
				#local Z = Z + 5;
			#end
		}
	#end
	object {
		TheSidewalk
		translate <NLanes*RoadLaneWidth/2,0,0>
	}
	
	object {
		TheSidewalk
		translate <NLanes*RoadLaneWidth/2,0,0>
		rotate 180*y
		translate <0,0,Length>
	}
#end

// *******************************************
// Street Intersection creation
// *******************************************
// Right now this is a cheesy intersection, I don't think you'll be able to see
// the roads OR it at 800x600.... whatever.
#macro MakeStreetIntersection(NLanesH,NLanesV,SidewalkTypeH,SidewalkTypeV)
	#local SizeV = NLanesH*RoadLaneWidth+ SideWalkWGrassWidth*2;
	#local SizeH = NLanesV*RoadLaneWidth+ SideWalkWGrassWidth*2;
	// ******************************************
	// Create Road slab
	box {
		<0,-1,0>
		<SizeH,0,SizeV>
		translate <-SizeH/2,0,-SizeV/2>
		texture{pigment {color rgb RoadGray_P}}
		texture {RoadOilStripe_T}
		texture {RoadOilStripe_T rotate 90*y}

	}
	// ******************************************
	// Create Sidewalk corners
	#if ((SidewalkTypeH != 0) & (SidewalkTypeV != 0))
		#local TransH = SizeH/2-SideWalkWGrassWidth;
		#local TransV = SizeV/2-SideWalkWGrassWidth;
		object{SideWalkWGrassCorner	rotate   0*y translate < TransH,0, TransV>}
		object{SideWalkWGrassCorner	rotate  90*y translate < TransH,0,-TransV>}
		object{SideWalkWGrassCorner	rotate 180*y translate <-TransH,0,-TransV>}
		object{SideWalkWGrassCorner	rotate 270*y translate <-TransH,0, TransV>}
	#end
#end

#include "Tower.inc"

// *******************************************
// Road set up
// *******************************************

#macro MakeStreets(NumRoadsH,RoadDescriptionH,NumRoadsV,RoadDescriptionV)
	#local StreetHCount = 0;
	#local StreetHPos = 0;
	#while (StreetHCount < NumRoadsH)
		#local StreetVCount = 0;
		#local StreetVPos = 0;
		#while (StreetVCount < NumRoadsV)
			union {
				MakeStreetIntersection(RoadDescriptionH[StreetHCount][aRoadNLanes],RoadDescriptionV[StreetVCount][aRoadNLanes],RoadDescriptionH[StreetHCount][aSidewalkType],RoadDescriptionV[StreetVCount][aSidewalkType])
				translate <StreetVPos,0,StreetHPos>
			}
			// Vertical street segment
			#if (StreetHCount < NumRoadsH-1)
				#local VShift1 = RoadDescriptionH[StreetHCount][aRoadNLanes]*RoadLaneWidth/2+SideWalkWGrassWidth; // + sidewalk?
				#local VShift2 = RoadDescriptionH[StreetHCount+1][aRoadNLanes]*RoadLaneWidth/2 +SideWalkWGrassWidth; // + sidewalk?
				#local VLength = RoadDescriptionH[StreetHCount][aRoadSpacing]-(VShift1+VShift2);
				union {
					MakeStreetSegment(RoadDescriptionV[StreetVCount][aRoadNLanes],RoadDescriptionV[StreetVCount][aRoadType],RoadDescriptionV[StreetVCount][aMedianType],RoadDescriptionV[StreetVCount][aSidewalkType],RoadDescriptionV[StreetVCount][aProbCars],VLength)
					translate <StreetVPos,0,StreetHPos+VShift1>
				}
			#end
			// Horizontal street segment
			#if (StreetVCount < NumRoadsV-1)
				#local HShift1 = RoadDescriptionV[StreetVCount][aRoadNLanes]*RoadLaneWidth/2+SideWalkWGrassWidth; // + sidewalk?
				#local HShift2 = RoadDescriptionV[StreetVCount+1][aRoadNLanes]*RoadLaneWidth/2 +SideWalkWGrassWidth; // + sidewalk?
				#local HLength = RoadDescriptionV[StreetVCount][aRoadSpacing]-(HShift1+HShift2);
				union {
					MakeStreetSegment(RoadDescriptionH[StreetHCount][aRoadNLanes],RoadDescriptionH[StreetHCount][aRoadType],RoadDescriptionH[StreetHCount][aMedianType],RoadDescriptionH[StreetHCount][aSidewalkType],RoadDescriptionH[StreetHCount][aProbCars],HLength)
					rotate 90*y
					translate <StreetVPos+HShift1,0,StreetHPos>
				}
			#end
			#if ((StreetHCount < NumRoadsH-1)&(StreetVCount < NumRoadsV-1))
				object {
					FillBlock(HLength,VLength,StreetHCount/NumRoadsH,StreetVCount/NumRoadsV)
					translate <StreetVPos+HShift1+HLength/2,0,StreetHPos+VShift1+VLength/2>
				}
			#end
			#local StreetVPos = StreetVPos + RoadDescriptionV[StreetVCount][aRoadSpacing];		
			#local StreetVCount = StreetVCount + 1;
		#end
		#local StreetHPos = StreetHPos + RoadDescriptionH[StreetHCount][aRoadSpacing];		
		#local StreetHCount = StreetHCount + 1;
	#end
#end
	
#if (Render = 1)

	#macro FillBlock(HLength,VLength,HPos,VPos)
		box {0,1 pigment {color rgb <1,0.5,0.5>}}
		//MakeTower(HLength,VLength)
	#end

	global_settings {ambient_light 3}
	/*union {
	 	MakeStreetSegment(5,1,1,1,0.0,10)
		MakeStreetIntersection(5,5,1,1)
		translate <4,0,5>
	}*/
	#include "colors.inc"
	// Create an infinite sphere around scene and allow any texture on it
	sky_sphere
	{
	 	pigment
	  	{
	  	  gradient y
	  	  color_map { 
	  	  	[0.0 MidnightBlue] 
	  	  	[1.0 CornflowerBlue*2] 
	  	  }
	  	}
	}
	//MakeStreets(NumRoadsH,RoadDescriptionH,NumRoadsV,RoadDescriptionV)
	MakeStreetSegment(5,1,1,2,.5,20)

	plane{
		y,
		-0.1
		pigment {color rgb <0,0.5,0.5>}
	}
	camera
	{
	  location  <0 , 5 , 0>
	  look_at   <4 , 0.0 , 10>
	}
	// create a regular point light source
	light_source
	{
	  0*x // light's position (translated below)
	  color <1,1,1>*1.5  // light's color
	  translate <-100 , 50 , -50>
	}
	
	background {color <1,1,1>}
#end


