// Persistence of Vision Ray Tracer Scene Description File
// File: TrainBuilder.inc
// Vers: 3.1
// Desc: Given a list of train car objects (or any objects :) and a list of track types, 
//         this will arange those objects along the track.  This involves a lot of trig,
//         and is quite complicated when transitioning between two different types of track.
//         Involves many applications of the Law of Cosines and the Law of Sines.  Lots of fun!
//         Probably it would have been simpler to just take small steps along the track 
//         curve until you achived the desired chord length.  But this was more fun.
// Date: 4/26/00
// Auth: Katherine Smith
// URL: http://www.bigfoot.com/~kathsth/
// EMail: kathsth@bigfoot.com
/*
	Copyright (C) 2000  Katherine Smith

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.
	
	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.
	
	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/


#ifndef (Render)
	// We're not being included, but rendered on our own,
	// so we need to create our own camera and lights
	#local Render = 1;
#else
	#local Render = 0;
#end

#ifndef (Detail_Level)
	#local Detail_Level = 10; // 0..10
#end

#declare kStraight = 0;
#declare kRightCurve = 1;
#declare kLeftCurve = 2;


		//  Compute the position for each end of the car, since that 
		// is what needs to be aligned with the track. (The wheels)
		// One end is prev. posit

#macro PlaceTrainCarsOnTrack(TrainCars,TrainCarLength,TrackType,TrackSize)
	#local NumCars = dimension_size(TrainCars,1);
	#local Car = 0;
	#local TrackCount = 0;
	#local Position = <0,0,0>;
	#local Angle = 0;
	#local CurrentTrackPosition = 0; // position allong length of track piece
	
	// I make the assumption here that no train car spans more than 2 pieces of track.
	#while(Car < NumCars)
		#local CurrentTrackType = TrackType[TrackCount];
		#local CurrentTrackSize = TrackSize[TrackCount];
		//#debug concat("<",str(CurrentTrackType,3,3),">")
		object {
			TrainCars[Car]
			
			#switch (CurrentTrackType)
				#case (kStraight)
					#local NextTrackPosition = CurrentTrackPosition + TrainCarLength[Car];
					#if (NextTrackPosition < CurrentTrackSize)
						#debug "<Straight>"
						// all on this nice straight piece of track.
						translate TrainCarLength[Car]/2*x
						rotate y*Angle
						translate Position
						#local CurrentTrackPosition = NextTrackPosition;
						#local Position = Position + TrainCarLength[Car]*<cos(radians(Angle)),0,-sin(radians(Angle))>;
						// Angle unchanged.
					#else
						#switch (TrackType[TrackCount + 1])
							#case(kStraight)
								#debug "<Straight to Straight>"
								translate TrainCarLength[Car]/2*x
								rotate y*Angle	
								translate Position
								#local CurrentTrackPosition = NextTrackPosition-CurrentTrackSize; // the over flow will start us off next time.
								#local Position = Position + TrainCarLength[Car]*<cos(radians(Angle)),0,-sin(radians(Angle))>;
								// Angle unchanged.
								#break
							#case(kRightCurve)
								#debug "<Straight to Right Curve>"
								// Need to do a funky partial rotate.
								#local A = CurrentTrackSize-CurrentTrackPosition;
								#local R = TrackSize[TrackCount + 1];
								#local B = sqrt(R*R + A*A);
								#local Th4 = atan2(A,R);
								#local CL = TrainCarLength[Car];
								#local Th3 = acos((CL*CL+B*B-R*R)/(2*CL*B));
								#local Th1 = 90-degrees(Th3+Th4);
								translate TrainCarLength[Car]/2*x
								rotate y*(Angle+Th1)
								translate Position
								
								#local Position = Position + CL*<cos(radians(Angle+Th1)),0,-sin(radians(Angle+Th1))>;
								#local C = sqrt(A*A+CL*CL-2*A*CL*cos(radians(Th1)));
								#local Th2 = acos(1-(C*C)/(2*R*R));
								#local Angle = Angle + degrees(Th2);
								#local CurrentTrackPosition = degrees(Th2);
								#break
							#case(kLeftCurve)
								#debug "<Straight to Left Curve>"
								// Need to do a funky partial rotate.
								#local A = CurrentTrackSize-CurrentTrackPosition;
								#local R = TrackSize[TrackCount + 1];
								#local B = sqrt(R*R + A*A);
								#local Th4 = atan2(A,R);
								#local CL = TrainCarLength[Car];
								#local Th3 = acos((CL*CL+B*B-R*R)/(2*CL*B));
								#local Th1 = 90-degrees(Th3+Th4);
								translate TrainCarLength[Car]/2*x
								rotate y*(Angle-Th1)
								translate Position
								
								#local Position = Position + CL*<cos(radians(Angle-Th1)),0,-sin(radians(Angle-Th1))>;
								#local C = sqrt(A*A+CL*CL-2*A*CL*cos(radians(Th1)));
								#local Th2 = acos(1-(C*C)/(2*R*R));
								#local Angle = Angle - degrees(Th2);
								#local CurrentTrackPosition = degrees(Th2);
								#break
							#else
								#debug "<Straight to ??>"
						#end
						#local TrackCount = TrackCount + 1;
					#end
					#break
				#case (kRightCurve)
					#local R = TrackSize[TrackCount];
					#local CL = TrainCarLength[Car];
					#local Th = degrees(acos(1-(CL*CL)/(2*R*R)));
					#if (CurrentTrackPosition + Th < 45)
						#debug "<Right Curve>"
						// All on the same arc.
						translate TrainCarLength[Car]/2*x
						rotate y*(Angle+Th/2)
						translate Position
						#local Position = Position + CL*<cos(radians(Angle + Th/2)),0,-sin(radians(Angle + Th/2))>;
						#local Angle = Angle + Th;
						#local CurrentTrackPosition = CurrentTrackPosition + Th;
					#else
						// Transitioning onto another straight or arc.
						#local Th1 = 45-CurrentTrackPosition;
						#local A = sqrt(2*R*R-2*R*R*cos(radians(Th1)));
						#local Th2 = (180-Th1)/2;
						
						#switch (TrackType[TrackCount + 1])
							#case (kStraight)
								#debug "<Right Curve to Straight>"
								#local Th3 = Th2+90;
								#local ThA = degrees(asin(A/CL*sin(radians(Th3))));
								#local Th4 = 180-Th3-ThA;
								
								translate TrainCarLength[Car]/2*x
								rotate y*(Angle + Th1/2 + Th4)
								translate Position
								#local Position = Position + CL* <cos(radians(Angle+ Th1/2 + Th4)),0,-sin(radians(Angle+ Th1/2 + Th4))>;
								#local Angle = Angle + Th1;
								#local CurrentTrackPosition = A*sin(radians(Th4))/sin(radians(ThA));
								#break
							#case (kRightCurve)
								#debug "<Right Curve to Right Curve>"
								#local R2 = TrackSize[TrackCount+1];
								#local Y = sqrt(A*A+R2*R2 - 2*A*R2*cos(radians(Th2)));
								#local Th6 = degrees(asin(A/Y*sin(radians(Th2))));
								#local Th5 = degrees(acos((Y*Y+R2*R2-CL*CL)/(2*Y*R2)))-Th6;
								#local Th3 = Th2 + (90-Th5/2);
								#local ThA = degrees(asin(A/CL*sin(radians(Th3))));
								#local Th4 = 180-Th3-ThA;
								#local ThRot = 90-Th2+Th4;
								
								translate TrainCarLength[Car]/2*x
								rotate y*(Angle + ThRot)
								translate Position
								#local Position = Position + CL*<cos(radians(Angle+ThRot)),0,-sin(radians(Angle+ThRot))>;
								#local Angle = Angle + Th1 + Th5;
								#local CurrentTrackPosition = Th5;
								#break
							#case (kLeftCurve)
								#debug "<Right Curve to Left Curve>"
								#local R2 = TrackSize[TrackCount+1];
								#local Th6 = 180-Th2;
								#local Y = sqrt(A*A+R2*R2-2*A*R2*cos(radians(Th6)));
								#local Th7 = degrees(asin(A/Y*sin(radians(Th6))));
								#local Th5 = degrees(acos((Y*Y+R2*R2-CL*CL)/(2*Y*R2)))-Th7;
								#local X = sqrt(2*R2*R2-2*R2*R2*cos(radians(Th5)));
								#local Th4 = degrees(acos((A*A+CL*CL-X*X)/(2*A*CL)));
								#local ThRot = 90-Th2-Th4;
								
/*								#debug concat("<R:",str(R,3,3),">")
								#debug concat("<R2:",str(R2,3,3),">")
								#debug concat("<CL:",str(CL,3,3),">")
								#debug concat("<A:",str(A,3,3),">")
								#debug concat("<X:",str(Y,3,3),">")
								#debug concat("<Y:",str(Y,3,3),">")
								#debug concat("<Th1:",str(Th1,3,3),">")
								#debug concat("<Th2:",str(Th2,3,3),">")
								#debug concat("<Th3:",str(Th3,3,3),">")
								#debug concat("<Th4:",str(Th4,3,3),">")
								#debug concat("<Th5:",str(Th5,3,3),">")
								#debug concat("<Th6:",str(Th6,3,3),">")
								#debug concat("<Th7:",str(Th7,3,3),">")*/
								translate TrainCarLength[Car]/2*x
								rotate y*(Angle + ThRot)
								translate Position
								#local Position = Position + CL*<cos(radians(Angle+ThRot)),0,-sin(radians(Angle+ThRot))>;
								#local Angle = Angle + Th1 - Th5;
								#local CurrentTrackPosition = Th5;
								#break
							#else
								#debug "<Right Curve to ??>"
						#end
						#local TrackCount = TrackCount + 1;
					#end
					#break
				#case (kLeftCurve)
					#local R = TrackSize[TrackCount];
					#local CL = TrainCarLength[Car];
					#local Th = degrees(acos(1-(CL*CL)/(2*R*R)));
					#if (CurrentTrackPosition + Th < 45)
						#debug "<Left Curve>"
						// All on the same arc.
						translate TrainCarLength[Car]/2*x
						rotate y*(Angle-Th/2)
						translate Position
						#local Position = Position + CL*<cos(radians(Angle - Th/2)),0,-sin(radians(Angle - Th/2))>;
						#local Angle = Angle - Th;
						#local CurrentTrackPosition = CurrentTrackPosition + Th;
					#else
						// Transitioning onto another straight or arc.
						#local Th1 = 45-CurrentTrackPosition;
						#local A = sqrt(2*R*R-2*R*R*cos(radians(Th1)));
						#local Th2 = (180-Th1)/2;
						#switch (TrackType[TrackCount + 1])
							#case (kStraight)
								#debug "<Left Curve to Straight>"
								#local Th3 = Th2+90;
								#local ThA = degrees(asin(A/CL*sin(radians(Th3))));
								#local Th4 = 180-Th3-ThA;
								
								translate TrainCarLength[Car]/2*x
								rotate y*(Angle - Th1/2 - Th4)
								translate Position
								#local Position = Position + CL* <cos(radians(Angle - Th1/2 - Th4)),0,-sin(radians(Angle - Th1/2 - Th4))>;
								#local Angle = Angle - Th1;
								#local CurrentTrackPosition = A*sin(radians(Th4))/sin(radians(ThA));
								#break
							#case (kLeftCurve)
								#debug "<Left Curve to Left Curve>"
								#local R2 = TrackSize[TrackCount+1];
								#local Y = sqrt(A*A+R2*R2 - 2*A*R2*cos(radians(Th2)));
								#local Th6 = degrees(asin(A/Y*sin(radians(Th2))));
								#local Th5 = degrees(acos((Y*Y+R2*R2-CL*CL)/(2*Y*R2)))-Th6;
								#local Th3 = Th2 + (90-Th5/2);
								#local ThA = degrees(asin(A/CL*sin(radians(Th3))));
								#local Th4 = 180-Th3-ThA;
								#local ThRot = 90-Th2+Th4;
								
								translate TrainCarLength[Car]/2*x
								rotate y*(Angle - ThRot)
								translate Position
								#local Position = Position + CL*<cos(radians(Angle-ThRot)),0,-sin(radians(Angle-ThRot))>;
								#local Angle = Angle - Th1 - Th5;
								#local CurrentTrackPosition = Th5;
								#break
							#case (kRightCurve)
								#debug "<Left Curve to Right Curve>"
								#local R2 = TrackSize[TrackCount+1];
								#local Th6 = 180-Th2;
								#local Y = sqrt(A*A+R2*R2-2*A*R2*cos(radians(Th6)));
								#local Th7 = degrees(asin(A/Y*sin(radians(Th6))));
								#local Th5 = degrees(acos((Y*Y+R2*R2-CL*CL)/(2*Y*R2)))-Th7;
								#local X = sqrt(2*R2*R2-2*R2*R2*cos(radians(Th5)));
								#local Th4 = degrees(acos((A*A+CL*CL-X*X)/(2*A*CL)));
								#local ThRot = 90-Th2-Th4;
								
								/*#debug concat("<R:",str(R,3,3),">")
								#debug concat("<R2:",str(R2,3,3),">")
								#debug concat("<CL:",str(CL,3,3),">")
								#debug concat("<A:",str(A,3,3),">")
								#debug concat("<X:",str(Y,3,3),">")
								#debug concat("<Y:",str(Y,3,3),">")
								#debug concat("<Th1:",str(Th1,3,3),">")
								#debug concat("<Th2:",str(Th2,3,3),">")
								#debug concat("<Th3:",str(Th3,3,3),">")
								#debug concat("<Th4:",str(Th4,3,3),">")
								#debug concat("<Th5:",str(Th5,3,3),">")
								#debug concat("<Th6:",str(Th6,3,3),">")
								#debug concat("<Th7:",str(Th7,3,3),">")*/
								translate TrainCarLength[Car]/2*x
								rotate y*(Angle - ThRot)
								translate Position
								#local Position = Position + CL*<cos(radians(Angle-ThRot)),0,-sin(radians(Angle-ThRot))>;
								#local Angle = Angle - Th1 + Th5;
								#local CurrentTrackPosition = Th5;
								#break
							#else
								#debug "<Left Curve to ??>"
						#end
						#local TrackCount = TrackCount + 1;
					#end
					#break
				#else
					#debug "<?>"
					translate 100
			#end
		}	
		#local Car = Car + 1;	
	#end
#end


#if (Render = 1)

	#include "TrackLayoutSamples.inc"
	
	#declare TrackNumSegments = 34;
	#declare Track = array[34] {kLongStraight,0,kMiniStraight,0,kLongStraight,0,kLongCurveR,0kLongCurveR,0,kShortCurveR,0,kLongStraight,0,kLongCurveR,0,kLongCurveR,0,kLongCurveR,0,kLongCurveR,0,kLongCurveR,0, kShortCurveL,0, kShortCurveL,0, kShortCurveL,0, kShortCurveL,0, kLongCurveR,0}
	
	union {
		#include "TrackBuilder.inc"
		//translate -10*x
	}
	
	#include "TrackCrossSection.inc"
	// 0 is straight, 1 is left curve, 2 is right curve
	#declare TrackUnderTrainType = array[17] {0,0,0,1,1,1,0,1,1,1,1,1,2,2,2,2,1}
	#declare TrackUnderTrainSize = array[17] {TrackUnitLength,TrackMiniUnitLength,TrackUnitLength,TrackUnitLength,TrackUnitLength,TrackShortUnitLength,TrackUnitLength,TrackUnitLength,TrackUnitLength,TrackUnitLength,TrackUnitLength,TrackUnitLength,TrackShortUnitLength,TrackShortUnitLength,TrackShortUnitLength,TrackShortUnitLength,TrackUnitLength}
	
	#declare NumCars = 42;
	#declare TrainCarLength  = array[NumCars]
	#declare TrainCars = array[NumCars]
	
	#include "TrainCars.inc"
	#local Count = 0;
	#while (Count < NumCars)
		#declare TrainCarLength[Count]  = TrainCarBaseSize.x;
		#declare TrainCars[Count] = object {MakeTrainBoxCar(color rgb <1,0,0>)}
		#local Count = Count + 1;
	#end
	// Featured object(s)
	PlaceTrainCarsOnTrack(TrainCars,TrainCarLength,TrackUnderTrainType,TrackUnderTrainSize)
	
	global_settings {ambient_light 2}
	plane{
		y,
		0
		pigment {color rgb <0,0.5,0.5>}
	}
	camera
	{
	  location  <2, 50 , -5>
	  look_at   <6 , 0.0 , 2>
	  orthographic
	}
	// create a regular point light source
	light_source
	{
	  0*x // light's position (translated below)
	  color <1,1,1>*1.5  // light's color
	  translate <-50 , 100 , -50>
	}
	
	background {color <1,1,1>}
#end
		
