// branches.inc
// David Morgan-Mar

/******
Usage:
Call this macro only inside a union{} with a tree bark texture defined after the Branches() call.
Before calling the macro, define a BranchingArray which contains the number of branches to produce
at each level of the macro.

Example:
    #declare BranchingArray = array[6] {7,2,3,4,4,3}
    #declare Leaf1 = object { fill in }
    union {
        Branches(6, 0.06, 0.6, 0.6, 0.4, 3, 1, Leaf1, 0,0,1, 0,1,0)
        texture { Bark_Texture }
    }

Parameters:
N:         Number of levels to recurse - should equal size of BranchingArray[]
Thickness: Initial thickness of trunk
Height:    Initial height of first trunk segment
Shorter:   Fraction by which additional segments are shorter
Thinner:   Fraction by which additional segments are thinner
Angle:     Relative opening angle of branchlets
Leaves:    Show leaves or not
Leaf:      Object identifier which contains a definition of a leaf object, located at <0,0,0>
x1,y1,z1:  Location of base of trunk
v1,v2,v3:  Vector of initial growth direction - set <v1,v2,v3> = <0,1,0> for vertical trunk
******/

#macro BranchLeaf(Size, x1,y1,z1)
sphere { <0,0,0>, Size
    scale <0.1,0.5,1>
    rotate <360*rand(Rseed),360*rand(Rseed),360*rand(Rseed)>
    texture {
        pigment { rgb <0,1,0> }
    }
    translate <x1,y1,z1>
}
#end // macro BranchLeaf


#macro Branches(N, Thickness, Height, Shorter, Thinner, Angle, Leaves, Leaf, x1,y1,z1, v1,v2,v3, seed1)
#ifndef (Rseed)
    #local Rseed = seed(seed1);
#end

#local norm = sqrt(v1*v1 + v2*v2 + v3*v3);
#local x2 = x1 + Height * v1/norm;
#local y2 = y1 + Height * v2/norm;
#local z2 = z1 + Height * v3/norm;

cone {
    <x1,y1,z1>, Thickness,
    <x2,y2,z2>, Thickness * Thinner
}

#if (N > 0)
    #local i = BranchingArray[dimension_size(BranchingArray, 1) - N];
    #while (i > 0)
        #local w1 = v1 + Angle*(rand(Rseed) - 0.5);
        #local w2 = v2 + Angle*(rand(Rseed) - 0.5);
        #local w3 = v3 + Angle*(rand(Rseed) - 0.5);
        Branches(N-1, Thickness*Thinner, Height*Shorter*(0.95+rand(Rseed)*0.1),
            Shorter, Thinner, Angle, Leaves, Leaf, x2,y2,z2, w1,w2,w3, seed1)
        #local i = i - 1;
    #end
#else
    #if (Leaves)
        object { Leaf
            rotate <360*rand(Rseed),360*rand(Rseed),360*rand(Rseed)>
            translate <x2,y2,z2>            
        }
    #end
#end

#end // macro Branches
