//****************************************************************************
//
// Author: Ali Lakhia
// File:   minars.inc
// Date:   January 15th, 2000
//
//***************************************************************************/

// Variable to determine the thickness of each section in the slice
#declare sec_slice_thickness = 0.001;

// Texture for green marble
#declare green_marble = pigment {
  marble
  turbulence 0.5
  lambda 1.5
  omega 0.8
  octaves 35
  frequency 1.2
  color_map {
    [0.0 color rgb <0.45, 0.47, 0.45>]
    [0.3 color rgb <0.7, 0.8, 0.7>]
    [0.75 color rgb <0.48, 0.51, 0.48>]
    [0.9 color rgb <0.6, 0.7, 0.6>]
  }
}

// Macro that returns the breadth given the full length
#macro Length_2_breadth(full_length)
  // Constant ratio between length of section and breadth
  #local sec_ratio = sqrt(2) + 1;
  (full_length/sec_ratio)
#end

// Create a single section for the minar, 
// Each section has to be rotated 4 times to complete slice of minar
#macro Minar_slice_section(lower_length, upper_length, height_start, height)
  #if (lower_length != upper_length)
    #local lower_breadth = Length_2_breadth(lower_length);
    #local upper_breadth = Length_2_breadth(upper_length);
    #local diff_breadth = (upper_length - lower_length)/2;
    #local adjusted_height =  sqrt(height*height + diff_breadth*diff_breadth);
    #local slice_angle = -degrees(atan2(height, diff_breadth));
    union {
      prism {
        linear_spline
        0, sec_slice_thickness, 5,
        <-lower_breadth/2, 0>,
        <lower_breadth/2, 0>,
        <upper_breadth/2, adjusted_height>,
        <-upper_breadth/2, adjusted_height>,
        <-lower_breadth/2, 0>
        rotate slice_angle*x
        translate height_start*y + lower_length*z/2
      }
      prism {
        linear_spline
        0, sec_slice_thickness, 5,
        <-lower_breadth/2, 0>,
        <lower_breadth/2, 0>,
        <upper_breadth/2, adjusted_height>,
        <-upper_breadth/2, adjusted_height>,
        <-lower_breadth/2, 0>
        rotate slice_angle*x
        translate height_start*y + lower_length*z/2
        rotate 180*y
      }
    }
  #else
    #local breadth = Length_2_breadth(lower_length);
    box {
      <-breadth/2, height_start, -lower_length/2>,
      <breadth/2, height_start + height, lower_length/2>
    }
  #end
#end

// Minar base
#declare Minar_base = union
{
  box {
    <-0.5, 0, -0.5>,
    <0.5, 1, 0.5>
  }
  box {
    <-0.5, -0.3536, -0.3536>,           // 0.3536 = sqrt(2)/4
    <0.5, 0.3536, 0.3536>
    rotate 45*x
    translate 1*y
  }
  pigment { green_marble }
}

// Minar base slice
#declare Minar_slice_base = union
{
  object {
    // Base
    Minar_slice_section(1, 1, 1.1, 0.45)
    pigment { green_marble }
  }

  // Brown stone base
  Minar_slice_section(1, 1.053, 1.55, 0.1)
  Minar_slice_section(1.053, 1.316, 1.65, 0.2)
}

// Slice of large frame of minar.
#declare Minar_slice_large_frame = union
{
  // Floor with cylinder cutout (for fence)
  difference {
    union {
      Minar_slice_section(1.316, 1.474, 1.85, 0.15)
      Minar_slice_section(1.474, 1.474, 2, 0.1)
    }
    cylinder {
      <0, 1.851, 0>,
      <0, 2.101, 0>,
      1.316/2
    }
  }

  // Lights in center of large frame
  #if (Lights_on > 0)
    object {
      Create_halo(Lights_color, 1.2, 1.7)
      scale <1, 0.45, 1>
      translate (2.1+2.55)/2*y
    }
    light_source
    {
      <0, (2.1+2.55)/2, 0.9/2.1>,
      Lights_color*Lights_on
      #if (Minar_area_lights_on)
        area_light <0, 0.45, 0>, <.3, 0, 0>, 
        Minar_area_lights_on, Minar_area_lights_on
      #end
      fade_distance 1
      fade_power 2
    }
    light_source
    {
      <0, (2.1+2.55)/22.1, -0.9/2.1>,
      Lights_color*Lights_on
      #if (Minar_area_lights_on)
        area_light <0, 0.45, 0>, <0.3, 0, 0>, 
        Minar_area_lights_on, Minar_area_lights_on
      #end
      fade_distance 1
      fade_power 2
    }
  #end
  
  // Frame middle column
  object {
    Minar_slice_section(0.8, 0.8, 1.851, 0.15+0.1+0.45)
    pigment { color Lights_color }
  }

  // Circumference of large frame
  cylinder {
    <Length_2_breadth(-1.474)/2 + 0.02, 2.1, 1.474/2 - 0.02>
    <Length_2_breadth(-1.474)/2 + 0.02, 2.55, 1.474/2 - 0.02>
    0.02
  }
  cylinder {
    <Length_2_breadth(1.474)/2 - 0.02, 2.1, -1.474/2 + 0.02>
    <Length_2_breadth(1.474)/2 - 0.02, 2.55, -1.474/2 + 0.02>
    0.02
  }

  // Rest of frame and roof (made with green marble)
  Minar_slice_section(1.474, 1.474, 2.55, 0.2)
  Minar_slice_section(1.947, 1.947, 2.75, 0.05)
  object {
    Minar_slice_section(1.947, 0.947, 2.8, 0.25)
    pigment { green_marble }
  }
}

// Slice of small frame
#declare Minar_slice_small_frame = union
{
  Minar_slice_section(0.895, 1.105, 5.85, 0.15)
  Minar_slice_section(1.105, 1.158, 6, 0.08)
  Minar_slice_section(1.158, 1.211, 6.08, 0.1)

  // Lights in center of small frame
  #if (Lights_on > 0)
    object {
      Create_halo(Lights_color, 1, 1.7)
      scale <1, 0.35, 1>
      translate (6.18+6.43)/2*y
    }
    light_source
    {
      <0, (6.18+6.43)/2, 0.7/2.1>,
      Lights_color*Lights_on
      fade_distance 1
      fade_power 2
    }
    light_source
    {
      <0, (6.18+6.43)/2, -0.7/2.1>,
      Lights_color*Lights_on
      fade_distance 1
      fade_power 2
    }
  #end

  // Frame middle column
  object {
    Minar_slice_section(0.5, 0.5, 5.85, 0.15+0.08+0.1+0.35)
    pigment { color Lights_color }
  }

  // Cicumference of small frame
  cylinder {
    <Length_2_breadth(-1.211)/2 + 0.015, 6.18, 1.211/2 - 0.015>,
    <Length_2_breadth(-1.211)/2 + 0.015, 6.43, 1.211/2 - 0.015>,
    0.015
  }
  cylinder {
    <Length_2_breadth(1.211)/2 - 0.015, 6.18, -1.211/2 + 0.015>,
    <Length_2_breadth(1.211)/2 - 0.015, 6.43, -1.211/2 + 0.015>,
    0.015
  }
}

// Create Minar slice rim
#declare Minar_slice_rim = union
{
  cylinder {
    <Length_2_breadth(-0.726)/2, 0, 0.726/2>,
    <Length_2_breadth(0.726)/2, 0, 0.726/2>,
    0.075
    open
  }    
  cylinder {
    <Length_2_breadth(-0.726)/2, 0, -0.726/2>,
    <Length_2_breadth(0.726)/2, 0, -0.726/2>,
    0.075
    open
  }
  sphere {
    <Length_2_breadth(-0.726)/2, 0, 0.726/2>, 0.075
  }
  sphere {
    <Length_2_breadth(0.726)/2, 0, -0.726/2>, 0.075
  }
}

// A single slice of the minar. Created in sections from the bottom up.
#declare Minar_slice = union
{
  // Base of Minar
  object {
    Minar_slice_base
  }

  // Large frame of minar
  object {
    Minar_slice_large_frame
  }

  // Long mid section and rim
  difference {
    #declare sec_slice_thickness = 0.08;
    Minar_slice_section(0.947, 0.842, 3.05, 2.4 + 0.05)
    #declare sec_slice_thickness = 0.001;
    union {
      cylinder {
        <0, 5.25, -0.9>,
        <0, 5.25, 0.9>,
        0.08
      }
      box {
        <-0.08, 5.07, -0.9>
        <0.08, 5.25,  0.9>
      }
      pigment { color Lights_color }
    }
  }
  #if (Lights_on > 0)
    object {
      Create_halo(Lights_color, 1.1, 0.9)
      scale <1, 0.26, 1>
      translate (5.07+5.25)/2*y
    }
  #end
  cylinder {
    <0, 5, 0>,
    <0, 5.5, 0>,
    0.18
    pigment { color Lights_color }
  }
  object {
    Minar_slice_rim
    translate (5.45+0.075)*y
  }
  Minar_slice_section(0.842, 0.895, 5.6 - 0.05, 0.25 + 0.05)
  
  // Small frame base
  object {
    Minar_slice_small_frame
  }

  // Rest of minar
  Minar_slice_section(1.211, 1.211, 6.43, 0.08)
  Minar_slice_section(1.211, 1.211, 6.51, 0.05)
  object {
    Minar_slice_section(1.421, 0.526, 6.56, 0.21)
    pigment { green_marble }
  }
  Minar_slice_section(0.526, 0.579, 6.77, 0.21)
  Minar_slice_section(0.579, 0.474, 6.98, 0.1)
  Minar_slice_section(0.474, 0.579, 7.08, 0.05)
  Minar_slice_section(0.579, 0.579, 7.13, 0.08)
  Minar_slice_section(0.579, 0.447, 7.21, 0.32)
  union {
    Minar_slice_section(0.447, 0.079, 7.53, 0.18)
    Minar_slice_section(0.079, 0.05, 7.71, 0.5)
    texture { T_Gold_1B }
  }
}

// Create crescant for minar
#declare Minar_crescent = difference 
{
  cylinder {
    <0, 0.2235, -0.0395>,
    <0, 0.2235, 0.0395>,
    0.2235
  }
  cylinder {
    <0.04, 0.2235+0.04, -0.04>,
    <0.04, 0.2235+0.04, 0.04>,
    0.2235-0.03
  }
  texture { T_Gold_1B }  
}

// Creates full minar by rotating slice 45 degrees
#macro Minar_create()
  union {
    object {
      Minar_slice
    }
    object {
      Minar_slice
      rotate 45*y
    }
    object {
      Minar_slice
      rotate 90*y
    }
    object {
      Minar_slice
      rotate 135*y
    }
  }
#end

// Minar
#declare Minar = union
{
  object {
    Minar_base
    scale <1, 1, 1.42>
  }
  object {
    Minar_create()
  }
  object {
    Minar_crescent
    translate 8.21*y
  }
  pigment { 
    brick rgb <0.62, 0.56, 0.4>, rgb <0.78, 0.7, 0.5>
    brick_size <0.2, 0.05, 0.2>
    mortar 0.008
  }
  normal {
    brick 0.05
    brick_size <0.2, 0.05, 0.2>
    mortar 0.008
  }
  finish {
    #if (Lights_on > 0)
      ambient 0.3
    #else
      ambient 0.03
    #end
  }
}

// These are holes in the wall that make the doorway
#declare Minar_wall_doorway = union
{
  cylinder {
    <0, 0.3, -0.3001>,
    <0, 0.3, 0.3001>,
    0.2
  }
  box {
    <-0.2, -0.0001, -0.3001>,
    <0.2, 0.3, 0.3001>
  }
}

// This is the wall that connects the two minars together
#declare Minar_wall_connection = difference
{
  box {
    <0, 0, -0.3>,
    <3*1.56, 0.6, 0.3>
  }
  object {
    Minar_wall_doorway
    translate (3*1.56/2 - 0.8)*x
  }
  object {
    Minar_wall_doorway
    translate 3*1.56/2*x
  }
  object {
    Minar_wall_doorway
    translate (3*1.56/2 + 0.8)*x
  }
}

// Railing between minars, on top of wall connector
#declare Minar_railing = union
{
  #local section_size = 1.56*3/64;      // 64 sections pieces
  #local section_height = section_size*8*0.3;
  #local l_section = section_size*1.5;
  #local l_section_height = l_section*8*0.4;
  #local i = 0;
  #local j = 0;

  #while (i <= 5)
    box {
      <25*i*section_size/2, 0, -l_section*2>,
      <25*i*section_size/2 + l_section, l_section_height, -l_section*2>
    }
    #if (i <= 4)
      #while (j < 7)
        box {
          <(25*i + 3*j + 4)*section_size/2, 0, -section_size/2>,
          <(25*i + 3*j + 6)*section_size/2, section_height, -section_size/2>
        }
        #local j = j + 1;
      #end                              // while for J
    #end                                // if for J
    #local i = i + 1;
    #local j = 0;
  #end                                  // while for I
  
  // Top of railing
  box {
    <0, section_height, -section_size/2>,
    <1.56*3, section_height+section_size, section_size/2>
  }
}

// Top level wall that connects the two minars, has doorways 
// and railing on top
#declare Minar_wall = union
{
  object {
    Minar_wall_connection
  }
  object {
    Minar_railing
    translate <0, 0.6, 0.3 - (1.56*3/64)*3>
  }
  object {
    Minar_railing
    translate <0, 0.6, -0.3 + (1.56*3/64)*3>
  }
}