//      Fortress
//      by Duncan Gray (c) 02/03 2k1

#declare main_render=on;

#include "colors.inc"
#include "textures.inc"
#include "metals.inc"
        
global_settings {
  assumed_gamma 1.6                                             
  max_intersections 350
  max_trace_level 350
}

// #declare palm_cutoff=.0001;     // cuts short the palm trees  
// #declare beach_cutoffz=.25;    // shortens beach
// #declare beach_cutoffx=.4;    // shallows beach
// #declare quickrender=on;       // mainly changes textures
// #declare quicksand=4;          // leaves holes in beach 4=horiz, 5=vert
// #declare posit_render=on;      // skips castle
// #declare quick_crab=1;         // reduce shell to matrix of blob-spheres

#declare _bcount=0;

#declare cam_pos=<-1.4,3,1>;
#declare cam_tgt=<.45,1.75,10>;
#declare cam_vew=40;

#declare light_pos=<-500,1000,-1000>;
#declare pen_light=.2;
#declare amb_light=.15;

#declare _cp=<0,-.7,10>;
#declare _ca=<0,-18,0>;

#declare crab_pos=<1.3,1,6>;
#declare crab_ang=<0,60,0>;

#declare boat_ang=<-10,-30,10>;
#declare boat_pos=<-7,1.5,90>;

#declare hut_ang=<0,30,0>;
#declare hut_pos=<32,6,133>;

// #declare cam_tgt=hut_pos;
// #declare cam_pos=hut_pos+<-25,4,-30>;

// returns a value representing a point along the path
// p1 to p4. leaving p1 in the direction of p2, and arriving
// at p4 from a direction to p3.
// cc is the current position on the spine which starts
// at sc and ends at ec (typ. 0,1)
// if cc<sc then p1 is returned. if cc>ec then p4 is returned
#macro flight_spline(_p1,_p2,_p3,_p4,_sc,_ec,_cc)
  #if (_cc<=_sc)
    (_p1)
  #else
    #if (_cc>=_ec)
      (_p4)
    #else
      #local _ci=(_cc-_sc)/(_ec-_sc);
      #local _co=(1-_ci);
      //  p = (p1*co^3)+(3*p2*co^2*ci)+(3*p3*ci^2*co)+(p4*ci^3) 
         ( ( _p1*pow(_co,3) ) + ( _p2*3*pow(_co,2)*_ci ) + ( _p3*3*pow(_ci,2)*_co ) + ( _p4*pow(_ci,3) ) )
    #end
  #end
#end  
// returns position on a straight linear line from
// p1 to p2 where cc represents the current position
// and sc and ec are time-indexes for endpoints (typ. 0,1)
// if cc<sc then p1 is returned. if cc>ec then p2 is returned
#macro linear_move(_p1,_p2,_sc,_ec,_cc)
  #if (_cc<=_sc)
    (_p1)
  #else
    #if (_cc>=_ec)
      (_p2)
    #else
      ((_p1)+((_p2-_p1)/(_ec-_sc))*(_cc-_sc))      
    #end
  #end
#end

// Blob construction macro's (adapted from Rune's - Cheers Rune)
// rebt(threshold) use this to set the threshold for the blob object
#macro rebt(_threshold)
   #declare reb_thresh=_threshold;
   threshold _threshold
#end
//  
#macro reb(_visibleradius,_blobbiness)
   #local _strength = reb_thresh*(1+1/_blobbiness);
   #local _actualradius =_visibleradius/sqrt(1-sqrt(reb_thresh/_strength));
   _actualradius, _strength
#end
//


#declare sky_color=
  texture {
    pigment {
      gradient y
      color_map {
        [ 0 color rgb <.4,.5,.8> ]
        [ 0.7 color rgb <.4,.5,.8> ]
        [ .9 color rgb <0.2,.2,.3> ]
        [ 1 color rgb <0.2,.2,.3> ]
      }
      rotate <0,0,-20>
      rotate <0,110,0>
    }
    finish {
      ambient 1.2
      diffuse 0
    }
  }

#declare sky_cloud=
  texture {
    pigment {
      bozo
      turbulence .2
      lambda 1.5
      omega .85
      octaves 11
      color_map {
        [ 0 color rgbt<1,1,1,1> ]
        [ 0.75 color rgbt<1,1,1,1> ]
        [ 0.95 color rgbt<1,1,1,.5> ]
        [ 2 color rgbt<1,1,1,0> ]
      }
    }
    finish {
      ambient 1
      diffuse 0
    }
    scale <.5,.05,.5>
    scale .25
    rotate <0,-60,-1>
  }


object {
  sphere { <0,0,0> 1 inverse  }
  texture { sky_color }
  texture { sky_cloud translate <0,-.006,0> rotate <0,-45,0> }
  translate <0,-.93,0>
  scale 10000
  translate <0,1,0>
  hollow
  no_shadow
}

// this next calculates the rotation and position required for
// the Gray message in the bottom right corner
  #declare cam_dir=vnormalize(cam_tgt-cam_pos);
  #declare cam_head=180*atan2(cam_dir.x,cam_dir.z)/pi;
  #declare cam_dir=vrotate(cam_dir,<0,-cam_head,0>);
  #declare cam_ang=-180*atan2(cam_dir.y,cam_dir.z)/pi;
  
  #declare dimmer=.85;
  #declare t2dim=.75;
  
  union {
    box { <-.1,-.025,0.001>,<.1,.025,0.002>
      texture { 
        pigment { 
          rgbf <1,1,1,dimmer>
        }
        finish {
          ambient 0
          diffuse 0
        }
      }
    }  
    object {
      text {
        ttf "c:\windows\comic.ttf"
        "Duncan Gray."
        1,0                      
        translate <-3,-.4,0>
        scale <.03,.03,.00001>
      }
      texture { 
        pigment { 
          rgbf <1,1,1,t2dim>
        }
        finish {
          ambient 1
          diffuse 0
        }
      }
    }  
    scale .25            
    translate <.27,-.215,.5>
    translate <0,0,.15>
    rotate <cam_ang,cam_head,0>
    translate cam_pos 
    no_shadow
  }
  
// this macro is a way of optimising the creation of bi-cubic blob-patches.
// by calling this function with the centre of the proposed blob component,
// the macro returns 1 if the position lies in the camera's field of view.
// the calling routine can therefore skip the inclusion of that component.
// n.b the angle cam_vew should be kept wide enough to ensure that components
// centred outside that area do not impact on the picture.
#macro isinview(_pos)
  // assumes {cam_ang},{cam_head},{cam_pos} and {cam_field} have been set
  #declare _inside=1;
  #local t_pos=_pos-cam_pos;
  #local t_pos=vrotate(t_pos,<0,-cam_head,0>);
  #local t_pos=vrotate(t_pos,<-cam_ang,0,0>);
  // camera axis and translation removed, now check to see is t_pos lies inside 
  // a cone extending from <0,0,0> in the z direction of angle cam_field.
  #local t_rad=sqrt(pow(t_pos.x,2)+pow(t_pos.y,2));
  #local t_ang=abs(degrees(atan2(t_rad,t_pos.z)));
  #if(t_ang>cam_vew)
    #declare _inside=0;
  #end
  (_inside)
#end

camera {
  location cam_pos
  up <0,1,0>
  right <4/3,0,0>
  direction <0,0,1.45>
  look_at cam_tgt
}  

fog { 
  fog_type 1
  distance 150
  //fog_offset 0
  //fog_alt 10
  //color rgbt<.6,.7,.8,.5>
  color rgbt<0,0,0,.9>
}

#local _fd=10;
#local _fp=2;
light_source {
  0,
  (1-pen_light)
//  area_light <30,0,-15>,<0,30,0>,5,5
  translate light_pos
//  (1-pen_light)*((1+pow(vlength(light_pos)/_fd,_fp))/2)           // adjust White by pen_light and counter the attenuation
//  fade_distance _fd
//  fade_power _fp
}
light_source {
  0,
  pen_light
  translate light_pos
  shadowless
}

#declare basesand=<.95,.92,.7>;

#declare sand_col1=basesand*.75;
#declare sand_col2=basesand*.95;
#declare sand_col3=basesand;

#ifdef(quickrender)
  #declare beach_sand_tex=
  texture {
    pigment { color basesand }
    finish { ambient amb_light diffuse .7 roughness .5 specular .01 }
  }
  #declare water_tex=
  texture {
    pigment {
      color rgb <0.2, 0.4, 0.8>
    }
    finish {
      ambient amb_light
      diffuse 0.8
      roughness 0.01
      specular 0.5      
    }
  }
#else
  #declare beach_sand_tex=
  texture {
    pigment {
      gradient y
      scale 30
      translate <0,-.2,0>
      pigment_map {
        [ 0 color sand_col1 ]
        [ .2 color sand_col2 ]
        [ .5 color sand_col3 ]
        [ 1 color sand_col3 ]
      }
    }
    finish {
      ambient amb_light
      diffuse .8
      roughness .5
      specular .01
    }
    normal {   
      granite -.05 poly_wave 2 scale .05/20 
    }
  }
  #declare water_tex=
  texture {
    pigment {
      color rgbt <0.1, 0.2, 0.2, 0.7>
    }
    finish {
      ambient amb_light/3
      diffuse 0.1
      brilliance 0.9
      roughness 0.01
      specular 0.9      
      reflection 0.3
      metallic 0.5
    }
  } 
#end     
#declare w_seed=seed(1);
#macro w_jitt(j_qty)
  #local j_rad=j_qty*rand(w_seed);
  #local j_ang=<360*rand(w_seed),360*rand(w_seed),360*rand(w_seed)>;
  ( vrotate(<0,1,0>,j_ang)*j_rad )
#end

#macro blob_watersurf(_p1,_p2,_p3,_p4, _br,_bb,_neg, _xs,_zs)
  #local _xf=1/_xs;
  #local _zf=1/_zs;
  
  #local _cs = reb_thresh*(1+1/_bb);
  #local _cr =_br/sqrt(1-sqrt(reb_thresh/_cs));
  
  #local _z=0;
  #while (_z<1-(_zf/2))
  
    #local _cp=(flight_spline(_p1,_p2,_p3,_p4,0,1,_z));
    #local _tp=_cp;
    #local _x=0;
    #while (isinview(_tp) )//& _x<1)
      
      #local _tp=_cp-(_x*x*2.5)+(flight_spline(<0,0,0>,<0,0.05,0>,<0,-.2,0>,<0,-.2,0>,0,1,_x))+w_jitt(.05);
      sphere { _tp,_cr,_cs*_neg }
                
      #local _x=_x+_xf;
    #end
    #render concat(str(100*_z/(1-(_zf/2)),0,0),"%\n")
    #local _z=_z+_zf;
  #end  
#end

merge { object { plane { y,0 } translate <0,-.05,0> }
//object {
  blob {
    rebt (.5)
    #render "Water patch 1\n"
    blob_watersurf(<-3.5,0,8>,<-3,0,10>,<-3,0,12>,<-3.5,0,14>, .1,10,1, 10,24)
    #render "Water patch 2\n"
    blob_watersurf(<-3.5,0,14>,<-3,0,17>,<-3,0,20>,<-3.5,0,23>, .1,10,1, 10,36)
    #render "Water patch 3\n"
    blob_watersurf(<-3.5,0,23>,<-3.5,0,27>,<-3.5,0,31>,<-3.5,0,35>, .1,10,1, 10,48)
    
    #render "Water patch 4\n"
    blob_watersurf(<-3.5,0,35>,<-3.5,0,41>,<-5,0,47>,<-6,0,53>, .1,10,1, 10,72)    
    #render "Water patch 5\n"
    blob_watersurf(<-6,0,53>,<-8,0,65>,<-16,-.2,77>,<-18,-.4,89>, .1,10,1, 10,144)
    sphere { <0,-20,0>,1,1 }
  }
  texture { water_tex }
}

#declare j_seed=seed(1);
#macro jitt(j_qty)
  #local j_rad=j_qty*rand(j_seed);
  #local j_ang=<360*rand(j_seed),360*rand(j_seed),360*rand(j_seed)>;
  ( vrotate(<0,1,0>,j_ang)*j_rad )
#end
                        
// blob_wall returns all the spherical blob components
// to build a wall at position p1, angle a1 of width and
// height specified. Components will be spaced bs apart
// and will have component radius's br. the blobyness is
// given by bb, jt is jitter in the wall construnction.
#macro blob_wall(_p1,_a1,_wid,_hei,_bs,_br,_bb,_neg,_jt)
   #local _cs = reb_thresh*(1+1/_bb);
   #local _cr =_br/sqrt(1-sqrt(reb_thresh/_cs));
   
   #local _xs=int(_wid/_bs);
   #if(_xs<1)
     #render "Wall warning - no blobs in width, vertical cylinder will be created\n"
     #local _xs=1;
   #end
   #local _ys=int(_hei/_bs);
   #if(_ys<1)
     #render "Wall warning - no blobs in height, horizontal cylinder will be created\n"
     #local _ys=1;
   #end
   #local _cx=0;
   #while (_cx<=_xs)
     
     #local _cy=0;
     #while (_cy<=_ys)
     
       #local _pp=_p1+vrotate(jitt(_jt)+<-_wid/2+_cx*_wid/_xs,-_hei/2+_cy*_hei/_ys,0>,_a1);
       #local _pp=_cp+vrotate(_pp,_ca);

       sphere { _pp,_cr,_cs*_neg }
       #declare _bcount=_bcount+1;
                        
       #local _cy=_cy+1;
     #end
     
     #local _cx=_cx+1;
   #end
#end
// blob_tower returns the spherical components required to make
// a tower at position p1 leaning in toward the top at an angle
// of wl. tower radius is tr, with blob components being spaced
// an average of bs units apart. blob radius is br and bb gives
// it's blobbyness. jt is jitter in the tower construnction.
#macro blob_tower(_p1,_wl,_tr,_th,_bs,_br,_bb,_neg,_jt,_cf)
   #local _cs = reb_thresh*(1+1/_bb);
   #local _cr =_br/sqrt(1-sqrt(reb_thresh/_cs));
   
   #local _rs=int(2*pi*_tr/_bs);
   #if(_rs<1)
     #render "Tower warning - no blobs in circumphrence, cylinder will be created\n"
     #local _rs=1;
   #end
   #local _ys=int(_th/_bs);
   #if(_ys<1)
     #render "Tower warning - no blobs in height, torus will be created\n"
     #local _ys=1;
   #end
   
   #local _ra=0;
   #while (_ra<_rs)
     
     #local _cy=0;
     #while (_cy<=_ys)
                     
       #local _pp=vrotate(jitt(_jt)+<0,-_th/2+_cy*_th/_ys,0>,<_wl,0,0>);
       #local _pp=_p1+vrotate(_pp+<0,0,-_tr>,<0,_ra*360/_rs,0>);
       #local _pp=_cp+vrotate(_pp,_ca);
       
       #if(_cf>0 & _cy>_ys*.75)
         #if ( mod((_ra/(_rs/_cf)),1)>.75)
           sphere { _pp,_cr*.95,-_cs*_neg*2.5 }
         #else
           sphere { _pp,_cr,_cs*_neg }
         #end
       #else
         sphere { _pp,_cr,_cs*_neg }
       #end
       #declare _bcount=_bcount+1;
       
       #local _cy=_cy+1;
     #end
     
     #local _ra=_ra+1;
   #end
#end
// adds sphericalcomponent, but rotates to castle coordinates.
#macro blob_sphere(_p1,_br,_bb,_neg)
   #local _cs = reb_thresh*(1+1/_bb);
   #local _cr =_br/sqrt(1-sqrt(reb_thresh/_cs));

   #local _pp=_cp+vrotate(_p1,_ca);
   sphere { _pp,_cr,_cs*_neg }
   #declare _bcount=_bcount+1;
   
#end
// adds a thing a bit like a wall, but with a spline for the lower or upper edge.
// used for the bridge arch, and window cutouts.
#macro blob_archedwall(_p1,_a1,_wid,_h1,_h2a,_h2b,_h2c,_h2d,_bs,_br,_bb,_neg,_jt)
   #local _cs = reb_thresh*(1+1/_bb);
   #local _cr =_br/sqrt(1-sqrt(reb_thresh/_cs));
   
   #local _xs=int(_wid/_bs);
   #if(_xs<1)
     #render "ArchedWall warning - no blobs in width, vertical cylinder will be created\n"
     #local _xs=1;
   #end
   #local _cx=0;
   #while (_cx<=_xs)
     
     #local _hei=(flight_spline(_h2a,_h2b,_h2c,_h2d,0,_xs,_cx));
     #local _ys=abs(int((_hei-_h1)/_bs));
     #if(_ys<1)
       #render "ArchedWall warning - no blobs in height, horizontal cylinder will be created\n"
       #local _ys=1;
     #end
     
     #local _cy=0;
     #while (_cy<=_ys)
     
       #local _pp=_p1+vrotate(jitt(_jt)+<-_wid/2+_cx*_wid/_xs,_h1+(_cy*(_hei-_h1)/_ys),0>,_a1);
       #local _pp=_cp+vrotate(_pp,_ca);
       
       sphere { _pp,_cr,_cs*_neg }
       #declare _bcount=_bcount+1;
                        
       #local _cy=_cy+1;
     #end
     
     #local _cx=_cx+1;
   #end
#end
// windowslit macro

#macro blob_windowslit(_p1,_a1,_hei,_dpt,_bs,_br,_bb,_neg,_jt)
   #local _cs = reb_thresh*(1+1/_bb);
   #local _cr =_br/sqrt(1-sqrt(reb_thresh/_cs));
   
   #local _ys=int(_hei/_bs);
   #if(_ys<1)
     #render "Windowslit warning - no blobs in height, cylindrical hole will be created.\n"
     #local _ys=1;
   #end
        
   #local _cy=0;
   #while (_cy<=_ys)
     
     #local _pp1=_p1+vrotate(jitt(_jt)+<0,-_hei/2+_cy*_hei/_ys,-_dpt/2>,_a1);
     #local _pp1=_cp+vrotate(_pp1,_ca);
     #local _pp2=_p1+vrotate(jitt(_jt)+<0,-_hei/2+_cy*_hei/_ys,_dpt/2>,_a1);
     #local _pp2=_cp+vrotate(_pp2,_ca);
      
     cylinder { _pp1,_pp2,_cr,_cs*_neg }
     #declare _bcount=_bcount+1;
     #if (_cy=0 | _cy=_ys)
       cylinder { _pp1,_pp2,_cr*1.5,_cs*_neg }
       #declare _bcount=_bcount+1;
     #end                        
     #local _cy=_cy+1;
   #end
#end


#macro blob_beachcurve(p1,p2,p3,p4, a1,a2,a3,a4, s1,s2,s3,s4 ,xs,zs )
  #local _xf=1/xs;
  #local _zf=1/zs;
  
  #local _points=array[_xf+1][_zf+1]
  
  #local _xp=0;
  #while (_xp<=_xf)
    #local _jf=(flight_spline(0,0.25,3,2,0,_xf,_xp));
    #local _zp=0;
    #while (_zp<=_zf)
      
      #local _points[_xp][_zp]=vrotate( (flight_spline(s1,s2,s3,s4,0,_xf,_xp)),
                         (flight_spline(a1,a2,a3,a4,0,_zf,_zp)) ) +
                         (flight_spline(p1,p2,p3,p4,0,_zf,_zp)) + jitt(_jf);
      
      #local _zp=_zp+1;
    #end
  
    #local _xp=_xp+1;
  #end

  #render "Generating Beach \n"

  #ifdef(beach_cutoffz)
    #local _ez=(_zf-1)*beach_cutoffz;
  #else
    #local _ez=_zf-1;
  #end 
  #ifdef(beach_cutoffx)
    #local _ex=(_xf-1)*beach_cutoffx;
  #else
    #local _ex=_xf-1;
  #end
  #local _xp=0;
  #while (_xp<=_ex)
    
    #render concat(str(100*_xp/_xf,0,0),"%\n")
    #local _zp=0;
    #while (_zp<=_ez)
    
      #local _r1c1=_points[_xp][_zp+1];
      #local _r1c4=_points[_xp+1][_zp+1];
      #local _r4c1=_points[_xp][_zp];
      #local _r4c4=_points[_xp+1][_zp];

      #local _inview=0+      isinview(_r1c1);   //      summ-up the number
      #local _inview=_inview+isinview(_r1c4);   //      of corners that are
      #local _inview=_inview+isinview(_r4c1);   //      within view of the
      #local _inview=_inview+isinview(_r4c4);   //      camera
      #if ( _inview>0 )         // and only create blob-patch if at least one corner is visible
      
        #local _v1=(_r4c1-_r1c1)*<1,0,1>;
        #local _r2c1=_r1c1+_v1/3;
        #local _r3c1=_r4c1-_v1/3;
      
        #local _v1=(_r4c4-_r1c4)*<1,0,1>;
        #local _r2c4=_r1c4+_v1/3;
        #local _r3c4=_r4c4-_v1/3;
      
      
        #local _v1=(_r1c4-_r1c1)*<1,0,1>;
        #local _r1c2=_r1c1+_v1/3;
        #local _r1c3=_r1c4-_v1/3;
      
        #local _v1=(_r2c4-_r2c1)*<1,0,1>;
        #local _r2c2=_r2c1+_v1/3;
        #local _r2c3=_r2c4-_v1/3;
      
        #local _v1=(_r3c4-_r3c1)*<1,0,1>;
        #local _r3c2=_r3c1+_v1/3;
        #local _r3c3=_r3c4-_v1/3;
      
        #local _v1=(_r4c4-_r4c1)*<1,0,1>;
        #local _r4c2=_r4c1+_v1/3;
        #local _r4c3=_r4c4-_v1/3;
           
        blob_patch( array[4] { _r1c1,_r1c2,_r1c3,_r1c4 } ,
                    array[4] { _r2c1,_r2c2,_r2c3,_r2c4 } ,
                    array[4] { _r3c1,_r3c2,_r3c3,_r3c4 } ,
                    array[4] { _r4c1,_r4c2,_r4c3,_r4c4 } ,
                    .04,  .2,10,.5,.2,0)
      #end
      #ifdef(quicksand)
        #if (quicksand<4)
          #local _zp=_zp+quicksand;
        #else
          #if (quicksand=5)
            #local _zp=_zp+2;
          #end
        #end
      #end
      #local _zp=_zp+1;
    #end
  
    #ifdef(quicksand)
      #if(quicksand<4)
        #local _xp=_xp+quicksand;
      #else
        #if (quicksand=4)
          #local _xp=_xp+2;
        #end
      #end
    #end
    #local _xp=_xp+1;
  #end

#end

#declare bplant_max=250;
#declare bplant_seed=array[bplant_max]  //      filled up during the blob_patch routine
#declare bp_rnd=seed(20);
#declare bplant_num=0;
#local _spl=0;
#while (_spl<bplant_max)
  #declare bplant_seed[_spl]=<0,0,0>;
  #local _spl=_spl+1;
#end

#declare splant_max=250;
#declare splant_seed=array[bplant_max]  //      filled up during the blob_patch routine
#declare sp_rnd=seed(2000);
#declare splant_num=0;
#local _spl=0;
#while (_spl<splant_max)
  #declare splant_seed[_spl]=<0,0,0>;
  #local _spl=_spl+1;
#end
// 4 arrays of four points, representing the 4 corners, and 12 control points
// order is as-per POV bi-cubic patch object.
#macro blob_patch(_r1,_r2,_r3,_r4,  _bs,_br,_bb,_neg,_jt,_ep)
                  
  #local _cs = reb_thresh*(1+1/_bb);
  #local _cr =_br/sqrt(1-sqrt(reb_thresh/_cs));
  
  #local _xp=0;
  #while (_xp<=1)
    
    #if (_xp<1.001-_bs | _ep=1)
      #local _c1=(flight_spline(_r1[0],_r1[1],_r1[2],_r1[3],0,1,_xp));
      #local _c2=(flight_spline(_r2[0],_r2[1],_r2[2],_r2[3],0,1,_xp));
      #local _c3=(flight_spline(_r3[0],_r3[1],_r3[2],_r3[3],0,1,_xp));
      #local _c4=(flight_spline(_r4[0],_r4[1],_r4[2],_r4[3],0,1,_xp));
    
      #local _zp=0;
      #while (_zp<=1)
        #if (_zp<1.001-_bs | _ep=1)
          #local _pp=(flight_spline(_c1,_c2,_c3,_c4,0,1,_zp))+jitt(_jt);
          
            sphere { _pp,_cr,_cs*_neg }
            #declare _bcount=_bcount+1;
            #local big_plant=0;
            #if (_pp.y>4.5)
              #if (bplant_num<bplant_max)
                #if (isinview(_pp))
                  #if (rand(bp_rnd)<((_pp.y-4.45)/100))   // plant probability increases with altitude
                    #declare bplant_seed[bplant_num]=_pp+<0,_br,0>;
                    #declare bplant_num=bplant_num+1;
                    #local big_plant=1;
                    #if (bplant_num=1)
                      #render "Started laying big plants\n"
                    #end      
                    #if (bplant_num=bplant_max)
                      #render concat("Run out of big plants - ",str(bplant_max,0,0)," allocated.!\n")
                    #end
                  #end
                #end
              #end            
            #end
            
            #if (_pp.y>3 & _pp.y<5.5 & big_plant=0)
              #if (splant_num<splant_max)
                #if (isinview(_pp))
                  #if (rand(sp_rnd)<((_pp.y-3)/100))   // plant probability increases with altitude
                    #declare splant_seed[splant_num]=_pp+<0,_br,0>;
                    #declare splant_num=splant_num+1;
                    #if (splant_num=1)
                      #render "Started laying small plants\n"
                    #end      
                    #if (splant_num=splant_max)
                      #render concat("Run out of small plants - ",str(bplant_max,0,0)," allocated.!\n")
                    #end
                  #end
                #end
              #end            
            #end
          
        #end
        #local _zp=_zp+_bs;
      #end
    #end  
    
    #local _xp=_xp+_bs;
  #end
#end

#declare beach=
  blob {
    rebt(.9)
    
    blob_beachcurve(<18,0,0>,<40,0,60>,<-14,0,120>,<-8,0,180>,
                    <0,0,0>,<0,20,0>,<0,-60,0>,<0,0,0>,
                    <-45,-1.7,0>,<-15,-1,0>,<15,6,0>,<45,7,0>,
                    8/120,8/180)

        // moat                                       
    blob_wall (<0,1.3,-2.8>,<-30,00,0>,5,.61,.3,  .2,10,-.5  ,.2)
    blob_wall (<+2.8,1.3,0>,<+30,90,0>,5,.61,.3,  .2,10,-.5  ,.2)
    blob_wall (<0,1.6,+2.8>,<+30,00,0>,5,.61,.3,  .2,10,-.5  ,.2)
    blob_wall (<-2.8,1.3,0>,<-30,90,0>,5,.61,.3,  .2,10,-.5  ,.2)
    // 4 walls of base
    blob_wall (<0,.4,-2.45>,<20,  0,0>,3,2,.08,  .1,10,+.5  ,.07)
    blob_wall (<0,.4,+2.45>,<20,180,0>,3,2,.08,  .1,10,+.5  ,.07)
    blob_wall (<-2.45,.4,0>,<20, 90,0>,3,2,.08,  .1,10,+.5  ,.07)
    blob_wall (<+2.45,.4,0>,<20,270,0>,3,2,.08,  .1,10,+.5  ,.07)
    
    // four corner towers of base
    blob_tower(<+1.75,.4,-1.75>,15,.6,2,.1,  .1,10,+.5, .07,0)
    blob_tower(<+1.75,.4,+1.75>,15,.6,2,.1,  .1,10,+.5, .07,0)
    blob_tower(<-1.75,.4,-1.75>,15,.6,2,.1,  .1,10,+.5, .07,0)
    blob_tower(<-1.75,.4,+1.75>,15,.6,2,.1,  .1,10,+.5, .07,0)
    // tighten edge of tower with additional, smaller ring
    blob_tower(<+1.75,1.45,-1.75>,0,.5,.21,.05,  .05,10,+.2,  .03,0)
    blob_tower(<+1.75,1.45,+1.75>,0,.5,.21,.05,  .05,10,+.2,  .03,0)
    blob_tower(<-1.75,1.45,-1.75>,0,.5,.21,.05,  .05,10,+.2,  .03,0)
    blob_tower(<-1.75,1.45,+1.75>,0,.5,.21,.05,  .05,10,+.2,  .03,0)
    // take hollow out of centre of lower towers
    blob_tower(<+1.75,1.5,-1.75>,0,.25,.05,.05,  .05,10,+.2,  .03,0)
    blob_tower(<+1.75,1.5,+1.75>,0,.25,.05,.05,  .05,10,+.2,  .03,0)
    blob_tower(<-1.75,1.5,-1.75>,0,.25,.05,.05,  .05,10,+.2,  .03,0)
    blob_tower(<-1.75,1.5,+1.75>,0,.25,.05,.05,  .05,10,+.2,  .03,0)

    // top of base
    blob_wall(<0,1.4,0>,<90,0,0>,4,4,.1,  .1,10,+.5,  .07)               

    #ifndef(posit_render)
        
        // upper towers
    // four corner towers
    blob_tower(<+1.5,2,+1.5>,3,.4,1.6,.03,  .04,10,+.5,  .03,0)
    blob_tower(<+1.5,2,-1.5>,3,.4,1.6,.03,  .04,10,+.5,  .03,0)
    blob_tower(<-1.5,2,+1.5>,3,.4,1.6,.03,  .04,10,+.5,  .03,0)
    blob_tower(<-1.5,2,-1.5>,3,.4,1.6,.03,  .04,10,+.5,  .03,0)

    #ifndef(quickrender)
      // windows in corner towers          
        //  fornt wall towers
      blob_windowslit(<-1.5,2.3,-1.95>,<3,0,0>,.4,.2, .02, .014,10,-1.5, .001)
      blob_windowslit(<+1.5,2.3,-1.95>,<3,0,0>,.4,.2, .02, .014,10,-1.5, .001)
        // left wall towers
      blob_windowslit(<-1.95,2.3,-1.5>,<3,90,0>,.4,.2, .02, .014,10,-1.5, .001)
      blob_windowslit(<-1.95,2.3,+1.5>,<3,90,0>,.4,.2, .02, .014,10,-1.5, .001)
        // left wall
      blob_windowslit(<-1.85,2.2,-.75>,<5,90,0>,.3,.3, .02, .013,10,-1.5, .001)
      blob_windowslit(<-1.85,2.2,+.75>,<5,90,0>,.3,.5, .02, .013,10,-1.5, .001)
      //                        _p1,    _a1,  _hei,_dpt,_bs,_br,_bb,_neg, _jt)
    #end
    
    // upper layer of 4 corners
    blob_tower(<+1.5,3,+1.5>,3,.41,.3,.025 .03,10,+.5,  .025,8)
    blob_tower(<+1.5,3,-1.5>,3,.41,.3,.025 .03,10,+.5,  .025,8)
    blob_tower(<-1.5,3,+1.5>,3,.41,.3,.025 .03,10,+.5,  .025,8)
    blob_tower(<-1.5,3,-1.5>,3,.41,.3,.025 .03,10,+.5,  .025,8)
    
    // 5 walls of upper level
    blob_wall (<-1.8,2,0>,<5,090,0>,2.8,1.4,.03,  .04,10,+.5  ,.03)
    blob_wall (<+1.8,2,0>,<5,270,0>,2.8,1.4,.03,  .04,10,+.5  ,.03)
    blob_wall (<0,2,+1.8>,<5,180,0>,2.8,1.4,.03,  .04,10,+.5  ,.03)
    
    blob_wall (<-.9,2,-1.8>,<5,0,0>,.6,1.4,.03,  .04,10,+.5  ,.03)
    blob_wall (<+.9,2,-1.8>,<5,0,0>,.6,1.4,.03,  .04,10,+.5  ,.03)
    
    // gate towers L/R
    blob_tower(<-.6,2,-1.7>,1,.2,1.2,.03,  .04,10,+.5,  .03,0)
    blob_tower(<+.6,2,-1.7>,1,.2,1.2,.03,  .04,10,+.5,  .03,0)
    
    blob_tower(<-.6,2.8,-1.7>,1,.24,.25,.025,  .03,10,+1,  .02,6)
    blob_tower(<+.6,2.8,-1.7>,1,.24,.25,.025,  .03,10,+1,  .02,6)
    // centre span between gate towers (arch)
    blob_archedwall(<0,2.7,-1.8>,<1,0,0>,.8, .1, -.6,.1,.1,-.6, .0125  ,.03,10,+.5,  .02)

    #ifndef(quickrender)
      blob_windowslit(<-.6,2.2,-1.95>,<1,0,0>,.25,.2, .02, .011,10,-1.5, .001)
      blob_windowslit(<+.6,2.2,-1.95>,<1,0,0>,.25,.2, .02, .011,10,-1.5, .001)
    #end               

    #end
// bridge
    blob_archedwall(<+.28,1.47,-2.7>,<0,99,0>,.75, 0, -.5,-.25,-.01,-.01,  .05,  .05,10,+.5,  .07)
    blob_archedwall(<+.10,1.42,-2.7>,<0,94,0>,.5, 0, -.5,-.35,-.01,-.01,  .05,  .05,10,+.5,  .07)
    blob_archedwall(<-.10,1.42,-2.7>,<0,86,0>,.5, 0, -.5,-.35,-.01,-.01,  .05,  .05,10,+.5,  .07)
    blob_archedwall(<-.28,1.47,-2.7>,<0,81,0>,.75, 0, -.5,-.25,-.01,-.01,  .05,  .05,10,+.5,  .07)
    
    blob_wall (<0,1.44,-3>,<82,0,0>,.4,2.2,.03,  .045,10,+.65  ,.03)


// centre tower
    //blob_tower(<0,2.1,0>,5,.85,2.5,.03,  .04,10,+.5,  .03,0)        
    //blob_tower(<0,3.3,0>,3,.84,.35,.025 .03,10,+.5,  .025,16)

 // blob_tower(_p1,_wl,_tr,_th,_bs,_br,_bb,_neg,_jt,_cf)
    
    texture {
      beach_sand_tex
    }
  }  

     
object { beach }  
object { plane { y,0 } translate <0,-1,0> texture { beach_sand_tex } }




#include "palm.inc"
// build a collection of palm trees to choose from
// (all the same looks false, all different eats memory; need something in-between)
#declare palm_col=array[3]
#declare palm1=object{ palm_tree(<0,3,0>,<-2,0,10>,<-4,0,15>,<-6,0,15>,<-8,0,10>,7,.45,1.55,.12) }
#declare palm2=object{ palm_tree(<0,3,-5>,<-2,0,1>,<-4,0,2>,<-6,0,4>,<-6,0,5>,7,.5,1.55,.12) }
#declare palm3=object{ palm_tree(<0,3,5>,<0,0,5>,<0,0,7>,<0,0,4>,<0,0,5>,8,.5,1.55,.12) }
#declare palm_col[0]=object { palm1 }
#declare palm_col[1]=object { palm2 }
#declare palm_col[2]=object { palm3 }

#local palm_num=0;
#local palm_loop=0;
#local palm_finish=1;
#ifdef(palm_cutoff)
  #local palm_finish=palm_cutoff;
#end
#while (palm_loop<palm_finish)
  #local _lp=(flight_spline(<18,0,0>,<40,0,60>,<-20,0,120>,<-10,0,180>,0,1,palm_loop));
  #local palm_dep=0;
  #while (palm_dep<=2)
    #local _pp=_lp+<30+50*palm_dep,1,10*palm_dep>+(jitt(4)*<2,.1,2>);
    #local _pa=<0,rand(j_seed)*360,0>;
    
    #if (isinview(_pp))
      #local palm_no=int(3*rand(j_seed));
      #local palm_num=palm_num+1;
      object {
        object { palm_col[palm_no] }
        rotate _pa
        rotate <2,0,-1> //      tend toward light slightly
        translate _pp
      }
    #end
    #local palm_dep=palm_dep+.25;
  #end
  #local palm_loop=palm_loop+.075;
#end

#local _spl=0;
#while (_spl<bplant_num)
  object { bspike_plant scale (.8+(rand(bp_rnd)/5)) rotate <0,360*rand(bp_rnd),0> rotate <-5,0,5> translate bplant_seed[_spl] }
  #local _spl=_spl+1;
#end

#local _spl=0;
#while (_spl<splant_num)
  object { sspike_plant scale (.8+(rand(sp_rnd)/5)) rotate <0,360*rand(sp_rnd),0> rotate <-5,0,5> translate splant_seed[_spl] }
  #local _spl=_spl+1;
#end

#declare fp_seed=seed(1);
#macro fp_jitt(j_qty)
  #local j_rad=j_qty*rand(fp_seed);
  #local j_ang=<360*rand(fp_seed),360*rand(fp_seed),360*rand(fp_seed)>;
  ( vrotate(<0,1,0>,j_ang)*j_rad )
#end

#declare fl_pole=
  blob {
    threshold .5
    #local _lp=<0,0,0>;
    #local _ta=<0,0,0>;
    #local _pc=.05;
    #while (_pc<=20)
      #local _br=(flight_spline(1,1,.75,.5,0,20,_pc));
      #local _ta=_ta+fp_jitt(10);
      #local _tp=_lp+vrotate(y/20,_ta);
      cylinder { _lp,_tp,.035*_br,1 }
      sphere { _lp,.035*_br,-1 }
      #local _lp=_tp;
      #local _pc=_pc+1;
    #end
    rotate <0,230,0>
    texture { trunk_tex }
  }
   
#macro ext_vpair(_ang,_vl,_vr,_w)
  #local _bcl=array[16]
  #local _bcr=array[16]
  #local _t1=0;
  #local _cp=<0,0,0>;
  #while (_t1<3.99)
    #local _t2=0;
    #while (_t2<3.99)
      #local _bcl[_t2+_t1*4]=_cp+vrotate(_vl[_t2]*_w[_t1],_ang[_t1]);
      #local _bcr[_t2+_t1*4]=_cp+vrotate(_vr[_t2]*_w[_t1],_ang[_t1]);
      #local _t2=_t2+1;
    #end
    #local _cp=_cp+vrotate(<.5,0,0>,_ang[_t1]);
    #local _t1=_t1+1;
  #end
  bicubic_patch {
    type 0 flatness 0.00010 u_steps 4 v_steps 4,
    _bcl[00],_bcl[01],_bcl[02],_bcl[03],
    _bcl[04],_bcl[05],_bcl[06],_bcl[07],
    _bcl[08],_bcl[09],_bcl[10],_bcl[11],
    _bcl[12],_bcl[13],_bcl[14],_bcl[15]
  }
  bicubic_patch {
    type 0 flatness 0.00010 u_steps 4 v_steps 4,
    _bcr[00],_bcr[01],_bcr[02],_bcr[03],
    _bcr[04],_bcr[05],_bcr[06],_bcr[07],
    _bcr[08],_bcr[09],_bcr[10],_bcr[11],
    _bcr[12],_bcr[13],_bcr[14],_bcr[15]
  }
#end

#declare fl_leaf1=
  union {
    #local _ang=array[4] { <90,0,0>,<30,0,-60>,<-30,0,-90>,<-60,30,-90> }
    #local _vl=array[4] { <0,-.2,0>,<0,0,.2>,<0,.1,.4>,<0,.2,.6> }
    #local _vr=array[4] { <0,-.2,0>,<0,0,-.2>,<0,.1,-.4>,<0,.2,-.6> }
    #local _w=array[4] { .16,.15,.13,.01 }
    ext_vpair(_ang,_vl,_vr,_w)
    texture { palm_leaf_tex }
    translate <0,.88,0>
    rotate <0,60,0>
  }

#declare fl_leaf2=
  union {
    #local _ang=array[4] { <90,0,0>,<140,0,-80>,<180,0,-90>,<210,0,-90> }
    #local _vl=array[4] { <0,-.2,0>,<0,0,.2>,<0,.1,.4>,<0,.2,.6> }
    #local _vr=array[4] { <0,-.2,0>,<0,0,-.2>,<0,.1,-.4>,<0,.2,-.6> }
    #local _w=array[4] { .16,.15,.13,.01 }
    ext_vpair(_ang,_vl,_vr,_w)
    texture { palm_leaf_tex }
    translate <0,.88,0>
    rotate <0,60,0>
  }
   
#declare flag1=
  union {
    object { fl_pole }
    object { fl_leaf1 }
  }


#declare flag2=
  union {
    object { fl_pole }
    object { fl_leaf2 }
  }
    
object { flag1 rotate <0,30,0> translate <-.6,2.5,-1.7> rotate _ca translate _cp }
object { flag2 rotate <0,43,0> translate <+.6,2.5,-1.7> rotate _ca translate _cp }

#include "crab.inc"
object { crab rotate crab_ang translate crab_pos }
#include "boat.inc"
object { boat rotate boat_ang translate boat_pos }
#include "hut.inc"
object { hut rotate hut_ang translate hut_pos }

#render concat("Total palm tree count =",str(palm_num,0,0),"\n")  
#render concat("Total big spiky plant count =",str(bplant_num,0,0),"\n")  
#render concat("Total small spiky plant count =",str(splant_num,0,0),"\n")  
#render concat("Total blob component count = ",str(_bcount,0,0),"\n")