/* 
 * skin.sl 
 * 
 * Copyright (C) 2000, Matt Pharr <mmp@Exluna.com> 
 * 
 * This software is placed in the public domain and is provided as is 
 * without express or implied warranty. 
 * 
 * Surface shader that implements a shading model that should have a visual 
 * appearence generall similar to that of skin.  Based on phenomenological 
 * information about skin reflectance from Hanrahan and Krueger, 
 * "Reflection from layered surfaces due to subsurface scattering", 
 * proceedings of Siggraph 1993. 
 */ 

/* Evaluate the Henyey-Greenstein phase function for two vectors with
   an asymmetry value g.  v1 and v2 should be normalized and g should 
   be in the range (-1, 1).  Negative values of g correspond to more
   back-scattering and positive values correspond to more forward scattering.
*/
float phase(vector v1, v2; float g) {
	float costheta = v1 . v2;
	return (1. - g*g) / pow(1. + g*g - 2.*g*costheta, 1.5);
}

/* Compute a the single-scattering approximation to scattering from
   a one-dimensional volumetric surface.  Given incident and outgoing
   directions wi and wo, surface normal n, asymmetry value g (see above),
   scattering albedo (between 0 and 1 for physically-valid volumes),
   and the thickness of the volume, use the closed-form single-scattering
   equation to approximate overall scattering.
*/
float singleScatter(vector wi, wo; normal n; float g, albedo, thickness) {
	float invWiDotN = 1. / -(wi . n);
	float invWoDotN = 1. /  (wo . n);

    return albedo * phase(wo, wi, g) / (invWiDotN + invWoDotN) *
		(1. - exp(-(invWiDotN + invWoDotN) * thickness));
}

/* Implements overall skin subsurface shading model.  Takes viewing and
   surface normal information, the base color of the skin, a
   color for an oily surface sheen, the ratio of the indices of 
   refraction of the incoming ray (typically ~1 for air) to the index
   of refraction for the transmitted ray (say something like 1.4 for
   skin), and the overall thickness of the skin layer.  Then loops
   over light sources with illuminance() and computes the reflected
   skin color.
*/
color subsurfaceSkin(vector Vf; normal Nn; color skinColor, sheenColor;
                     float eta, thickness) {
	extern point P;
	float Kr, Kt;
	vector R, T;
	color subsurf = 0, glossy = 0;

	/* Use frensel() to both compute the vector of the transmitted
	   direction into the skin as well as how much light is reflected
	   at the surface, Kr, and how much is transmitted into the
	   subsurface skin layer.
	*/
	fresnel(-Vf, Nn, eta, Kr, Kt, R, T);

	/* Fudge Kr a bit to increase glossy reflection at grazing
	   angles. */
	Kr = smoothstep(0., .5, Kr);
	/* Force Kt to be consistent.  prman and BMRT give different values
	   of Kt from fresnel() (prman always says Kt = 1-Kr, which is 
	   wrong). */
	Kt = 1. - Kr;

	/* Hmm. Is this necessary? */
	T = normalize(T);

	illuminance(P, Nn, PI/2) {
                extern vector L;
                extern color Cl;
		vector Ln = normalize(L);

		vector H = normalize(Ln + Vf);
		if (H . Nn > 0)
			glossy += Cl * (Ln . Nn) * pow(H . Nn, 4.);
		glossy += Cl * (Ln . Nn) * .2;

		/* We need to compute the refracted light ray into the subsurface
		   layer, Lsub, and use that for the lighting model below the
		   surface.  Again, not sure if this normalization is needed.
		   Flip its direction so that it's pointing away from the surface. 
		*/
		vector Lsub = -normalize(refract(Vf, Nn, eta));
		if (Lsub . Lsub > 0.)
			/* If the refracted ray exists (which it always should, since
			   we're transmitting from air to something with a higher index
			   of refraction, so there should be no total internal
			   reflection issues at the boundary), use the single
			   scattering approximation up above.  Call it with two
			   different g values, one with slight backwards scattering,
			   and one with isotropic scattering.  The isotropic term is a
			   very empirical approximation to the multiple scattering
			   within the medium.
			*/
			subsurf += Cl * (Ln . Nn) *
				(singleScatter(T, Lsub, Nn, -.15, .5, thickness) +
				 singleScatter(T, Lsub, Nn,   0., .75, thickness));
	}

	/* Finally, combine the subsurface scattering with the scattering from
	   the fake oily glossy layer at the top.  Note that we should scale
	   the subsurface stuff by two Kt terms, since light is transmitted
	   twice: once going into the subsurface layer and once on the way back
	   out.  One Kt seems to look better, however. */
	return (Kt * skinColor * subsurf +
			Kr * sheenColor * glossy);
}

/* Basic surface shader that uses the skin reflection model implemented
   above.
*/
surface skin(color Ka = .5; color skinColor = color(.8, .5, .5);
             color sheenColor = 1.;
			 float eta = 1./1.4, thickness = 0.5) {
	normal Nn = faceforward(normalize(N), I);
	vector Vf = -normalize(I);

	Oi = Os;
	Ci =   subsurfaceSkin(Vf, Nn, skinColor, sheenColor, eta, thickness);
	Oi = 0 ;/* smoothstep(0, 1, comp(ctransform("hsv", Ci), 2) );*/
}
