// "Worlds within worlds" - Glass
// Author: Paul Houx, Utrecht, The Netherlands
// Email: paul.houx@xs4all.nl

// Render this scene at a 1:1 ratio at highest quality.
// Antialias settings: type 2 (adaptive), threshold 0.1, recursion 1, jitter 0, should be sufficient

// Use MegaPov to render this file, since we need both GLOW and FRESNEL effects
#version unofficial MegaPov 0.7;

// Some switches to quickly change certain settings
#declare hasAreaLights = no;				// since you can't really see 'em, don't use 'em
	
#declare isGlowing = yes;						// the glow effect adds that little bit of magic to the scene
#declare cameraBumps = yes;				// the bump applied to the camera
#declare transparentSpheres = yes;		// the glass-like material
#declare colouredSpheres = yes;			// if the spheres are not transparent, this one alters the colours
#declare innerSpheres = yes;				// the small marbles inside the glass spheres
#declare whitePlane = yes;					// the white plane that is only visible as a reflection

// Set up the environment
global_settings {
	ambient_light 0.25				// there's nothing wrong with a little ambient light
	assumed_gamma 1.8			// I'm a mac user, but since the majority of you isn't, let's assume PC gamma
	max_trace_level 4				// after four recursions, the reflections will be complex enough, me thinks
}

background { color rgb 0 }		// I like black, can't see what's wrong with that! Gives the scene some contrast.

camera {
	orthographic						// an orthographic camera enables us to create a hexagon out of a box-like object
	location <200, 200, -200>
	right <250, 0, 0>				// make sure we capture everything without too much background
	up <0, 250, 0>
	look_at <0, 0, 0>
	
	// Use a bumped camera lens - this is the 'Special Ingredient X'
	#if (cameraBumps)
		normal { bumps 0.05 scale 0.1 }
	#end
}

light_source {							// this one's the main light (or key light)
	<0, 0, -5000>
	color rgb 0.75
	
	spotlight
	radius 5
	tightness 1
	falloff 10
	point_at <0, 0, 0>
	
	#if (hasAreaLights)
		area_light
		<100, 0, 0>, <0, 100, 0>
		3, 3
		adaptive 1
		jitter
	#end
	
	media_interaction off			// left-over from some dismissed experiments
}

light_source {							// this one is for the top of the cube
	<0, 5000, 0>
	color rgb 0.5
	
	spotlight
	radius 5
	tightness 1
	falloff 10
	point_at <0, 0, 0>
	
	#if (hasAreaLights)
		area_light
		<100, 0, 0>, <0, 0, 100>
		3, 3
		adaptive 1
		jitter
	#end
	
	shadowless
	
	media_interaction off
}

light_source {								// this one is for the right side of the cube
	<5000, 0, 0>
	color rgb 0.25
	
	spotlight
	radius 5
	tightness 1
	falloff 10
	point_at <0, 0, 0>
	
	#if (hasAreaLights)
		area_light
		<0, 0, 100>, <0, 100, 0>
		3, 3
		adaptive 1
		jitter
	#end
	
	media_interaction off
}

// Add a white floor plane to the scene, which is only visible as a reflection in the spheres.
// This enhances the glass-like qualities of the spheres without altering the mood of the scene.
// If omitted, the spheres don't look as spectacular as they do now.
#if (whitePlane)
	plane {
		y, -400
		
		pigment { color rgb <1, 1, 1> }
		finish { ambient 1.0 diffuse 0.75 specular 0.0 reflection 0.0 }
		
		// Make sure it is not in the way of my lights. Don't know if it is really neccesary.
		no_shadow
		
		// Do not render the object directly, but only as a reflection. This is a neat MegaPov-feature.
		no_image
	}
#end

// Define some textures
#declare texture1 = texture {
	// this is the texture for the darker small spheres
	pigment { color rgb <0, 0, 0> }
	
	// the finish is quite important: I used a Fresnel-type of reflection, since it's more realistic. Reflection_type is a MegaPov-feature.
	finish { ambient 0.25 diffuse 0.5 specular 0.2 reflection_type 1 reflection_min 0.15 reflection_max 0.50 reflection_falloff 2 conserve_energy }
}

#declare texture2 = texture {
	// this is the texture for the bright glass spheres
	pigment { color rgb <0.9, 0.7, 0.3> * 2 }
	
	// both hilite and reflection use the Fresnel equation. Blinn is a MegaPov-feature.
	finish { ambient 0.0 diffuse 0.6 blinn 2 facets 0.2 reflection_type 1 reflection_min 0.15 reflection_max 0.50 reflection_falloff 2 conserve_energy }
}


#declare stone_spacing = 30.0;	// determines the size of the spheres
#declare group_scale = 2.0;			// determines the size of the group of spheres

// create the spheres. The radius is calculated precisely, because I want the spheres to *just* touch eachother
#declare obsidian = sphere {
	<0, 0, 0>, (stone_spacing - (stone_spacing / sqrt(2.0)))
	texture { texture1 }
	interior { ior 1.5 }	// needed for Fresnel-equations
}

#declare barnstone = sphere {
	<0, 0, 0>, (stone_spacing / sqrt(2.0))
	texture { texture2 }
	interior { ior 15 }		// needed for Fresnel-equations and refraction
}

// create the inner particles
#declare randseed = seed(10);

#declare inner_world = union {
	#declare n = 0;
	#while (n < 10)
		sphere { <rand(randseed)-0.5, rand(randseed)-0.5, rand(randseed)-0.5>, 0.1 }
		#declare n = n + 1;
	#end
	scale (stone_spacing / sqrt(2.0)) * 0.75
	texture { texture1 }
}

// draw the stones
#declare dep = -group_scale;
#while (dep <= group_scale)
	#declare hor = -group_scale;
	#while(hor <= group_scale)
		#declare ver = -group_scale;
		#while(ver <= group_scale)
			// the place in the group determines the kind of stone
			#if ( mod(dep + ver + hor, 2) = 0)				
				// draw a glass one
				object {
					barnstone
					translate <hor, ver, dep> * stone_spacing
					
					// the color of the stone is determined by its position
					#if (transparentSpheres)
						pigment { color rgbf <1 + hor / group_scale, 1 + ver / group_scale, 1 - dep / group_scale, 0.85> }
					#else
						#if (colouredSpheres)
							pigment { color rgb <1 + hor / group_scale, 1 + ver / group_scale, 1 - dep / group_scale> }
						#end
					#end
				} 
				// each glass sphere has inner particles, bright and dark ones
				#if (transparentSpheres)
					#if (innerSpheres)
						object {
							inner_world
							rotate <360 * rand(randseed), 360 * rand(randseed), 0>
							translate <hor, ver, dep> * stone_spacing
							pigment { color rgb 3 }	// bright ones
						} 
						object {
							inner_world
							rotate <360 * rand(randseed), 360 * rand(randseed), 0>
							translate <hor, ver, dep> * stone_spacing
							pigment { color rgb 0.5 }	// dark ones
						}
					#end
				#end
				// each glass sphere glows from the inside. Glows are another neat MegaPov-feature.
				#if (transparentSpheres)
					#if (isGlowing)
						glow {
							type 1
							location <hor, ver, dep> * stone_spacing
							size (stone_spacing / sqrt(2.0)) * 0.01
							radius (stone_spacing / sqrt(2.0)) * 1.5
							fade_power 4
							color rgb 1
						}
					#end
				#end
			#else 
				// draw a dark one
				object {
					obsidian
					translate <hor, ver, dep> * stone_spacing
				}
			#end
			#declare ver = ver + 1;
		#end
		#declare hor = hor + 1;
	#end
	#declare dep = dep + 1;
#end

