//
// spiral.inc v1.0
// Written by Felix Morley Finch (felix@crowfix.com)
// Extended comments, textures, default variables, and variable renaming
//   by Michael D Johnson (mdjohnson@voyager.net or redbeard@quark.gmi.edu)
// 
// Current version located at
// http://quark.gmi.edu/~redbeard/POVRay-Util.html
//
// Use:
// 1. Define the varaiables listed below.
// 2. #include "spiral.inc"
// 3. Clear your floor for the landing at the top
// difference {
//   object { Your_Floor translate y*(Spiral_Height - Spiral_Landing_H) }
//   object { Spiral_Hole translate <Spiral_X, 0, Spiral_Z> }
// }
// 4. Use the landing and staircase
// union {
//   object { Spiral_Steps }
//   object { Spiral_Landing }
//   translate <Spiral_X, 0, Spiral_Z>
// }
//
// Variables:
// Declare these in your main file before including spiral.inc.  Or, use
// the defaults, listed in brackets ([]), with all dimensions in inches.
//
// Spiral_Clues		[false]	Set to true to mark entry/exit and one side
// Spiral_Clockwise	[true]	Set to true for clockwise climbing spiral,
//				false for counterclockwise
// Spiral_Top_A		[0]	Angle of leading edge of top tread
// Spiral_Diameter	[48]	Outside diameter of staircase
// Spiral_Height	[108]	Height of staircase (9 feet)
// Spiral_Riser_H	[9]	Height rise from tread to tread
// Spiral_Tread_A	[30]	Rotation from tread to tread
// Spiral_Tread_H	[2]	Thickness of each tread
// Spiral_Tread_Overlap	[1]	How much overlap on tread edges
// Spiral_Column_W	[5]	Diameter of central column
// Spiral_Rail_H	[36]	Height of step posts
// Spiral_Rail_T	[2]	Thickness of railing
// Spiral_Post_Delta	[6]	Spacing between railing posts
// Spiral_Post_R	[1]	Radius of railing posts
// Spiral_Landing_H	[12]	Landing thickness at top
//
// The following are the textures to use.  They default to unique colors
// so you can identify the various parts.
// T_Spiral_Tread	T_Spiral_Post	T_Spiral_Rail
// T_Spiral_Support	T_Spiral_Column
//
// 
// Output: (See also above)
// Spiral_Hole		An object used to cut out the upper floor
// Spiral_Steps		The actual steps of the staircase (includes rail)
// Spiral_Landing	The landing of the staircase
// 
// 
// All other variables used by this file are prefixed by _spiral_ and should
// be assumed to be unknown.
// 


#ifndef (Spiral_Clues) #declare Spiral_Clues = false; #end
#ifndef (Spiral_Clockwise) #declare Spiral_Clockwise = true; #end
#ifndef (Spiral_Top_A) #declare Spiral_Top_A = 0; #end 
#ifndef (Spiral_Diameter) #declare Spiral_Diameter = 48 ;#end
#ifndef (Spiral_Height) #declare Spiral_Height = 108; #end
#ifndef (Spiral_Riser_H) #declare Spiral_Riser_H = 9 ;#end 
#ifndef (Spiral_Tread_A) #declare Spiral_Tread_A = 30; #end 
#ifndef (Spiral_Tread_H) #declare Spiral_Tread_H = 2; #end
#ifndef (Spiral_Tread_Overlap) #declare Spiral_Tread_Overlap = 1; #end
#ifndef (Spiral_Column_W) #declare Spiral_Column_W = 5; #end
#ifndef (Spiral_Rail_H) #declare Spiral_Rail_H = 36; #end
#ifndef (Spiral_Rail_T) #declare Spiral_Rail_T = 2; #end
#ifndef (Spiral_Post_Delta) #declare Spiral_Post_Delta = 6; #end
#ifndef (Spiral_Post_R) #declare Spiral_Post_R = 1; #end
#ifndef (Spiral_Landing_H) #declare Spiral_Landing_H = 0; #end  //12 #end
#ifndef (T_Spiral_Tread) 
  #declare T_Spiral_Tread = texture { pigment { rgb <0.9, 0.1, 0.1> } };
#end
#ifndef (T_Spiral_Post)
  #declare T_Spiral_Post = texture { pigment { rgb <0.5, 0.9, 0.1> } };
#end
#ifndef (T_Spiral_Rail)
  #declare T_Spiral_Rail = texture { pigment { rgb <0.1, 0.9, 0.1> } };
#end
#ifndef (T_Spiral_Support)
  #declare T_Spiral_Support = texture { pigment { rgb <0.1, 0.5, 0.9> } };
#end
#ifndef (T_Spiral_Column)
  #declare T_Spiral_Column = texture { pigment { rgb <0.1, 0.1, 0.9> } };
#end


// Brochure says the landing box has to be 2 inches larger than the diameter.
#declare _spiral_Box_W = Spiral_Diameter + 2;
#declare _spiral_Box_Half_W = _spiral_Box_W / 2;

#if (Spiral_Clockwise)
#declare _spiral_Multiplier	= +1;
#else
#declare _spiral_Multiplier	= -1;
#end

#declare _spiral_Landing_rail_post = cylinder {
  <0, 0, 0>,
  <0, Spiral_Rail_H, 0>,
    Spiral_Post_R
    pigment{Clear}//texture { T_Spiral_Post }
}

// Approximate a pie section for each tread.
// The top is at floor level.  The "center of column" is at <0,0>.
// Posts are positioned along the outer edge, at the midpoint and front.
// To use a tread, rotate it, then translate it the appropriate y.

#declare _spiral_tread_length = Spiral_Diameter / 2;
#declare _spiral_tread_X = (_spiral_tread_length *
			    cos(radians(Spiral_Tread_A / 2)));
#declare _spiral_tread_ZZ = (_spiral_tread_length *
			     sin(radians(Spiral_Tread_A / 2)));
#declare _spiral_tread_Z = (_spiral_tread_ZZ +
			    (Spiral_Rail_T / 2 + Spiral_Tread_Overlap) *
			    cos(radians(Spiral_Tread_A / 2)));

#declare _spiral_post_length = (_spiral_tread_length - Spiral_Rail_T / 2 - 
				Spiral_Tread_Overlap);
#declare _spiral_post_X = (_spiral_post_length *
			   cos(radians(Spiral_Tread_A / 2)));
#declare _spiral_post_Z = (_spiral_post_length * 
			   sin(radians(Spiral_Tread_A / 2))) ;

#declare _spiral_middle_Y = Spiral_Rail_H + Spiral_Riser_H / 2;
#declare _spiral_front_Y = _spiral_middle_Y + Spiral_Riser_H / 2;

#declare _spiral_middle_post = union {
  cylinder {
    <0, 0, 0>,
    <0, _spiral_middle_Y, 0>,
      Spiral_Post_R
  }
  sphere {
    <0, _spiral_middle_Y, 0>,
      Spiral_Post_R
  }
    texture { T_Spiral_Post }
}
#declare _spiral_front_post = union {
  cylinder {
    <0, 0, 0>,
    <0, _spiral_front_Y, 0>,
      Spiral_Post_R
  }
  sphere {
    <0, _spiral_front_Y, 0>,
      Spiral_Post_R
  }
    texture { T_Spiral_Post }
}
#declare _spiral_rail_bannister = cylinder {
  <0, 0, 0>,
  <_spiral_post_X - _spiral_post_length, Spiral_Riser_H / 2,
   _spiral_post_Z * -_spiral_Multiplier>
  Spiral_Rail_T / 2
    texture { T_Spiral_Rail }
}



// Could replace this with a union of this prism, and a cylinder clipped
// by a larger version of this, to get the rounded edge.  I guess.

#declare _spiral_tread = union {
  prism {		// tread
    -Spiral_Tread_H, 0, 6,
    <0, _spiral_Multiplier*-(Spiral_Column_W / 2)>,
    <_spiral_tread_X, _spiral_Multiplier*-_spiral_tread_Z>,
    <_spiral_tread_length, 0>,
    <_spiral_tread_X, _spiral_Multiplier*_spiral_tread_Z>,
    <0, _spiral_Multiplier*(Spiral_Column_W / 2)>,
    <0, _spiral_Multiplier*-(Spiral_Column_W / 2)>
      texture { T_Spiral_Tread }
  }

  prism {		// The support underneath.
    -Spiral_Rail_T/2, Spiral_Rail_T/2, 5
    <0, 0>
    <_spiral_tread_length, 0>
    <_spiral_tread_length, 1>
    <0, Spiral_Riser_H - Spiral_Tread_H - 1>
    <0, 0>
    rotate x*90
    translate y*-Spiral_Tread_H
      texture { T_Spiral_Support }
  }
  object { _spiral_middle_post translate <_spiral_post_length, 0, 0> }
  object
    { _spiral_front_post translate <_spiral_post_length, 0, 0>
      rotate y*(Spiral_Tread_A / 2)*_spiral_Multiplier }
  rotate y*-(Spiral_Tread_A / 2)*_spiral_Multiplier
}

#declare Spiral_Steps = union {
  #declare _spiral_Bottom_A = Spiral_Top_A;
  #declare _spiral_y = Spiral_Height - Spiral_Riser_H ;
  #declare _spiral_top_post = 1 ;
  #while (_spiral_y > 0)
  union {
    object { _spiral_tread }
    object {
      _spiral_rail_bannister
      translate <_spiral_post_length, _spiral_middle_Y, 0>
      rotate y*-(Spiral_Tread_A / 2)*_spiral_Multiplier
    }
    #if (_spiral_top_post)
    #declare _spiral_top_post = 0;	// No bannister ahead of the top post
    #else
    object {
      _spiral_rail_bannister
      translate <_spiral_post_length, _spiral_front_Y, 0>
      rotate y*(Spiral_Tread_A / 2)*_spiral_Multiplier
      rotate y*-(Spiral_Tread_A / 2)*_spiral_Multiplier
    }
    #end
    translate y*_spiral_y
    rotate y*_spiral_Multiplier*_spiral_Bottom_A
  }
  #declare _spiral_Bottom_A = _spiral_Bottom_A - Spiral_Tread_A ;
  #declare _spiral_y = _spiral_y - Spiral_Riser_H ;
  #end // while

  union {		// The center column, with a ball on top
    cylinder {
      <0, 0, 0>,
      <0, Spiral_Height + Spiral_Rail_H - Spiral_Column_W/2 + 1, 0>,
      Spiral_Column_W / 2
    }
    sphere {
      <0, Spiral_Height + Spiral_Rail_H, 0>,
      Spiral_Column_W / 2
    }
      texture { T_Spiral_Column }
  }

  #if (Spiral_Clues)		// directional box and arrows
  #declare _spiral_Clue_side = Spiral_Diameter + 4;
  union {
    box { <0, -2, 0>, <_spiral_Clue_side, 0, 2> pigment {Clear}}//{ rgb <0, 1, 0> } }
    box
      {
	  <_spiral_Clue_side - 2, -2, 0>,
	  <_spiral_Clue_side, 0, _spiral_Clue_side>
      }
    box
      {
	  <0, -2, _spiral_Clue_side - 2>,
	  <_spiral_Clue_side, 0, _spiral_Clue_side>
      }
    box { <0, -2, 0>, <2, 0, _spiral_Clue_side> }
    pigment {Clear}//{ rgb <1,0,0> }
    translate <-_spiral_Clue_side / 2, Spiral_Height, -_spiral_Clue_side / 2>
  }
  box {
    <-1, -2, 0>,
    <+1, 0, -_spiral_Multiplier*Spiral_Diameter*2>
    translate x*Spiral_Diameter/4
    rotate y*_spiral_Multiplier*Spiral_Top_A
    translate y*Spiral_Height
    pigment { Clear}//rgb <1,1,0> }
  }
  box {
    <-1, -2, 0>,
    <+1, 0, _spiral_Multiplier*Spiral_Diameter*2>
    translate x*Spiral_Diameter/4
    rotate y*_spiral_Multiplier*_spiral_Bottom_A
    translate y*Spiral_Riser_H
    pigment { Clear}//rgb <0,0,1> }
  }
  #end // Spiral_Clues
}

// Here's a cutout to clear out the landing area in the top floor.
// This is defined with the top surface at y=0.
#declare _spiral_i = _spiral_Box_Half_W + .001;
#declare Spiral_Hole = prism {
  -(Spiral_Landing_H + .001), .001, 7
  <0, 0>,
  <0, -_spiral_i * _spiral_Multiplier>,
  <-_spiral_i, -_spiral_i * _spiral_Multiplier>,
  <-_spiral_i, +_spiral_i * _spiral_Multiplier>,
  <+_spiral_i, +_spiral_i * _spiral_Multiplier>,
  <+_spiral_i, 0>,
  <0, 0>
  rotate y*Spiral_Top_A*_spiral_Multiplier
  translate y*Spiral_Height
  pigment { rgbf 1 }
}



// Define the posts centered at 0,0 like the stairs.
// There are two full length sides and three half sides.
#declare _spiral_Offset = _spiral_Box_Half_W + Spiral_Rail_T/2;
#declare _spiral_Full_N = int((_spiral_Offset*2 + Spiral_Post_Delta/2) /
			      Spiral_Post_Delta);
#declare _spiral_Full_0 = (-_spiral_Offset + Spiral_Post_Delta +
			   (_spiral_Box_W + Spiral_Post_Delta/2 -
			    _spiral_Full_N * Spiral_Post_Delta) / 2);
#declare _spiral_Half_N = int((_spiral_Offset + Spiral_Post_Delta/2) /
			      Spiral_Post_Delta);
#declare _spiral_Half_0 = (Spiral_Post_Delta +
			   (_spiral_Box_Half_W + Spiral_Post_Delta/2 -
			    _spiral_Half_N * Spiral_Post_Delta) / 2);
#declare Spiral_Landing = union {
  union {	// Defined from the center column out to the spiral bannister
    cylinder {
      <0, 0, 0>
      <0, 0, _spiral_Multiplier*-_spiral_Offset>
      Spiral_Rail_T/2
    }
    sphere {
      <0, 0, _spiral_Multiplier*-_spiral_Offset>
      Spiral_Rail_T/2
    }
    cylinder {
      <0, 0, _spiral_Multiplier*-_spiral_Offset>
      <-_spiral_Offset, 0, _spiral_Multiplier*-_spiral_Offset>
      Spiral_Rail_T/2
    }
    sphere {
      <-_spiral_Offset, 0, _spiral_Multiplier*-_spiral_Offset>
      Spiral_Rail_T/2
    }
    cylinder {
      <-_spiral_Offset, 0, _spiral_Multiplier*-_spiral_Offset>
      <-_spiral_Offset, 0, _spiral_Multiplier*_spiral_Offset>
      Spiral_Rail_T/2
    }
    sphere {
      <-_spiral_Offset, 0, _spiral_Multiplier*_spiral_Offset>
      Spiral_Rail_T/2
    }
    cylinder {
      <-_spiral_Offset, 0, _spiral_Multiplier*_spiral_Offset>
      <_spiral_Offset, 0, _spiral_Multiplier*_spiral_Offset>
      Spiral_Rail_T/2
    }
    sphere {
      <_spiral_Offset, 0, _spiral_Multiplier*_spiral_Offset>
      Spiral_Rail_T/2
    }
    cylinder {
      <_spiral_Offset, 0, _spiral_Multiplier*_spiral_Offset>
      <_spiral_Offset, 0, 0>
      Spiral_Rail_T/2
    }
    sphere {
      <_spiral_Offset, 0, 0>
      Spiral_Rail_T/2
    }
    cylinder {
      <_spiral_Offset, 0, 0>
      <_spiral_Offset - Spiral_Tread_Overlap - Spiral_Rail_T, 0, 0>
      Spiral_Rail_T/2
    }
    translate y*(Spiral_Rail_H)
      texture { T_Spiral_Rail }
  }

  union {
    object
      {
	  _spiral_Landing_rail_post
	  translate <-_spiral_Offset, 0, _spiral_Multiplier*-_spiral_Offset>
      }
    object
      {
	  _spiral_Landing_rail_post
	  translate <-_spiral_Offset, 0, _spiral_Multiplier*_spiral_Offset>
      }
    object
      {
	  _spiral_Landing_rail_post
	  translate <_spiral_Offset, 0, _spiral_Multiplier*_spiral_Offset>
      }
    object { _spiral_Landing_rail_post translate <_spiral_Offset, 0, 0> }
    object
      {
	  _spiral_Landing_rail_post
	  translate <0, 0, _spiral_Multiplier*-_spiral_Offset>
      }

    #declare _spiral_i = 0;
    #declare _spiral_j = _spiral_Full_0;
    #while (_spiral_i < _spiral_Full_N - 1)
    object
	{
	    _spiral_Landing_rail_post
	    translate <_spiral_j, 0, _spiral_Multiplier*_spiral_Offset> 
	}
    object
	{ 
	    _spiral_Landing_rail_post
	    translate <-_spiral_Offset, 0, _spiral_Multiplier*_spiral_j> 
	}
    #declare _spiral_i = _spiral_i + 1;
    #declare _spiral_j = _spiral_j + Spiral_Post_Delta;
    #end	

    #declare _spiral_i = 0;
    #declare _spiral_j = _spiral_Half_0 ;
    #while (_spiral_i < _spiral_Half_N - 1)
    object
	{
	    _spiral_Landing_rail_post
	    translate <_spiral_j - _spiral_Offset, 0,
	    	       _spiral_Multiplier*-_spiral_Offset> 
	}
    object
	{ 
	    _spiral_Landing_rail_post
	    translate <_spiral_Offset, 0,
	    	       _spiral_Multiplier*(_spiral_j - Spiral_Rail_T)>
	}
    object
	{
	    _spiral_Landing_rail_post
	    translate <0, 0, _spiral_Multiplier*(_spiral_j - _spiral_Offset - 
						 Spiral_Rail_T)>
	}
    #declare _spiral_i = _spiral_i + 1;
    #declare _spiral_j = _spiral_j + Spiral_Post_Delta;
    #end
  }
  rotate y*_spiral_Multiplier*Spiral_Top_A
  translate y*Spiral_Height
}
