//fossing macros. To add detail to sci-fi style objects.
//by Tek
//todo: split this into seperate headers - foss, scene specific textures, text macros.


/*
Cell probability.

Select each unit square randomly from a bunch of textures, according to the probability of each texture.
e.g. if you have 26 textures, each with a different letter of the alphabet tiled infinitely across space
(i.e. 1 that was all A's, one all B's) then this pattern would generate a texture of random letters!

usage:
[pigment|texture|normal] {

	m_CellProbStart(17,1) //set random seed and scale (component textures will be inverse scaled to compensate)
	
	[pigment|texture|normal]_map {
		m_CellProb(.1,pDecals_Num) 
		m_CellProb(.1,pDecals_Symbols) 
		m_CellProb(.1,pDecals_Text) 
		m_CellProb(-1,pNothing) //special case, use this for all unused possibilities, must come last
	}
}
*/

#macro m_CellProbStart(nSeed,vScale)
	#declare fCellProbIndex = 0;
	#declare vCellProbScale = vScale;
	#declare nCellProbSeed = nSeed;
	cells
	translate nCellProbSeed
	scale vCellProbScale
	
#end

#macro m_CellProb(fProb, _Item)
	[fCellProbIndex    _Item scale 1/vCellProbScale translate -nCellProbSeed]

	#if ( fProb = -1 )
		//make sure no other items go after this one.
		#declare fCellProbIndex = 1;
 	#else
 		//step on to end of this item.
		#declare fCellProbIndex = fCellProbIndex + fProb;
	#end

	[fCellProbIndex    _Item scale 1/vCellProbScale translate -nCellProbSeed]

	#if ( fCellProbIndex > 1.01 )
		#warning "Probabilities add up to more than 1"
	#end
#end


/*
Text Block

Fill a unit cube with text.

parameters:
	szFont - typeface file to use
	szText - the text to create
	vScale - amount to scale font by (x and y, z ignored because it fills a unit cube)
	eAlignX, eAlignY -
			Type of alignment, applied using bounding box of text after scaling.
			(if scaled alignement specified, it overrides scaling)

required headers:
	transforms.inc
*/

#declare TEXT_ALIGN_LEFT		= 0;
#declare TEXT_ALIGN_BOTTOM	= 0;
#declare TEXT_ALIGN_RIGHT		= 1;
#declare TEXT_ALIGN_TOP			= 1;
#declare TEXT_ALIGN_CENTRE	= 2; //use for either axis
#declare TEXT_ALIGN_SCALE		= 3;

#macro m_TextBlock( szFont, szText, vfScale, vOffset, eAlignX, eAlignY )

	#local vScale = vfScale*<1,1,1>; //deal with vfScale possibly being a float.

	//count newlines
	#local nStrLen = strlen(szText);
	#local nLinesNum = 0;
	#local nStrPos = 0;
	#while ( nStrPos < nStrLen )
		#if ( strcmp(substr(szText,nStrPos+1,1),"\n") = 0 )
			#local nLinesNum = nLinesNum + 1;
		#end
		#local nStrPos = nStrPos + 1;
	#end //while
	
	#if ( strcmp(substr(szText,nStrLen,1),"\n") != 0 )
		//not newline terminated, need an extra line.
		#local nLinesNum = nLinesNum + 1;
	#end

	#local oText =
		#if ( nLinesNum > 1 )
		merge {
		#else
		object {
		#end
		
			#local nLastNewLine = 0;
			#local nLine = 0;
			#while ( nLine < nLinesNum )
						
				#local nStrPos = nLastNewLine;
				#while ( nStrPos < nStrLen & strcmp(substr(szText,min(nStrPos+1,nStrLen),1),"\n") != 0 )
					#local nStrPos = nStrPos + 1;
				#end //while

				#local oLine = text { ttf szFont substr(szText,nLastNewLine+1,nStrPos-nLastNewLine) 1, 0 scale vScale*(x+y) + z }
				#local nLastNewLine = nStrPos + 1;

				object {
					oLine

					#switch ( eAlignX )
						#case (TEXT_ALIGN_LEFT)
							Align_Trans(oLine, -x, 0)
							#break
			
						#case (TEXT_ALIGN_RIGHT)
							Align_Trans(oLine, x, 1)
							#break
			
						#case (TEXT_ALIGN_SCALE)
							//scale by 1/bounding box, then centre
							Align_Trans(oLine, -x, 0)
							scale <1/(max_extent(oLine).x - min_extent(oLine).x),1,1>
							#break

						#case (TEXT_ALIGN_CENTRE)
							Center_Trans(oLine, x)
							translate x/2 //move to middle of cube
							#break
			
					#end

					translate -y*(nLine+1)*vScale.y
				}
				
				#local nLine = nLine + 1;
			#end//while

			#local fHeight = nLinesNum*vScale.y;
			
			translate fHeight*y
		}
		
	object {
		oText

		//the bounding box on the text is wrong, so do vertical alignment based on computed height
		#switch ( eAlignY )
			#case (TEXT_ALIGN_BOTTOM)
				//do nothing, already aligned
				#break

			#case (TEXT_ALIGN_TOP)
				translate y*(1-fHeight)
				#break

			#case (TEXT_ALIGN_CENTRE)
				translate y*(1-fHeight)/2
				#break

			#case (TEXT_ALIGN_SCALE)
				//scale by 1/bounding box, and keep within unit cube
				scale <1,1/fHeight,1>
				#break
		#end
		
		clipped_by { box { 0, 1 } }
		
		translate vOffset
	}
#end


/*
Details

Scatter details over the surface of an object.

uses a spherical projection from a given "safe" radius.
*/
#macro m_ApplyDetails( oBaseObj, fObjectBoundingRad, aoDetails, nDetailArraySize, nDetailsToApply, nDetailSeed, nPosSeed )
	union {
		object { oBaseObj }
	
		#local rsDetail = seed(nDetailSeed);
		#local rsPos = seed(nPosSeed);

		//scatter loop. we may want to use regularity here		
		#local nObj = 0;
		#while ( nObj < nDetailsToApply )
		
			//choose/compute position of detail
			#local vPos = <rand(rsPos),rand(rsPos),rand(rsPos)>;
			//sphere distribution
			#local vPos = vPos*<2,2,0> - <0,1,0>;
			#local vPos = pi*<vPos.x,select(vPos.y,-pow(vPos.y,2)/2,pow(vPos.y,2)/2),0>;
			#local vPos = <cos(vPos.x)*cos(vPos.y),sin(vPos.y),sin(vPos.x)*cos(vPos.y)>;
			
			//project to object
			#local vNorm = <0,0,0>;
			#local vPos = trace( oBaseObj, vPos*fObjectBoundingRad, -vPos, vNorm );

			#if ( vNorm.x != 0 | vNorm.y != 0 | vNorm.z != 0 )
			
				//select detail to apply
				#local nDetail = (nDetailArraySize - .001) * rand(rsDetail);
				
				//apply it
				object {
					aoDetails[nDetail]

					//rotate to point out of surface, keeping it facing up.
					#if ( vNorm.x != 0 )
						Reorient_Trans(z, <vNorm.x,0,vNorm.z>)
					#else
						#if ( vNorm.z < 0 )
							rotate 180*y
						#end
					#end
					Reorient_Trans(<vNorm.x,0,vNorm.z>, vNorm)
					
					translate vPos
				}
				
			#end

			#local nObj = nObj + 1; //only step on if obj created.
		#end //while ( nObj )
	} //union
#end



/*
Cube Map

Apply a texture designed to by mapped onto plane { -z } to
a given object, using a slope map to change the angle every
90 degrees. the texture may be offset for each rotation.

vAxisMask specifies which axes to map to, and uses rgbt 1
on axes not being mapped to.
*/
#macro m_pCubeMap( _ItemInNegZ, vOffset, vAxisMask )
    #local pXZ =
        pigment {
            slope { x }
            pigment_map {
                [.25
                    #if ( vdot(vAxisMask,x) != 0 )
                        _ItemInNegZ rotate 90*y translate vOffset*1
                    #else
                        rgbt 1
                    #end
                ] 
                [.25
                    #if ( vdot(vAxisMask,z) != 0 )
                        _ItemInNegZ translate vOffset*1
                    #else
                        rgbt 1
                    #end
                ] 
                [.75
                    #if ( vdot(vAxisMask,z) != 0 )
                        _ItemInNegZ rotate 180*y translate vOffset*1
                    #else
                        rgbt 1
                    #end
                ]
                [.75
                    #if ( vdot(vAxisMask,x) != 0 )
                        _ItemInNegZ rotate -90*y translate vOffset*1
                    #else
                        rgbt 1
                    #end
                ]
            } //map
        } //pigment

    slope { y }
    pigment_map {
        [.25
            #if ( vdot(vAxisMask,y) != 0 )
                _ItemInNegZ rotate -90*x translate vOffset*1
            #else
                rgbt 1
            #end
        ] 
        [.25
            pXZ
        ] 
        [.75
            pXZ
        ]
        [.75
            #if ( vdot(vAxisMask,y) != 0 )
                _ItemInNegZ rotate 90*x translate vOffset*1
            #else
                rgbt 1
            #end
        ]
    }
#end




//standard textures (useful to keep them here to save clutter in the main source file)
#declare pRedPaint =
	pigment { rgb <1,.08,.03> }

#declare pBluePaint =
	pigment { rgb <.02,.1,.6> }
	
#declare pBlackPaint =
	pigment { rgb .1 }
	
#declare pYellowPaint =
	pigment { rgb <1,.9,.03> }
	
#declare pWhitePaint =
	pigment { rgb <.98,.97,.93> }
	
#declare pStripesBY =
	pigment {
		gradient x+y
		scale 2
		pigment_map { [.5 pBlackPaint] [.5 pYellowPaint] }
	}

#declare pLoudStuff =
	pigment {
		bozo
		turbulence 4
		scale 10
		sine_wave
		colour_map {
			[0	rgb <1,1,0>]
			[.5	rgb <1,0,0>]
			[1	rgb <0,0,0>]
		}
	}

#declare pRockTwist =
	pigment {
		bozo
		turbulence 4
		scale 10
		sine_wave
		colour_map {
			[0	rgb .4]
			[1	rgb .2]
		}
	}


#declare tBoltGun =
	texture {
		pigment {
			spotted
			scale 0.04
			colour_map {
				[0	rgb .1]
				[1	rgb .3]
			}
		}
		finish {
			specular 1.0
			roughness 0.1
			reflection 0.2
		}
		normal {
			spotted 1
			scale 0.04
		}
	}

#declare tChainMetal =
	texture {
		pigment {
			spotted
			scale 0.04
			colour_map {
				[0	rgb .2]
				[1	rgb .3]
			}
		}
		finish {
			specular 1.0
			roughness 0.1
			reflection 0.5
		}
		normal {
			spotted .1
			scale 0.02
		}
	}

#declare tMithrilSilver =
	texture {
		pigment {
			spotted
			scale 0.04
			colour_map {
				[0	rgb .5]
				[1	rgb .7]
			}
		}
		finish {
			specular 1.0
			roughness 0.1
			reflection 0.4
		}
		normal {
			spotted .4
			scale 0.04
		}
	}

#declare nPanels =
	normal {
		cells 3
		slope_map {
			[.5	<0,0>]
			[.5	<1,0>]
		}
	}







