//---------------------------------------------------------------------------------------------------//
//                                           Tree.inc                                                //
// Objects included:                                                                                 //
// - Foreground Tree Group                                                                           //
// - Background Tree Group                                                                           //
//---------------------------------------------------------------------------------------------------//

//function.inc included in howl.pov

//---------------------------------------------------------------------------------------------------//
//                                   Basic Geometric Functions                                       //
//---------------------------------------------------------------------------------------------------//
#declare fn_cylinder = function(x,y,z,r) { sqrt(pow(x,2) + pow(z,2)) - r };
#declare fn_cone = function { sqrt(pow(x,2) + pow(z,2)) + y };

//---------------------------------------------------------------------------------------------------//
//                                        Test Blob Functions                                        //
//---------------------------------------------------------------------------------------------------//
#declare blob_threshold = 0.3;

//---------------------------------------------------------------------------------------------------//
//                                       Pigment Functions                                           //
//---------------------------------------------------------------------------------------------------//
//Agate pigment function, ideal for background level 1 trees
#declare fn_Pigm1=function {
    pigment {
      agate
      turbulence 0.2
      colour_map {
        [0 color rgb 0]
        [1 color rgb 1]
      }
      //scale <1,1,3.5>
    }
  };

//Crackle function, ideal for foreground (closeup) trees
#declare fn_Pigm2=function {
    pigment {
      crackle
      turbulence 0.1
      colour_map {
        [0 color rgb 0]
        [0.1 color rgb 0]
        [0.1 color rgb 0.7]
        [1 color rgb 0.8]
      }
      scale <0.4,0.7,0.4>
    }
  };

#declare agate_pigm_scale = <0.65,0.08,0.65>;

#declare pigment1 = pigment {
    agate
      //turbulence 0.2
      colour_map {
        [0 color rgb 0]
        [1 color rgb 0.1]
      }
      scale agate_pigm_scale
}     

#declare pigment2 = pigment {
     agate
     //turbulence 0.2
     colour_map {
        [0 color rgb 0.2]
        [1 color rgb 1]
     }
     scale agate_pigm_scale
}     

#declare fn_Pigm3 = function {
     pigment {
          gradient y
          pigment_map {
               [0 pigment2]
               [1 pigment1]
          }
     }
}

//---------------------------------------------------------------------------------------------------//
//                                      Tree Shape Variables                                         //
//---------------------------------------------------------------------------------------------------//
//Tree top isosurface scaling factors
#declare wood_variation = 6;       //variation in lengths of the woodchips (7)
#declare wood_density = 4;         //density of the woodchips (4)
#declare wood_h_diff = 4;          //overall height variation of the top surface (4)
#declare wood_h_threshold = wood_variation+1;
//---------------------------------------------------------------------------------------------------//
//                                   Tree Shape Geo Function                                         //
//---------------------------------------------------------------------------------------------------//
//tree's middle trunk: r = radius
#declare fn_mid_trunk = function(x,y,z,r) { sqrt(pow(x,2) + pow(z,2)) - r };

//tree's base (roots): s = slope, h = tip height
#declare fn_tree_cone = function(x,y,z,s,h) { sqrt(pow(x*s,2) + pow(z*s,2)) + y - h };

//tree's top - supposed to be "destroyed": h = y distance of tip of the fractured surface
//For tall trees
#declare fn_tree_top = function(x,y,z,s,h) { -fn_tree_cone(x,y,z,s,h)+(f_noise3d(x*wood_density,0,z*wood_density)*wood_variation)+(f_noise3d(x/2,0,z/2)*wood_h_diff)-wood_h_threshold };

//For short-cut trees
#declare fn_tree_top2 = function(x,y,z,s,h) { -fn_tree_cone(x,y-h,z,s,h)+(f_noise3d(x*wood_density,0,z*wood_density)*wood_variation)/*+(f_noise3d(x/2,0,z/2)*wood_h_diff)*/-wood_h_threshold };

//#declare fn_tree_top3 = function(x,y,z,h) { -y+(f_noise3d(x*wood_density,0,z*wood_density)*wood_variation)+(f_noise3d(x/2,0,z/2)*wood_h_diff)+h-wood_h_threshold };

//---------------------------------------------------------------------------------------------------//
//                                   Tree Texture / Image Map                                        //
//---------------------------------------------------------------------------------------------------//
#declare wood_colour = RGBColour (214, 194, 146); //(215, 187, 124);
#declare wood_colour2 = RGBColour (134, 134, 139);

//For barkImage2
#declare hori_tex_scl = 0.005;
#declare vert_tex_scl = 2;   //higher number = compression (1=no compression, no zero!)

//For short tree
#declare Bark_Image = pigment {
     image_map {
          tga "texturez/bark3.tga"
          map_type 2
          interpolate 2
     }
     scale <0.1,0.3,0.1>*0.2
}

#declare Bark_Image2 = pigment {
     image_map {
          tga "texturez/bark3.tga"
          map_type 2
          interpolate 2
     }
}

#declare N_Inner_Wood = normal {
     wood 0.3
     bump_size 0.5
     turbulence 0.25
}

#declare Bark_Finish = finish {
     ambient 0.4
     diffuse 0.3
}
     
#declare Wood_Finish = finish {
     ambient 0.2
     diffuse 0.4
}

#declare T_Tree_Test = texture { pigment { colour rgb 1} finish {ambient 0.3}}

//---------------------------------------------------------------------------------------------------//
//                                   Tree Shape Gen Macro                                            //
//---------------------------------------------------------------------------------------------------//
//Foreground tree (root remains)
#declare overall_tex_scl = 0.4;

#macro fg_tree (base_r, base_h, trunk_r, trunk_h)

     #local base_slope = base_h / (base_r - trunk_r);
     #local base_tip_h = base_slope * trunk_r;
     #local fn_trunk_base = function {
          (1+blob_threshold)
          -pow(blob_threshold, fn_mid_trunk(x,y,z,trunk_r))
          -pow(blob_threshold, fn_tree_cone(x,y,z,base_slope,base_tip_h))
     };
     //#local fn_final = function { max(fn_trunk_base(x,y,z), -fn_tree_top(x,y,z,cut_angle,trunk_h+cut_level)) }

     #if (fg_high_detail)
          isosurface {
               function { fn_trunk_base(x,y,z) - fn_Pigm3(x/2, (y+base_h)/(trunk_h+base_h), z/2).grey*0.5 }
               contained_by { box {<base_r+1,trunk_h,base_r+1>,<-base_r-1,-base_h,-base_r-1>} }
               //evaluate 2, 1.1, 0.95    //2,1.1,0.95=>1m40s@320x240
               //No Extra Settings 51s@320x240
               //640x480 AA0.3 1 tree  18m17s
               //max_gradient 4
          }
     #else
          union {
               #if (base_h > 0) cone { <0,-base_h,0>, base_r, <0,0,0>, trunk_r } #end
               cylinder { <0,0,0>,<0,trunk_h,0>,trunk_r }
          }
     #end
     #if (detail_texture)
          texture { 
               pigment { Bark_Image2 scale <hori_tex_scl, trunk_h/vert_tex_scl, hori_tex_scl>*overall_tex_scl}
               finish { Bark_Finish }
          }
     #else
          texture { T_Tree_Test }
     #end
     translate base_h*y
#end

//Parameter Order
//Base radius (root base), base height (root height), trunk radius (max @ base), trunk height...
//...y level of top cut, Cut angle, ofs_x/y/z: cut offset (translation)
//positive cut_level = up
#macro short_tree (base_r, base_h, trunk_r, trunk_h, cut_level, cut_angle, ofs_x, ofs_y, ofs_z, burnt, colour_shift)

     #if (base_h > 0)
          #local base_slope = base_h / (base_r - trunk_r);
     #else
          #local base_slope = 1;
     #end
     #local base_tip_h = base_slope * trunk_r;
     #local fn_trunk_base1 = function {
          (1+blob_threshold)
          -pow(blob_threshold, fn_mid_trunk(x,y,z,trunk_r))
     };
     #if (base_h > 0)
          #local fn_trunk_base2 = function { fn_trunk_base1(x,y,z)-pow(blob_threshold, fn_tree_cone(x,y,z,base_slope,base_tip_h)) };
     #else
          #local fn_trunk_base2 = fn_trunk_base1;
     #end 
     #local fn_final = function { max(fn_trunk_base2(x,y,z)-fn_Pigm1(x/2, y/trunk_h, z/2).grey*0.5, -fn_tree_top2(x-ofs_x,y-ofs_y,z-ofs_z,cut_angle,trunk_h+cut_level)) }

     //-------- Texture Declarations
     #local t_trunk = texture {
          pigment { Bark_Image scale trunk_h }
          finish { Bark_Finish }
     };
     #if (base_h > 0)
          #local t_base = texture { pigment { Bark_Image scale base_h } finish { Bark_Finish }};
     #else
          #local t_base = texture { pigment { Bark_Image } finish { Bark_Finish }};
     #end
     #local t_burnt = texture { 
          pigment { rgb wood_colour - burnt } 
          normal { N_Inner_Wood scale <0.5,0.3,0.5> }
          finish { Wood_Finish }
     }
     
     //---------- Some variables
     #local total_h = base_h + trunk_h + ofs_y;        //total height
     #local burnt_y = (total_h + cut_level) / total_h;
          
     #if (st_high_detail)
          isosurface {
               //function {fn_tree_top(x,y,z,cut_angle,trunk_h+cut_level)}
               function { fn_final(x,y,z) }
               contained_by { box {<base_r+1,trunk_h,base_r+1>,<-base_r-1,-base_h,-base_r-1>} }
               max_gradient 4
               //evaluate 4, 10, 0.95
          }
     #else
          union {
               #if (base_h > 0) cone { <0,-base_h,0>, base_r, <0,0,0>, trunk_r } #end
               cylinder { <0,0,0>,<0,trunk_h,0>,trunk_r }
          }
     #end
     //Tree Texture
     #if (detail_texture)
     texture {
          gradient y
          turbulence 0.18
          texture_map {
               [0.00000 t_base]
               [burnt_y t_trunk]
               [burnt_y t_burnt]
               [1.00000 t_burnt]
          }
          scale <1,total_h,1>
          translate (-base_h+colour_shift)*y
     }
     #else
          texture { T_Tree_Test }
     #end
  
     translate base_h*y
#end

//---------------------------------------------------------------------------------------------------//
//                                   Foreground Branch Objects                                       //
//---------------------------------------------------------------------------------------------------//

#if (branch_high_detail)
//-------------------------- high detail - use tomtree ----------
     //More branches background trees
     #include "branch.inc"
     #include "TOMTREE.inc"
     #declare obj_branch1 = object { TREE }

     //Less branches than the branch1
     #include "branch2.inc"
     #include "TOMTREE.inc"
     #declare obj_branch2 = object { TREE }

#else
//------------------------ cylinder tree ------------------
     #local test_radius = 0.01;
     #local test_scale = 0.7; 
     #declare obj_branch1 = cylinder { <0,0,0>,<0,1,0>,test_radius
          texture { pigment { colour rgb 0.3 }}
          scale test_scale
     }

     #declare obj_branch2 = cylinder { <0,0,0>,<0,1,0>,test_radius
          texture { pigment { colour rgb 0.3 }}
          scale (test_scale-<0,0.2,0>)
     }            
#end

//coord = tree center  r=tree radius vert=vertical distance from ground
//v/h_ang=vertical/horizontal angle of rotation scl=scale ofs=in/out offset (+ out, - in)
#macro branch1 (coord,r,vert,v_ang,h_ang,scl,ofs)
     object { obj_branch1
          scale scl
          rotate v_ang*z
          translate <-(r+ofs),0,0>
          rotate h_ang*y
          translate coord+<0,vert,0>
     }
#end

//coord = tree center  r=tree radius vert=vertical distance from ground
//v/h_ang=vertical/horizontal angle of rotation scl=scale ofs=in/out offset
#macro branch2 (coord,r,vert,v_ang,h_ang,scl,ofs)
     object { obj_branch2
          scale scl
          rotate v_ang*z
          translate <-(r+ofs),0,0>
          rotate h_ang*y
          translate coord+<0,vert,0>
     }
#end

//---------------------------------------------------------------------------------------------------//
//                             Foreground Tree Object (Branches included)                            //
//---------------------------------------------------------------------------------------------------//
//Cylinder tree
#declare test_tree = cylinder { <0,0,0>,<0,1,0>,0.5
     texture { pigment { colour rgb 0.8 }}
}
 
//------------------------- High detail ISOSURFACE tree --------------------
#declare fg_tree_group = union {
          // --------- Fallen tree trunk ---------
          object { short_tree (1.6,0,1.6,33,-10,2,0,-10,0,0.2,13) rotate <-90,-28.5,0> translate <-7,1.5,7> rotate wolf_rotation translate <-2,0,2.5>}
          //object { short_tree (1.6,0,1.6,30,-13,4,0,0,0,0.2,0) rotate <-90,-28.5,0> translate <-7,1.5,7> rotate wolf_rotation}
          //Branch that goes with the fallen tree
          //#declare fallen_branch1 = branch2 (<0,0,0>,1.5,25,130,-5,18,-0.5)
          //object { fallen_branch1 rotate <-90,-27,0> translate <-6,1.5,5> }

          // ---------- Right side tree --------
          object { short_tree (2.1,1.8,1,29,-4,2,9,-5,0,0.3,2) translate <5,0,3> }
          branch2 (<5,0,3>,1,22,140,30,<18,15,18>,-0.2)
          
          object { fg_tree (2.5,2.5,1.4,50) rotate 45*y translate <8,0,-8> }
          branch1 (<8,0,-8>,1.4,30,140,0,<8,15,8>*1.2,-0.2) //left
          branch2 (<8,0,-8>,1.4,28,40,90,<20,20,20>,-0.2)      //on the back
          branch2 (<8,0,-8>,1.4,27,120,120,22,-0.2)
          
          object { short_tree (2.7,2.5,1.3,9,-2,2,0,-1,-0.5,0.1,0.5) rotate <0,90,0> translate <15,0,10> }
          
          object { fg_tree (2.5,2.5,1.5,50) translate <15,0,-15> }
          branch1 (<15,0,-15>,1.5,30,140,0,<8,14,8>*1.2,-0.2)    //left
          branch2 (<15,0,-15>,1.5,29,100,90,22,-0.2)
          branch2 (<15,0,-15>,1.5,22,120,-40,20,-0.2)
          branch2 (<15,0,-15>,1.5,19,140,-10,23,-0.2)         //lowest
          
          
          object { fg_tree (2.5,2.5,1.5,50) translate <32,0,5> }
          branch2 (<32,0,5>,1.5,38,50,-30,26,-0.2)
          branch2 (<32,0,5>,1.5,34,60,30,26,-0.2)

          // -------- Left Side tree ----------
          object { short_tree (3,2.5,1.5,10,-3,2,-0.8,0,0.5,0,0) scale 0.8 translate <-9,0,20> }
          object { short_tree (3,2.5,1.6,15,-4,2,2,0,3,0,-0.8) translate <-9,0,0> }
          
          object { fg_tree (4,3,2.2,55) rotate 45*y translate <-24,0,5> }
          branch2 (<-24,0,5>,2.2,44,60,-20,25,-0.2)       //left
          //branch2 (<-24,0,5>,2.2,48,50,-180,25,-0.2)
         
          object { fg_tree (3,2.5,1.5,50) translate <-20,0,-12> }
          branch2 (<-20,0,-12>,1.5,30,30,-70,20,-0.2)       //left
          //branch2 (<-20,0,-12>,1.5,30,50,-170,22,-0.2)
          branch2 (<-20,0,-12>,1.5,28,80,170,<26,22,26>,-0.2)
          branch2 (<-20,0,-12>,1.5,20,70,-150,25,-0.2)
                    
          object { short_tree (3,2.5,1.6,7,-2,2,0,0,0,0,-1.5) translate <-5,0,-21> }
     
          //------------- Other Branches -----------------
          #declare fallen_branch2 = branch1 (<0,0,0>,0,0,70,0,<14,14,14>,0)
          object { fallen_branch2 rotate 90*x translate <18,1.3,-5>}
          
          #declare Not_Really_A_Branch = branch2 (<0,0,0>,0,0,-10,0,16,0)
          object { Not_Really_A_Branch translate <-12,1.5,-3.4> }
 }


//---------------------------------------------------------------------------------------------------//
//                                   Background Tree Objects                                         //
//---------------------------------------------------------------------------------------------------//

#if (bg1_high_detail)
     //-------------------------- high detail - use tomtree -------------------
     #include "bg_tree1.inc"
     #include "TOMTREE.inc"
     //More branches background trees
     #declare bg_tree_1 = object { TREE }

     //Less branches than the bg_tree_1
     #declare BRANCHES=10;
     #declare TWIGS=2*BRANCHES;
     #declare BRANCH_RANDOM=36.480;     //25 hmm, 125 ok, 534.473,143.473(not bad), 36.480
     #include "TOMTREE.inc"
     #declare bg_tree_2 = object { TREE }

     //Almost no branches left - burnt
     #declare BRANCHES=5;
     #declare TWIGS=2*BRANCHES;
     #declare BRANCH_RANDOM=832.423;//63.123;

     /*#declare TRUNK_STRENGTH=1.5;
     #declare TRUNK_BEND_X=30;
     #declare TRUNK_BEND_Z=-20;
     #declare TRUNK_BEND_X_FREQUENCY=1;
     #declare TRUNK_BEND_Z_FREQUENCY=2;
     #declare TRUNK_CURVE=-10;
     #declare TRUNK_CURVE_FREQUENCY=3;
     #declare TRUNK_SNAG=3;
     #declare TRUNK_SNAG_FREQUENCY=7;*/

     #include "TOMTREE.inc"
     #declare bg_tree_3 = object { TREE }

#else
     //------------------------ cylinder tree ------------------
     #local test_radius = 0.01;
     #local test_scale = <0.7,0.9,0.7>; 
     #declare bg_tree_1 = cylinder { <0,0,0>,<0,1,0>,test_radius
          texture { pigment { colour rgb 0.5 }}
          scale test_scale
     }

     #declare bg_tree_2 = cylinder { <0,0,0>,<0,1,0>,test_radius
          texture { pigment { colour rgb 0.5 }}
          scale test_scale
     }            

     #declare bg_tree_3 = cylinder { <0,0,0>,<0,1,0>,test_radius
          texture { pigment { colour rgb 0.5 }}
          scale test_scale
     }

#end

//------- Background tree group declaration -------------
//Broken in half tree: Param: 
//half: cut top half (1) / lower half (0)
//dist: distance from end point to cut
//rnd: random seed
//move: spatial adjustment
#declare cone_count = 10;
#declare box_size = 0.01;
#declare cone_inc = (box_size*2)/cone_count;

#macro bg_tree_4 (tophalf, dist, rnd, move)
     //Destroy previous declarations
     #ifdef (this_broken_tree) #undef this_broken_tree #end
     #ifdef (CIH_Block) #undef CIH_Block #end
     #ifdef (t4_randseed) #undef t4_randseed #end
     
     #declare t4_randseed = seed(rnd);
     
     //Cut-in-half difference block
     #declare CIH_Block = union {
          box { <box_size+1,0,box_size+1>,<-box_size-1,2,-box_size-1> }
          #local i = 0;
          #local x_val = (i*cone_inc)-box_size-(cone_inc/2);
          #while (i < cone_count)
               #local j = 0;
               #local x_val = x_val + cone_inc;
               #local z_val = (j*cone_inc)-box_size-(cone_inc/2);
               #while (j < cone_count)
                    #local z_val = z_val + cone_inc;
                    #local random_rot = (rand(t4_randseed)/50)-0.01;
                    cone { <x_val+random_rot, random_rot, z_val+random_rot>, (cone_inc/2)-(random_rot/100), <x_val+random_rot, -(rand(t4_randseed)/10)-(j/50)-random_rot, z_val+random_rot>, +random_rot/100 }
                    #local j = j + 1;
               #end
               #local i = i + 1;
          #end
          texture { 
               pigment { rgb wood_colour-0.6 } 
               normal { N_Inner_Wood scale <0.5,0.3,0.5> }
               //finish { Wood_Finish }
          }
     }

     #declare this_broken_tree = difference {
          object { bg_tree_2 }
          object { CIH_Block
               #if (!tophalf) rotate 180*z #end
               translate dist*y
               translate move
          }
     }
   
#end


#declare rand_pos = seed(23);
#declare rand_y = seed(22);
#declare rand_rot = seed(1254);

#declare bg_tree_group = union {

     //--------------- Final trees - Second row
     object { bg_tree_1 scale 100 rotate <0,60,-26> rotate -18*y translate <-1,0,25> }
     object { bg_tree_2 scale 150 rotate 20*y translate <20.5,0,20> }
     object { bg_tree_1 scale 80 translate <34,0,45> }
     object { bg_tree_1 scale 90 translate <40,0,30> }
     object { bg_tree_3 scale 100 translate <42,0,20> }
     
     //object { bg_tree_1 scale 100 translate <50,0,15> }
     //object { bg_tree_1 scale 100 translate <55,0,10> }
     /*object { bg_tree_3 scale 100 translate <56,0,4> }
     object { bg_tree_3 scale 100 translate <47,0,27> }*/
     
     object { bg_tree_3 scale 90 rotate <0,180,-90> rotate 20*y translate <-12,1.5,35> }
     
     object { bg_tree_3 scale 100 rotate <0,90,-90> rotate -30*y translate <-40,2,0> }
     object { bg_tree_1 scale 80 rotate 65*y translate <-20,0,40>}          //-16*x
     bg_tree_4 (1,0.35,38.231,0)
     object { this_broken_tree scale 160 rotate 180*y translate <-27,0,23> }
     //object { bg_tree_2 scale 160 rotate 180*y translate <-27,0,23>}
     object { bg_tree_2 scale 100 rotate 15*z translate <-27,0,25>}
     object { bg_tree_1 scale 100 rotate 30*z translate <-30,0,30>}
     
     //----------------- Background tree, last row
     object { bg_tree_2 scale 80 translate <-2,0,55> }
     object { bg_tree_2 scale 80 rotate 15*z rotate -60*y translate <14,0,58> }
     object { bg_tree_2 scale 80 rotate 0*z translate <26,0,70> }
     object { bg_tree_2 scale 80 rotate 10*z translate <40,0,60> }
     
     object { bg_tree_2 scale 80 translate <-10,0,80> }
     object { bg_tree_1 scale 80 translate <-15,0,80> }
     object { bg_tree_2 scale 80 rotate -6*z translate <-10,0,40> }  //**
     bg_tree_4 (1,0.4,63.371,0)
     object { this_broken_tree scale 80 rotate 30*z rotate 0*y translate <-4,0,45> }
     object { bg_tree_1 scale 80 translate <-25,0,83> }
     //object { bg_tree_2 scale 80 rotate 6*z translate <-30,0,60> }
       object { bg_tree_3 scale 100 rotate <0,45,-62.5> rotate 80*y translate <-15,0,35> }
     object { bg_tree_1 scale 80 rotate 10*z translate <-28,0,40> }
     object { bg_tree_2 scale 80 translate <-60,0,70> }
     object { bg_tree_3 scale 80 rotate -10*z translate <-80,0,60> }
     object { bg_tree_1 scale 80 translate <-60,0,40> }
     
     //----------------- Really Background Now
     object { bg_tree_2 scale 80 translate <0,0,95> }
     //object { bg_tree_1 scale 80 rotate 45*y translate <30,0,90> }
     object { bg_tree_2 scale 80 rotate 5*z translate <40,0,90> }
     object { bg_tree_1 scale 80 translate <45,0,95> }
     object { bg_tree_2 scale 80 translate <50,0,95> }
     
     object { bg_tree_2 scale 80 rotate 3*z translate <-20,0,60> }
     object { bg_tree_1 scale 80 translate <-33,0,60> }
     object { bg_tree_2 scale 80 translate <-100,0,95> }
     object { bg_tree_1 scale 80 translate <-110,0,90> }
     
     #declare i = 0;
     #while (i < 130)
          //Row 1
          object { bg_tree_2
               scale 80
               rotate (rand(rand_rot)*16-8)*z
               translate <i+rand(rand_pos),-(rand(rand_y)*3),100+(rand(rand_pos)*30)>
               rotate 10*y
          }
          object { bg_tree_2
               scale 80
               rotate (rand(rand_rot)*16-8)*z
               translate <-i-15-rand(rand_pos),-(rand(rand_y)*3),100+(rand(rand_pos)*30)>
               rotate -30*y
          }

          //Row 2
          object { bg_tree_2
               scale 80
               rotate (rand(rand_rot)*10-5)*z
               translate <i-10+rand(rand_pos),(rand(rand_y)*5),140+(rand(rand_pos)*50)>
          }
          object { bg_tree_2
               scale 80
               rotate (rand(rand_rot)*10-5)*z
               translate <-i-50-rand(rand_pos),(rand(rand_y)*5),110+(rand(rand_pos)*50)>
          }

          //Row 3
          //object { bg_tree_3 scale 70 translate <i+30+rand(rand_pos),(rand(rand_y)*8),180+(rand(rand_pos)*70)> }
          //object { bg_tree_3 scale 70 translate <i-50+(rand(rand_pos)*5),(rand(rand_y)*8),200+(rand(rand_pos)*80)> }
          //object { bg_tree_3 scale 70 translate <-i-100-rand(rand_pos),(rand(rand_y)*8),180+(rand(rand_pos)*70)> }
          
          #declare i = i + 10;
     #end

}    //union


//---------------------------------------------------------------------------------------------------//
//                                        Test Objects                                               //
//---------------------------------------------------------------------------------------------------//

#declare test_tree2 = cylinder { <0,0,0>,<0,1,0>,0.5
     texture { pigment { colour rgb <1,1,1> }}
}

#declare tree = isosurface {
     function { y-f_ridged_mf(x,y,z, 0.2, 1, 1.5, 0, 0, 1)}
     //function { test_tree(x,y,z) }
     contained_by { box {-3,3} }
     max_gradient 3
     //accuracy 0.005
     texture {
          normal {
               wrinkles 0.5
               turbulence 0.2
               scale 0.3
          }
          pigment { colour rgb <0.8,0.78,0.78> }
          //finish { ambient 0.3 diffuse 0.5 specular 0.2 roughness 0.1 }
     }
     scale 1.5
     rotate 0*x
}

//Background level 1 tree
#macro bg_tree (base_r, base_h, trunk_r, trunk_h, top_r)

     #local base_slope = base_h / (base_r - trunk_r);
     #local base_tip_h = base_slope * trunk_r;
     #local trunk_slope = trunk_h / (trunk_r - top_r);
     #local trunk_tip_h = trunk_slope * trunk_r;
     #local fn_final = function {
          (1+blob_threshold)
          -pow(blob_threshold, fn_tree_cone(x,y,z,trunk_slope, trunk_tip_h)) //tree trunk
          -pow(blob_threshold, fn_tree_cone(x,y,z,base_slope,base_tip_h))
     };
     //#local fn_final = function { max(fn_trunk_base(x,y,z), -fn_tree_top(x,y,z,4,trunk_h)) }

     isosurface {
          //function { fn_tree_top(x,y,z,trunk_h-2) }
          function { fn_final(x,y,z) }//-fn_Pigm1(x/2, y/5, z/2).grey*0.5 }
          contained_by { box {<base_r,trunk_h,base_r>,<-base_r,-base_h,-base_r>} }
          max_gradient 4
          pigment {colour White}
     }
#end


/*
     //--------------------------------- cylinder tree ---------------------------
     #declare fg_tree_group = union {
          //Lie down tree
          object { test_tree scale <3,30,3> rotate <-90,-27,0> translate <-7,1.5,7> rotate wolf_rotation} //<-8,2,6>

          //Closest Row right (+x)
          object { test_tree scale <2.4,35,2.4> translate <5,0,3> }
          object { test_tree scale <3,50,3> translate <8,0,-8> }
          object { test_tree scale <3,10,3> translate <13,0,0> }
          object { test_tree scale <2.8,50,2.8> translate <15,0,-15> }
          object { test_tree scale <2.3,50,2.3> translate <32,0,5> }

          //Closest Row left (-x)
          object { test_tree scale <2.6,10,2.6> translate <-9,0,20> }
          object { test_tree scale <2.6,15,2.6> translate <-6,0,-10> }
          object { test_tree scale <2.8,60,2.8> translate <-24,0,5> }
          object { test_tree scale <2.8,50,2.8> translate <-20,0,-12> }
          object { test_tree scale <3,5,3> translate <-5,2,-21> }

          //------------- Branches -----------------
          branch1 (<8,0,-8>,1.4,33,130,0,<8,15,8>*1.2,-0.2)
          branch2 (<8,0,-8>,1.4,27,150,150,25,-0.2)

          branch2 (<-20,0,-12>,1.5,30,30,-70,20,-0.2)
          branch2 (<-20,0,-12>,1.5,20,40,-150,23,-0.2)
          branch2 (<-20,0,-12>,1.5,28,100,-180,25,-0.2)

          branch1 (<15,0,-15>,1.5,31,160,0,<8,15,8>*1.3,-0.2)
          branch2 (<15,0,-15>,1.5,24,110,70,15,-0.2)

          branch2 (<32,0,5>,1.5,38,160,30,25,-0.5)

          #declare fallen_branch1 = branch2 (<0,0,0>,1.5,25,130,-5,15,-0.5)
          object { fallen_branch1 rotate <-90,-27,0> translate <-6,1.5,5> }

          #declare fallen_branch2 = branch1 (<0,0,0>,0,0,70,0,<14,14,14>,0)
          object { fallen_branch2 rotate 90*x translate <18,1.3,-5>}
     }
*/
