// Persistence of Vision Ray Tracer Scene Description File
// File: icedland.pov
// Vers: 3.5
// Desc: A nice winter scene
// Date: 02.07.02 (dd.mm.yy)
// Auth: Tim Nikias Wenclawiak
// Last Update: 20.08.02 (dd.mm.yy)

//Required files:
// None

//Description:
//An icy, smooth winter landscape, nice weather along with a sunset, and
//a lonely iglu...

//You may make use of the objects, textures, algorithms... anything, basically,
//as long as you mention that they're deriven from my code, which is all I ask
//for. Before putting this image on a webpage or use it in any way, please
//ask for permission. Thanks!


//Homepage:
// www.digitaltwilight.de/no_lights
//Email:
// Tim.Nikias@gmx.de

global_settings{assumed_gamma 1 ambient_light 1*<.8,.8,1> max_trace_level 15}

//Command-Line options
//+W1024 +H512 +AM2 +A0.2 +R5

//Imprint area (I tend to trace the image once, and then 
// trace the section with the watermark later)
//+sc0.806452 +sr0.000000 +ec1.000000 +er0.131115

//Heightfield-Area (for LOD changes)
//+sc0.000000 +sr0.594912 +ec1.000000 +er1.000000

//Bump map imprint
//(requires name.tga, a 400x100 image I use to bump-map my
// name onto the image, not supplied with ZIP)
#declare Watermark = on;

//Use the different media-objects?
#declare Media_Sky = on;
#declare Media_Sun = on;
#declare Media_Haze = on;

//High-Sampling on media?
#declare High_MSamples = yes;

//LOD on heightfield
#declare HF_Detail = 2000;

//Camera Position
#declare Cam_Pos = <6,2,-8>;

//Sun-Position and -Colors
#declare Sun_Pos = <-10,10,40>;
#declare Sun_Col = <1,1,1>;
#declare SunBeam_Col = <1,.5,.3>;

//Ambient Color of light/haze
#declare Ambient_Col = <.5,.6,1>;

/***************************************/
//Watermark calculations/setup
//Setup
#declare Imprint_Size = .15; //Width-Proportion of scene
#declare Imprint_Position = <1,1>; //X,Y (0>1 fit into image)
#declare Imprint_Depth = .25;
#declare Imprint_Pic_Size = <400,100>; //Pixel size of imprint-image
//Calculations
#declare _IP_X_Pos = 1-Imprint_Size;
#declare _IP_Y_Pos = 1-(Imprint_Pic_Size.v/image_height)/(Imprint_Pic_Size.u/image_width)*Imprint_Size;
/***************************************/
//Very simple skysphere as backdrop
sky_sphere{
 pigment{
  gradient y 
  color_map{[0 rgb 1][.4 rgb .4][.7 rgb <0,.05,.3>][1 rgb <0,0,.3>]}
  scale 2 translate -1
  }
 }

/***************************************/
//Camera automatically adjusts to widescreen
camera{
 location Cam_Pos
 up y right x*image_width/image_height
 look_at <0,3,0>
 #if (Watermark)
 normal{
  bump_map{tga "name.tga" bump_size Imprint_Depth once}
  scale <1,(Imprint_Pic_Size.v/image_height)/(Imprint_Pic_Size.u/image_width),1>*Imprint_Size
  translate <-.5,-.5,0>+x*_IP_X_Pos*Imprint_Position.u+y*_IP_Y_Pos*Imprint_Position.v
  }
 #end
 }

/***************************************/
//The sun / main & only lightsource
#declare SunCam_Dir = vnormalize(Sun_Pos-Cam_Pos);

light_source{Cam_Pos+SunCam_Dir*300 rgb Sun_Col*.6+.4*vlength(Sun_Col)} //.8,.2

//Sun, simulated with scattering cone pointing on camera
#if (Media_Sun)

 cone{Cam_Pos+SunCam_Dir*150,.7 Cam_Pos+SunCam_Dir*1,1.2  //.9,1.4
  hollow pigment{rgbt <0,0,0,1>}
  interior{
   media{
    scattering{4,(.04*(SunBeam_Col*.7+.3*vlength(SunBeam_Col))) *vnormalize(<.5,.5,1>) extinction .1}
    method 3 intervals 1 samples 5,25
    }}
  no_shadow
  }

#end

/***************************************/
//Haze, simulated with emission/absorption media
#if (Media_Haze)

 sphere{0,1 scale <20,4,20> hollow pigment{rgbt <0,0,0,1>}
  interior{media{emission Ambient_Col*.025 absorption Ambient_Col*.02 intervals 1 method 3 samples 5,25}}
  }

#end
/***************************************/
//Clouds
#if (Media_Sky)

 //Model a hemispheric hull for the clouds
 intersection{
  sphere{0,145 inverse}
  plane{-y,0}
  sphere{0,190}
  
  scale <1.5,.2,1.5>
  hollow
  double_illuminate
  pigment{rgbt <1,1,1,1>}
  interior{

   media{
    scattering{4,.1*vnormalize(Ambient_Col) extinction .6}
    absorption .05
    method 3 intervals 1
    #if (High_MSamples) samples 35,250 #else samples 5,10 #end
    density{
     agate scale 30*<1,.5,1> color_map{[0 rgb 0][.89 /*.85*/rgb 0][.95 rgb 1][1 rgb 1]}
     warp{turbulence .9 omega .35 octaves 6 lambda 3}
     scale 2
     warp{turbulence .6 omega .4 octaves 7 lambda 4}
     //Positioning to un-obscure the sun
     translate <-15,0,25>
     }
    }

   }
  translate y*-20
  }

#end

/***************************************/
//Heightfield: Constructed with patterns
#declare Just_Uneven = pigment{bozo color_map{[0 rgb 0][1 rgb .05]}poly_wave 2 scale .175 translate 5}
#declare MainHF = pigment{bozo color_map{[0 rgb 0][1 rgb .7]}poly_wave 1 scale .175 translate 5 warp{turbulence .1 omega .4 octaves 10 lambda 3}}

#declare Sphere_Scale = 1;
#declare Sphere_Trans = <1,1,0>;

#declare Sec_HF = pigment{spherical scale Sphere_Scale translate Sphere_Trans poly_wave 3
 pigment_map{
 [0 MainHF scale 1/Sphere_Scale translate -Sphere_Trans]
 [1 Just_Uneven scale 1/Sphere_Scale]
 }}

#declare HF = function{ pigment{Sec_HF}}

#declare HF_Size = <18,2.75,18>;

#declare HF_Object = 
 height_field{
  function HF_Detail,HF_Detail{HF(x,y,z).x}
  smooth
  scale HF_Size
  }

/***************************************/
//Textures for the heightfield
#declare Snow_Pigment = pigment{rgb 1.6*<.8,.8,1>}
#declare Snow_Texture = 
texture{
 pigment{Snow_Pigment}
 finish{brilliance .7 diffuse 1.3}
 }

//Foreground heightfield
object{HF_Object
 texture{Snow_Texture}
 translate -<.5,0,.5>*HF_Size
 double_illuminate
}

//Snow/dust on foreground heightfield
intersection{
 object{HF_Object translate y*.05}
 box{<1,0,1>*.1,<1,1,1>*HF_Size-<1,0,1>*.1}
 hollow double_illuminate
 pigment{rgbt <1,1,1,1>}
 interior{
  media{
   scattering{5,.05*3 eccentricity .9 extinction 0}
   method 3 intervals 1 samples 5,5
   }
  }
 translate -<.5,0,.5>*HF_Size+y*.001
 }
/***************************************/
//Background (ASCII-Art):
//   3
// 852
//0741 Z
//96M  ^
//     |
//  X<-0
//
//M = Foreground Heightfield

//1
object{HF_Object
 texture{Snow_Texture}
 translate -<.5,0,.5>*HF_Size
 scale <-1,.9,-1>
 translate <1,0,1>*HF_Size
 double_illuminate
}

//2
object{HF_Object
 texture{Snow_Texture}
 translate -<.5,0,.5>*HF_Size
 scale <-1,.8,1>
 translate <1,0,2>*HF_Size
 double_illuminate
}

//3
object{HF_Object
 texture{Snow_Texture}
 translate -<.5,0,.5>*HF_Size
 scale <1,.7,-1>
 translate <1,0,3>*HF_Size
 double_illuminate
}

//4
object{HF_Object
 texture{Snow_Texture}
 translate -<.5,0,.5>*HF_Size
 scale <1,.9,-1>
 translate <0,0,1>*HF_Size
 double_illuminate
}

//5
object{HF_Object
 texture{Snow_Texture}
 translate -<.5,0,.5>*HF_Size
 scale <1,.8,1>
 translate <0,0,2>*HF_Size
 double_illuminate
}

//6
object{HF_Object
 texture{Snow_Texture}
 translate -<.5,0,.5>*HF_Size
 scale <-1,.9,1>
 translate <-1,0,0>*HF_Size
 double_illuminate
}

//7
object{HF_Object
 texture{Snow_Texture}
 translate -<.5,0,.5>*HF_Size
 scale <-1,.9,-1>
 translate <-1,0,1>*HF_Size
 double_illuminate
}

//8
object{HF_Object
 texture{Snow_Texture}
 translate -<.5,0,.5>*HF_Size
 scale <-1,.8,1>
 translate <-1,0,2>*HF_Size
 double_illuminate
}

//9
object{HF_Object
 texture{Snow_Texture}
 translate -<.5,0,.5>*HF_Size
 scale <1,.8,1>
 translate <-2,0,0>*HF_Size
 double_illuminate
}

//0
object{HF_Object
 texture{Snow_Texture}
 translate -<.5,0,.5>*HF_Size
 scale <1,.9,-1>
 translate <-2,0,1>*HF_Size
 double_illuminate
}


/***************************************/
//The Iglu

//Use trace() to find position...
#declare Iglu_Position = <.9,0,-5.9>;
#declare Trace_IgluPos = trace(HF_Object, Iglu_Position+HF_Size/2+y*4, <0,-1,0>);

//Model at least the horizontal lines of a built iglu
#declare InCut_Depth = .02;
#declare InCut_Amount = 7;

//CSG-Iglu
union{
 difference{
  sphere{0,1}
  #declare C=0; #while (C<=InCut_Amount)
   torus{sqrt(1-pow((1/InCut_Amount*C),2)),InCut_Depth scale <1,1,1> translate y*1/InCut_Amount*C}
   #declare C=C+1; #end
  }  
 difference{
  cylinder{0,z*(-1.3),.3}
  cylinder{0,z*(-1.4),.24}
  }
 torus{.27,.03 rotate x*90 translate z*-1.3}
 pigment{Snow_Pigment}
 finish{specular .8 roughness .02}
 scale .24 rotate y*-125
 translate Trace_IgluPos-<.5,0,.5>*HF_Size
 }

//The outer, scattering hull (sort of subsurface-scattering, just a very crude approach
//cause we only see the iglu from a distance) 
sphere{0,1.1 hollow
 pigment{rgbt 1}
 interior{
  media{
   scattering{5,20 eccentricity .3 extinction 0} intervals 1 method 3 samples 10,60
   density{spherical scale 1.1 color_map{[0 rgb 0][1  rgb 1]}}
   }}
 scale .24
 translate Trace_IgluPos-<.5,0,.5>*HF_Size
}

//Place emitting (artistic) glow by hand
sphere{0,1 hollow pigment{rgbt <1,0,0,1>}
 interior{media{
  emission Ambient_Col intervals 1 method 3 samples 10,60
  density{spherical poly_wave 4 color_map{[0 rgb 0][1 rgb 2]}}
  }}
 scale .5
 translate z*.24
 rotate <-25,60,0>
 translate Trace_IgluPos-<.5,0,.5>*HF_Size 
 }
