// Persistence of Vision Ray Tracer Scene Description File
// File: chain.inc version 1.1
// Vers: 3
// Desc: Include file for hanging chains
//       Allows either a parabola or catenary
// Date: 24 Feb 97
// Auth: Rob Antonishen
//
// Required Variables: link = object { see documentation }, default=white/blue cylinder
//                     link_length = float, the link scaling factor, default=0.1
//                     link_rotation = float, the degrees to rotate each link, default=90
//                     link_error =float, error allowed in itteration, default=0.01
//                     chain_start = vector, one anchor point, default=<-1,1,0>
//                     chain_end = vector, the other anchor point, default=<1,1,0>
//                     chain_scale = float, controls the chain tension default=1
//                     chain_debug = boolean, turns on/off endpoint markers, default=off
//                     chain_type = int, 1=parabola or 2=catenary, default 1 *for now

// Initialize required variables to defaults if not defined
#ifndef (link)
  #declare link =
    cylinder {
      <0,0,0>,
      <1,0,0>, 0.2
      pigment { checker color rgb <1, 1, 1>, color rgb <0, 0, 1> translate z*0.3 }
    }
#end
#ifndef (link_length)
  #declare link_length=0.1;
#end
#ifndef (link_rotation)
  #declare link_rotation=90;
#end
#ifndef (link_error)
  #declare link_error=0.01;
#end
#ifndef (chain_start)
  #declare chain_start=<-1,1,0>;
#end
#ifndef (chain_end)
  #declare chain_end=<1,1,0>;
#end
#ifndef (chain_scale)
  #declare chain_scale=1;
#end
#ifndef (chain_debug)
  #declare chain_debug=off;
#end
#ifndef (chain_type)
  #declare chain_type=1;
#end

// Start of Routine:
// remap the chain to start at the origin and extend in the +x direction
// to fit a 2d curve
// the links will be rotated and translated later
//
#declare chain_end_trans=chain_end-chain_start; // start at origin
#declare rotate_angle=atan2(vdot(chain_end_trans,z),vdot(chain_end_trans,x))*180/pi;
#declare chain_end_rot=vaxis_rotate(chain_end_trans,y,rotate_angle); // rotate to xy plane

// Extract the x,y coordinates from the endpoints
#declare x1=0;
#declare y1=0;
#declare x2=vdot(chain_end_rot,x);
#declare y2=vdot(chain_end_rot,y);

// Calculate the curve's low point (offset from origin)
#if (chain_type=1)      //parabola
//  #debug "Parabola..."
  #declare x0=(y1-y2-chain_scale*(x1*x1-x2*x2))/(2*chain_scale*(x2-x1));
  #declare y0=y1-chain_scale*(x1-x0)*(x1-x0);
#else                     //catenary
  #debug "Catenary..."
  #declare chain_scale=1/chain_scale;     //invert scale factor so larger #=longer chain
  #declare K1=exp(x2/chain_scale)-1;
  #declare K2=1/exp(x2/chain_scale)-1;

  #declare C1=-2*y2/chain_scale;

  #declare P=(-C1-sqrt(C1*C1-4*K2*K1))/(2*K2);      // solve for roots of eqn.

  #declare x0=chain_scale*log(P);
  #declare y0=chain_scale-chain_scale*(exp(-x0/chain_scale)+exp(x0/chain_scale))/2; //formula for catenary
#end //if

// determine coordinates of low point of chain, for user.
#declare chain_low= vaxis_rotate(<x0,y0,0>,-y,rotate_angle)+chain_start;

// The following spheres were for debugging, showing the locations of chain_start
// chain_end, and chain_low.
//
#if (chain_debug=on)
  sphere { chain_start,link_length/2 pigment { color rgbf <0, 1, 0, 0.8> } finish { ambient 0.5 } }
  sphere { chain_end,link_length/2 pigment { color rgbf <1, 0, 0, 0.8 > } finish { ambient 0.5 } }
  sphere { chain_low,link_length/2 pigment { color rgbf <0, 0, 1, 0.8 > } finish { ambient 0.5 } }
#end // if

// Initialize looping variables, link position, etc.
#declare done=0;
#declare link_count=0;

// First link position, orientation
#declare x_start=x1;
#if (chain_type=1)     //parabola
  #declare y_start=chain_scale*pow(x_start-x0,2)+y0;       // generic formula
#else                    //catenary
  #declare y_start=chain_scale*(exp(-(x_start-x0)/chain_scale)+exp((x_start-x0)/chain_scale))/2-chain_scale+y0; // generic formula
#end //if
#declare link_twist = 0;

// Main loop
//
#while (done=0)

  // initialize variables for binary search of next link position
  #declare x_low=x_start;
  #if (chain_type=1)       //parabola
    #declare y_low=chain_scale*pow(x_low-x0,2)+y0;       // generic formula
  #else                      //catenary
    #declare y_low=chain_scale*(exp(-(x_low-x0)/chain_scale)+exp((x_low-x0)/chain_scale))/2-chain_scale+y0; // generic formula
  #end //if
  #declare x_high=x_start+link_length;

  // loop until error is reached.
  #declare bail=0;
  #while (bail=0)

    #declare x_mid=(x_low+x_high)/2;
    #if (chain_type=1)      //parabola
      #declare y_mid=chain_scale*pow(x_mid-x0,2)+y0;       // curve function
    #else
      #declare y_mid=chain_scale*(exp(-(x_mid-x0)/chain_scale)+exp((x_mid-x0)/chain_scale))/2-chain_scale+y0; // generic formula
    #end  //if
    #declare test_length=sqrt(pow(x_mid-x_start,2)+pow(y_mid-y_start,2));

    #if (test_length<link_length*(1+link_error))
      #if (test_length>link_length*(1-link_error))
        #declare bail=1;
      #end // if
    #end // if
    #if (bail=0)
      #if (test_length<link_length)
        #declare x_low=x_mid;
      #else
        #declare x_high=x_mid;
      #end // if
    #end // if bailout
  #end // while - link endpoint found within specified error

  #declare link_angle=atan2(y_mid-y_start,x_mid-x_start)*180/pi; // determine link pitch

  object { link
    scale link_length               // scale to length
    rotate x*link_twist             // rotate on length of link axis
    rotate z*link_angle             // pitch the link
    translate <x_start,y_start,0>   // translate to its place on the 2d curve
    rotate -y*rotate_angle          // rotate and
    translate chain_start           // translate it to its real world place.
  }

  #declare link_count=link_count+1; // increment statistical counter
  #declare x_start = x_mid;         // set up for the next link
  #declare y_start = y_mid;
  #declare link_twist=mod(link_twist+link_rotation,360);  // increment the link rotation

  #if (x_start>=x2) // see if we've gone far enough
    #declare done=1;
  #end //if
#end  //while

//#debug concat("Number of links:",str(link_count,5,0),"\n") // report number of links
