
// ---------------------------------------------------

/*

  POV-Ray 3.5 Scene File Feb 2003

  Written by Richard Webster 

  Contact irtc_mail@yahoo.co.uk

  Created for the IRTC Jan-Feb 2003 Stills Competition

  www.irtc.org

*/

// ----------------------------------------------------

#include "textures.inc"
#include "functions.inc"
#include "shapes.inc"

global_settings{max_trace_level 8}
global_settings{ambient_light rgb 1}

background{color rgb 1}
#declare film_size = 0.035;
#declare focal_length = 0.1;
#declare aspect_ratio = image_height / image_width;
camera{perspective
      location <0,0,0>
      direction <0,0.08,0>
      right <film_size,0,0>
      up <0,0,film_size*aspect_ratio>
      rotate<0.5,0,7>
      translate<4,-20,2.0>
}

light_source{<1e6,-7e5,1e6> color rgb 1}
light_source{<1e6,1e5,-1e5> color rgb 0.7 shadowless}
light_source{<0,0,-1e6> color rgb 0.2 shadowless}
light_source{<0,0, 1e6> color rgb 0.2 shadowless}
light_source{<1e5,-1e6,1e5> color rgb 0.3 shadowless}

// ------------------------------------------------------

// atmosphere

fog{distance 1000 color rgbt<0.8,0.9,1,0.6>}

// visible sky

#declare T_sky = texture{
  pigment{bozo turbulence 0.7 octaves 8 omega 0.7 lambda 2
          color_map{[0.0 color rgb<0.0,0.0,0.2>]
                    [0.4 colour rgb<0.25,0.3,0.5>]
                    [1.0 colour rgb<1,1,1>]}}
  finish{ambient 1}
  translate <-0.8,0,0.5> scale <3,1,1>*600 
};

plane{y, 0 translate y*4000 texture{T_sky} hollow no_shadow}

// sky seen in reflections

plane{z, 1000 pigment{color rgb<0.8,0.9,1>} finish{ambient 1} hollow no_shadow}

// -----------------------------------------------------------------------------

// river water

plane{z, 0 texture{
  pigment{color rgb<0.02,0.02,0>}
  finish{ambient 1.0 diffuse 0.6 phong 1 phong_size 20 reflection 0.5}
  normal{bumps 0.03 scale 0.3 }}
clipped_by{plane{-y (-1.0)}}}

// foreground water

#declare fn_water = function {
  0.5*sin(sqrt(x*x+y*y)*8) * (1/(pow(x*x+y*y,1)+1))
  + 0.09*f_noise3d(5*x, 5*y, 0) * (1/(x*x+y*y+1))
};

isosurface{
  function{ (z)-fn_water(x,y,z)  }
  contained_by{box{<-3,-10,-0.4>,<6,0.3,0.4>}}
  max_gradient 4
  accuracy 0.0001

  pigment{color rgb<0.03,0.03,0>}
  finish{ambient 0.8 diffuse 0.6 phong 2 phong_size 5 reflection 0.4}
  normal{cylindrical poly_wave 1.0 rotate<90,0,0> scale 11
         normal_map{[0 granite 0][1 granite 100 scale 0.1]}
        }
}

// --------------------------------------------------------------------------------

// wheel

#declare T_wheel = texture{
   pigment{bozo turbulence 0.65 octaves 6 omega 0.7 lambda 2
           colour_map{[0 colour rgb<0.1,0.1,0.0>][1 colour rgb<0.3,0.4,0.3> ]}}
   finish{ambient 0.4 diffuse 0.8 phong 1 phong_size 10}
   normal{bumps 1 scale 0.5}
};

#declare wheel_radius = 2;
#declare wheel_width = 0.8;
#declare wheel_shaft_height = 1.6;
#declare outer_rim_radius = 0.8*wheel_radius;
#declare inner_rim_radius = 0.7*wheel_radius;
#declare n_spokes = 8;
#declare n_slats = n_spokes * 3;
#declare wheel_turn = -10;

#declare outer_rim = difference{
  cylinder{<-0.04,0,0><0.04,0,0>, outer_rim_radius}
  cylinder{<-0.041,0,0><0.041,0,0>, 0.975*outer_rim_radius}
  texture{T_wheel scale <0.02,0.02,0.1>}
};

#declare inner_rim = difference{
  cylinder{<-0.04,0,0><0.04,0,0>,  inner_rim_radius}
  cylinder{<-0.041,0,0><0.041,0,0>, 0.975*inner_rim_radius}
  texture{T_wheel scale <0.02,0.02,0.1>}
};

#declare spoke = box{
  <-0.04,0,-0.05><0.04,outer_rim_radius,0.05>
   texture{T_wheel scale<0.02,0.2,0.02>}
};

#declare slat = box{
  <-0.5*wheel_width,inner_rim_radius,-0.02>
  <0.5*wheel_width,wheel_radius,0.02>
   texture{pigment{bozo turbulence 0.65 octaves 6 omega 0.7 lambda 2
           color_map{[0 colour rgb<0.15,0.2,0.05>][1 colour rgb<0.35,0.4,0.3> ]}
             scale <0.02,0.2,0.02>        }
           finish{ambient 0.4 diffuse 0.5 phong 2 phong_size 10}
          normal{bumps 2 scale <0.01,0.05,0.01>} }
};

#declare wheel = union{

  // outer rim

  object{outer_rim translate<-0.5*wheel_width,0,wheel_shaft_height>}
  object{outer_rim rotate x*180 translate<0.5*wheel_width,0,wheel_shaft_height>}

  // inner rim

  object{inner_rim translate<-0.5*wheel_width,0,wheel_shaft_height>}
  object{inner_rim rotate x*180 translate<0.5*wheel_width,0,wheel_shaft_height>}

  // spokes

  #declare i=0;
  #while(i<n_spokes)
    #declare theta = wheel_turn + 360 * i / n_spokes;
    object{spoke rotate x*theta translate<-0.5*wheel_width,0,wheel_shaft_height>}
    object{spoke rotate x*theta translate< 0.5*wheel_width,0,wheel_shaft_height>}
    #declare i=i+1;
  #end

  // slats

  #declare R4 = seed(2);
  #declare i=0;
  #while(i<n_slats)
    #declare theta = wheel_turn + 360 * i / n_slats;
    object{slat rotate x*theta translate<0,0,wheel_shaft_height>}

    #if ( (theta > 0) & (theta <= 180))
       #declare p0 = <0,inner_rim_radius,0>;
    #else
       #declare p0 = <0,wheel_radius,0>;
    #end

    // drops

    #if( (theta < 220) & (theta > 90) )
        #declare tdrop = theta/270;
        #declare p0 = vrotate(p0,<theta,0,0>);
        #declare nj= 3 + int(40*tdrop*tdrop);
        #declare j=0;
        #while(j<nj)
          #declare tj=j/(nj-1);
          #declare x_drop = 0.05*(2*rand(R4)-1);
          #declare z_drop = 0.05+0.4*rand(R4);

          #declare nk = 5;
          #declare k=0;
          #while(k<nk)
            #declare tk=k/(nk-1);

            #declare p1 = <p0.x,p0.y,p0.z - z_drop*tk*tk>;
            sphere{p1, 0.006 pigment{color rgb<1,1,1>}finish{ambient 0.6 phong 2 phong_size 20}
                  translate<-0.3+0.6*tj + x_drop,0,wheel_shaft_height>}
            #declare k=k+1;
          #end

          #declare j=j+1;
        #end

      #end

    #declare i=i+1;
  #end

  // shaft

  union{
    cylinder{<-1.2*wheel_width,0,0><1.2*wheel_width,0,0>,0.05
             pigment{color rgb 0.2}
             finish{ambient 0.4 diffuse 0.6 phong 1 phong_size 10}
             normal{bumps 1 scale<0.0075,0.0075,0.0075>}
             }

    cylinder{<-0.49*wheel_width,0,0><-0.52*wheel_width,0,0>,0.45
             texture{T_wheel scale 0.02}}

    cylinder{< 0.49*wheel_width,0,0><0.52*wheel_width,0,0>,0.45
             texture{T_wheel scale 0.02}}

    box{<-0.4*wheel_width,-0.3,-0.3><-0.6*wheel_width,0.3,0.3>
        texture{T_wheel scale 0.02}}
    box{<0.4*wheel_width,-0.3,-0.3><0.6*wheel_width,0.3,0.3>
        texture{T_wheel scale 0.02}}

    rotate x*wheel_turn
    translate z*wheel_shaft_height
  }

}; // end wheel union

object{wheel translate 0.1*x}

// -----------------------------------------------------------------------------

// mill

#declare mill_x = -0.75*wheel_width;
#declare mill_y = -2.1;
#declare mill_width = 8;
#declare window_x = -1.5;
#declare window_y = mill_y-0.1;
#declare window_z = 1.5;
#declare window_width = 1;
#declare window_height = 1.4;

// mill interior texture

#declare T_wall = texture{
  pigment{bozo turbulence 0.65 octaves 6 omega 0.7 lambda 2
          colour_map{[0 colour rgb<0.8,0.8,0.7>][1 colour rgb<0.3,0.3,0.1> ]}}
  finish{ambient 0.4 diffuse 0.6}
  scale <1,1,2>
};

// mill exterior texture

#declare P_stonewall_wet1 = pigment{
  bozo turbulence 0.65 octaves 6 omega 0.7 lambda 2
  colour_map{[0.0 colour rgb<0.8,0.8,0.8>]
             [0.49 colour rgb<0.8,0.8,0.8>]
             [0.5 colour rgb<0.5,0.5,0.4>]
             [1.0 colour rgb<0.5,0.5,0.4>]}
  scale 0.45
};

#declare P_stonewall_wet2 = pigment{
  bozo turbulence 0.65 octaves 6 omega 0.7 lambda 2
  colour_map{[0.0 colour rgb<0.3,0.5,0.3>]
             [0.49 colour rgb<0.3,0.5,0.3>]
             [0.5 colour rgb<0.0,0.2,0.0>]
             [1.0 colour rgb<0.0,0.2,0.0>]}
  scale 0.45
};

#declare P_stonewall_dry1 = pigment{
  bozo turbulence 0.65 octaves 6 omega 0.7 lambda 2
  colour_map{[0.0 colour rgb<0.8,0.8,0.5>]
             [0.49 colour rgb<0.8,0.8,0.5>]
             [0.5 colour rgb<0.7,0.7,0.3>]
             [1.0 colour rgb<0.7,0.7,0.3>]}
  scale 0.45
};

#declare P_stonewall_dry2 = pigment{
  bozo turbulence 0.65 octaves 6 omega 0.7 lambda 2
  colour_map{[0.0 colour rgb<1.0,1.0,0.6>]
             [0.49 colour rgb<1.0,1.0,0.6>]
             [0.5 colour rgb<0.9,0.9,0.5>]
             [1.0 colour rgb<0.9,0.9,0.5>]}
  scale 0.45
};

#declare T_stonewall1 = texture{
  pigment{bozo turbulence 0.65 octaves 6 omega 0.7 lambda 2
          scale <1,1,2>/5
          pigment_map{[0 P_stonewall_wet1][0.4 P_stonewall_wet1]
                      [0.45 P_stonewall_wet2][1 P_stonewall_wet2]}
          }
  finish{ambient 0.2 diffuse 0.6 phong 1 phong_size 30}
   normal{granite 1 scale 0.01}

  scale 1/8
};

#declare T_stonewall2 = texture{
  pigment{bozo turbulence 0.65 octaves 6 omega 0.7 lambda 2
          scale <1,1,2>/5
          pigment_map{[0 P_stonewall_dry1][0.6 P_stonewall_dry1]
                      [0.65 P_stonewall_dry2][1 P_stonewall_dry2]}
          }
  finish{ambient 0.2 diffuse 0.6 phong 1 phong_size 30}
  normal{granite 1 scale 0.01}

  scale 1/8
};

#declare T_stonewall = texture{gradient z scale 7 translate -0.5*z
  texture_map{[0 T_stonewall1][1 T_stonewall2]}
};

// -----------------------------------------------------------------

// mill windows

// window slabs texture

#declare T_slab =texture{T_stonewall translate 0.5*z};

#declare window_gap = box{<-0.5*window_width,-0.3,0><0.5*window_width,0.3,window_height>
                      pigment{color rgbt<1,1,1,1>}
};

#declare slab_dy = 0.15;

#declare fn_slab = function {
 f_rounded_box(x,y,z, 0.01,   0.6*window_width,slab_dy,0.04) - 0.02*f_noise3d(10*x, 10*y, 10*z)
};

#declare slab = isosurface{
 function{ fn_slab(x, y, z) }
 contained_by{box{<-0.65*window_width,-slab_dy-0.05,-0.1>,<0.65*window_width,slab_dy+0.05,0.1>}}
  max_gradient 1.9
  accuracy 0.001
}

#declare window_slabs = union{

  // bottom

  object{slab scale<1.05,1,1.5> translate<0,0.05,0>}

  // sides

  object{slab rotate y*90 scale<1,1,1.2> translate<-0.5*window_width,0.1,0.5*window_height>}
  object{slab rotate y*90 scale<1,1,1.2> translate<0.5*window_width,0.1,0.5*window_height>}

  // top

  object{slab scale<1.1,1,2>translate<0,0.05,window_height>}

};

// wood window frame

#declare T_frame = texture{
  pigment{bozo turbulence 0.65 octaves 6 omega 0.7 lambda 2
          colour_map{[0.0 colour rgb<0.6,0.6,0.5>]
                     [1.0 colour rgb<0.2,0.2,0.1>]}
          scale <0.01,0.01,0.05>}
  finish{ambient 0.3 diffuse 0.6 phong 1 phong_size 30}
};

#declare frame_dy = 0.5*window_width - 0.08;
#declare frame0 = box{<-0.03,-0.03,0><0.03,0.03,window_height>}
#declare frame1 = box{<-0.5*window_width,-0.03,-0.03><0.5*window_width,0.03,0.03>}

#declare window_frame = union{

  object{frame0 translate<-frame_dy,0,0>texture{T_frame}}
  object{frame0 scale<0.5,1,1> translate<-0.15*window_width,0,0>texture{T_frame}}
  object{frame0 scale<0.5,1,1> translate<0.15*window_width,0,0>texture{T_frame}}
  object{frame0 translate<frame_dy,0,0>texture{T_frame}}
  object{frame1 translate<0,0,0.08>texture{T_frame rotate y*90}}
  object{frame1 scale<1,1,0.5> translate<0,0,0.35*window_height>texture{T_frame rotate y*90}}
  object{frame1 scale<1,1,0.5> translate<0,0,0.65*window_height>texture{T_frame rotate y*90}}
  object{frame1 translate<0,0,window_height-0.1>texture{T_frame rotate y*90}}

};

// window glass textures

#declare P_clear_glass0 =
  pigment{bozo turbulence 0.65 octaves 6 omega 0.7 lambda 2
          colour_map{[0.0 colour rgbt<0.2,0.4,0.1,0.6>]
                     [0.5 colour rgbt<0.2,0.4,0.1,0.6>]
                     [0.51 colour rgbt<0.2,0.6,0.1,0.6>]
                     [1.0 colour rgbt<0.2,0.6,0.1,0.6>]}
        scale 0.08
};

#declare P_clear_glass1 =
  pigment{bozo turbulence 0.65 octaves 6 omega 0.7 lambda 2
          colour_map{[0.0 colour rgbt<0.2,0.2,0.1,0.8>]
                     [0.5 colour rgbt<0.2,0.2,0.1,0.8>]
                     [0.51 colour rgbt<0.4,0.4,0.1,0.8>]
                     [1.0 colour rgbt<0.4,0.4,0.1,0.8>]}
       scale 0.08
};

#declare T_clear_glass = texture{
  pigment{bozo turbulence 0.65 octaves 6 omega 0.7 lambda 2
          pigment_map{[0 P_clear_glass0][0.5 P_clear_glass0]
                      [0.51 P_clear_glass1][1 P_clear_glass1]}
         }
  finish{ambient 0.4}
  scale <0.05,0.05,0.2> translate -0.5*z
};

#declare T_reflecting_glass = texture{
  pigment{bozo turbulence 0.65 octaves 6 omega 0.7 lambda 2
          colour_map{[0.0 colour rgb<0.2,0.3,0.4>]
                     [1.0 colour rgb<0.6,0.7,0.8>]}
          scale <0.5,0.5,0.1>}
  finish{ambient 0.5}
};

// window glass

#declare window_glass = polygon{5,
  <-0.5*window_width,0,0><-0.5*window_width,0,window_height>
  <0.5*window_width,0,window_height><0.5*window_width,0,0><-0.5*window_width,0,0>
};

// facing window

#declare Tr = transform{translate<window_x,window_y,window_z>};
object{window_slabs transform{Tr} texture{T_slab}}
object{window_frame transform{Tr}}
object{window_glass transform{Tr} texture{T_clear_glass}}

// side windows

#declare Tr = transform{rotate z*90 translate<mill_x+0.16,mill_y+2,4.4>};
object{window_slabs transform{Tr} texture{T_slab}}
object{window_frame transform{Tr}}
object{window_glass transform{Tr} texture{T_reflecting_glass}}

#declare Tr = transform{rotate z*90 translate<mill_x+0.16,mill_y+6,4.4>};
object{window_slabs transform{Tr} texture{T_slab}}
object{window_frame transform{Tr}}
object{window_glass transform{Tr} texture{T_reflecting_glass}}

// -------------------------------------------------------------------

// mill walls 

#declare fn_side_stonelayout = function{
  pigment{Irregular_Bricks_Ptrn( 0.03, 1, 0.4, 0.3) rotate<90,0,90>
          color_map {[0 color rgb 0][1 color rgb 1]}}
};

#declare fn_side_stonewall = function {
    0.08*fn_side_stonelayout(2*x, 2*y, 3*z).gray
    + 0.1*f_noise3d(5*x, 5*y, 5*z)
};

#declare fn_face_stonelayout = function{
  pigment{Irregular_Bricks_Ptrn( 0.03, 1, 0.4, 0.3) rotate<90,0,0>
          color_map {[0 color rgb 0][1 color rgb 1]}}
};

#declare fn_face_stonewall = function {
    0.08*fn_face_stonelayout(2*x, 2*y, 3*z).gray
    + 0.1*f_noise3d(5*x, 5*y, 5*z)
};

// mill building

difference{

  union{

    // interior wall

    box{<mill_x-mill_width,mill_y,-0.5><mill_x-0.05,6,8> texture{T_wall} }

    // facing stone wall

    isosurface{
    function{ (y-mill_y) + fn_face_stonewall(x, y, z)}
    contained_by{box{<mill_x-1.8,mill_y,-0.5>,<mill_x+0.06,mill_y-0.5,6>}}
    max_gradient 90
    accuracy 0.001
    open
    texture{T_stonewall}
    }

  }

  object{window_gap translate<window_x,window_y,window_z>}

}

// side stone wall 

isosurface{
  function{ (x-mill_x) - fn_side_stonewall(x, y, z)}
  contained_by{box{<mill_x-0.1,mill_y-0.05,-0.5>,<mill_x+0.5,6,7>}}
  max_gradient 90
  accuracy 0.001
  open
  texture{T_stonewall}
}

// join the walls

#declare stone_h = 0.333;
#declare n = 10;
#declare R5 = seed(2);
#declare i = 0;
#while(i<n)

  #declare dz = (i-1) * stone_h * 2 - 0.17;

  #declare rx = 6 + rand(R5);
  #declare ry = 6 + rand(R5);
  #declare rz = 6 + rand(R5);

  // large stone

  isosurface{
  function{ f_rounded_box(x,y,z, 0.05,  0.25,0.25,0.35*stone_h)
           - 0.08*f_noise3d( rx*x, ry*y, rz*z ) 
           }
  contained_by{box{<-0.35,-0.35,-0.45*stone_h>,<0.4,0.4,0.45*stone_h>}}
  max_gradient 1.5
  accuracy 0.001
  translate<mill_x-0.12,mill_y+0.14,dz>
  texture{T_stonewall}
  }

  #declare rx = 10*rand(R5);
  #declare ry = 10*rand(R5);
  #declare rz = 10*rand(R5);

  // small stone

  isosurface{
  function{ f_rounded_box(x,y,z, 0.05,  0.15,0.15,0.38*stone_h)
            - 0.06*f_noise3d(5*x+rx, 5*y+ry, 5*z+rz)
          }
  contained_by{box{<-0.24,-0.24,-0.5*stone_h>,<0.24,0.24,0.5*stone_h>}}
  max_gradient 1.5
  accuracy 0.001
  translate<mill_x-0.01,mill_y+0.04,dz+stone_h>
  texture{T_stonewall}
  }

  #declare i=i+1;
#end

// ---------------------------------------------------------------------------

// mill interior

// gear

#declare T_gear = texture{
  pigment{bozo turbulence 0.65 octaves 6 omega 0.7 lambda 2
          colour_map{[0.0 colour rgb<0.6,0.6,0.4>]
                     [1.0 colour rgb<0.4,0.4,0.2>]}
          scale 0.02}
  finish{ambient 0.4 diffuse 0.7 phong 1 phong_size 30}
  normal{bumps 1 scale 0.01}
};

#declare rim_radius = 1.2;
#declare n_spokes = 8;
#declare n_teeth = n_spokes * 5;
#declare gear_turn = 10;

#declare big_gear = union{

  // rim

  difference{
    cylinder{<-0.05,0,0><0.05,0,0>, rim_radius}
    cylinder{<-0.051,0,0><0.051,0,0>, 0.8*rim_radius}
    texture{T_gear scale 0.02}
    }
  cylinder{<-0.05,0,0><0.05,0,0>,0.3 texture{T_gear}}

  // spokes

  #declare spoke = box{ <-0.04,0,-0.05><0.04,rim_radius+0.01,0.05>
     texture{T_gear scale<0.02,0.2,0.02>}
  };
  #declare i=0;
  #while(i<n_spokes)
    #declare theta = gear_turn + 360 * i / n_spokes;
    object{spoke rotate x*theta}
    #declare i=i+1;
  #end

  // teeth

  #declare tooth = box{ <-0.15,0.85*rim_radius,-0.04><0.1,0.95*rim_radius,0.04>
     texture{T_gear scale<0.02,0.2,0.02>}
  };
  #declare i=0;
  #while(i<n_teeth)
    #declare theta = gear_turn + 360 * i / n_teeth;
    object{tooth rotate x*theta}
    #declare i=i+1;
  #end
};

#declare small_gear = union{
  cylinder{<0,0,-0.07><0,0,0.07>, 0.25}
  cylinder{<0,0,0><0,0,1>, 0.03}
  #declare tooth = box{ <0,-0.035,-0.05><0.35,0.035,0.05>}
   #declare n_teeth=n_teeth/3;
  #declare i=0;
  #while(i<n_teeth)
    #declare theta = 20 + 360 * i / n_teeth;
    object{tooth rotate z*theta}
    #declare i=i+1;
  #end
   texture{T_gear scale 0.02}
};

// gears

object{big_gear translate<-1.9,0,wheel_shaft_height>}
object{small_gear translate<-2.3,0,wheel_shaft_height+0.9*rim_radius>}

// vertical shaft

cylinder{<-1.9,0,wheel_shaft_height><-1.5,0,wheel_shaft_height>,0.07 texture{T_gear}}

// -----------------------------------------------------------------------------------

// Ground textures

// far ground texture

#declare P_far_ground1 = pigment{
  bozo turbulence 0.65 octaves 6 omega 0.7 lambda 2
  colour_map{[0.0 colour rgb<0.3,0.6,0.3>]
             [0.3 colour rgb<0.3,0.6,0.3>]
             [0.7 colour rgb<0.5,0.7,0.5>]
             [1.0 colour rgb<0.5,0.7,0.5>]}
  scale 0.2
};

#declare P_far_ground2 = pigment{
  bozo turbulence 0.65 octaves 6 omega 0.7 lambda 2
  colour_map{[0.0 colour rgb<0.6,0.9,0.5>]
             [0.3 colour rgb<0.6,0.9,0.5>]
             [0.7 colour rgb<0.5,0.7,0.3>]
             [1.0 colour rgb<0.5,0.7,0.3>]}
  scale 0.2
};

#declare T_far_ground = texture{
  pigment{bozo turbulence 0.65 octaves 6 omega 0.7 lambda 2 scale 2
          pigment_map{[0 P_far_ground1][0.45 P_far_ground1]
                      [0.55 P_far_ground2][1 P_far_ground2]}
         }
  finish{ambient 0.2 diffuse 0.6 phong 0.5 phong_size 30}
  normal{bumps 1 scale 0.01}
};

// river bank texture

#declare P_riverbank1 = pigment{
  bozo turbulence 0.65 octaves 6 omega 0.7 lambda 2
  colour_map{[0.0 colour rgb<0.3,0.3,0.1>]
             [0.3 colour rgb<0.3,0.3,0.1>]
             [0.7 colour rgb<0.1,0.1,0.0>]
             [1.0 colour rgb<0.1,0.1,0.0>]}
  scale 0.1
};

#declare P_riverbank2 = pigment{
  bozo turbulence 0.65 octaves 6 omega 0.7 lambda 2
  colour_map{[0.0 colour rgb<0.7,0.7,0.5>]
             [0.49 colour rgb<0.7,0.7,0.5>]
             [0.5 colour rgb<0.4,0.4,0.2>]
             [1.0 colour rgb<0.4,0.4,0.2>]}
  scale 0.1
};

#declare T_riverbank = texture{
  pigment{gradient z poly_wave 0.3 scale 0.6
          pigment_map{[0 P_riverbank1][1 P_riverbank2]}}
  finish{ambient 0.3 diffuse 0.7 phong 1 phong_size 20}
  normal{bumps 3 scale 0.05}
};

// -----------------------------------------------------------------

// River path

#macro river_path( tr )
 #local rx = 2.7-pow(tr,4)*100;
 #local ry = 20+tr*(250+250);
 #local rz = 0;
 (<rx,ry,rz>)
#end

#macro river_bank( tr, d )
  #local p0 = river_path(tr);
  #local p1 = river_path(tr+0.0001);
  #local vr = vnormalize(vrotate(p1-p0,<0,0,-90>));
  #local p = p0 + vr * d;
  (p)
#end

#declare river_bowl = sor{5,
  <1,0.0>,<2.5, 0.24>,<3.7,0.45>,<4,0.755>,<4.1,1.0>
  translate -0.25*y rotate 90*x
};


// land with river cut out

difference{

  // distant ground

  plane{z, 0.5 texture{T_far_ground} hollow no_shadow 
        clipped_by{plane{-y (-20.0)}}}

  // dig out the river

  union{
    #declare dx = 0;
    #declare max_offset = 0.5;
    #declare R3 = seed(3);
    #declare ni= 250;
    #declare i=0;
    #while(i<ni)
      #declare ti=i/(ni-1);
      #declare dx = dx + (2*rand(R3)-1) * 0.2;
      #if(dx < -max_offset)
        #declare dx = -max_offset;
      #end
      #if(dx > max_offset)
        #declare dx = max_offset;
      #end
      #declare p = river_path(ti) + dx*x;
      object{river_bowl  translate p  }
      #declare i=i+1;
    #end
    texture{T_riverbank}

  } // end river union


} // end ground difference

// ----------------------------------------------------------------

// right foreground wall

isosurface{
  function{ f_rounded_box(x,y,z, 0.1, 5,0.5,0.5)
           - fn_face_stonewall(x+0.135, y, z-0.15)
           + 0.07*f_noise3d(10*x, 10*y, 10*z) }
  contained_by{box{<-5.5,-0.6,-0.6>,<5.5,0.6,0.6>}}
  max_gradient 90
  accuracy 0.001
  translate <5+0.64,0.5-2.08,0.5-0.55>
  texture{T_stonewall}
}

// wheel support wall

isosurface{
  function{ f_rounded_box(x,y,z, 0.1,  0.3,5.5,0.5)
           - fn_side_stonewall(x, y, z-0.15)
           + 0.08*f_noise3d(10*x, 10*y, 10*z) }
  contained_by{box{<-0.35,-6.0,-0.6>,<0.35,6.0,0.6>}}
  max_gradient 90
  accuracy 0.001
  translate <0.3+0.64,6.0-2.1,0.5-0.5>
  texture{T_stonewall}
}

// ----------------------------------------------------------------------

// wheel support

// wood bearing mount

object{Round_Box_Union(<0.85*wheel_width,-0.2,wheel_shaft_height-0.13>,
                       <wheel_width+0.35,0.2,wheel_shaft_height+0.18>, 0.03)
       texture{T_wheel scale 0.05}
}

// stone pillar

isosurface{
  function{ f_rounded_box(x,y,z, 0.04,  0.175,0.5,0.775)
           - 0.05*f_noise3d(10*x, 10*y, 10*z) }
  contained_by{box{<-0.25,-0.55,-0.9>,<0.25,0.55,0.9>}}
  max_gradient 2
  accuracy 0.001
  translate<wheel_width+0.1,0,0.95>
  texture{T_stonewall}
}

// ------------------------------------------------------------------------

// sluice gate

#declare sluice_y = 8.2;

box{<-wheel_width,1,-0.5><-20,19,0.5> texture{T_stonewall}}
box{<-0.8*wheel_width,sluice_y-0.1,-0.1><0.8*wheel_width,sluice_y+0.1,1.0>
    texture{T_wheel scale<0.04,0.04,0.1>}}
box{<-0.05,-0.15,-0.1><0.05,0.15,1.2>translate<0.8*wheel_width,sluice_y,0>
    texture{T_wheel scale<0.04,0.04,0.2>}}

// ------------------------------------------------------------------------

// hills

#declare fn_hills = function{
 ( 0.9*f_noise3d(4*x, 4*y, 0) - 0.2*f_noise3d(20*x, 20*y, 0))
   * (1-x*x)*(1-y*y)
};
#declare hills = isosurface{
  function{ (z)-fn_hills(x,y,z)  }
  contained_by{box{<-1,-1, 0>,<1,1,1>}}
  max_gradient 5
  accuracy 0.001
  texture{pigment{bozo scale 0.06
                     color_map{[0 color rgb <0.3,0.5,0.2>]
                               [0.4 color rgb <0.3,0.5,0.2>]
                               [0.41 color rgb <0.3,0.3,0.1>]
                               [0.5 color rgb <0.3,0.3,0.1>]
                               [0.51 color rgb <0.3,0.5,0.2>]
                               [1 color rgb<0.3,0.5,0.2>]}}
             finish{ambient 0.3 diffuse 0.8}
             normal{bumps 1 scale 0.005}
             }
scale <1000,250,40> rotate -z*70 translate y*1200
}

object{hills}

// ----------------------------------------------------------------------------------------

// bridge

#declare P_bridge = pigment{
  bozo turbulence 0.65 octaves 6 omega 0.7 lambda 2
  colour_map{[0.0 colour rgb<0.9,0.9,0.6>]
             [0.3 colour rgb<0.9,0.9,0.6>]
             [0.7 colour rgb<0.6,0.6,0.3>]
             [1.0 colour rgb<0.6,0.6,0.3>]}
  scale <0.05,0.005,0.005>
};

#declare T_bridge = texture{
  pigment{P_bridge }
  finish{ambient 0.3 diffuse 0.7 phong 1 phong_size 10}
};

#declare bridge_length = 8.0;
#declare bridge_width = 3.0;
#declare bridge_height = 0.35;
#declare n_posts = 10;
#declare p_bridge = river_path(0.02) + 0.5*z;
#declare post = box{<-0.05,-0.15,-0.6><0.05,0.15,1.3*bridge_height>
                    texture{T_bridge rotate 90*y}};

#declare bridge_side = union{

  // side rail

  box{<-0.5*bridge_length,-0.05,bridge_height-0.05><0.5*bridge_length,0.05,bridge_height+0.05>}

  // posts

  #declare i=0;
  #while(i<n_posts)
    #declare ti=i/(n_posts-1);
    #declare p = <-0.5*bridge_length + ti * bridge_length,0,0>;
    object{post translate p}
    #declare i=i+1;
  #end
}

#declare bridge_deck = box{
 <-0.5*bridge_length,-0.5*bridge_width,-0.14>
 < 0.5*bridge_length, 0.5*bridge_width,0.02>
};

#declare bridge = union{
  object{bridge_side translate -0.5*bridge_width*y texture{T_bridge}}
  object{bridge_side translate  0.5*bridge_width*y texture{T_bridge}}
  object{bridge_deck texture{T_bridge}}
};

object{bridge translate p_bridge}

// ----------------------------------------------------------------

// river bank shrubs

#declare near_shrub = 
isosurface{
  function{f_sphere(x,y,z,0.2)        
           -0.4*f_noise3d(50*x, 50*y, 50*z)
           -0.6*f_noise3d(3*x, 3*y, 3*z) }
  contained_by{sphere{<0,0,0>,1.5}}
  max_gradient 26
  accuracy 0.001
  open
 pigment{color rgb<0.2,0.4,0.1>}
 finish{ambient 0.2 diffuse 0.7 phong 2 phong_size 10}
 normal{bumps 1 scale 0.03}
};

#declare mid_shrub = 
isosurface{
  function{f_sphere(x,y,z,0.1)        
           -0.5*f_noise3d(30*x, 30*y, 30*z)
           -0.4*f_noise3d(3*x, 3*y, 3*z) }
  contained_by{sphere{<0,0,0>,1.1}}
  max_gradient 22
  accuracy 0.001
  open
 pigment{color rgb<0.2,0.4,0.1>}
 finish{ambient 0.2 diffuse 0.7 phong 2 phong_size 20}
 normal{bumps 1 scale 0.03}
};

#declare large_shrub = 
isosurface{
  function{f_sphere(x,y,z,0.9)
     -1*f_noise3d(16*x, 16*y, 16*z)
     -1*f_noise3d(2*x, 2*y, 2*z) 
    }
  contained_by{sphere{<0,0,0>,3.1}}
  max_gradient 22
  accuracy 0.001
  open
 pigment{color rgb<0.2,0.4,0.1>}
 finish{ambient 0.2 diffuse 0.7 phong 2 phong_size 20}
 normal{bumps 1 scale 0.03}
};

// river bank trees

#declare R5 = seed(1);
#declare ni= 40;
#declare i=0;
#while(i<ni)
  #declare ti=i/(ni-1);

  #declare p = river_bank(0.01+0.8*pow(ti,0.9), 10) + 1.5*z;
  object{large_shrub scale<0.7+0.3*rand(R5),0.7+0.3*rand(R5),0.5+0.9*rand(R5)>
         rotate z*rand(R5)*360 translate p}
 
  #declare i=i+1;
#end

#declare R5 = seed(3);
#declare ni= 10;
#declare i=0;
#while(i<ni)
  #declare ti=i/(ni-1);

  #declare p = river_bank(0.87+0.05*ti, 10) + 1.5*z;
  object{large_shrub scale<0.7+0.3*rand(R5),0.7+0.3*rand(R5),0.5+0.9*rand(R5)>
         rotate z*rand(R5)*360 translate p}

  #declare i=i+1;
#end

// -----------------------------------------------------------------------------

// reeds

#macro make_reed( R )
 sphere_sweep {b_spline
    5,
    <0,0, -0.5>, 0.007
    <0,0, 0>, 0.007
    <0.08,0, 0.5>, 0.006
    <0.1+rand(R)*0.1,0, 1.0>, 0.004
    <0.3+rand(R)*0.2,0, 1.5>, 0.002
  scale <1,1,1>
  texture{pigment{gradient z scale 1.05
                  color_map{[0 color rgb<0.5,0.6,0.2>][1 color rgb<0.3,0.5,0.1>]}}
          finish{ambient 0.5 diffuse 0.7 phong 1 phong_size 10}
         }
 }
#end

// wheel support wall reeds

#declare R1 = seed(1);
#declare ni=300;
#declare i=0;
#while(i<ni)
  #declare tx = rand(R1);
  #declare ty = (1-tx)*rand(R1);
  #declare p = <1.5+5.5*tx,-1.2+2*ty,-0.1>;
  #declare reed = make_reed(R1);
  object{reed rotate z*(2*rand(R1)-1)*(-180)
         scale <1,1,1-0.2*rand(R1)>
         translate p}
  #declare i=i+1;
#end

// right bridge wall reeds

#declare R1 = seed(1);
#declare ni=100;
#declare i=0;
#while(i<ni)
  #declare tx = rand(R1);
  #declare ty = tx * rand(R1);
  #declare p = <6+2.8*tx,19-7*ty,-0.1>;
  #declare reed = make_reed(R1);
  object{reed rotate z*(2*rand(R1)-1)*180
         scale <1,1,1-0.2*rand(R1)>
         translate p}
  #declare i=i+1;
#end

// bridge reeds

#declare R1 = seed(1);
#declare ni=n_posts-1;
#declare i=1;
#while(i<ni)

  #declare px = -0.5*bridge_length + i/(n_posts-1) * bridge_length;

  #declare nj=30;
  #declare j=0;
  #while(j<nj)

    #if(rand(R1)<0.5)
       #declare py = -1.7;
    #else
       #declare py = 1.7;
    #end
    #declare p = <px+0.1*(2*rand(R1)-1),py,-0.9>;
    #declare reed = make_reed(R1);
    object{reed rotate z*(2*rand(R1)-1)*180
           scale <1,1,1-0.2*rand(R1)>
           translate p + p_bridge}

    #declare j=j+1;
  #end

  #declare i=i+1;
#end

// right foreground reeds

#declare R2 = seed(1);
#declare ni=250;
#declare i=0;
#while(i<ni)
  #declare t1 = rand(R2);
  #declare p = <1.5+4.5*pow(t1,0.5), -2.3-3*(0.2+0.8*t1)*rand(R2),-0.1>;
  #declare reed = make_reed(R2);
  object{reed rotate z*(2*rand(R2)-1)*180
           scale <1,1,0.6+1.1*t1*rand(R1)>
           translate p}
  #declare i=i+1;
#end

// left foreground reeds

#declare R2 = seed(3);
#declare ni=40;
#declare i=0;
#while(i<ni)
  #declare t1 = rand(R2);
  #declare p = <-0.5-0.5*t1, -2.3-0.4*rand(R2),-0.1>;
  #declare reed = make_reed(R2);
  object{reed rotate z*(2*rand(R2)-1)*180
           scale <1,1,0.6+1.1*t1*rand(R1)>
           translate p}
  #declare i=i+1;
#end

// ---------------------------------------------------------------------------

// tall grass

#declare grass_patch = isosurface{
  function{ z+2-4.0*f_noise3d(30*x, 30*y, 0.01*z) }
  contained_by{sphere{<0,0,0>, 3}}
  max_gradient 10
  accuracy 0.001
  open
};

// grass along path

#declare R5 = seed(1);
#declare ni= 100;
#declare i=0;
#while(i<ni)
  #declare ti=i/(ni-1);
  #declare tr=rand(R5);
  #declare p = river_bank(0+0.7*ti, 7.8+2*tr) + <0,0,0.5>;
  object{grass_patch scale <1,1,0.05+0.9*tr*tr>
         rotate z*360*rand(R5)
         translate p + x*0.5*rand(R5)
         pigment{color rgb<0.6,0.9,0.5>}
         finish{ambient 0.3 diffuse 0.7 phong 1 phong_size 30}
        }
  #declare i=i+1;
#end

// grass in field

#declare R5 = seed(2);
#declare ni= 50;
#declare i=0;
#while(i<ni)
  #declare ti=i/(ni-1);
  #declare tr=rand(R5);
  #declare p = river_bank(0.02+0.9*ti, -(7+10*tr)) + <0,0,0.5>;
  object{grass_patch 
         scale <1,1,0.1+0.9*tr>
         rotate z*360*rand(R5)
         translate p + x*0.5*rand(R5)
         pigment{color rgb<0.6,0.9,0.5>}
         finish{ambient 0.3 diffuse 0.7 phong 1 phong_size 30}
         }
  #declare i=i+1;
#end

// -------------------------------------------------------------------------

// stones right of bridge

box{<5.5,19.5,-0.5><9,20.7,0.05>texture{T_stonewall}}
box{<5.8,19.6, 0.05><9,20.7,0.2>texture{T_stonewall}}
box{<6.1,19.7, 0.2><9,20.7,0.35>texture{T_stonewall}}
box{<6.4,19.8, 0.35><9,20.7,0.5>texture{T_stonewall}}

// shrubs right of bridge

object{mid_shrub scale 0.5 rotate z*90 translate <6.5,19,0.4>}
object{mid_shrub scale 0.5 rotate z*120 translate <7,19,0.45>}
object{mid_shrub scale 0.5 rotate z*240 translate <7.5,19,0.5>}

// left foreground wall and shrubs

box{<-3,mill_y,-0.5><-1.1,mill_y-3,0.5>texture{T_stonewall}}
object{near_shrub rotate z*20 translate <-2,mill_y-0.5,0.6>}
object{near_shrub rotate z*60 translate <-1.7,mill_y-3,0.6>}

// millstone

#declare F_millstone = function{ sqrt(x*x+y*y)-0.5*(1-pow(abs(5.9*z),30))};
#declare millstone =
 isosurface{
 function{ F_millstone(x,y,z)
          + 0.05*f_noise3d(20*x, 20*y,20*z)}
 contained_by{box{<-0.7,-0.7,-0.17>,<0.7,0.7,0.17>}}
 max_gradient 95
 accuracy 0.001
 translate 0.15*z
};

object{millstone rotate y*60 translate<-0.9,-4,0.1> 
       texture{T_stonewall translate -0.4*z}
}

// -------------------------------------------------------------------------------

// painter

#macro make_painter( rgb_colour )

  #local cloth = isosurface{
     function{ f_torus(x,y-0.05,z, 1, 0.1)
              + 0.02*f_noise3d(5*x, 5*y, 5*z) }
     contained_by{box{<-1.11,-0.2,-1.11>,<1.11,0.2,1.11>}}
     max_gradient 1.5
     accuracy 0.001*1

     rotate x*90 scale<1,1,10>
     pigment{color rgb rgb_colour}
     finish{ambient 0.5 diffuse 0.7 phong 1 phong_size 10}
     normal{bumps 1 scale<0.1,0.1,2>}
  }

  #local T_easel = texture{pigment{color rgb<0.5,0.5,0.2>}
                           finish{ambient 0.5 diffuse 0.7 phong 1 phong_size 20}
                           normal{bumps 0.1 scale 0.02}};
  #local easel_height = 0.9;
  #local easel_leg =  union{
      box{<-0.015,-0.015,-easel_height><0.015,0.015,-0.5*easel_height>}
      box{<-0.02,-0.02,-0.5*easel_height><0.02,0.02,0>}
      texture{T_easel}
  };
  #local easel = union{
   
    object{easel_leg rotate -10*x translate easel_height*z}
    object{easel_leg    rotate 20*y translate easel_height*z}
    object{easel_leg    rotate -20*y translate easel_height*z}
    object{easel_leg   rotate 180*x translate 0.7*easel_height*z}

    // painting

    union{

      // canvas

      difference{
        box{<-0.8, 0,0><0.8,0.075,1.1> scale 0.65}
        box{<-0.725,-0.01,0.05><0.725,0.074,1.025> scale 0.65}

         texture{pigment{color rgb<1,1,0.9>}
                 finish{ambient 0.5 diffuse 0.5 phong 1 phong_size 20}
                 normal{granite 0.5 scale 0.02}}
         }
      box{<-0.35,-0.02,-0.02><0.35,0.02,0.04> texture{T_easel}}
      translate z*0.7
    }
  } // end easel
  
  // painter

  union{

  // easel

  object{easel rotate 5*x translate <0,-0.6,-0.1>}

  // smock

  object{cloth scale<0.23,0.19,0.5> translate<0,0,0.5>}

  // legs

  object{cloth scale <0.08,0.08,0.5> translate <-0.1,0,0.2>}
  object{cloth scale <0.08,0.08,0.5> translate  <0.1,0,0.2>}

  // arm

  union{

     // sleeve

     object{cloth scale <0.1,0.07,0.5> }

     // brush

     cone{<-0.15,0,-0.25>, 0.02 <0.2,0,-0.25>, 0.015
          pigment{rgb<1,1,0.7>}finish{ambient 0.4 diffuse 0.7 phong 1 phong_size 10}}

     // hand

     sphere{<0,0,0>, 0.035 scale<1.5,1,1> translate <-0.04,0,-0.25>
            pigment{rgb<1,0.8,0.6>}finish{ambient 0.3 diffuse 0.6} }

     rotate  y*85 translate <-0.4,0,0.9>
    }

  } // end painter

#end

#declare painter = make_painter( <0.9,0.9,0.8> );
object{painter rotate 20*z translate p_bridge+<1.65,0,0>}

