// Persistence of Vision Ray Tracer Scene Description File
// File: columns.inc
// Vers: 3.5
// Desc: CreateBlobbedObj macro
// Date: 2003/03/30
// Auth: Maurizio Tomasi
//

#macro CreateColumn (Radius, N, BaseHeight, BrickHeight,
                     BrickTextures, InsertStatue, RndSeed)

    #local NumOfTextures = dimension_size (BrickTextures, 1);
    #local LastUsedTexture = -1;
      
    // Top of the column
    #local TopHeight = 2.0;

    union
    {
        #local PrismShape =
        prism
        {
            linear_sweep
            linear_spline
            0,
            1.0,
            N+1
            
            #local Count = 0;
            #while (Count <= N)
                #if (Count < N)
                    <cos (2*pi*Count/N), sin(2*pi*Count/N)>,
                #else
                    <1, 0>
                #end
                #declare Count = Count + 1;
            #end
        }
      
        #local CurRowPos = 0;
        
        #while (CurRowPos < BaseHeight)
            #local CurRowHeight = (1.0 + RRand (-0.3, 0.3, RndSeed)) * BrickHeight;
            
            object
            {
                PrismShape
                
                scale Radius/2 * 0.925 * <1, 0, 1> + y*(-1)
                scale <1, CurRowHeight * 0.98, 1>
                translate (BaseHeight - CurRowPos)*y
                
                texture
                {
                    // Search for a texture different from the previous one we have used
                    #local CurTexture = LastUsedTexture;
                    #while (CurTexture = LastUsedTexture)
                        #declare CurTexture = floor(RRand(0, NumOfTextures, RndSeed));
                    #end
                    #declare LastUsedTexture = CurTexture;
                    
                    BrickTextures[CurTexture]
                    
                    rotate 360 * VRand (RndSeed)
                    scale RRand (0.8, 1.2, RndSeed)
                }
            }
            
            #declare CurRowPos = CurRowPos + CurRowHeight;
        #end

        // Matte
        
        object
        {
            PrismShape
            
            scale Radius/2 * 0.85 * <1, 0, 1> + y
            scale <1, BaseHeight, 1>
            
            texture { MatteTexture }
        }

        #ifdef (USE_ARCHES)
                
            #local CurSide = 0;
            #while (CurSide < N)
                union
                {
                    object { CreateArch (BaseHeight, RndSeed) }
                    object { CreateBottomArch (RndSeed) }
                    
                    scale <1.05, 1, 1> * Radius/2 * sin (2*pi/N)
    
                    rotate (90 - 360*(CurSide + 0.5)/N) * y
                    
                    translate 0.8 * Radius * cos (2*pi/N) *
                        <cos (2*pi*(CurSide + 0.5)/N), 0, sin(2*pi*(CurSide + 0.5)/N)>
                        
                    translate y*BaseHeight
                    
                    CreateArchTexture (RndSeed)
                }
                
                #declare CurSide = CurSide + 1;
            #end
            
        #end // USE_ARCHES
                
        #local CurPos = 0.00;
        #macro CreateColumnBox (Height, Side)
            box
            {
                <-Side, CurPos, -Side>,
                #declare CurPos = CurPos + Height;
                < Side, CurPos,  Side>
            }
        #end

        #local Column =
        union
        {
            CreateColumnBox (0.050, 0.8)
            CreateColumnBox (0.005, 0.9)
            CreateColumnBox (0.005, 1.0)
            CreateColumnBox (0.005, 0.9)
            CreateColumnBox (0.060, 0.8)
            CreateColumnBox (0.005, 0.9)
            CreateColumnBox (0.005, 1.0)
            CreateColumnBox (0.005, 0.9)

            CreateColumnBox (0.300, 0.7)
            CreateColumnBox (0.005, 0.8)
            CreateColumnBox (0.005, 0.9)
            CreateColumnBox (0.005, 0.8)
            CreateColumnBox (0.060, 0.7)
            CreateColumnBox (0.005, 0.8)
            CreateColumnBox (0.005, 0.9)
            CreateColumnBox (0.005, 0.8)

            CreateColumnBox (0.200, 0.6)
            CreateColumnBox (0.005, 0.7)
            CreateColumnBox (0.005, 0.8)
            CreateColumnBox (0.005, 0.7)
            CreateColumnBox (0.080, 0.6)
            CreateColumnBox (0.005, 0.7)
            CreateColumnBox (0.005, 0.8)

            scale Radius/16*(x + z) + TopHeight*y
        }
        
        #local Ratio = 1/sqrt(1 + pow(0.5, 2));
        #local LittleChapelRoofElement =
        box
        {
            <-0.05, 0, -0.5>, <0, 1, 0.5>
            rotate 180/pi * acos(Ratio) * z
            translate 0.5*x
        }
        
        #local LittleChapelHeight = TopHeight / 8;
        
        #local LittleChapelRoof =
        union
        {
            object { LittleChapelRoofElement }
            object { LittleChapelRoofElement scale <-1, 1, 1> }
            
            union
            {
                object { LittleChapelRoofElement }
                object { LittleChapelRoofElement scale <-1, 1, 1> }
                
                rotate 90*y
            }
            
            box
            {
                <-0.5, 0, -0.5>, <0.5, 1 - Ratio, 0.5>
                
                translate y * Ratio
            }
            
            scale Radius / 16 * (x + z) + LittleChapelHeight * y
        }
        
        #local ChapelBasePos = CurPos * TopHeight;
        #local CurPos = ChapelBasePos + LittleChapelHeight * 1.5;

        #local LittleChapel =
        union
        {
            object
            {
                LittleChapelRoof
                scale <1.3, 0.5, 1.3>
                translate Radius/6 * y
            }

            box
            {
                <-Radius/24, 0.00, -Radius/24>,
                < Radius/24, 0.03,  Radius/24>
                
                translate -0.03 * y
            }
            
            cylinder
            {
                <-Radius/30, 0.0, Radius/30>,
                <-Radius/30, Radius/6, Radius/30>, Radius/128
            }
                            
            cylinder
            {
                <Radius/30, 0.0, Radius/30>,
                <Radius/30, Radius/6, Radius/30>, Radius/128
            }

            cylinder
            {
                <-Radius/30, 0.0, -Radius/30>,
                <-Radius/30, Radius/6, -Radius/30>, Radius/128
            }
                            
            cylinder
            {
                <Radius/30, 0.0, -Radius/30>,
                <Radius/30, Radius/6, -Radius/30>, Radius/128
            }
        } // LittleChapel

        #local Count = 0;
        #while (Count < 2 * N)
            
            union
            {
                object { Column }
                
                object
                {
                    #local Pt1 = <-Radius/32, 0, -Radius/32>;
                    #local Pt2 = <Radius/32, LittleChapelHeight * 0.5, Radius/32>;
                    
                    CreateBlobbedObj_Box (80, Pt1, Pt2, Radius/48, RndSeed)
                    
                    translate (ChapelBasePos + LittleChapelHeight) * y
                }
                
                object
                {
                    LittleChapelRoof 
                    translate ChapelBasePos * y
                }
    
                object
                {
                    LittleChapel
                
                    translate CurPos * y
                }
                
                #if (InsertStatue)
                    object
                    {
                        #if (odd(Count))
                            Statue1
                        #else
                            Statue2
                        #end
                        
                        scale Radius/6 * 0.8
                        //rotate 180*y
                        translate CurPos * y
                    }
                #end

                difference
                {
                    box
                    {
                        <-0.35, LittleChapelHeight, -0.35>, <0.35, 1, 0.35>
                    }

                    box
                    {
                        <-0.25, LittleChapelHeight + 0.05, 0.4>, <0.25, 1, 0.3>
                    }

                    scale Radius / 16 * (x + z) + 1*y
                    scale <1.3, 0.5, 1.3>
                    translate (CurPos + Radius/6) * y
                }
                
                rotate (90 - 360 * Count / (2 * N)) * y
                translate 0.7 * Radius/2
                    * <cos(2*pi * Count / (2 * N)), 0, sin(2*pi * Count / (2 * N))>
                    + BaseHeight * y
                
                CreateArchTexture (RndSeed)
            }

            union
            {
                #local CylCount = 0;
                #while (CylCount < 16)
                    cylinder
                    {
                        <0, 0, 0>, <0, CurPos - Radius/32, 0>, Radius/32/8
                        
                        translate Radius/32 
                            * <cos(2*pi*CylCount/16), 0, sin(2*pi*CylCount/16)>
                    }
                    #declare CylCount = CylCount + 1;
                #end
                
                torus
                {
                    Radius/32, Radius/32/2
                    translate y * (CurPos - Radius/32/2)
                }
                
                #local CylColBase =
                union
                {
                    torus
                    {
                        Radius/32, Radius/32/2
                    }

                    torus
                    {
                        Radius/32 * 0.9, Radius/32/2 * 2/3
                        translate -y * Radius/32 / 2
                    }

                    torus
                    {
                        Radius/32 * 0.9, Radius/32/2 * 2/3
                        translate +y * Radius/32 / 2
                    }
                }
                
                object
                {
                    CylColBase
                    translate y * (CurPos/2 * 0.86)
                }

                object
                {
                    CylColBase
                    translate y * (CurPos/2 * 1.02)
                }
                
                object
                {
                    CylColBase
                    translate y * (CurPos/2 * 1.44)
                }

                object
                {
                    CylColBase
                    translate y * (CurPos/2 * 1.62)
                }
                

                cylinder
                {
                    <0, CurPos + Radius/32, 0>, <0, CurPos + 1.0, 0>, Radius/32 * 2/3
                }

                object
                {
                    #local Pt1 = <-Radius/32, 0, -Radius/32>;
                    #local Pt2 = <Radius/32, LittleChapelHeight, Radius/32>;

                    CreateBlobbedObj_Box (160, Pt1, Pt2, Radius/48, RndSeed)
                    translate y * (CurPos)
                }
                
                object
                {
                    LittleChapel
                    
                    translate y * (CurPos + LittleChapelHeight)
                }
                
                translate y*BaseHeight
                rotate (90 - 360 * (Count + 0.5) / (2 * N)) * y
                translate 0.7*Radius/2*<cos(2*pi*(Count + 0.5)/(2 * N)), 0, sin(2*pi*(Count + 0.5)/(2 * N))>

                CreateArchTexture (RndSeed)
            }

            #declare Count = Count + 1;
        #end


        // Black shadow
        
        object
        {
            PrismShape
            scale 0.6*Radius/2 * (x + z) + (CurPos + 1.0) * y
            translate BaseHeight * y
            
            pigment { color rgb 0 }
        }
    }
#end // CreateColumn macro
