// Persistence of Vision Ray Tracer Scene Description File
// File: mtdomine.pov
// Vers: 3.5
// Desc: Non nobis sed Tibi Domine
// Date: 2003/03/29
// Auth: Maurizio Tomasi
//

#version 3.5;

#include "stdinc.inc"
#include "skies.inc"

#declare USE_WALLS = 1;
#declare USE_DOME = 1;
#declare USE_LADDER = 1;
#declare USE_COLUMNS = 1;
#declare USE_ARCHES = 1;
#declare USE_BLOBS = 1;
#declare USE_BRICKS = 1;
#declare USE_TRANSVERSAL_WALL = 1;
#declare USE_WINDOW = 1;
#declare USE_SHADOWS = 1;

//#declare USE_RADIOSITY = 1;
#declare CAMERA_NUM = 1;

#include "walls.inc"
#include "blob.inc"
#include "julia.inc"
#include "window.inc"

global_settings {
  assumed_gamma 2.1
  
  #ifdef (USE_RADIOSITY)
    radiosity
    {
        brightness 0.5
    }
    
    ambient_light 0
  #end
}


#include "statue1.inc"

#local Statue1 =
object
{
    O_SPatch
    scale 1/5.8
}

#include "statue2.inc"

#local Statue2 =
object
{
    O_SPatch
    scale 1/5.8
}


// ----------------------------------------

#switch (CAMERA_NUM)
    #case (1)
    #declare CameraLocation = <14.0, 5.4, -25.0>;
    #declare CameraDirection = 3.5*z;
    #declare CameraRight = x;
    #declare CameraLookAt = <6.5, 9.7, -2.0>;
    #break  

    #case (2)
    #declare CameraLocation = <10.0, 80.0, -12.0>;
    #declare CameraDirection = 1.5*y;
    #declare CameraRight = x;
    #declare CameraLookAt = <10.0, 0.0, -12.0>;
    #break  

    #case (3)
    #declare CameraLocation = <14.5, 5.4, -25.0>;
    #declare CameraDirection = 3.5*z;
    #declare CameraRight = x;
    #declare CameraLookAt = <6.5, 9.7, -2.0>;
    #break  
#end


camera {
    location  CameraLocation
    direction CameraDirection
    right     CameraRight*image_width/image_height
    look_at   CameraLookAt
}


// The sky sphere
/*
sphere {
    0, 1000

    pigment { color rgb <0.7, 0.8, 1.0> }
    
    finish
    {
        diffuse 0.4
        ambient 0.0
    }
    
    hollow
}
*/

sphere {
    0, 1000
    
    hollow
    
    pigment { color rgb <0.7, 0.8, 1.0> }
    texture { T_Cloud3 scale <10, 4, 10> }
}

light_source {
    <-150, 300, -300>
    color rgb 1
}

light_source {
    <100, 200, -50>
    color rgb 0.3
    shadowless
}

light_source {
    <0, 50, -150>
    color rgb 0.1
    shadowless
}


// --------------------------------

// Arches

#declare ArchThickness = 0.2;
#declare TopArchHeight = 1143/784;
#declare MiddleArchHeight = 541/484;
#declare BottomArchPos = -5.18; //-5.75;

#declare HalfTopArch =
height_field
{
    png "arch1.png"
    water_level 0.01
    smooth
    
    translate <-0.5, 0, 0>
    scale <1, 1, TopArchHeight>
}

#declare HalfMiddleArch =
height_field
{
    png "arch2.png"
    water_level 0.01
    smooth
    
    translate <-0.5, 0, 0>
    scale <1.05, 1.05, MiddleArchHeight>
}

#declare TopArch =
union
{
    object
    {
        HalfTopArch
        scale <1, 0.5, 1>
    }

    object
    {
        HalfTopArch
        rotate 180*z
        scale <-1, 0.5, 1>
    }
    
    rotate -90*x            
    scale <1, 1, ArchThickness>
}


#declare ArchTransform =
transform
{
    scale <1.1, 1.1, 1>
    translate 0.4*y
}

/* Create a decoration which goes from <-0.5, 0, -ArchThickness> to
   <0.5, WallHeight, ArchThickness> */

#macro CreateArch (WallHeight, RndSeed)
union
{
    object { TopArch }
        
    // Column near the arch
    
    union
    {
        difference
        {
            box
            {
                <-0.05, 0, -0.05>, <0.05, 1.50, 0.05>
            }
            
            union
            {
                box
                {
                    <-0.07, 0.10, -0.035>,
                    <-0.035, 1.45,  0.035>
                }

                box
                {
                    <0.07, 0.10, -0.035>,
                    <0.035, 1.45,  0.035>
                }
                
                box
                {
                    <-0.035, 0.10, -0.07>,
                    < 0.035, 1.45, -0.035>
                }

                box
                {
                    <-0.035, 0.10, 0.07>,
                    < 0.035, 1.45,  0.035>
                }
            }
        }
                
        box
        {
            <-0.07, 1.50, -0.07>, <0.07, 1.55, 0.07>
        }

        #local Blob = object { CreateBlobbedObj (40, RndSeed) scale 0.06 }
        
        object { Blob translate <-0.07, 1.60,  0.00> }
        object { Blob translate < 0.07, 1.60,  0.00> }
        object { Blob translate < 0.00, 1.60, -0.07> }
        object { Blob translate < 0.00, 1.60,  0.07> }
        
        box
        {
            <-0.03, 1.65, -0.03>, <0.03, 1.69, 0.03>
        }
        
        box
        {
            <-0.1, 1.69, -0.1>, <0.1, 1.70, 0.1>
        }

        object
        {
            CreateBlobbedObj (40, RndSeed)
            scale 0.09
            translate 1.77*y
        }
                
        rotate 45*y
        translate -0.45*x
    }    

    // Column on the top arch
    
    union
    {
        #local Base = TopArchHeight - 0.2;
        cone
        {
            <0.0, Base, 0.0>, 0.045,
            #declare Base = Base + 0.35;
            <0.0, Base, 0.0>, 0.025
        }
        
        cylinder
        {
            <0.0, Base, 0.0>, 
            #declare Base = Base + 0.01;
            <0.0, Base, 0.0>, 0.07
        }
        
        object
        {
            CreateBlobbedObj (40, RndSeed)
            scale 0.1
            
            #declare Base = Base + 0.07;
            translate y*Base
        }
        
        box
        {
            <-0.01, Base,      -0.01>,
            < 0.01, Base + 0.3, 0.01>
        }
        
        box
        {
            <-0.04, Base + 0.25, -0.01>,
            < 0.04, Base + 0.27,  0.01>
        }
    }

    /* Middle arch frame
    
                1
       +-----------------+
       |       ___       |
       |      /   \      |
     2 |      \   /      | 3
       |  4    | |       |
       +-----------------+
       |       | |       |
       |      /   \      |
       |    _/     \_    |
       |  _/         \_  |
       |_/             \_|
       |                 |
       |                 |
       |                 |
       |                 |
       |                 |
     
     */
    
    // 1a
    box
    {
        <-0.5, -0.01, -ArchThickness/2>, <0.5, 0.01, ArchThickness/2>
    }

    // 1b
    box
    {
        <-0.5, -0.03, -ArchThickness/4>, <0.5, 0.03, ArchThickness/4>
    }
    
    // 2a
    box
    {
        <-0.50, -WallHeight, -ArchThickness/2>, <-0.48, 0, ArchThickness/2>
    }

    // 2b
    box
    {
        <-0.50, -WallHeight, -ArchThickness/4>, <-0.42, 0, ArchThickness/4>
    }

    // 3a
    box
    {
        <0.48, -WallHeight, -ArchThickness/2>, <0.50, 0, ArchThickness/2>
    }

    // 3b
    box
    {
        <0.42, -WallHeight, -ArchThickness/4>, <0.50, 0, ArchThickness/4>
    }
    
    // 4a
    box
    {
        <-0.5, -0.41, -ArchThickness/6>, <0.5, -0.39, ArchThickness/6>
    }

    // 4b
    box
    {
        <-0.5, -0.42, -ArchThickness/6>, <0.5, -0.38, 0>
    }


    // Middle arch

    union
    {
        union
        {
            object
            {
                HalfMiddleArch
                scale <1, 0.5, 1>
            }
        
            object
            {
                HalfMiddleArch
                rotate 180*z
                scale <-1, 0.5, 1>
            }
    
            rotate -90*x            
            scale <0.96, 1, ArchThickness/2>
        }
    

        #local MinAngle = -40;
        #local MaxAngle = 220;
        #local AngleStep = 20;
        #local Radius = 0.01;
            
        #local FractalSupport =
        union
        {
            union
            {
                sphere
                {
                    <0, 0, 0>, 0.1
                    scale <1, 1, 0.05>
                }
                
                torus
                {
                    0.1, Radius
                    rotate 90*x
                }
    
                translate -0.14 * 0.05 / 2 * z
            }
            
            sphere_sweep
            {
                cubic_spline
                2 + (MaxAngle - MinAngle) / AngleStep + 1,
                
                (0.13 + 0.15) * <cosd(MinAngle), sind(MinAngle), 0>, Radius
                (0.13 + 0.15 * 0.25) * <cosd(MinAngle), sind(MinAngle), 0>, Radius
                
                #local Count = MinAngle + AngleStep;
                #while (Count < MaxAngle)
                
                    0.13 * <cosd (Count), sind(Count), 0>, Radius
                    #declare Count = Count + AngleStep;
                
                #end

                (0.13 + 0.15 * 0.25) * <cosd(MaxAngle), sind(MaxAngle), 0>, Radius
                (0.13 + 0.15) * <cosd(MaxAngle), sind(MaxAngle), 0>, Radius
                
                translate Radius/2 * z
            }
        }
        
        #local FractalDistance = 0.15;

        union
        {
            object
            {
                CreateFractal (RndSeed)
                scale <0.3, 0.3, 0.15>
            }
            
            object { FractalSupport }
                
            translate <0, 0.6, 0>
        }

        union
        {
            object
            {
                CreateFractal (RndSeed)
                scale <0.3, 0.3, 0.15>
            }
            
            object { FractalSupport rotate (90 + 30) * z }
                
            translate <-FractalDistance, 0.6 - FractalDistance * sqrt(3), 0>
        }

        union
        {
            object
            {
                CreateFractal (RndSeed)
                scale <0.3, 0.3, 0.15>
            }
            
            object { FractalSupport rotate - (90 + 30) * z }
                
            translate <FractalDistance, 0.6 - FractalDistance * sqrt(3), 0>
        }
        
        translate -1.3*y
    }
    
    // Column on the middle arch
    
    union
    {
        #local Base = -0.4;
        cone
        {
            <0.0, Base, 0.0>, 0.045,
            #declare Base = Base + 0.15;
            <0.0, Base, 0.0>, 0.025
        }
        
        cylinder
        {
            <0.0, Base, 0.0>, 
            #declare Base = Base + 0.01;
            <0.0, Base, 0.0>, 0.07
        }
        
        object
        {
            CreateBlobbedObj (40, RndSeed)
            scale 0.1
            
            #declare Base = Base + 0.07;
            translate y*Base
        }
    }

    transform { ArchTransform }
}
#end // CreateArch macro


#macro CreateBottomArch (RndSeed)
    union
    {
        #local BottomArch =
        union
        {
            object
            {
                HalfMiddleArch
                scale <1, 0.5, 1>
            }
        
            object
            {
                HalfMiddleArch
                rotate 180*z
                scale <-1, 0.5, 1>
            }
    
            rotate -90*x            
            scale <0.97, 0.97, ArchThickness/2>
            translate -0.5*y
        }
        
        object { BottomArch }
        object { BottomArch scale <1, -1, 1> translate 1*y }

        box
        {
            <-0.5, -0.05, -ArchThickness/2>,
            < 0.5,  0,     ArchThickness/2>
        }

        box
        {
            <-0.45, -0.03, -ArchThickness/2 * 1.1>,
            < 0.45, -0.02,     ArchThickness/2 * 1.1>
        }
        
        box
        {
            <-0.5,  1.03,   -ArchThickness/2>,
            < 0.5,  1.08, ArchThickness/2>
        }

        box
        {
            <-0.45, 1.04,   -ArchThickness/2 * 1.1>,
            < 0.45, 1.07, ArchThickness/2 * 1.1>
        }

        translate BottomArchPos*y
        transform { ArchTransform }
    }
#end


// ----------------------------------------

// Walls

#declare WallThickness = 0.1;

#declare LowWallHeight = 6;
#declare HighWallHeight = 10;

#declare LowColumnHeight = LowWallHeight;
#declare HighColumnHeight = HighWallHeight;

#declare MeanBrickWidth = 0.225;
#declare MeanRowHeight = 0.175;

#local BrickNormal1 =
normal
{
    granite 0.15
    turbulence 0.8
    slope_map {
        [0.00 <0, 1>]
        [0.45 <0, 1>]
        [0.50 <1, 1>]
        [0.55 <0, 1>]
    }
    
    scale 0.2
}

#local BrickNormal2 =
normal
{
    wrinkles 0.5
    turbulence 0.5
    
    scale 0.03
}

#local BrickColor1 = <1.0, 0.84, 0.74> * 1.3;
#local BrickColor2 = <1.0, 1.0, 0.79> * 1.3;

#declare BrickTexture = array[4]

#declare BrickTexture[0] =
texture
{
    pigment
    {
        marble        
        turbulence 0.4
        color_map
        {
            [0.05 color rgb <0.94, 0.86, 0.87> * 1.1] // 1.4?
            [0.08 color rgb <1.0, 0.91, 0.88> * 1.1] // 1.4?
            [0.5 color BrickColor1]
            [0.9 color BrickColor2]
        }
        scale 0.02
        rotate 90*z
    }

    normal { BrickNormal1 }
}

#declare BrickTexture[1] =
texture
{
    pigment
    {
        marble        
        turbulence 1.4
        color_map
        {
            [0.2 color BrickColor1]
            [0.8 color BrickColor2]
        }
        scale 0.1
    }
    
    normal
    {
        average
        normal_map
        {
            [0.8 BrickNormal1]
            [0.2 BrickNormal2]
        }
    }
}

#declare BrickTexture[2] =
texture
{
    pigment
    {
        wrinkles
        //turbulence 1.4
        color_map
        {
            [0.33 color rgb BrickColor1]
            [0.66 color rgb BrickColor2]
        }
        scale 0.05
    }
    
    normal
    {
        average
        normal_map
        {
            [0.1 BrickNormal1]
            [0.1 BrickNormal2]
        }
    }
}

#declare BrickTexture[3] =
texture
{
    pigment
    {
        marble        
        turbulence 1.4
        color_map
        {
            [0.4 color rgb BrickColor1]
            [0.7 color rgb BrickColor2]
        }
        scale 0.06
    }
    
    normal
    {
        average
        normal_map
        {
            [0.8 BrickNormal1]
            [0.2 BrickNormal2]
        }
    }
}


#declare MatteTexture =
texture
{
    pigment { color rgb <0.8, 0.8, 0.7> }
    normal
    {
        wrinkles 0.5
        turbulence 0.6
        scale 0.01
    }
}

#macro CreateArchTexture (RndSeed)
    texture
    {
        pigment
        {
            granite
            turbulence 0.7
            color_map
            {
                [0.5 color rgb <0.90, 0.90, 0.76>]
                [0.8 color rgb <0.80, 0.71, 0.69>]
            }
            
            scale 0.3
            
            rotate 360 * VRand_In_Sphere (RndSeed)
            translate VRand (RndSeed)
        }
        
        normal
        {
            wrinkles 0.5
            turbulence 0.3
            scale 0.005
        }
    }
#end


// Wrapper for the CreateBrickedWall macro.  If UseArches is true,
// place a set of arches on the wall.  If CreateBottomArches is
// true, skip the creation of the wall and the arches and create
// only the bottom arches

#macro CreateWall (CreateBottomArches, Length, Height, UseArches, ArchSize, 
    ArchOffset, Thickness, RndSeed, IntersectFlag, Window, Texture)

    union
    {
        #if (! CreateBottomArches)
            #ifdef (USE_BRICKS)
            
                CreateBrickedWall(Length, Height, Thickness, 
                    MeanBrickWidth, MeanRowHeight, 
                    Texture, MatteTexture, RndSeed)
                
            #else
            
                box
                {
                    <0, 0, 0>, <Length, Height, Thickness> 
                    texture { Texture[0] } 
                }
                
            #end
        #end
                
        #ifdef (USE_ARCHES)
            #if (UseArches | CreateBottomArches)
                #local Count = 0;
                #while (Count * ArchSize * 0.95 + ArchOffset < Length)
    
                    #ifdef (USE_BLOBS)
                        #if (! CreateBottomArches)
                    
                            #declare Blob1 =
                            object
                            {
                                CreateBlobbedObj (40, RndSeed)
                                scale 0.09
                            }
                            
                            #declare Blob2 =
                            object
                            {
                                CreateBlobbedObj (40, RndSeed)
                                scale 0.09
                            }
                            
                            #declare Blob3 =
                            object
                            {
                                CreateBlobbedObj (30, RndSeed)
                                scale 0.05
                            }
                            
                        #end    
                    #end
                    
                    #if (CreateBottomArches)
                    union
                    #else
                        #ifdef (USE_BLOBS)
                        union
                        #else
                        object
                        #end
                    #end
                    {
                        #local Transf =
                        transform
                        {
                            scale ArchSize
                            translate <Count * ArchSize * 0.98 + ArchOffset, Height, 0>
                        }
                        
                        #if (! CreateBottomArches)
                            #if (IntersectFlag)
                            difference
                            {
                            #end
                                object
                                {
                                    CreateArch (Height / ArchSize, RndSeed)

                                    transform { Transf }
                                }
                            #if (IntersectFlag)
                                object { Window }
                                cutaway_textures
                            }
                            #end
    
                            #ifdef (USE_BLOBS)
                                union
                                {
                                    object
                                    {
                                        Blob1
                                        translate <-0.35, 1.35, 0>
                                    }
                                    
                                    object
                                    {
                                        Blob1
                                        translate < 0.35, 1.35, 0>
                                    }
            
                                    object
                                    {
                                        Blob2
                                        translate <-0.2, 1.65, 0>
                                    }
            
                                    object
                                    {
                                        Blob2
                                        translate < 0.2, 1.65, 0>
                                    }
                                    
                                    object
                                    {
                                        Blob3
                                        translate <-0.28, 1.51, 0>
                                    }
                                    
                                    object
                                    {
                                        Blob3
                                        translate <0.28, 1.51, 0>
                                    }
                                    
                                    object
                                    {
                                        Blob3
                                        translate <-0.12, 1.76, 0>
                                    }
                                    
                                    object
                                    {
                                        Blob3
                                        translate <0.12, 1.76, 0>
                                    }

                                    transform { Transf }
                                }
                            #end // USE_BLOBS
                        #end // ! CreateBottomArches
                        
                        #if (CreateBottomArches & Count != 13 & Count != 14)
                        object
                        {
                            CreateBottomArch (RndSeed)

                            transform { Transf }
                        }
                        #end
                            
                        CreateArchTexture (RndSeed)
                    }
                    
                    #declare Count = Count + 1;
                #end // #while
                

                // Metallic bar which passes through the arches
                
                cylinder
                {
                    <0, Height + 0.25, 0>, <Length, Height + 0.25, 0>,
                    ArchThickness*ArchSize/20
                    
                    texture
                    {
                        pigment
                        {
                            color rgb 0.2
                        }
    
                        finish
                        {
                            ambient 0
                            diffuse 0.2
                            
                            specular 1.3
                            reflection { 1 metallic }
                        }
                        
                    }
                }
            #end // UseArches <> 0
        #end // USE_ARCHES
    }
#end // CreateWall macro


// How much smaller is the inner Aisle with respect to the outer one?
#local SmallAisleRatio = 0.84/1.20;

#ifdef (USE_WALLS)
    
    object
    {
        CreateWall (false, 10, LowWallHeight, true, 0.4, 0.35, 
            WallThickness, seed (16), false, 0, BrickTexture)

        translate <-10, 0, -WallThickness/2>
        rotate -40*y
        translate <12, 0, -15>
    }
    
    union
    {
        object
        {
            CreateWall (false, 15 * SmallAisleRatio, HighWallHeight, true, 0.4, 1.8, 
                WallThickness, seed(12), false, 0, BrickTexture)

            rotate 90*y
            scale <-1, 1, 1>
            translate <8.4, 0, 0>
        }    
    
        object
        {
            CreateWall (false, 1, HighWallHeight, true, 0.4, 1.8, 
                WallThickness, seed(12), false, 0, BrickTexture)

            translate <8.4, 0, 0>
        }    

        #local Transf =
        transform
        {
            scale 0.84
            translate (0.8845 * 10 * SmallAisleRatio - 0.4) * x
                + (LowWallHeight + 0.4 * (HighWallHeight - LowWallHeight) - 0.3) * y
        }

        #local WindowCSGShape =        
        object
        {
            WindowShape
            transform { Transf }
        }
        
        #local WallObj =
        object
        {
            CreateWall (false, 10 * SmallAisleRatio, HighWallHeight, 
                true, 0.4, 0.50, WallThickness, seed (7), defined (USE_WINDOW),
                WindowCSGShape, BrickTexture)
        }
        
        #local WallArches =
        object
        {
            CreateWall (true, 10 * SmallAisleRatio, HighWallHeight, true, 0.4, 0.50, 
                WallThickness, seed (5), false, 0, BrickTexture)
        }
        
        #ifdef (USE_WINDOW)
        
            union
            {
                object { WallArches }

                object
                {
                    WindowFrame
                    transform
                    {
                        scale <1.0, 1.02, 0.1>
                        Transf
                        translate -0.04*y
                    }
                    
                    CreateArchTexture (seed(4))
                }


                // Glass

                object
                {
                    WindowShape
                    
                    scale <1, 1, 0.01>
                    transform Transf
                    
                    #local GlassColor = color rgb <0.15, 0.23, 0.20>/2;
                    texture
                    {
                        pigment { color GlassColor }
                        normal
                        {
                            bozo 0.033
                            scale 0.04
                        }
                        
                        finish
                        {
                            specular 0.6
                            roughness 0.075
                            reflection 
                            {
                                0.15
                                exponent 2
                            }
                        }
                    }
                    
                    texture
                    {
                        pigment
                        {
                            wrinkles
                            color_map
                            {
                                [0.42 color rgb 0 transmit 1]
                                [0.46 color rgb 0 transmit 0]
                                [0.50 color rgb 0 transmit 0.6]
                                [0.70 color rgb 0 transmit 0.6]
                                [0.74 color rgb 0 transmit 0]
                                [0.78 color rgb 0 transmit 1]
                            }
                        }
                        
                        scale 0.04
                    }
                }
                    
                object { WallObj }

        #else // USE_WINDOW
        
            union
            {
                object { WallObj }
                object { WallArches }
        
        #end // ! USE_WINDOW
        
            translate <-10 * SmallAisleRatio, 0, -WallThickness/2>
            rotate -40*y
            translate <8.4, 0, -15 * SmallAisleRatio>
        }
    }
#end // USE_WALLS


// ------------------------------------------------------

// Tranversal wall

#include "transv.inc"

// -------------------------------------------------------

// Columns

#include "columns.inc"

#ifdef (USE_COLUMNS)
    
    #local ColumnSize = 0.8 / sin(2*pi/6);
    #local ColumnSides = 6;
    
    object
    {
        CreateColumn (ColumnSize, ColumnSides, LowColumnHeight, 
            MeanRowHeight, BrickTexture, true, seed(13))
    
        rotate (40+360/6/2)*y
        translate <12, 0, -15>
    }
    
    object
    {
        CreateColumn (ColumnSize, ColumnSides, HighColumnHeight, 
            MeanRowHeight, BrickTexture, false, seed(14))
    
        rotate (40+360/6/2)*y
        translate <8.4, 0, -15 * SmallAisleRatio>
    }
    
    object
    {
        CreateColumn (ColumnSize, ColumnSides, HighColumnHeight, 
            MeanRowHeight, BrickTexture, true, seed(15))
    
        rotate 360/6/2*y
        translate <8.4, 0, -15 * SmallAisleRatio / 3>
    }

    object
    {
        CreateColumn (ColumnSize, ColumnSides, HighColumnHeight, 
            MeanRowHeight, BrickTexture, false, seed(16))
       
        rotate 360/6/2*y
        translate <8.4, 0, -15 * SmallAisleRatio * 2 / 3>
    }

#end // USE_COLUMNS

// -------------------------------------------------------

// Central dome

#ifdef (USE_DOME)

    #ifdef (USE_LADDER)
    
        #include "ladder.inc"
    
    #end
    
    #local UpperBrickColor1 = <0.94, 0.94, 0.76> * 1.3;
    #local UpperBrickColor2 = <0.98, 0.91, 0.85> * 1.3;
    
    #declare UpperBrickText = array[4]
    
    #declare UpperBrickText[0] =
    texture
    {
        pigment
        {
            marble        
            turbulence 0.4
            color_map
            {
                [0.05 color rgb <0.94, 0.94, 0.89> * 1.1]
                [0.08 color rgb <1.0, 0.95, 0.90> * 1.1]
                [0.5 color UpperBrickColor1]
                [0.9 color UpperBrickColor2]
            }
            scale 0.02
            rotate 90*z
        }
    
        normal { BrickNormal1 }
    }
    
    #declare UpperBrickText[1] =
    texture
    {
        pigment
        {
            marble        
            turbulence 1.4
            color_map
            {
                [0.2 color UpperBrickColor1]
                [0.8 color UpperBrickColor2]
            }
            scale 0.1
        }
        
        normal
        {
            average
            normal_map
            {
                [0.8 BrickNormal1]
                [0.2 BrickNormal2]
            }
        }
    }
    
    #declare UpperBrickText[2] =
    texture
    {
        pigment
        {
            wrinkles
            //turbulence 1.4
            color_map
            {
                [0.33 color rgb UpperBrickColor1]
                [0.66 color rgb UpperBrickColor2]
            }
            scale 0.05
        }
        
        normal
        {
            average
            normal_map
            {
                [0.1 BrickNormal1]
                [0.1 BrickNormal2]
            }
        }
    }
    
    #declare UpperBrickText[3] =
    texture
    {
        pigment
        {
            marble        
            turbulence 1.4
            color_map
            {
                [0.4 color rgb UpperBrickColor1]
                [0.7 color rgb UpperBrickColor2]
            }
            scale 0.06
        }
        
        normal
        {
            average
            normal_map
            {
                [0.8 BrickNormal1]
                [0.2 BrickNormal2]
            }
        }
    }

    #declare MeanBrickWidth = 0.175;
    #declare MeanRowHeight = 0.1375;
   
    union
    {
        object
        {    
            CreateWall (false, 5, 2, false, 0.4, 0.35, 
                WallThickness, seed (12), false, 0, UpperBrickText)
        }
    
        #ifdef (USE_LADDER)
            
            object
            {
                Ladder
                texture
                {
                    pigment { color rgb 0.7 }
                }
                scale 0.5
                translate <3.7, 0.2, -0.25>
            }
        
        #end
                
        translate <-5, 0, -WallThickness/2>
        rotate -40*y
        translate <5 + 1, HighWallHeight + 1, -5>
    }
    
    union
    {
        object
        {
            CreateWall (false, 5, 4, true, 0.3, 0.25
                WallThickness, seed (14), false, 0, UpperBrickText)
        }

        translate <-5, 0, -WallThickness/2>
        rotate -90*y
        translate <5 + 1, HighWallHeight + 1, -5 + 5>
    }

    object
    {
        CreateWall (false, 2.5, 4, true, 0.3, 0.25
            WallThickness, seed (6), false, 0, UpperBrickText)

        translate <5 + 1, HighWallHeight, -WallThickness/2>
    }

#end // USE_DOME

// -------------------------------------------------------

// Buildings behind the camera

#ifdef (USE_SHADOWS)
    
    box
    {
        <18, 0, -22>, <20, 11, -18.5>
        
        pigment { color rgb <0.7, 0.7, 0.65> }
    }
    
    box
    {
        <18, 0, -18.5>, <20, 11.5, -14>
        
        pigment { color rgb <0.7, 0.7, 0.65> }
    }
    
    
    // Columns (look at the shadows casted on the walls)
    
    box
    {
        <3, 0, -18>, <3, HighWallHeight + 5.2, -20>
        
        pigment { color rgb 0 }
    }
    
    
    union
    {
        box
        {
            0, <1, HighWallHeight + 2.5, 3>
            
            pigment { color rgb 0 }
        }

        julia_fractal
        {
            <-0.083,0.0,-0.83,-0.025>
            quaternion
            sqr
            max_iteration 8
            precision 15
            
            scale 1.5
            rotate 70*y
            rotate 36*x
            translate (HighWallHeight + 2.5)*y
        }
        
        translate <5, 0, -17.5>
    }    
    
    union
    {
        box
        {
            0, <1, HighWallHeight + 2.5, 1.5>
        }
        
        julia_fractal
        {
            <-0.083,0.0,-0.83,-0.025>
            quaternion
            sqr
            max_iteration 8
            precision 15
            
            scale 1.5
            translate (HighWallHeight + 2.5)*y
        }
    
        rotate 60*y
        translate <5, 0, -18.5>
    
        pigment { color rgb 0 }
    }

#end // USE_SHADOWS
