////////////////////////////////////////////////////////////////////////////////
//
// SpatialCrunch.pov
//
// Peter Duthie, 2003.
// rdx_irtc[at]warlordsofbeer.com
// http://art.warlordsofbeer.com/
//
////////////////////////////////////////////////////////////////////////////////

#include "glass.inc"
#include "stars.inc"
  

#declare DEF_HIRES_MAPS = on;
#declare DEF_MEDIA = on;

#declare DEF_RADIOSITY = off;
#declare DEF_RADIOSITY_PRESAVE = off;
#declare DEF_RADIOSITY_PRELOAD = off;

// Beginning and end of the segment of space station that we wish to display
#declare Var_ArcLimit_Begin	= 240;
#declare Var_ArcLimit_End	= 62;

// Sunlight
#declare Var_Sunlight = <1.0, 0.85, 0.60>;
 
global_settings {
#if (DEF_RADIOSITY)
	radiosity {
		media on
#if (DEF_RADIOSITY_PRELOAD)
		load_file "space1.rad"
		always_sample off
#end
#if (DEF_RADIOSITY_PRESAVE)
		save_file "space1.rad"
#end
	}
#end
  	max_trace_level 20
}

 
camera {
   location <100, -50, 35>
   look_at <0, -50, 0>
} 

////////////////////////////////////////////////////////////////////////////////
//
// Building Textures
//
////////////////////////////////////////////////////////////////////////////////

#declare Tex_Build_Wall = array[7];

// Black
#declare Tex_Build_Wall[0] = texture {
	pigment {
		rgb 0
	}
}

// White
#declare Tex_Build_Wall[1] = texture {
	pigment {
		rgb 1
	}
}

// Grey
#declare Tex_Build_Wall[2] = texture {
	pigment {
		rgb 0.5
	}
}

// Pale Grey
#declare Tex_Build_Wall[3] = texture {
	pigment {
		rgb 0.8
	}
}

// Light Blue
#declare Tex_Build_Wall[4] = texture {
	pigment {
		rgb <0.7,0.7,1> 
	}
}

// Beige
#declare Tex_Build_Wall[5] = texture {
	pigment {
		rgb <0.97, 0.86, 0.74>
	}
}

// Brown
#declare Tex_Build_Wall[6] = texture {
	pigment {
		rgb <0.46, 0.345, 0.22>
	}
}

#declare Tex_Build_Window = array[3];

// Black
#declare Tex_Build_Window[0] = texture {
	pigment {
		rgb 0
	}
	finish {
		reflection 0.5
	}
}

// Faint Blue
#declare Tex_Build_Window[1] = texture {
	pigment {
		rgb <0.3,0.3,0.35>
	}
	finish {
		reflection 0.5
	}
}

// Silver
#declare Tex_Build_Window[2] = texture {
	pigment {
		rgb 1
	}
	finish {
		reflection 1
	}
}



#declare Tex_Build_Dome = array[6];

// Black
#declare Tex_Build_Dome[0] = texture {
	pigment {
		rgb 0
	}
}

// Black Reflective
#declare Tex_Build_Dome[1] = texture {
	pigment {
		rgb 0
	}
	finish {
		reflection 0.5
	}
}

// Light Blue
#declare Tex_Build_Dome[2] = texture {
	pigment {
		rgb <0.7,0.7,1>
	}
}

// White
#declare Tex_Build_Dome[3] = texture {
	pigment {
		rgb 1
	}
}

// Silver
#declare Tex_Build_Dome[4] = texture {
	pigment {
		rgb 1
	}
	finish {
		reflection 1
	}
}

// Radial
#declare Tex_Build_Dome[5] = texture {
	radial
	texture_map {
		[0.0 pigment {rgb 0}]
		[0.5 pigment {rgb <1,1,0>}]
		[1.0 pigment {rgb 0}]
	}  
	frequency 20
}

#declare Tex_Metal_Chrome = texture {
	// Modified from textures.inc
	pigment {
		rgb <0.658824, 0.658824, 0.658824>
	}
	finish {
		diffuse 0.7
		reflection 0.15
		brilliance 8
		specular 0.8
		roughness 0.1
	}
}

////////////////////////////////////////////////////////////////////////////////
//
// Materials
//
////////////////////////////////////////////////////////////////////////////////

#declare Mat_Default = material {
	texture {
		pigment {
			color <1,1,1>
		}
	}
}

#declare Mat_Stars = material {
	texture {
		bozo
		texture_map {
			[0.00 pigment {rgb <0,0,0>}]
			[0.30 pigment {rgb <0,0,0>}]
			[0.60 Starfield1]
			[0.85 Starfield3]
			[1.00 Starfield3]
		}
		scale <1,1,5>
	}
}

#declare Mat_Space_Station_Shell = material {
	texture {
		pigment {
			//rgbf <0.97, 0.99, 0.98, 0.9>
			rgbf <1,1,1,1>
		}
		finish {
			specular 0.7
			roughness 0.001
			ambient 0
			diffuse 0
			reflection {
				0.2, 1.0
				fresnel on
			}
			//conserve_energy
		}
	}
	interior {
		I_Glass1
	}
}

#declare Mat_Space_Station_Ground = material {
	texture {
		pigment {
			rgb <1,1,1>
		}
		
		finish {
			reflection 0.15
		}
		normal {
			crackle metric 3 offset 0.5
			normal_map {
				[0.0 crackle 0.5 ]
				[0.03 crackle 0 ]
				[0.03 dents 0.2]
				[1.0 dents 0.2]
			}
		}
		scale 2
	}
}

#declare Mat_Space_Station_Center = material {
	texture {
		hexagon
		texture {
			pigment {
				color <0.9,0.9,0.9>
			}
		}
		texture {
			pigment {
				color <0.8,0.8,0.8>
			}
		}
		texture {
			pigment {
				color <1,1,1>
			}
		}
		rotate <90,0,0>
	}
}

#declare Mat_Elevator_Shaft = material {
	texture {
		pigment {
			rgbf <0.97, 0.99, 0.98, 0.9>
		}
		finish {
			specular 0.7
			roughness 0.001
			ambient 0
			diffuse 0
			reflection {
				0.2, 1.0
				fresnel on
			}
		}
	}
	interior {
		I_Glass1
	}
}

#declare Mat_Support_Beam = material {
	texture {
		Tex_Metal_Chrome
	}
}

#declare Mat_Space_Ship = material {
	texture {
		// Modified from textures.inc
		pigment {
			rgb <0.658824, 0.658824, 0.658824>
		}
		finish {
			diffuse 0.7
			reflection 0.15
			brilliance 8
			specular 0.8
			roughness 0.1
		}
	}
}

#declare Mat_Space_Station_LightHousing = material {
	texture {
		pigment {
			gradient y
			color_map {
				[0.00 rgb Var_Sunlight * 2]
				[0.05 rgb Var_Sunlight * 2]
				[0.05 rgb <0.5,0.5,0.5>]
				[0.95 rgb <0.5,0.5,0.5>]
				[0.95 rgb Var_Sunlight * 2]
				[1.00 rgb Var_Sunlight * 2]
			}
		}
		finish {
			specular 0.3
			roughness 0.001
			ambient 0.5
			diffuse 0.7
		}
	}
}

#declare Mat_Space_Station_Light = material {
	texture {
		pigment {
			spherical
			pigment_map {
				[0.0 rgbf <1,1,1,1>]
				[0.6 rgbf <1,1,1,0>]
				[0.6 rgbf <1,1,1,0> * 10]
				[1.0 rgbf <1,1,1,0> * 10]
			}
		}
	}
}
 

#declare MatRand = seed(2);
#macro Mat_Building_1()
	#local Wall = int(rand(MatRand * 0.999999) * 7);
	#local Window = int(rand(MatRand * 0.999999) * 3);
	material {
		texture {
			gradient y
			texture_map {
				[0.25 Tex_Build_Wall[Wall]]
				[0.25 Tex_Build_Window[Window]]
				[0.75 Tex_Build_Window[Window]]
				[0.75 Tex_Build_Wall[Wall]]
			}
		}
	}
#end

#macro Mat_Building_2()
	#local Dome = int(rand(MatRand * 0.999999) * 6);
	material {
		texture {
			Tex_Build_Dome[Dome]
		}
	}
#end

#macro Mat_Building_3()
	#local Wall = int(rand(MatRand * 0.999999) * 7);
	#local Window = int(rand(MatRand * 0.999999) * 3);
	material {
		texture {
			gradient y
			texture_map {
				[0.5 Tex_Build_Wall[Wall]]
				[0.5 Tex_Build_Window[Window]]
				[0.6 Tex_Build_Window[Window]]
				[0.6 Tex_Build_Wall[Wall]]
			}
		}
	}
#end

// Image maps from http://www.space-graphics.com/
#declare Mat_Earth = material {
	texture {
		pigment {
			image_map {
				#if(DEF_HIRES_MAPS)
					jpeg "maps/earth_pigment.jpg"
					interpolate 2
				#else
					jpeg "maps/earth_pigment_low.jpg"
				#end					
				map_type 1
			}
		}
		normal {
			bump_map {
				#if(DEF_HIRES_MAPS)
					jpeg "maps/earth_bump.jpg"
				#else
					jpeg "maps/earth_bump_low.jpg"
				#end
				map_type 1
				bump_size 4
			}
		}	
	}
}

#declare Mat_Earth_Atmosphere  = material {
	texture {
		pigment {
			rgb <1,1,1> transmit 1
		}
	}
#if (DEF_MEDIA)
	interior {
		media {
			scattering { 3, <1,1,1> * 0.00002 }
			
			density {
				spherical
				density_map {
					[0.0 rgb <0,0,0>]
					[0.010 rgb <0.3,0.3,1>]
					[0.015 rgb <0.9,0.9,1>]
				}
			}			
		}
	}
#end
}	


// Warp macro courtesy http://www.3dluvr.com/intercepto/povray/tutorials/expmedia/media01.htm
#macro warptexture(times,fal,strn,pos,rad)
   #local wcounter = 0; // set counter to 0
   #local maxc = times; // number of black holes
   #while (wcounter<maxc)
   warp {
      black_hole pos,rad
      falloff fal
      strength strn
      turbulence 0.4 // some turbulence for the black hole
   }
   #local wcounter = wcounter +1;
   #end
#end

   

#declare Mat_Sun_Gas = material {
	texture {
		pigment {
			rgb <1,1,1> transmit 1
		}
	}
#if (DEF_MEDIA)
	interior {
		media {
			emission Var_Sunlight * 0.00003 
			
			density {
				spherical
				density_map {
					[0 rgb 0]
					[ 0.10
						granite
						color_map {
							[ 0   color rgb 0.3 ]
							[ 0.5 color rgb 0 ]
							[ 1   color rgb 0 ]
						}
						warptexture(4,1,1,<0,0,0>,1)
						scale 2
					]
					[ 0.90
						granite
						color_map {
							[ 0   color rgb 1 ]
							[ 0.7 color rgb 0.5 ]
							[ 1   color rgb 0 ]
						}
						warptexture(4,1,1,<0,0,0>,1)
						scale 2
					]
					[0.95 rgb 15]
				}
			}			
		}
	}
#end
}

#declare Mat_Space_Ship_Blast = material {
	texture {
		pigment {
			rgb <1,1,1> transmit 1
		}
	}
#if (DEF_MEDIA)
	interior {
		media {
			emission 1
			density {
				spherical
				density_map {
					[0.0 rgb Var_Sunlight * 0.5]
					[0.5 rgb Var_Sunlight]
					[1.0 rgb 5]
				}
				turbulence 0.5
			}			
		}
	}
#end
}


////////////////////////////////////////////////////////////////////////////////
//
// Objects
//
////////////////////////////////////////////////////////////////////////////////

#declare Sun = object {
	union {
		sphere {
			<0,0,0>, 1
			hollow
			material {
				Mat_Sun_Gas
			}
			no_shadow
		}
		light_source {
			<0,0,0> color 7
		} 
	}
	scale 16000
}	

#declare Earth = object {
	union {
		sphere {
			<0,0,0>,1
			material {
				Mat_Earth
			}
		} 
#if (DEF_MEDIA)
		sphere {
			<0,0,0>,1.05
			hollow
			material {
				Mat_Earth_Atmosphere
			}
			scale 1.02
		}
#end
	}
	rotate <0,135,0>
	rotate <7, 0,0>
	scale 10000
}

#declare StarField = object {
	sphere {
		<0,0,0>, 10
		material {
			Mat_Stars
		}	
	}
	inverse
	scale 100000
}

#macro Space_Station_Shell(Rad_Outer, Rad_Inner, Thickness)
	// Main shell is a torus subtracted from a torus
	#local Minor_Radius_Outside = (Rad_Outer - Rad_Inner) / 2;
	#local Major_Radius_Outside = Rad_Outer - Minor_Radius_Outside;
	#local Minor_Radius_Inside = Minor_Radius_Outside - Thickness;
	difference {
		torus {
			Major_Radius_Outside, Minor_Radius_Outside
		}
		torus {
			Major_Radius_Outside, Minor_Radius_Inside
		}
		material {
			Mat_Space_Station_Shell
		}
		rotate <90,0,0>
	}
#end

#macro Space_Station_Ground(Rad_Edge_Outer, Rad_Edge_Inner, Rad_Ground)
	// Ground is a sphere subtracted from a torus
	#local Minor_Radius = (Rad_Edge_Outer - Rad_Edge_Inner) / 2;
	#local Major_Radius = Rad_Edge_Outer - Minor_Radius;
	difference {
		torus {
			Major_Radius, Minor_Radius
		}
		sphere {
			<0,0,0>, Rad_Ground
		}
		material {
			Mat_Space_Station_Ground
		}
		rotate <90,0,0>
	}
#end

#macro Space_Station_Lights(Center, Radius, Frequency)
	union {
		difference {
			torus {
				Center, Radius
			}
			torus {
				(Center + (Radius / 100)), (Radius * 1.001)
			}
			material {
				Mat_Space_Station_LightHousing
			}
			rotate <90,0,0>
		}    
		
		#local cnt = 0;
		#while (cnt < Frequency)
			#local Arc = ((360 / Frequency) * cnt);
			#if((Arc >= Var_ArcLimit_Begin) | (Arc <= Var_ArcLimit_End))
				difference {
					sphere {
						<0,0,0>, 1
					}
					box {
						<2,2,2>,<-2,0,-2>
						rotate <45,0,0>
					}
					box {
						<2,2,2>,<-2,0,-2>
						rotate <-45,0,0>
					}
					material {
						Mat_Space_Station_Light
					}
					scale <4,1,1>
					translate <0, -Center, 0>
					rotate <0, 0, Arc>
					no_shadow
				} 
				
					
				light_source {
					<0,0,0>, rgb <1,1,1> * 1.3
					spotlight
					point_at <0,-1,0>
					radius 60
					falloff 90
					fade_power 1
					fade_distance 10
					
					translate <0, -Center, 0>
					rotate <0, 0, Arc>
					media_interaction off
				}  
			#end
			#local cnt = cnt + 1;
		#end
	}	
#end

#macro Space_Station_Center(Radius, Length)
	union {
		#local Torus_Central_Radius = Radius * 0.6;
		#local Torus_External_Radius = Radius - Torus_Central_Radius;
		#local Torus_Start_Offset = (-Length / 2) + Torus_External_Radius;
		#local Torus_End_Offset = (Length / 2) - Torus_External_Radius;
		#local Torus_Steps = 7;
		#local Torus_Step_Length = (Torus_End_Offset - Torus_Start_Offset) / (Torus_Steps - 1);
		#local t_total = Torus_Steps;
		#local t_count = 0;
		#while (t_count < t_total)
			#local offset_size = Torus_Start_Offset + (t_count * Torus_Step_Length);
			torus {
				Torus_Central_Radius, Torus_External_Radius
				rotate <90,0,0>			
				translate <0,0,offset_size>
			}
			#local t_count = t_count + 1;
		#end
		cylinder {
			<0, 0, (-Length/2)>, <0, 0, (Length/2)>, Torus_Central_Radius
		}
		material {
			Mat_Space_Station_Center
		}
	}
#end

#macro Elevator_Shaft(Rad_Start, Rad_Finish, Rad_Outer, Thickness)
	// cylinder subtracted from a cylinder
	difference {
		cylinder {
			<0, Rad_Start, 0>, <0, Rad_Finish, 0>, Rad_Outer
		}
		cylinder {
			<0, Rad_Start * 0.9, 0>, <0, Rad_Finish * 1.1, 0>, Rad_Outer - Thickness
		}
		material {
			Mat_Elevator_Shaft
		}
	}
#end

#macro Elevator_Shafts(Rad_Start, Rad_Finish, Rad_Outer, Thickness, Number)
	#local cnt = 0;
	#while (cnt < Number)
		object {
			Elevator_Shaft(Rad_Start, Rad_Finish, Rad_Outer, Thickness)
			rotate <0, 0, ((360 / Number) * cnt)>
		}
		#local cnt = cnt + 1;
	#end
#end

#macro Support_Beam(Disp_Top, Disp_Bottom, Rad_Base, Width, Thickness, Members)
	// Longitude beam
	union {
		cylinder {
			<Disp_Bottom, Rad_Base, Width/2>,
			<Disp_Top, 0, Width/2>,
			Thickness
		}
		cylinder {
			<Disp_Bottom, Rad_Base, -Width/2>,
			<Disp_Top, 0, -Width/2>,
			Thickness
		}
		#local cnt = 0;
		#local step_x = (Disp_Top - Disp_Bottom) / Members;
		#local step_y = Rad_Base / Members;
		#while (cnt < Members)
			cylinder {
				<Disp_Top - (step_x * cnt), (step_y * cnt), -Width/2>,
				<Disp_Top - (step_x * (cnt + 1)), (step_y * (cnt + 1)), Width/2>,
				Thickness
			}
			cylinder {
				<Disp_Top - (step_x * cnt), (step_y * cnt), Width/2>,
				<Disp_Top - (step_x * (cnt + 1)), (step_y * (cnt + 1)), -Width/2>,
				Thickness
			}
			#local cnt = cnt + 1;
		#end
		material {
			Mat_Support_Beam
		}
	}
#end

#macro Support_Beams(Disp_Top, Disp_Bottom, Rad_Base, Width, Thickness, Members, Number)
	#local cnt = 0;
	#while (cnt < Number)
		object {
			union {
				object {
					Support_Beam(Disp_Top, Disp_Bottom, Rad_Base, Width, Thickness, Members)
				}
				object {
					Support_Beam(Disp_Top, Disp_Bottom, Rad_Base, Width, Thickness, Members)
					rotate <0,180,0>
				}
			}
			rotate <0, 0, ((360 / Number) * cnt)>
		}
		#local cnt = cnt + 1;
	#end
#end      

#macro BuildingPosition(Base_Radius, RotateX, RotateZ)
	scale 0.2
	translate <0,-Base_Radius,0>
	rotate <RotateX, 0, 0>
	rotate <0,0,RotateZ>
#end
 
#macro Building_1(Height, Scale, Base_Radius, RotateX, RotateZ)
	object {
		box {
			<-1,0,-1>, <1,1,1>
		}
		scale <1, Height, 1>
		scale Scale 
		Mat_Building_1()
		BuildingPosition(Base_Radius, RotateX, RotateZ)
	}	
#end 

#macro Building_2(Height, Scale, Base_Radius, RotateX, RotateZ)
	object {
		difference {
			sphere {
				<0,0,0>,1
			}
			box {
				<-1,-1,-1>,<1,0,1>
			}
			scale Scale
			Mat_Building_2()
			BuildingPosition(Base_Radius, RotateX, RotateZ)
		}	
	}
	
#end
#macro Building_3(Height, Scale, Base_Radius, RotateX, RotateZ)
	object {
 		cone {
			<0,0,0>, 1, <0,1,0>, 0
		}
		scale <1, Height, 1>
		scale Scale 
		Mat_Building_3()
		BuildingPosition(Base_Radius, RotateX, RotateZ)
	}	
#end
#macro Building_4(Height, Scale, Base_Radius, RotateX, RotateZ)
	object {
		cylinder {
			<0,0,0>, <0,1,0>, 1
		}
		scale <1, Height, 1>
		scale Scale 
		Mat_Building_1()
		BuildingPosition(Base_Radius, RotateX, RotateZ)
	}	
#end
#macro Building_5(Height, Scale, Base_Radius, RotateX, RotateZ)
	object {
		difference {
			torus {     
				0.80, 0.20
			}
			box {
				<-1,-1,-1>,<1,0,1>
			}
		}
		scale Scale 
		Mat_Building_2()
		BuildingPosition(Base_Radius, RotateX, RotateZ)
	}	
#end        

#macro Buildings(Number, BaseRadius, AngleX)
	#local cnt = 0;
	#declare BRand = seed(1);
	#while (cnt < Number)                    
		#local RotateZ = rand(BRand) * 360; 
		#if((RotateZ >= Var_ArcLimit_Begin) | (RotateZ <= Var_ArcLimit_End))
			#local RotateX = (rand(BRand) - 0.5) * AngleX; 
			#local Height = int(rand(BRand) * 10) + 1;
			#local Scale = (int(rand(BRand) * 3) + 1) * 2;
			#switch(int(rand(BRand) * 4.99999))
				#case (0)
					Building_1(Height, Scale, BaseRadius, RotateX, RotateZ)
				#break
				#case (1)
					Building_2(Height, Scale, BaseRadius, RotateX, RotateZ)
				#break
				#case (2)
					Building_3(Height, Scale, BaseRadius, RotateX, RotateZ)
				#break  
				#case (3)
					Building_4(Height, Scale, BaseRadius, RotateX, RotateZ)
				#break
				#case (4)
					Building_5(Height, Scale, BaseRadius, RotateX, RotateZ)
				#break 
			#end
		#end
		
		#local cnt = cnt + 1;
	#end
#end

#declare Space_Ship_Wing = difference {
	box {
		<0, 0, -0.01>, <4, 12, 0.01>
	}
	box {
		<0, 0, -0.02>, <4, 16, 0.02>
		rotate <0,0,18.43> 
		translate <4,0,0>
	}
	translate <3,4,0>
}


#macro Space_Ship(TransX, TransY, TransZ, RotX, RotY, RotZ)
	union {
		union {
			cylinder {
				<0,4,0>,<0,22,0>, 3
			}
			sphere {
				<0,0,0>,3
				scale <1,2,1>
				translate <0,22,0>
			}
			difference {
				sphere {
					<0,0,0>, 3
					scale <1,1.5,1>
				}
				box {
					<-3,-6,-3>,<3,0,3>
				}
			}
			
			
			difference {
				torus {     
					0.80, 0.20
				}
				box {
					<-1,-1,-1>,<1,0,1>
				}
			}
			object {
				Space_Ship_Wing
			}
			object {
				Space_Ship_Wing
				rotate <0,90,0>
			}
			object {
				Space_Ship_Wing
				rotate <0,180,0>
			}
			object {
				Space_Ship_Wing
				rotate <0,270,0>
			}
			material {
				Mat_Space_Ship
			}
		}
		object {
			difference {
				sphere {
					<0,0,0>, 1
					translate <0,-0.5,0>
				}
				box {
					<-1, 0, -1>, <1, 1, 1>
				}
				material {
					Mat_Space_Ship_Blast
				}
				hollow
				no_shadow
			}
			scale 2.8
			scale <1,8,1>
		}
		light_source {
			<2.5, -4, 2.5> 
			color <1.0, 0.95, 0.85> * 0.75
			fade_power 1
			fade_distance 40
			media_interaction off
		}
		light_source {
			<2.5, -4, -2.5> 
			color <1.0, 0.95, 0.85> * 0.75
			fade_power 1
			fade_distance 40
			media_interaction off
		}
		light_source {
			<-2.5, -4, -2.5> 
			color <1.0, 0.95, 0.85> * 0.75
			fade_power 1
			fade_distance 40
			media_interaction off
		}
		light_source {
			<-2.5, -4, 2.5> 
			color <1.0, 0.95, 0.85> * 0.75
			fade_power 1
			fade_distance 40
			media_interaction off
		}
		
		scale 0.3
		rotate <RotX, RotY, RotZ>
		translate <TransX, TransY, TransZ>
	}	
#end        


////////////////////////////////////////////////////////////////////////////////
//
// Main bit
//
////////////////////////////////////////////////////////////////////////////////


#declare Var_Space_Station_Outer_Radius = 100;
#declare Var_Space_Station_Inner_Radius = 75;
#declare Var_Space_Station_Thickness = 1;

#declare Var_Space_Station_Ground_Radius = 90;

#declare Var_Space_Station_LightHousing_Radius = 0.5;

#declare Var_Space_Station_Center_Radius = 10;
#declare Var_Space_Station_Center_Length = (Var_Space_Station_Outer_Radius - Var_Space_Station_Inner_Radius);

#declare Var_Elevator_Tube_Radius = 0.5;
#declare Var_Elevator_Tube_Width = 0.01;

#declare Var_Support_Radius = 0.1;
#declare Var_Support_Members = 10;

#declare Var_Space_Station_Spokes = 6;
#declare Var_Space_Station_Lights = 48;

#declare Var_Buildings = 5000;



union {
	object {
		Sun
		translate <-200000, 0, 5000>
	} 
	object {
		StarField
	}
     
	object {
		Earth
		translate <-5000,0,-15000>
	}

	light_group {	
		union {
			Space_Station_Shell(Var_Space_Station_Outer_Radius, Var_Space_Station_Inner_Radius, Var_Space_Station_Thickness)

			Space_Station_Ground(Var_Space_Station_Outer_Radius + 0.00001, Var_Space_Station_Inner_Radius + 0.00001, Var_Space_Station_Ground_Radius)
        
			Space_Station_Center(Var_Space_Station_Center_Radius , Var_Space_Station_Center_Length)

			Elevator_Shafts(Var_Space_Station_Center_Radius , Var_Space_Station_Ground_Radius, Var_Elevator_Tube_Radius, Var_Elevator_Tube_Width, Var_Space_Station_Spokes)
        
			Support_Beams(Var_Space_Station_Center_Radius, Var_Elevator_Tube_Radius, Var_Space_Station_Ground_Radius, Var_Space_Station_Center_Length * 0.75, Var_Support_Radius, Var_Support_Members, Var_Space_Station_Spokes)
	
			Buildings(Var_Buildings, Var_Space_Station_Ground_Radius, 12)
	
			Space_Station_Lights(Var_Space_Station_Inner_Radius + (Var_Space_Station_Thickness * 1.05), Var_Space_Station_LightHousing_Radius, Var_Space_Station_Lights)
		
			rotate <0,15, 0>
			translate <0,0,5>
		}
		
		Space_Ship(10,-70,55,15,0,0)
		global_lights on
	}	
}
