// Persistence of Vision Ray Tracer Scene Description File
// File: crop.pov
// Vers: 3.5
// Desc: A simple crop circle
// Date: 6/1/2003
// Auth: Tim Attwood
//

#version 3.5;
                       
                       
#include "colors.inc" 
#include "functions.inc"
#include "handy.inc" 

global_settings {
  assumed_gamma 1
}

// ----- scene set up -----

camera {
  location  <0.0, 1, -5.0>
  direction 1.5*z
  right     x*image_width/image_height
  look_at   <0.0, 0.0,  0.0>
  focal_point < 0, 0, 5>    // circle in focus
  aperture 0.1              
  blur_samples 4      
}

light_source { // artificial lighting upper left
  <0, 0, 0>            
  color rgb <1, 1, .8>*0.1  
  area_light <5, 0, 0>, <0, 0, 5>, 5, 5
  adaptive 1
  jitter
  translate <-30, 30, -30>
}

// ----- macros ----- 

// randomized grid placement
// S = seed, Num = number of objects in column, Num2 = number of objects in row,
// StartPos = first object location, 
// IncPos = increment vector column, IncPos2 = increment vector row
#macro GridOfRan(Obj StartPos IncPos IncPos2 Num Num2 S)
  #local r1 = seed(S); 
  #local _COUNT2 = 0;
  #local _CPOS2 = StartPos;
  #while (_COUNT2 < Num2)
    #local _COUNT = 0;
    #local _CPOS = _CPOS2;
    #while (_COUNT < Num)
      object {Obj        
        #local rot1 = rand(r1)*360;
        #local tra1 = rand(r1)*IncPos.x*2 - IncPos.x;
        #local tra2 = rand(r1)*IncPos2.z*2 - IncPos2.z;        
        rotate <0,rot1,0>
        #local tra3 = _CPOS + <tra1,0,tra2>;
        translate tra3}
      #local _CPOS = _CPOS + IncPos;
      #local _COUNT = _COUNT + 1;
    #end
    #local _CPOS2 = _CPOS2 + IncPos2;
    #local _COUNT2 = _COUNT2 + 1;
  #end
#end
 
// randomized line placement
// S = seed, Num = number of objects, StartPos = first object location, IncPos = increment vector
#macro LineOfRan(Obj StartPos IncPos Num S)
  #local r1 = seed(S); 
  #local _COUNT = 0;
  #local _CPOS = StartPos;
  #while (_COUNT < Num)
    #local rot1 = 90*int(rand(r1)*4);
    object {Obj
      rotate <0,rot1,0> 
      translate _CPOS}
    #local _CPOS = _CPOS + IncPos;
    #local _COUNT = _COUNT + 1;
  #end
#end  

// randomized radial placement
// S = seed, N = number of objects, dxM = major - minor radius, dxA = minor radius
#macro RadialRand(Obj S N dxM dxA)  
 #local r1 = seed(S); 
 #local C = 0;
 #while (C < N)
   #local C = C + 1;
   #local dx = rand(r1)*dxM + dxA;
   #local rot1 = rand(r1)*360;
   #local rot2 = rand(r1)*360;
   object { Obj 
     rotate <0,rot1,0>
     translate <dx,-1,0>
     rotate <0,rot2,0>
   }   
 #end 
#end

// ----- textures -----
#declare TEX_ground_stain = texture { //ground contact stain
  pigment {
    gradient y
    color_map {
      [0 Gray25]
      [0.03 rgbf <.4,.5,.1,0.5>]
      [0.1 Clear]
      [1 Clear]
    }
  } 
  scale <1,3,1> 
}
#declare TEX_woodgrain = texture { // woodgrain
  pigment {  
    marble           
    color_map {      
      [0.1 MediumWood]
      [0.5 DarkWood]
    }
    turbulence 0.5   
    rotate <0,90,0>
    scale <1/30,1/3,1/30>    // grain upward
  }
  finish {
    ambient 0.1          
    diffuse 0.6          
    brilliance 1.0       
    specular 0.5      
    roughness .05     
    conserve_energy   // more realistic
  } 
}

#declare TEX_stalk = texture { //upright stalks
  pigment { 
    gradient y 
    color_map {
      [0 Yellow] 
      [1 ForestGreen]
    } 
  } 
}
#declare TEX_wheat = texture {  // bent wheat
     pigment { 
       gradient x 
       color_map {
        [0 Yellow] 
        [1 ForestGreen] 
       } 
     }
     scale <0.85,1,1>
     translate <-0.1,0,0> 
}    
 
#declare TEX_ground = texture { // ground 
  pigment { 
    crackle solid
    color_map {
      [0   rgb <0.7,0.7,0.7>]
      [0.2 rgb <0.1,0.6,0.1>]      
      [0.8 rgb <0.7,0.7,0.4>]
      [1   rgb <0.0,0.7,0>] 
    }   
    scale 0.1
  }
}

#declare MAT_post = material { // fence posts 
  texture { TEX_woodgrain} // woodgrain
  texture { TEX_ground_stain} //ground contact stain
}   
#declare MAT_stalk = material { // upright stalk
  texture { TEX_stalk} 
  texture { TEX_ground_stain} //ground contact stain
}   

#declare MAT_wheat = material { // bent stalk
  texture { TEX_wheat} 
  texture { TEX_ground_stain} //ground contact stain
}   

// ----- declared objects ----

#declare Stalk = union { // straight stalk
  cone{ <0,.8,0>, 0.0125, <0,1,0>, 0}
  cylinder{ <0,0,0>,<0,0.8,0>, 0.0125 }
  material{MAT_stalk}
}

#declare bend_object = Stalk; 
#declare object_axis1 = <0,0,0>;
#declare object_axis2 = <0,1,0>;
#declare bend_angle = 15;
#declare TREE = object {  // curved upright stalk
  #include "bend.inc"
}

#declare wheat1a = union {  // bent wheat (one way)
  cylinder{ <0,0,0>,<0,0.1,0>, 0.0125 }
  union {
    cylinder{ <0,0,0>,<0,0.5,0>, 0.0125}
    cone{ <0,.5,0>, 0.0125, <0,0.7,0>, 0} 
    translate <-0.1,0,0>
    rotate <0,0,-75>
    translate <0.1,0.1,0>  
  }
  difference {
    torus { 0.1, 0.0125
      rotate <90,0,0> 
    } 
    box {<-1,-1,-1>,<1,0,1>}
    box {<-1,-1,-1>,<1,0,1> rotate <0,0,105>}
    translate <0.1,0.1,0>
  }         
  material{MAT_wheat}
  rotate <0,15,0>
} 
 
#declare wheat2a = object {wheat1a rotate <0,-210,0>}

//GridOf( TREE, <0,-1,0>, <0.15,0,0.15>, <-.15,0,.15>, 10, 10) 


//------- fencing
#declare fn_box2 = function(x,y,z) { max(abs(z)+abs(y)-0.1,
                                         abs(x*cos(radians(45)) + y*sin(radians(45)))
                                         +abs(-x*sin(radians(45)) + y*cos(radians(45)))-2 )};
#declare fn_E = function { fn_box2(x,y,z)-f_noise_generator(x*10,y*33,z*33,3)/33 };
#declare fn_F = function { fn_E(x,y,z)-f_noise_generator(x,y,z,2)/3 };
#declare fence_post = isosurface {
  function { fn_F(x,y,z) }          
  contained_by { box { -1, 1 } }  
  accuracy 0.001                     
  max_gradient 4                     
  rotate <0,0,90> 
  rotate <0,60,0>
  scale <1,1.5,1>     // resize
  translate <0,1.5,0> // place base on origin
}   
// apply texture here
#declare fence_post2 = object { 
   fence_post 
   material {MAT_post}
}                  

// ----- farmhouse
#declare farmhouse = height_field {
  png "farmhouse_hf.png"
  water_level 32/255
  texture {
    pigment {
      image_map { png "farmhouse.png" map_type 0 interpolate 2 once }
      rotate x*90 // lay X-Y image map down onto X-Z height field plane
    }
  }
  rotate <-90,0,0>
  scale <2.5902,1,1>
  scale 10
} 

// ----- spooky woods backdrop
#declare spooky = height_field {
  png "forest03_hf.png"
  water_level 32/255
  texture {
    pigment {
      image_map { png "forest02.png" map_type 0 interpolate 2 once }
      rotate x*90 // lay X-Y image map down onto X-Z height field plane  
      //Black
    }
  }
  rotate <-90,0,0>
  translate <-0.5,0,0>
  scale <2.56,1,0.01>
}

// ----- scene ----   
plane {  // the ground
  y, -1
  texture {TEX_ground}
}

// the fog
fog {
    distance 150
    color rgb<0.05, 0.1, 0.05>
}

// moon
sphere { <150,50,600>, 25 
  texture {
    pigment { rgb <0.8,0.8,1> } 
    finish {
      ambient 1
    }
  }
}

// farmhouse
object { farmhouse
  rotate y*(-10) 
  translate <-28,-1,80>
} 

// woods     
object { spooky
  scale <170,70,1>
  translate <0,-1,500>
}  

// fence posts
LineOfRan( fence_post2, <-25,-1,41.5>,<5,0,0>,10,1)

// field beyond circle
GridOfRan(TREE, <-20,-1,1.5>, <0.4,0,0>, <0,0,0.4>, 100, 100,1 )

RadialRand(TREE, 1, 75, 1, 0) //inner
RingOf(TREE, <-1,-1,0>, <0,3.75,0>, 96)

RadialRand(TREE, 1, 750, 3, 2) //outer 
RingOf(TREE, <2,-1,0>, <0,1.875,0>, 192)

// bent wheat
RingOf(wheat1a, <0,-1,1.1>, <0,3.71,0>,  97)  //  97 (# of stalks)
RingOf(wheat2a, <0,-1,1.2>, <0,3.36,0>, 107)  // 107
RingOf(wheat1a, <0,-1,1.3>, <0,3.05,0>, 118)  // 118
RingOf(wheat2a, <0,-1,1.4>, <0,2.79,0>, 129)  // 129
RingOf(wheat1a, <0,-1,1.5>, <0,2.57,0>, 140)  // 140
RingOf(wheat2a, <0,-1,1.6>, <0,2.4 ,0>, 150)  // 150
RingOf(wheat1a, <0,-1,1.7>, <0,2.25,0>, 160)  // 160
RingOf(wheat2a, <0,-1,1.8>, <0,2.1 ,0>, 171)  // 171
RingOf(wheat1a, <0,-1,1.9>, <0,1.98,0>, 181)  // 181
        