#!/usr/bin/python

#**************************************************************************
#
#    Stem.py  - classes Segment, Leaf and Stem - here is all the logic of 
#               the tree generating algorithm
#
#    Copyright (C) 2003  Wolfram Diestel
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
#    Send comments and bug fixes to diestel@steloj.de
#
#**************************************************************************/


from Vector import *
import sys

class ErrorNotYetImplemented(Exception):
	pass
	
class Segment:
      "A segment class, multiple segments form a stem"
      def __init__(self,tree,stem,index,pos,dir,rad1,rad2):
	  self.tree = tree
	  self.stem = stem
	  self.position=pos.copy()
	  self.direction=dir.copy()
	  self.index=index
	  self.rad1=rad1
	  self.rad2=rad2

  
      def position2(self):
	  return self.position+self.direction

      def povray(self):
	  indent = " "*(self.stem.level*2+4)

# FIXME: if direction is not 1*y, there is a gap between earth and tree base
# best would be to add roots to the trunk(?)	  
#	  print indent+\
#		   "difference { sphere { %s, %.5f } plane { %s, 0 translate %s} }\n" \
#		   	% (str(self.position),self.rad1-0.0001,\
#		   	   str(self.direction.cartesian()/(-self.direction.radius)),str(self.position))

	  # FIXME: if nCurveRes[0] > 10 this division into several
	  # cones should be extended over more then one segments?
	  if self.stem.level==0 and self.tree.Flare>0 and self.index==0:
	  	# draw segment as several cones
	  	pos1 = 0
		len = self.direction.radius
	  	for i in range(9,-1,-1):
			pos2 = len/2**i
			rad1 = self.stem.stem_radius(pos1)
			rad2 = self.stem.stem_radius(pos2)
			print indent+\
			"cone   { %s, %.5f, %s, %.5f }" % \
				(str(self.position+self.direction*(pos1/len)),rad1,\
				 str(self.position+self.direction*(pos2/len)),rad2)
			pos1 = pos2
	  else:
		print indent+\
			"cone   { %s, %.5f, %s, %.5f }" % \
				(str(self.position),self.rad1,str(self.position2()),self.rad2)
	  if self.rad2 > 0: 
	     print indent+\
		   "sphere { %s, %.5f }\n" % (str(self.position2()),self.rad2-0.0001)


class Leaf:
      "A class for the leaves of the tree"
      
      def __init__(self,tree,par,level,pos,dir,offset=0):
	  self.position = pos.copy()  # the base position
	  self.direction = dir.copy() # the base direction
	  self.tree = tree     # the tree
	  self.parent = par    # the parent stem
	  self.level = level   # which branch level
	  self.offset = offset # how far from the parent's base
	  # get a vector orthogonal to the layer build by self.direction and parent.direction
	  self.normal = self.direction.normal(self.parent.direction) 
	  # the leaf's normal is the normal of the layer build by this vector and the
	  # leaf's stem direction
	  self.normal = self.normal.normal(self.direction)
	  self.shape = self.tree.LeafShape
	  (self.length,self.width) = self.leaf_dimension()
	  # point direction to the middle of the leaf 
	  # FIXME: stem len is half of leaf len here - should this be variable?
	  # FIXME: stem_radius seams not to be added, why?
	  #print self.parent.stem_radius(self.offset)
	  self.direction.radius = self.parent.stem_radius(self.offset)+self.length
	  self.leaf_orient()

	  	  
      def rotatez(self,angle):
          self.direction.rotatez(angle)
	  self.normal.rotatez(angle)
	  
      def rotatey(self,angle):
          self.direction.rotatey(angle)
	  self.normal.rotatey(angle)
      
      def leaf_orient(self):
	  "leaf rotation toward light"
	  if self.tree.bend==0: return
	  
	  # rotation outside 
	  tpos = self.position.spherical().phi.degree #atan2(self.position.y,self.position.x)
	  tbend=(tpos-self.normal.phi.degree)%360 #tpos-atan2(norm.y,norm.x)
	  if tbend>180: tbend=360-tbend
	  
	  bend_angle = Angle(self.tree.bend*tbend)
	  self.rotatez(bend_angle)
	  
	  # rotation up
	  norm = self.normal.cartesian()
	  fbend=atan2(sqrt(norm.x**2+norm.y**2),norm.z)
	  
	  orientation = Angle(self.normal.phi)
	  bend_angle = Angle(self.tree.bend*fbend)
	  
	  self.rotatez(-orientation)
	  self.rotatey(bend_angle)
	  self.rotatez(orientation)

	        
      def leaf_dimension(self):
         "return the length and width of a leaf"
	 return (self.tree.LeafScale/sqrt(self.tree.quality),\
	 	self.tree.LeafScale*self.tree.LeafScaleX/sqrt(self.tree.quality))
		
      def make(self):
         "makes the leaf shape"
	 # FIXME: add code here if necessary
	 pass
		
      def povray(self):
	  "prints povray code for the leaf"
	  indent = " "*(self.parent.level*2+4)
	  # FIXME: only shape 0 (disc) supported yet
	  mid = self.position+self.direction # what about connecting stem len?
	  # FIXME: the disc must be oval, so draw the disc first, scale
	  # it, turn it, that its normal is paralel with self.normal and then
	  # translate it to the point mid
#	  print indent+\
#		"disc   { ",mid,", ",\
#		self.normal.cartesian(),", "+\
#		str(round(self.length,5))+" }\n"
	  
#	  print indent+\
#	  	"cylinder { ",pos,",",mid,",",self.parent.base_radius/2,"}"
#	  print "/*dir:",dir,"norm:",self.normal,"*/"

#	  print indent+\
#		"union {disc   { <0,0,0>, <0,1,0>, 0.5 } disc {<0,0.0001,0>,<0,1,0>,0.5 pigment {color rgb <0,1,0>}}"+\
	  print indent+\
	  	"disc {<0,0,0>,<0,1,0>,0.5 pigment {color rgb <0,1,0>} "+\
	  	"scale <%.5f,1.0,%.5f> %s}" % (self.length,self.width,\
	  		self.direction.transformation_matrix_povray(self.normal,mid))

      def dump(self):
		indent = " "*(self.parent.level*2+4)
		print indent+"LEAF:"
		print indent+"position: ",self.position
		print indent+"direction: ",self.direction
		print indent+"offset: ",self.offset
		print indent+"normal: ",self.normal
		   
class Stem:
      "A helper class for making 3d trees, this class makes a stem"+\
      " (trunk or branch)"

      def __init__(self,tree,par,level,pos,dir,offset=0):
	  self.position = pos  # the base postion
	  self.direction = dir # the base direction
	  self.tree = tree     # the tree
	  self.parent = par    # the parent stem
	  self.level = level   # which branch level
	  self.offset = offset # how far from the parent's base
	  self.segments = []   # the segments forming the stem
	  self.substems = []   # the substems
	  self.leaves = []     # the leaves
	  self.leaves_per_segment = 0 
	  self.split_cnt = 0
	  self.split_corr = () #Direction(0,0,0)	  
	  self.start_segment=0 # only informational
	  self.debug = self.tree.debug
	  self.verbose = self.tree.verbose
      
      def DBG(self,str):
          if self.debug: sys.stderr.write(str)

      def clone(self,pos,dir,start_segm):
	  clone = Stem(self.tree,self.parent,self.level,\
		pos.copy(),dir.copy(),self.offset)
	  clone.segment_len = self.segment_len
	  clone.start_segment = start_segm
	  clone.segment_cnt = self.segment_cnt
	  clone.length = self.length
	  clone.base_radius = self.base_radius
	  clone.length_child_max = self.length_child_max
	  clone.substem_cnt = self.substem_cnt
	  clone.substems_per_segment = self.substems_per_segment
	  clone.seg_splits = self.seg_splits
	  clone.split_corr = self.split_corr[:] # copy
	  # FIXME: for more then one clone this angle should somehow
	  # correspond to the rotation angle of the clone
	  clone.substem_rotangle=self.substem_rotangle+180
	  clone.leaves_per_segment=self.leaves_per_segment
	  return clone

      def var(self,variation):
	  return self.tree.random.uniform(-variation,variation)

      def make(self):
	  self.segment_cnt = self.tree.nCurveRes[self.level]
	  self.length = self.stem_length()
	  self.base_radius = self.stem_base_radius()
          self.prepare_substem_params()      
	  self.make_segments(0,self.segment_cnt)

      def stem_length(self):
	  if self.level == 0:
	     return (self.tree.nLength[0]\
		      + self.var(self.tree.nLengthV[0])) \
		    * ( self.tree.Scale\
		      + self.var(self.tree.ScaleV) ) 

	  elif self.level == 1:
	     parlen = self.parent.length
	     baselen = self.tree.BaseSize*parlen
	     ratio  = (parlen-self.offset)/(parlen-baselen)
	     return parlen * self.parent.length_child_max \
		    * self.tree.shape_ratio(ratio)
          else:
	     parlen = self.parent.length
	     return self.parent.length_child_max*(parlen-0.6*self.offset)
  
      def make_segments(self,start_seg,end_seg):
          "makes the segments of the stem"
	  if self.verbose: 
	  	if self.level==0: sys.stderr.write("=")
	  	elif self.level==1 and start_seg==0: sys.stderr.write("/")
		elif self.level==2 and start_seg==0: sys.stderr.write(".")
      
	  pos = self.position.copy()
	  dir = self.direction.copy()
	  self.segment_len = float(self.length)/self.tree.nCurveRes[self.level]
	  dir.radius = self.segment_len
	  for s in range(start_seg,end_seg):
	      if self.verbose:
	          if self.level==0: sys.stderr.write("|")
	  
	      # curving
	      dir=self.new_direction(dir,s)
	      
	      # segment radius
	      rad1 = self.stem_radius(s*dir.radius)
	      rad2 = self.stem_radius((s+1)*dir.radius)
	      segment = Segment(self.tree,self,s,pos,dir,rad1,rad2)
	      self.segments.append(segment)
	      
	      #print "spos: "+self.position.str()+" sdir: "+self.direction.str()
	      #print " pos: "+pos.str()+" dir: "+dir.str()

	      # create substems
	      if self.level<self.tree.Levels-1:
		 self.make_substems(segment)
		 
	      if self.level==self.tree.Levels-1 and self.tree.Leaves>0:
	         if self.debug: sys.stderr.write("MAKING LEAVES...\n")
	         self.make_leaves(segment)
	
	      # shift to next position
	      pos=pos+dir

	      # new direction (curving)
	      # weiter oben. dir=self.new_direction(dir)

	      # splitting (create clones)
	      if s<end_seg-1: 
		 dir = self.make_clones(pos,dir,s) # dir is changed by make_clones


      def new_direction(self,dir,nsegm): 
          # next segments  direction
	  
	  # I think the first segment should't get another
	  # direction because there is yet a variation from down and rotation angle
	  # otherwise the first trunk segment should get a random rotation(?)
	  if nsegm == 0 and self.level>0: return dir
	  
	  # regular curving 
	  self.DBG("CURVING: dir: %s," % (dir))
	  
	  if self.tree.nCurveBack==0:
	  	dir.addtheta(self.tree.nCurve[self.level]\
	  	    	/ self.tree.nCurveRes[self.level])
	  	    	# + self.var(self.tree.nCurveV[self.level]))\
	  else:
	  	if nsegm<(self.tree.nCurveRes[self.level]+1)/2:
	  		dir.addtheta(self.tree.nCurve[self.level]*2.0\
	  			/self.tree.nCurveRes[self.level])
		else: dir.addtheta(self.tree.nCurveBack[self.level]*2.0\
	  			/self.tree.nCurveRes[self.level])
	  
	  self.DBG("=> %s\n" % (dir))

	  # add random rotation
	  if nsegm==0 and self.level==0: # first_trunk_segment
		# random rotation more moderate
		delta = Angle(\
		  	(abs(self.var(self.tree.nCurveV[self.level]))-abs(self.var(self.tree.nCurveV[self.level])))\
				/self.tree.nCurveRes[self.level])
	  else:
	  	# full random rotation
	  	delta = Angle(self.var(self.tree.nCurveV[self.level])/self.tree.nCurveRes[self.level])
		self.DBG("curvV (delta): %s\n" % str(delta))
	  rho = Angle(180+self.var(180))

	  dir.rotate_away_and_about(delta,rho)
	  
	  self.DBG("trunkdir: %s\n" % str(dir))
	  	
          # compensate earlier splitting angle
#	  if self.debug: sys.stderr.write("APPLY_SPLIT_CORE: %s\n" % self.split_corr)
#	  r = dir.radius
#	  dir = dir+self.split_corr
#	  dir.radius = r
#          dir.addtheta(self.split_corr.theta)
#          dir.addphi(self.split_corr.phi)
	  for sc in self.split_corr:
	  	self.DBG("SPLITCORR: dir: %s, axis: %s, angle: %s =>" % (dir,sc[0],sc[1]))
	  	dir = dir.cartesian().rotateaxis(sc[1],sc[0]).spherical()
		self.DBG(str(dir)+"\n")
	  
	  # attraction up/down
	  if self.tree.AttractionUp != 0 and self.level>=2:
	  	decl = dir.theta.degree*pi/180
	  	orient = abs(dir.theta.degree-90)*pi/180
	  	self.DBG("ATTRAC decl: %f, orient %f, cos(orient): %f\n" % (decl,orient,cos(orient)))
	  	curve_up = self.tree.AttractionUp * abs(decl * cos(orient) \
	  		/ self.tree.nCurveRes[self.level])
	  	self.DBG("ATTRAC curve_up: %f\n" % (curve_up))
	  	dir.addtheta(-curve_up*180/pi)
	  
	  return dir
	  
      def stem_base_radius(self):
          if self.level == 0: # trunk
	     # radius at the base of the stem
	     return self.length * self.tree.Ratio \
			 * (self.tree.nScale[0] \
			   + self.var(self.tree.nScaleV[0]))
	  else:
	     return self.parent.base_radius * \
	     		(self.length/self.parent.length)**self.tree.RatioPower

      # gets the stem width at a given position within the stem
      def stem_radius(self,h):
	  Z = h/self.length

	  if self.tree.nTaper[self.level] <= 1:
	     unit_taper = self.tree.nTaper[self.level]
 	  else:
	     raise ErrorNotYetImplemented,"nTaper>1 not implemented yet"

	     
	  radius = self.base_radius * (1 - unit_taper * Z)
	  
	  if self.level==0:
	     # add flaring (thicker stem base)
	     if self.tree.Flare > 0:
	     	flare = self.tree.Flare * (100 ** (1-8*Z) -1) / 100 +1
	     	radius = radius*flare
	  else:
             # max radius is the radius of the parent at offset
	     max_radius = self.parent.stem_radius(self.offset)
	     radius = min(radius,max_radius)
		
	  return radius		
	

      def prepare_substem_params(self):
	  level = self.level+1
	  if level>3: level=3
	  
	  # splitting ratio
	  self.seg_splits = self.tree.nSegSplits[self.level]

	  # maximum length of a substem
	  self.length_child_max = self.tree.nLength[level]\
				+self.var(self.tree.nLengthV[level])
				
	  # maximum number of substems
	  stems_max = self.tree.nBranches[level]
	  # actual number of substems and substems per segment
          if level==1:
             self.substem_cnt = stems_max
             self.substems_per_segment = float(self.substem_cnt) \
                 / self.segment_cnt / (1-self.tree.BaseSize)
          elif level==2:
             self.substem_cnt = stems_max * \
	           (0.2 + 0.8*(self.length/self.parent.length)\
		   / self.parent.length_child_max)
             self.substems_per_segment = self.substem_cnt \
		   / self.segment_cnt
             self.DBG("SS-PRP-%d: smax: %f, len: %f, parlen: %f, lenchildmax: %f\n" % (self.level,\
	  	stems_max,self.length,self.parent.length,self.parent.length_child_max))
          else:
             self.substem_cnt = stems_max * \
		   (1.0 - 0.5 * self.offset/self.parent.length)
             self.substems_per_segment = self.substem_cnt \
                   / self.segment_cnt
     
          self.DBG("SS-PRP-%d: sscnt: %f, ss/segm: %f\n" % (self.level,\
	  	self.substem_cnt,self.substems_per_segment))
		   
          self.substem_rotangle = 0
	  
	  if self.level == self.tree.Levels-1:
	     self.leaves_per_segment = self.leaves_per_branch() / self.segment_cnt

      def leaves_per_branch(self):
        "calcs the number of leaves for a stem"
	if self.tree.Leaves==0: return 0
	else: return self.tree.Leaves * self.tree.shape_ratio(self.offset/self.parent.length,4)\
		* self.tree.quality	  
	  
      def make_substems(self,segment):
	  "creates substems_per_segment substems for the current segment"
	  # FIXME: the first segments of the trunk up to
	  #  BaseSize*length shouldn't have substems
	  #  the first segment could have less, when it's
	  #  lower part is free of substems because of BaseSize>0
	  
	  if self.level>0:
	  	# full length of stem can have substems
	  	subst_per_segm = self.substems_per_segment
	  	if segment.index==0:
	  		offs = self.parent.stem_radius(self.offset)/self.segment_len
		else: offs = 0
	  elif segment.index*self.segment_len > self.tree.BaseSize*self.length:
	  	# segment is fully out of the bare trunk region => normal nb of substems
	  	subst_per_segm = self.substems_per_segment
	  	offs = 0
	  elif (segment.index+1)*self.segment_len <= self.tree.BaseSize*self.length:
	  	# segment is fully part of the bare trunk region => no substems
	  	return
	  else:
	  	# segment has substems in the upper part only
		offs = (self.tree.BaseSize*self.length - segment.index*self.segment_len)/self.segment_len
		if self.debug: 
			sys.stderr.write("base: %f seg: %d seglen: %f offs: %f\n" %\
				(self.tree.BaseSize*self.length,segment.index*self.segment_len,\
				self.segment_len,offs))
		subst_per_segm = self.substems_per_segment*(1-offs)
	  
	  # how many substems in this segment
	  substems_eff = int(subst_per_segm+self.tree.\
			 substemErrorValue[self.level]+0.5)
			 
          # adapt error value
	  self.tree.substemErrorValue[self.level] = \
	      self.tree.substemErrorValue[self.level] \
	      - substems_eff + subst_per_segm;
	  
	  self.DBG("SS-%d/%d: subst/segm: %f, subst_eff: %f, err: %f\n" % (self.level,segment.index,\
	  	subst_per_segm,substems_eff,self.tree.substemErrorValue[self.level]))
	  
	  if self.debug: sys.stderr.write("level: %d, segm: %d, substems: %d\n" % \
	  	(self.level,segment.index,substems_eff))
	  
	  if substems_eff <= 0: return
	  
	  # what distance between the segements substems
	  dist = (1.0-offs)/substems_eff
	  
	  for s in range(substems_eff):
	      # where on the segment add the substem
	      where = offs+dist/2+s*dist+self.var(dist/2)
	      # offset from stembase
	      offset = (segment.index+where)*self.segment_len
	      # get a new direction for the substem
	      dir = segment.direction.copy()
	      dir.rotate_away_and_about(self.downangle(offset),self.rotation())
	      substem = Stem(self.tree,self,self.level+1,\
		      # position of substem
		      segment.position+(segment.direction*where),\
		      dir,offset)
	      substem.make()
	      self.substems.append(substem)
	      
      def make_leaves(self,segment):
	  "creates leaves for the current segment"
	  
	  #if self.debug: sys.stderr.write("make_leaves"+"-"*15+"\n")
	  
	  # how many leaves in this segment
	  leaves_eff = int(self.leaves_per_segment+self.tree.\
			 leavesErrorValue[self.level]+0.5)
	  
          # adapt error value
	  self.tree.leavesErrorValue[self.level] = \
	      self.tree.leavesErrorValue[self.level] \
	      - leaves_eff + self.leaves_per_segment;
	  
	  if leaves_eff <= 0: return
	  
	  if segment.index==0:
	  	offs = self.parent.stem_radius(self.offset)/self.segment_len
	  	if self.debug: sys.stderr.write("leaves-offs: %f\n" % offs)
	  else:
	  	offs = 0
	  
	  # what distance between the leaves
	  dist = (1.0-offs)/leaves_eff
	  
	  for s in range(leaves_eff):
	      # where on the segment add the leaf
	      #if self.debug: sys.stderr.write("dist: %f, s: %d, wmid: %f" % (dist,s,offs+dist/2+s*dist)
	      where = offs+dist/2+s*dist+self.var(dist/2)
	      
	      #if self.debug: print "w:",where
	      #if self.debug: print "sdir:",segment.direction,"lpos:",segment.position+(segment.direction*where)
	      
	      # offset from stembase
	      offset = (segment.index+where)*self.segment_len
	      # get a new direction for the leaf
	      dir = segment.direction.copy()
	      dir.rotate_away_and_about(self.downangle(offset),self.rotation())
	      leaf = Leaf(self.tree,self,self.level+1,\
		      # position of leaf
		      segment.position+(segment.direction*where),\
		      dir,offset)
	      leaf.make()
	      self.leaves.append(leaf)
	      
      def downangle(self,offset):
          level = min(self.level+1,3)
          if self.tree.nDownAngleV[level]>=0:
		#self.DBG("DOWNANGLE: %s\n" % str(self.tree.nDownAngle[level]+self.var(self.tree.nDownAngleV[level])))
		#self.DBG("DOWN: %d %f %f\n" % (level,self.tree.nDownAngle[level],self.var(self.tree.nDownAngleV[level])))
	  	return Angle(self.tree.nDownAngle[level]+self.var(self.tree.nDownAngleV[level]))
	  else:
	  	if level==1: len = self.length*(1-self.tree.BaseSize)
	  	else: len = self.length
		#self.DBG("DOWNANGLE: %s\n" % str(self.tree.nDownAngle[level]+(\
	  	#	self.var(self.tree.nDownAngleV[level])*(1 - 2 *\
	  	#		self.tree.shape_ratio(self.length-offset/len,0)))))
		#self.DBG("DOWNA: %d %f %f\n" % (level,self.tree.nDownAngle[level],self.var(self.tree.nDownAngleV[level])))
	  	#return Angle(self.tree.nDownAngle[level]+(\
	  	#	self.var(self.tree.nDownAngleV[level])*(1 - 2 *\
	  	#		self.tree.shape_ratio((self.length-offset)/len,0))))
	  	return Angle(self.tree.nDownAngle[level]+(\
	  		  self.tree.nDownAngleV[level]*(1 - 2 *\
	  			self.tree.shape_ratio((self.length-offset)/len,0))))
	  
      def rotation(self):
          level = min(self.level+1,3)
          self.substem_rotangle = Angle(self.substem_rotangle \
	  	+ self.tree.nRotate[level]+self.var(self.tree.nRotateV[level]))
	  #if self.debug: print "rotangle:",self.substem_rotangle
          return self.substem_rotangle
	      
      # splitting
      def make_clones(self,pos,dir,nseg):

          if self.tree.nBaseSplits[0]>0 and self.level==0 and nseg==0:
	  	seg_splits_eff = self.tree.nBaseSplits[0]
	  else:
	  	# how many clones?
	  	#seg_splits = self.tree.nSegSplits[self.level]
	  	seg_splits_eff = int(self.seg_splits+self.tree.\
	  			 splitErrorValue[self.level]+0.5)
	  
          	# adapt error value
	  	self.tree.splitErrorValue[self.level] = \
	  		self.tree.splitErrorValue[self.level] \
	  		- seg_splits_eff + self.seg_splits;

			
	  if seg_splits_eff<1: return dir
	  
	  # print "cloning... height: "+str(nseg)+" split_eff: "+str(seg_splits_eff)   
	  # print "  error_val: "+ str(self.tree.splitErrorValue[self.level])
	  self.DBG("CLONING: seg_splits_eff: %f\n" % (seg_splits_eff))
	  
	  s_angle = 180/(seg_splits_eff+1)
	  
	  # make clones
          # if seg_splits_eff > 0:
          for i in range(0,seg_splits_eff): 

             # copy params
	     clone = self.clone(pos,dir,nseg+1)
          
	     # NOTE: its a little bit problematic here
	     # when the clone is given as a parent to
	     # the substems, it should have the same
	     # params for length and segment_cnt like
	     # the original stem, but this could be
	     # somewhat confusing(?)
	     # clone.segment_cnt = remaining_segs;
	     # clone.length = remaining_segs * self.segment_len

	     # change the direction for the clone
	     if self.debug: sys.stderr.write("-SPLIT_CORE_BEFOR: %s, dir: %s\n" % \
	     	(str(clone.split_corr),str(clone.direction)))
	     clone.direction = clone.split(clone.direction,s_angle*(1+2*i),nseg)
	     if self.debug: sys.stderr.write("-SPLIT_CORE_AFTER: %s, dir: %s\n" % \
	     	(str(clone.split_corr),str(clone.direction)))
       
	     # make segments etc. for the clone
	     clone.make_segments(nseg+1,clone.segment_cnt)
	     # FIXME: should clones and substems be in separated lists?
	     self.substems.append(clone);
	  
	  # got another direction for the original stem too   
	  dir = self.split(dir,-s_angle,nseg)
	  return dir


      # applies a split angle to the stem
      def split(self,direction,s_angle,nseg):
	    remaining_seg = self.segment_cnt-nseg-1
	    dir = direction.copy()
	    
    	    # the splitangle
	    # FIXME: don't know if it should be nSplitAngle or nSplitAngle/2
            split_angle = Angle(self.tree.nSplitAngle[self.level]) # /2 ?
	    dir.rotate_away_and_about(split_angle,Angle(s_angle))

	    # in the Weber/Penn paper they add an rotation
	    # split_diverge = 20 + 0.75 * (30 + abs(dir.theta-90)) * rand(1)**2
            # split_diverge = - split_diverge if (rand(1)>=0.5)
            # but this split_diverge for the trunk makes the tree a little
	    # bit out of equilibry, 
	    # I used nSplitV instead for the random rotation, so
	    # you can still make trees without random setting nSplitV to 0

	    # add some random rotation
	    if self.tree.nSplitAngleV[self.level] != 0:
	    	delta = Angle(self.var(self.tree.nSplitAngleV[self.level])) # /2 ?
	    	rho = Angle(180+self.var(180))
	    	dir.rotate_away_and_about(delta,rho)
	    	    
            # the splitting angle is applied in reverse direction
            # to the following segments
	    # FIXME: what if only one segment follows?
	    # does this hide the split?
	    
	    # calc split correction
	    axis = direction.normal(dir).cartesian()
	    angle = Angle(acos(direction.cartesian().cos2(dir.cartesian()))*180/pi/remaining_seg)
	    self.DBG("NEWSPLITCORR: direc: %s, dir: %s, axis: %s, angle: %s\n" % (direction,dir,axis,angle))
            self.split_corr = ((axis,angle),)+self.split_corr
		
	    # adjust other parameters	
	    self.split_cnt = self.split_cnt+1
	    # lower substem prospensity (FIXME: should be /n for n-1 clones?)
	    self.substems_per_segment = self.substems_per_segment/2.0
	    # Don't know if this is correct 
	    ### self.seg_splits = self.seg_splits/2
	    
	    # FIXME same reduction for leaves_per_segment?
	    
	    return dir
		 
      def povray(self):
	  indent = " "*(self.level*2+2)
	  if len(self.segments)>1 or len(self.substems)>0 or len(self.leaves)>0: union=1
	  else: union=0
	  if union: print indent+"union { /* level "+str(self.level)+" */"
	  for s in self.segments:
	      s.povray()
	  for ss in self.substems:
	      ss.povray()
	  for l in self.leaves:
	      l.povray()
	  if union: print indent+"}"

      def count_real_substems(self,follow_clones=0):
          "returns number of real (with level+1) substems of th stem and all its clones"
	  sum = 0
	  for ss in self.substems:
	  	if ss.level > self.level: sum = sum+1
		elif follow_clones: sum = sum+ss.count_real_substems()
	  return sum
	  	  
      def dump(self):
	  indent = " "*(self.level*2+2)
	  print indent+"STEM:"
	  print indent+"debug:",self.debug
	  print indent+"level:",self.level
	  print indent+"position:",self.position
	  print indent+"direction:",self.direction
	  print indent+"offset:",self.offset
	  print indent+"length:",self.length
	  print indent+"base_radius:",self.base_radius
	  if self.level>0:
	  	print indent+"parent_radius:",self.parent.stem_radius(self.offset)
	  #...
	  print indent+"substems (own/clones/all): %d/%d/%d" %\
	  	(self.count_real_substems(),len(self.substems)-self.count_real_substems(),\
		self.count_real_substems(1))
	  print indent+"leaves: ",len(self.leaves)
	  for ss in self.substems:
	  	ss.dump()
	  for l in self.leaves:
	  	l.dump()
      

	      














