#!/usr/bin/python

#**************************************************************************
#
#    Stem.py  - classes Segment, Leaf and Stem - here is all the logic of 
#               the tree generating algorithm
#
#    Copyright (C) 2003  Wolfram Diestel
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
#    Send comments and bug fixes to diestel@steloj.de
#
#**************************************************************************/

from math import *
from types import *

class Location:
      "A cartesain vector class"
      def __init__(self,x=0,y=0,z=0):
	  self.x=x
	  self.y=y
	  self.z=z

      def copy(self):
          return Location(self.x,self.y,self.z)

      def __neg__(self):
          return self*-1  
	  
      def __add__(self,vector):
          "adds a vector of class Location or Direction"
	  if vector.__class__ == Location :     
	  	return Location(self.x+vector.x,self.y+vector.y,self.z+vector.z)
	  elif vector.__class__ == Direction:
	  	return self+vector.cartesian()
	  else:
	  	raise TypeError,"cannot add object of type "+str(vector.__class__)+" to a vector"
		
      def __sub__(self,value):
          return self+(-value)

      def __mul__(self,factor):
          "returns scalar product when multiplied with a vector of class Location, scales \
	  the vector, when multiplied with a number"
	  if type(factor) == IntType or type(factor) == FloatType:
	  	# scale vector
	  	return Location(factor*self.x,factor*self.y,factor*self.z)
	  elif factor.__class__ == Location:
	  	# scalar product
	  	return factor.x*self.x+factor.y*self.y+factor.z*self.z
	  else:
	  	raise TypeError,"cannot multiply object of type "+str(factor.__class__)+" with a vector"
      
      def __div__(self,value):
          if type(value) == IntType or type(value) == FloatType:
          	return self*(1.0/value)
	  else:
	  	raise TypeError,"cannot divide vector by an object of type "+str(factor.__class__)
	  
      def __abs__(self):
          "returns the length of the vector"
	  return sqrt(self.x**2+self.y**2+self.z**2)
	  

      def cos2(self,vec):
	"returns the direction cosinus of the angle (self,vec)"
	return (self*vec)/abs(self)/abs(vec)
      
      def normal(self,vec):
	   "returns the normal of the layer buildt by the vectors self and vec"
           # n= (a1*b2 - a2*b1, a2*b0 - a0*b2, a0*b1 - a1*b0)
	   n = Location(\
            	self.y*vec.z-self.z*vec.y,\
	    	self.z*vec.x-self.x*vec.z,\
	    	self.x*vec.y-self.y*vec.x)   
	   return n*(1/abs(n))
	   
      def rotatex(self,angle):
          "rotates the vector about the x-axis"
          self.y, self.z = \
	  	angle.cos()*self.y - angle.sin()*self.z,\
	  	angle.sin()*self.y + angle.cos()*self.z
	  return self

      def rotatey(self,angle):
          "rotates the vector about the y-axis"
          self.x, self.z = \
	  	angle.cos()*self.x - angle.sin()*self.z,\
		angle.sin()*self.x + angle.cos()*self.z
	  return self

      def rotatez(self,angle):
          "rotates the vector about the z-axis"
          self.x, self.y = \
	  	angle.cos()*self.x - angle.sin()*self.y,\
		angle.sin()*self.x + angle.cos()*self.y
	  return self
	  
      def rotateaxis(self,angle,axis):
          "rotates the vector about axis"
	  axis=axis/abs(axis)
	  (a,b,c) = (axis.x,axis.y,axis.z)
#	  print "a:",a,"b:",b,"c:",c
	  (si,co) = (angle.sin(), angle.cos())
#	  print "sin:",si,"cos:",co
	  x = self.x*(co+a*a*(1-co)) + self.y*(c*si+a*b*(1-co)) + self.z*(-b*si+a*c*(1-co))
	  y = self.x*(-c*si+b*a*(1-co)) + self.y*(co+b*b*(1-co)) + self.z*(a*si+b*c*(1-co))
	  z = self.x*(b*si+c*a*(1-co)) + self.y*(-a*si+c*b*(1-co)) + self.z*(co+c*c*(1-co))
#	  print "x:",x,"y:",y,"z:",z
	  return Location(x,y,z)
	  
#      def rotateaxis_matrix
	  
      def spherical(self):
          "return a Direction object in spherical coordinates"
	  r = sqrt(self.x**2+self.y**2+self.z**2)
	  if abs(self.x)<0.0000001 and abs(self.y)<0.0000001: 
	  	phi = 0
		if self.z>0: theta=0 
		else: theta=pi
	  else:
	  	theta = acos(max(min(self.z/r,1),-1))
	  	phi = acos(max(min(self.x/r/sin(theta),1),-1))
	  	if self.y<0: phi = 2*pi-phi
	  return Direction(theta*180/pi,phi*180/pi,r)
	  
      def __str__(self): # output as a (left handed system) povray vector
	  return "<"+str(round(self.x,4))+","\
		    +str(round(self.z,4))+","\
		    +str(round(self.y,4))+">"

class Angle:
      def __init__(self,degree):
          if type(degree) == InstanceType and degree.__class__ == Angle:
	  	degree = degree.degree
	  self.degree = degree%360
	  
      def rad(self):
          return self.degree*pi/180
	  
      def __neg__(self):
          return Angle(-self.degree)
	  
      def __add__(self,value):
          if type(value) == IntType or type(value) == FloatType:
	  	return Angle((self.degree+value)%360)
	  elif value.__class__ == Angle:
	  	return Angle((self.degree+value.degree)%360)
		
      def __sub__(self,value):
          return self+(-value)

      def cos(self):
          return cos(self.rad())
	  
      def sin(self):
          return sin(self.rad())
			  
      def __str__(self):
	  return str(round(self.degree,5))

class Direction:
      "A spherical vector class"
      def __init__(self,theta,phi,radius):
	  self.theta=Angle(theta)
	  self.phi=Angle(phi)
	  self.radius=radius

      def copy(self):
	  return Direction(self.theta.degree,self.phi.degree,self.radius)

      def addtheta(self,theta):
          if type(theta) == IntType or type(theta) == FloatType:
	  	deg = theta
	  else: deg = theta.degree
	  x = self.theta.degree + deg
	  if x>=0 and x<=180:
	     self.theta = self.theta + deg
	  elif x>180:
	     self.theta = self.theta - deg
	     self.phi = self.phi + 180
	  else: # x < 0
	     self.theta = -self.theta + deg
	     self.phi = self.phi + 180

      def addphi(self,phi):
	  self.phi = self.phi+phi
	  
      def __add__(self,vec):
          return (self.cartesian()+vec).spherical()
	  
      def __sub__(self,vec):
          return (self.cartesian()-vec).spherical()
	  
      def __mul__(self,value):
          if type(value) == IntType or type(value) == FloatType:
	  	# scale vector
	  	return Direction(self.theta,self.phi,self.radius*value)
		
      def  __div__(self,value):
          return self*(1.0/value)
		
      def __neg__(self):
          return self*(-1)
	  
      def cartesian(self):
	  return Location(\
	  	self.theta.sin()*self.phi.cos(),\
	  	self.theta.sin()*self.phi.sin(),
		self.theta.cos())*self.radius

      def rotatex(self,angle):
          newdir = self.cartesian().rotatex(angle).spherical()
	  self.theta.degree=newdir.theta.degree
	  self.phi.degree=newdir.phi.degree
		 
      def rotatey(self,angle):
          newdir = self.cartesian().rotatey(angle).spherical()
	  self.theta.degree=newdir.theta.degree
	  self.phi.degree=newdir.phi.degree
      
      def rotatez(self,angle):
          newdir = self.cartesian().rotatez(angle).spherical()
	  self.theta.degree=newdir.theta.degree
	  self.phi.degree=newdir.phi.degree
	  
      def normal(self,dir):
          "returns the normal of the layer buildt by the vectors self and dir"
	  return self.cartesian().normal(dir.cartesian()).spherical()
	  
      def transformation_matrix(self,norm,pos):
          "returns a transformation matrix for the system self,norm,normal(self,norm)"
	  x = self.cartesian() # dir
	  y = -self.cartesian().normal(norm.cartesian()) # to side
	  z = norm.cartesian() # up
	  u = Location(1,0,0) # leaf prototyp points in x-direction
	  v = Location(0,1,0) # to side
	  w = Location(0,0,1) # up
	  #print "/*"
	  #print "x:",x,"y:",y,"z:",z
	  #print "ux:",round(u.cos2(x),2),"uy:",round(u.cos2(y),2),"uz:",round(u.cos2(z),2)
	  #print "vx:",round(v.cos2(x),2),"vy:",round(v.cos2(y),2),"vz:",round(v.cos2(z),2)
	  #print "wx:",round(w.cos2(x),2),"wy:",round(w.cos2(y),2),"wz:",round(w.cos2(z),2)
	  #print "*/"
	  return ((u.cos2(x),v.cos2(x),w.cos2(x)),\
		(u.cos2(y),v.cos2(y),w.cos2(y)),\
		(u.cos2(z),v.cos2(z),w.cos2(z)),\
		(pos.x,pos.y,pos.z))
		

      def transformation_matrix_povray(self,norm,pos):
	  "returns the transformation matrix in the form <.,.,.,.,....>"
	  m = self.transformation_matrix(norm,pos)
	  # y and z exchanged for left hand povray system	  
	  s = "matrix <"
	  for i in (0,2,1,3):
	     for k in (0,2,1):
	     	s = s+str(round(m[i][k],5))
		if i!=3 or k!=1: s = s+','
		else: s = s+'>'
	     s = s+" "
	  return s

      def rotate_away_and_about(self,delta,rho):
          "rotates away (delta) from the direction-axis and then (rho) about it"
	  phi = Angle(self.phi)
	  theta = Angle(self.theta)
	  
	  # rotate dir so, that it's parallel to the z-axis
	  self.rotatez(-phi)
	  self.rotatey(theta)
	  
	  # rotate delta*y + rho*z
	  self.rotatey(-delta)
	  self.rotatez(rho)
          
	  # rotate back into original system
	  self.rotatey(-theta)
	  self.rotatez(phi)
	  
      def rotate_away_and_about_matrix(self,delta,rho):
	  phi = Angle(self.phi)
	  theta = Angle(self.theta)
          	
	  sys = (Location(1,0,0),Location(0,1,0),Location(0,0,1))
	  for vec in sys:
	  	# rotate dir so, that it's parallel to the z-axis
	  	vec.rotatez(-phi)
	  	vec.rotatey(theta)
	  	# rotate delta*y + rho*z
	  	vec.rotatey(-delta)
	  	vec.rotatez(rho)
          	# rotate back into original system
	  	vec.rotatey(-theta)
	  	vec.rotatez(phi)
	  (u,v,w) = (Location(1,0,0),Location(0,1,0),Location(0,0,1))
	  (x,y,z) = sys
	  return ((u.cos2(x),v.cos2(x),w.cos2(x)),\
		(u.cos2(y),v.cos2(y),w.cos2(y)),\
		(u.cos2(z),v.cos2(z),w.cos2(z)))
#	  return ((u.cos2(x),u.cos2(y),u.cos2(z)),\
#		(v.cos2(x),v.cos2(y),v.cos2(z)),\
#		(w.cos2(x),w.cos2(y),w.cos2(z)))
		
      def apply_transform_matrix(self,A):
          "rotates the vector using the tranformation matrix A"
	  vec = [0,0,0]
	  v = self.cartesian()
	  print "v:",v
	  for i in (0,1,2):
	  	vec[i] = vector_mul(A[i],(v.x,v.y,v.z))
	  print "vec:",vec
	  return Location(vec[0],vec[1],vec[2]).spherical()    
		    
      def __str__(self):
	  return "<"+str(self.theta)+","+str(self.phi)+","\
		 +str(round(self.radius,5))+">"
		 
def matrix_mul(A,B):
	"returns the matrix product of to transformation matrices, t.e. the transformation matrix \
	for the to tranformations B and A one applied after the other"
	AB = [[0,0,0],[0,0,0],[0,0,0]]
	for i in (0,1,2):
		for j in (0,1,2):
			AB[i][j] = vector_mul(A[i],matrix_column(B,j))
	return AB	
		
def matrix_column(A,i):
	"returns the i'th column of A as a tuple - used in matrix_mul"
	return (A[0][i],A[1][i],A[2][i])
	
def vector_mul(a,b):
	"returns the scalar vector product - used for matrix_mul"
	return a[0]*b[0]+a[1]*b[1]+a[2]*b[2]
	

def atan2(v,u):
	"returns the angle of a 2-dimensional vector (u,v) with the u-axis (-180..180)"
	if u==0:
		if v>=0: return 90
		else: return -90
	elif u>0:
		return atan(v/u)*180/pi
	elif v>=0:
		return 180+atan(v/u)*180/pi
	else: return atan(v/u)*180/pi-180




