//////////////////////////////////////////////////////////////////////////
// Door.pov  Copyright(c) 2003, K.E. Ayers Designs                      //
//////////////////////////////////////////////////////////////////////////
// A somewhat whimsical scene that depicts what looks like a door into  //
// a normal house.  There is a front porch and steps leading up to it,  //
// and the door is flanked by two narrow windows.                       //
//                                                                      //
// A seemingly normal entry way.                                        //
//                                                                      //
// However, the door and porch appear to be floating in space!  And,    //
// the door is partially open, revealing a glimpse of what appears to   //
// be a bright sunny day in the backyard!                               //
//////////////////////////////////////////////////////////////////////////


//////////////////////////////////////////////////////////////////////////
// The animation clock is used to control various rendering options:    //
//                                                                      //
//  CLOCK VALUE     OPTION                                              //
//  ==============  =================================================== //
//  0 <= K <= 1     Normal scene rendering                              //
//                                                                      //
//  +Kxxxxx         Dumps scene help information and aborts rendering.  //
//                                                                      //
//  +K1xxxxx        Use the "test" camera setup, in the section named:  //
//                      TEST CAMERA SETUP                               //
//                                                                      //
//  +K11xxxx        Use the "closeup" test camera setup.  The closeup   //
//                  settings are defined in the section named:          //
//                      CLOSEUP TEST CAMERA SETUP                       //
//                  otherwise, the "regular" test setup is used from:   //
//                      NORMAL TEST CAMERA SETUP                        //
//                                                                      //
//  +K1x1xxx        Eliminates the dimpled glass in the entry way       //
//                  windows.                                            //
//  +K1xx1xx        Suppresses the shrubbery around the front porch.    //
//                                                                      //
//  +K1xxx1x        Makes the scene's sky transparent.                  //
//                                                                      //
//  +K1xxxx1        Leaves the front door closed.                       //
//////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////
// Set the default scene rendering options.                             //

// Set to false to render test protions of the complete scene.
#declare RenderWholeScene = true;
// Set to true if there are two spotlights illuminating the door.
#declare TwoSpotlights = false;
// Set to true when the door is open.
#declare DoorIsOpen = true;
// Set to false to use a test camera position.
#declare UseStandardCamera = true;
// Set to true, when using the test camera, to zoom in on the door.
#declare UseCloseupView = false;
// Set to false to disable the dimpled glass side panels
// ... speeds up rendering while developing the scene.
#declare UseDimpledGlass = true;
// Set to true to include plants around the porch.
#declare IncludePorchFoliage = true;
// Set to true to make the scene's sky transparent.
#declare UseTransparentSky = false;
//////////////////////////////////////////////////////////////////////////


//////////////////////////////////////////////////////////////////////////
// Decode the clock to change the scene rendering options.              //
#local CLK = clock;

#if (CLK > 1)
    #if (CLK < 100000)
        // Print help on using the scene rendering options.
        #debug "\n\n==== HELP ON SCENE RENDERING OPTIONS ==============================\n"
        #debug "\nThe animation clock is used to control various rendering options:\n\n"
        #debug "CLOCK VALUE     OPTION\n"
        #debug "==============  ===================================================\n"
        #debug "0 <= K <= 1     Normal scene rendering\n"
        #debug "+Kxxxxx         Dumps scene help information and aborts rendering.\n"
        #debug "+K1xxxxx        Use the \"test\" camera setup, in the section named:\n"
        #debug "                    TEST CAMERA SETUP\n"
        #debug "+K11xxxx        Use the \"closeup\" test camera setup.  The closeup\n"
        #debug "                settings are defined in the section named:\n"
        #debug "                    CLOSEUP TEST CAMERA SETUP\n"
        #debug "                otherwise, the \"regular\" test setup is used from:\n"
        #debug "                    NORMAL TEST CAMERA SETUP\n"
        #debug "+K1x1xxx        Eliminates the dimpled glass in the entry way windows.\n"
        #debug "+K1xx1xx        Suppresses the shrubbery around the front porch.\n"
        #debug "+K1xxx1x        Makes the scene's sky transparent.\n"
        #debug "+K1xxxx1        Leaves the front door closed.\n"
        #debug "===================================================================\n"
        #debug "\n"
        #debug "\n"
        #debug "\n"
        #debug "\n"
        #error "\n"
    #else
        #debug "Using the test camera setup.\n"
        #declare UseStandardCamera = false;
        #local CLK = CLK - 100000;

        #if (CLK >= 10000)
            #debug "Using the closeup camera setting.\n"
            #declare UseCloseupView = true;
            #local CLK = CLK - 10000;
        #end

        #if (CLK >= 1000)
            #debug "Eliminate dimpled glass.\n"
            #declare UseDimpledGlass = false;
            #local CLK = CLK - 1000;
        #end

        #if (CLK >= 100)
            #debug "Suppress the shrubbery around the front porch.\n"
            #declare IncludePorchFoliage = false;
            #local CLK = CLK - 100;
        #end

        #if (CLK >= 10)
            #debug "Make the scene's sky transparent.\n"
            #declare UseTransparentSky = true;
            #local CLK = CLK - 10;
        #end

        #if (CLK >= 1)
            #debug "Close the front door!\n"
            #declare DoorIsOpen = false;
        #end
    #end
#end

//////////////////////////////////////////////////////////////////////////


// This is how far the door will be opened for the scene.

#if (DoorIsOpen)
    #declare DoorOpenAngle = -85;
#else
    #declare DoorOpenAngle = 0;
#end


//////////////////////////////////////////////////////////////////////////
// Include the definitions for the scene's components.                  //
//////////////////////////////////////////////////////////////////////////

#include "Entry.inc"
#include "Wall.inc"
#include "Porch.inc"
#include "Sky.inc"
#include "Scene.inc"


//////////////////////////////////////////////////////////////////////////
// Place the scene's components.                                        //
//////////////////////////////////////////////////////////////////////////

#if (RenderWholeScene)
    object { EntryWay }
    object { Wall }
    object { Porch }
    object { Sky }
    object { Scene }
#else
    //////////////////////////////////////////////////////
    // Objects for rendering the test scene go here.    //
    //////////////////////////////////////////////////////
    object { Scene }
    object { Floor }
    object { DoorFrame }
#end


//////////////////////////////////////////////////////////////////////////
// Place the camera.                                                    //
//////////////////////////////////////////////////////////////////////////

#if (UseStandardCamera)
    //////////////////////////////////////////////////////
    // Define the standard view's camera settings here. //
    //////////////////////////////////////////////////////
    #declare EyeX    =  DoorXOfs;
    #declare EyeY    =  DoorYOfs + DoorHgt * 0.5;
    #declare EyeZ    = -DoorHgt * 2.0;
    #declare LookAtX =  DoorXOfs;
    #declare LookAtY =  DoorYOfs + DoorHgt * 0.4;
    #declare LookAtZ =  DoorZOfs;
#else
    //////////////////////////////////////////////////////
    // TEST CAMERA SETUP                                //
    //////////////////////////////////////////////////////
    #if (UseCloseupView)
        //////////////////////////////////////////////////
        // CLOSEUP TEST CAMERA SETUP                    //
        //////////////////////////////////////////////////

        /*
        // Zoom-in on the door latch.
        #declare EyeX    =  DoorXOfs + LatchXOfs;
        #declare EyeY    =  DoorYOfs + LatchYOfs;
        #declare EyeZ    =  DoorZOfs - 10;
        #declare LookAtX =  DoorXOfs + LatchXOfs;
        #declare LookAtY =  DoorYOfs + LatchYOfs;
        #declare LookAtZ =  DoorZOfs;
        */
        /*
        // Zoom-in on the door knocker.
        #declare EyeX    =  DoorXOfs + KnockerXOfs;
        #declare EyeY    =  DoorYOfs + KnockerYOfs;
        #declare EyeZ    =  DoorZOfs - 10;
        #declare LookAtX =  DoorXOfs + KnockerXOfs;
        #declare LookAtY =  DoorYOfs + KnockerYOfs;
        #declare LookAtZ =  DoorZOfs;
        // Zoom-in on the welcome mat.
        #declare EyeX    =  DoorXOfs;
        #declare EyeY    =  DoorYOfs + DoorHgt * 0.25;
        #declare EyeZ    = -DoorHgt * 0.6;
        #declare LookAtX =  PorchCenterX;
        #declare LookAtY =  PorchTop;
        #declare LookAtZ =  PorchCenterZ;
        // Zoom-in on the front porch.
        #declare EyeX    =  DoorXOfs;
        #declare EyeY    =  DoorYOfs + DoorHgt * 1.0;
        #declare EyeZ    =  PorchFront - 54;
        #declare LookAtX =  PorchCenterX;
        #declare LookAtY =  PorchTop;
        #declare LookAtZ =  PorchCenterZ;
        // Zoom-in on the address above the front door.
        #declare EyeX    =  DoorXOfs;
        #declare EyeY    =  DoorYOfs + DoorHgt + WallHgtAboveDoor * 0.5;
        #declare EyeZ    =  PorchFront - 24;
        #declare LookAtX =  DoorXOfs;
        #declare LookAtY =  EyeY;
        #declare LookAtZ =  0;
        // Zoom-in on the right-side window.
        #declare EyeX    =  FrameRight  + WindowXOfs * 0.75;
        #declare EyeY    =  FrameBottom + WindowYOfs * 1.0;
        #declare EyeZ    =  PorchFront - 36;
        #declare LookAtX =  EyeX;
        #declare LookAtY =  EyeY;
        #declare LookAtZ =  0;
        */
        // Zoom-in on the scene beyond the door.
        #declare EyeX    =  DoorXOfs;
        #declare EyeY    =  DoorYOfs + DoorHgt * 0.5;
        #declare EyeZ    =  DoorZOfs + DoorWid;
        #declare LookAtX =  DoorXOfs;
        #declare LookAtY =  DoorYOfs + DoorHgt * 0.4;
        #declare LookAtZ =  EyeZ + 1200;
    #else
        //////////////////////////////////////////////////
        // NORMAL TEST CAMERA SETUP                     //
        //////////////////////////////////////////////////

        // View the entire scene from above and to the right.
        #declare EyeX    =  DoorXOfs * 4.5;
        #declare EyeY    =  DoorYOfs + DoorHgt * 2.0;
        #declare EyeZ    = -DoorHgt * 3.0;
        #declare LookAtX =  DoorXOfs * 3.5;
        #declare LookAtY =  DoorYOfs + DoorHgt * 0.75;
        #declare LookAtZ =  DoorZOfs;
    #end
#end

camera {
    location < EyeX, EyeY, EyeZ >
    look_at  < LookAtX, LookAtY, LookAtZ >
}


//////////////////////////////////////////////////////////////////////////
// Place the lighting.                                                  //
//////////////////////////////////////////////////////////////////////////

#declare LightColor     =  color rgb < 1.0,  0.95, 0.85 >;

#declare LightXOfs      =  DoorWid * 0.5;
#declare LightYOfs      =  WallHgt;
#declare LightZOfs      = -DoorHgt * 0.75;

#declare SpotXOfs       =  0.0;
#declare SpotYOfs       =  DoorHgt * 0.667;
#declare SpotZOfs       = -DoorThick;

#declare SpotRad        = 70;
#declare SpotFalloff    = 50;

light_source {
    < DoorXOfs,
      DoorYOfs + LightYOfs,
      DoorZOfs + LightZOfs >
    color LightColor
    spotlight
    point_at < DoorXOfs + SpotXOfs,
               DoorYOfs + SpotYOfs,
               DoorZOfs + SpotZOfs >
    radius  SpotRad
    falloff SpotFalloff
}

